/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file command.c
 * \brief Interactive command-line interpreter 
 * \author Peter Corke and Pavan Sikka
 */

#include 	<stdio.h>
#include 	<stdlib.h>
#include 	<string.h>

#include "rtx/command.h"
#include "rtx/defines.h"
#include "rtx/error.h"
#include "rtx/thread.h"
#include "config.h"

#ifdef	HAVE_LIBREADLINE
#  include <readline/readline.h>
#  include <readline/history.h>
#endif

static char rcsid[] RTX_UNUSED = "$Id: command.c 2846 2008-03-31 04:07:22Z roy029 $";


#define	MAX_ARG_LENGTH 80
#define	MAX_NUM_OF_ARGS 10

static void
menu_help(RtxCommandMenu *menu)
{
        int i;
 
        printf("Options:\n");
        for (i=0; menu[i].name != NULL; i++)
                printf("%s\t%s\n", menu[i].name, menu[i].help);
	printf("help/?\tthis help display\n");
        printf("quit\texit menu\n");
        printf("exit\texit menu\n");
}
 
#ifdef HAVE_LIBREADLINE
/*
 * Reset the terminal if readline dies while waiting, 
 * if the readline thread gets cancelled for instance
 * */
static
void readline_cleaner(void * arg)
{
	rl_cleanup_after_signal();
}
#endif /* HAVE_READLINE */

/*
 * Get a line of input from stdin and eliminate \r characters that 
 * may get there from telnet connections
 */
static int
get_keyboard_input(in, prompt)
char	*in;
char	*prompt;
{
#ifdef HAVE_LIBREADLINE
		char	*cmd;
		/*
		 * prompt and get command via readline package
		 */
		rtx_thread_cleanup_push(readline_cleaner,NULL)
		cmd = readline(prompt);
		rtx_thread_cleanup_pop(0);
		if (cmd == NULL)		/* EOF */
			return -1;
		if (*cmd != '\0')		/* empty string */
		    add_history(cmd);
		strcpy(in, cmd);
		free(cmd);
#else
	printf(prompt);
	if (fgets(in, BUFSIZ, stdin) == NULL)
		return -1;

#endif /* HAVE_LIBREADLINE */

	return 0;

}


static void
string_free(argc, argv)
int argc;
char *argv[];
{
	int i;

	for (i=0; i<argc; i++)
		free(argv[i]);
}

/*
 * Parse the string `in' and place pointers to successive space separated 
 * tokens into the array `out[]'.
 */
static int 
parse_input(in, out)
char *in, *out[];
{
        int	i = 0, args = 0, inblank=1;
        char	*p, *s = in;
        char	temp[MAX_ARG_LENGTH];

        for (;; s++) {
 	       switch (*s) {
                case ' ':
		case '\t':
			if (inblank)
				break;
                        temp[i] = '\0';
                        p = (char *) malloc(strlen(temp)+1);
                        strcpy(p, temp);
                        out[args++] = p;
                        i = 0;
			inblank++;
			break;
		case '\n':
		case '\r':
		case '\0':
			if (!inblank) {
				temp[i] = '\0';
				p = (char *) malloc(strlen(temp)+1);
				strcpy(p, temp);
				out[args++] = p;
			}
			return args;
                default:
			inblank = 0;
                        temp[i++] = *s;
			if (i > MAX_ARG_LENGTH)
				return rtx_error("rtx_command: argument too long");
                        break;
                }
		if (args > MAX_NUM_OF_ARGS)
			return rtx_error("rtx_command: too many arguments");
	}
}

/**
 * Main menu handler.
 *
 * Given a menu and prompt string, it will prompt, accept lines of
 * input and dispatch to handler functions.  Returns only on the
 * keyword 'quit' or 'exit'.
 *
 * @param menu Pointer to menu of commands.
 * @param prompt Pointer to prompt string.
 *
 * Each command is invoked as \p int func(int argc, char *argv[]), if
 * the return value is negative an rtx_error_flush()
 *
 * @note The function never returns.
 * @note \p readline() is used to get the line from the user, this provides
 * command history and editing capability.
 */
void
rtx_command_handler(RtxCommandMenu *menu, char *prompt)
{
	int	which, i, argc, chosen = 0;
	char	input[BUFSIZ];
	char	*argv[MAX_NUM_OF_ARGS];
	static char	*func = "rtx_command_handler";

#define	NO_COMMAND		-1
#define	AMBIGUOUS_COMMAND	-2
	for (;;) {
		if (get_keyboard_input(input, prompt) < 0)
			break;
		argc = parse_input(input, argv);
		if (argc <= 0)
			continue;

		if (	strcmp(argv[0], "help") == 0 ||
			strcmp(argv[0], "?") == 0
		) {
			menu_help(menu);
			continue;
		}
		if (strncmp(argv[0], "exit", strlen(argv[0])) == 0)
			return;
		if (strncmp(argv[0], "quit", strlen(argv[0])) == 0)
			return;
		chosen = 0;
		
		/*
		 * try to match the first token with the menu
		 */
		which = NO_COMMAND;
		for (i=0; menu[i].name != NULL; i++)
			if (strncmp(menu[i].name, argv[0], strlen(argv[0])) == 0) {
				/*
				 * we have a match, do we already have one?
				 */
				if (which >= 0) {
					which = AMBIGUOUS_COMMAND;
					break;
				} else
					which = i;
			}
		
		switch (which) {
		case AMBIGUOUS_COMMAND:
			printf("%s: command <%s> is ambiguous\n", func, argv[0]);
			break;
		case NO_COMMAND:
			printf("%s: unknown command <%s>\n", func, argv[0]);
			break;
		default:
			/*
			 * invoke the handler
			 */
			if ((*(menu[which].handler)) (argc, argv) < 0)
				rtx_error_flush("%s: Error handling command <%s>",
					func,
					argv[0]);
			break;
		}
		/*
		 * free memory allocated for arguments
		 */
		string_free(argc, argv);
		
	}
}
