/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/

#ifdef __GNUC__
#define DDX_STATIC_ATTRIBUTE __attribute__((unused))
#else
#define DDX_STATIC_ATTRIBUTE
#endif
static char rcsid[] DDX_STATIC_ATTRIBUTE = "$Id: circbuf.c 3132 2008-05-19 06:14:40Z roy029 $";

/**
 * \file circbuf.c
 * \brief circular buffer
 * \author Pavan Sikka
 *
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <rtx/sem.h>
#include <rtx/thread.h>
#include <rtx/error.h>
#include <rtx/time.h>
#include <rtx/timer.h>

#include "rtx/circbuf.h"

#ifndef DOXYGEN_SHOULD_SKIP_THIS

static void *
rtx_circbuf_refill_thread (
			   void * arg
			   )
{
    RtxCircbuf * b = (RtxCircbuf *) arg;

    while (1) {
        if (rtx_sem_wait (b->sem) == -1) {
	    rtx_error_flush ("rtx_circbuf_refill_thread: rtx_sem_wait() failed");
	    rtx_timer_sleep (0.1);
	    continue;
	}
	b->refill_fn (arg);
    }
    return (NULL);
}

#endif

RtxCircbuf * 
rtx_circbuf_init (
		  int n,
		  int elSize,
		  void * (* refill_fn) (void *),
		  void * refillArg, 
		  int refillThPrio,
		  int refillThreashold
		  )
{
    RtxCircbuf * b = NULL;

    if (n <= 0)
        return (rtx_error_null ("rtx_circbuf_init: invalid n, should be > 0 (%d)", n));
    if (elSize <= 0)
        return (rtx_error_null ("rtx_circbuf_init: invalid element size, should "
				"be > 0 (%d)", elSize));
    if ((b = calloc (1, sizeof (RtxCircbuf))) == NULL)
        return (rtx_error_null ("rtx_circbuf_init: no memory"));
    if ((b->d = calloc (n, elSize)) == NULL)
        return (rtx_error_null ("rtx_circbuf_init: no memory"));
    b->eSize = elSize;
    b->n = n;
    b->front = b->d;
    b->back = b->d;
    b->lastElem = b->d + (n-1) * elSize;
    if (refill_fn != NULL) {
        if ((b->sem = rtx_sem_init (NULL, 0, 0)) == NULL)
	    return (rtx_error_null ("rtx_circbuf_init: rtx_sem_init() failed"));
	b->refill_fn = refill_fn;
	b->refillArg = refillArg;
	b->refillThreashold = refillThreashold;
	b->refillThPrio = refillThPrio;
	if ((b->th = rtx_thread_create ("rtx_circbuf_refill_fn", 0,
					RTX_THREAD_SCHED_OTHER, refillThPrio, 0,
					RTX_THREAD_CANCEL_DEFERRED,
					rtx_circbuf_refill_thread, refillArg, 
					NULL, NULL)) == NULL)
	    return (rtx_error_null ("rtx_circbuf_init: rtx_thread_create() failed"));
    }
    return (b);
}

void * 
rtx_circbuf_get (
		 RtxCircbuf * c
		 )
{
    void * b = NULL;

    if (c->numInBuf == 0)
        return (NULL);
    b = c->front;
    c->front += c->eSize;
    if (c->front > c->lastElem)
        c->front = c->d;
    c->numInBuf--;
    if (c->refill_fn != NULL)
        if (c->numInBuf < c->refillThreashold)
	    if (rtx_sem_post (c->sem) == -1)
	        rtx_error_flush ("rtx_circbuf_get: rtx_sem_post() failed");
    return (b);
}

void * 
rtx_circbuf_peek (
		  RtxCircbuf * c
		  )
{
    if (c->numInBuf == 0)
        return (NULL);
    return (c->front);
}

int
rtx_circbuf_next (
		  RtxCircbuf * c
		  )
{
    if (c->numInBuf == 0)
        return (0);
    c->front += c->eSize;
    if (c->front > c->lastElem)
        c->front = c->d;
    c->numInBuf--;
    return (0);
}

int 
rtx_circbuf_put (
		 RtxCircbuf * c, 
		 void * d, 
		 int n
		 )
{
    if (c->numInBuf == c->n)
        return (rtx_error ("rtx_circbuf_put: buf full"));
    if (n > c->eSize)
        return (rtx_error ("rtx_circbuf_put: data elem too big"));
    memcpy (c->back, (char *) d, n);
    c->back += c->eSize;
    if (c->back > c->lastElem)
        c->back = c->d;
    c->numInBuf++;
    return (0);
}

int
rtx_circbuf_done (
		  RtxCircbuf * c
		  )
{
    return (0);
}

