/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/


/**
 * \file byte.c
 * \brief Byte stream utility functions
 * \author Jonathan Roberts
 */

#ifdef	__sun__
#include	<strings.h>
#else
#include	<string.h>
#endif
#include <stdlib.h>

#include "rtx/byte.h"
#include "rtx/defines.h"
#include "rtx/error.h"

static char rcsid[] RTX_UNUSED = "$Id: byte.c 2274 2007-12-23 05:37:32Z roy029 $";

/**
 * Convert a byte string to an int
 * @param v a long pointer in which is returned the integer value
 * @param p a pointer to the bytes to be converted
 * @param n number of bytes to be converted
 * @param flags to control conversion
 * @return 0 if OK, -1 on error (n > sizeof (long), or invalid flag)
 *
 * The flags are specified in byte.h and are:
 * - RTX_BYTE_SIGNED\n
 *    convert to a signed integer
 * - RTX_BYTE_UNSIGNED\n
 *    convert to an unsigned integer
 * - RTX_BYTE_LSBFIRST\n
 *    least significant byte is first
 * - RTX_BYTE_MSBFIRST\n
 *    most significant byte is first
 */
int
rtx_byte_to_int(long *v, unsigned char *p, int n, int flag)
{
	int	t = 0;
	int	i;

	if (n > sizeof(long))
		return rtx_error("rtx_byte_to_int: long is only %d bytes", sizeof(long));
	
	if (flag & RTX_BYTE_MSBFIRST) {
		for (i=0; i<n; i++) {
			t <<= 8;
			t |= *p++;
		}
	} else if (flag & RTX_BYTE_LSBFIRST) {
		for (i=0; i<n; i++)
			t |= ((*p++) << (8*i));
	} else
		return rtx_error("rtx_byte_to_int: no order flag specified");
	/*
	 * sign extend if needed
	 */
	if ((flag & RTX_BYTE_SIGNED) && (n < 4)) {
		int	mask =0;

		for (i=0; i<n; i++)
			mask |= (mask<<8 | 0xff);

		t |= ((0x80 << ((n-1)*8)) & t) ? ~mask : 0;
	}
	*v = t;
	return 0;
}

/**
 * Compute the CRC16 polynomial on byte string.
 *
 * @param p pointer to start of byte string
 * @param n length of byte string
 * @return the CRC16 result
 */
int
rtx_byte_crc16(unsigned char *p, int n)
{
        unsigned int crc16, crc_data, i;
 
        crc16 = 0;
        crc_data = 0;
        for (i=0; i<n; i++) {
                crc_data <<= 8;
                crc_data |= (unsigned int)p[i];
                crc16 <<= 1;
                if ((crc16 & 0x10000) > 0)
                                crc16 ^= 0x8005;
                crc16 ^= crc_data;
                crc16 &= 0xffff;
        }
 
        return (crc16);
}

/**
 * Compute the CRC16 polynomial on byte string.
 *
 * @param p pointer to start of byte string
 * @param n length of byte string
 * @return the CRC16 result
 */
unsigned int
rtx_byte_crc16_2(unsigned char *p, int n)
{
	unsigned int    i, checksum = 0;

	for (i=0; i<n; i++) {
		if ((i%2) == 0)
			checksum += ((unsigned int) p[i]) << 8;
		else
			checksum += (unsigned int) p[i];
	}
	checksum += checksum >> 16;
	checksum &= 0x0000FFFF;

	return (checksum);
}

/**
 * Swap bytes end-to-end within a 32-bit word, i.e., 0x01020304 becomes 0x04030201.
 * @param data pointer to an \c int to be byte swapped.
 */
void
rtx_byte_swap_int(int *data)
{
        char    t, *p = (char *)data;
	
        /* swap bytes 0 & 3 */
        t = p[0];
        p[0] = p[3];
        p[3] = t;

        /* swap bytes 1 & 2 */
        t = p[1];
        p[1] = p[2];
        p[2] = t;

}

/**
 * Swap bytes end-to-end within a 16-bit word, i.e., 0x01020304 becomes 0x04030201.
 * @param data pointer to an \c int to be byte swapped.
 */
void
rtx_byte_swap_short(short *data)
{
        char    t, *p = (char *)data;
	
        /* swap bytes 0 & 1 */
        t = p[0];
        p[0] = p[1];
        p[1] = t;
}
