/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file auth.c
 * \brief Simple front-end to RTC authentication
 * \author Jonathan Roberts
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <rtc.h>

#include "rtx/auth.h"
#include "rtx/defines.h"
#include "rtx/error.h"
#include "rtx/message.h"

static char rcsid[] RTX_UNUSED = "$Id: auth.c 2274 2007-12-23 05:37:32Z roy029 $";

/**
 * add hosts to the list of acceptable clients
 *
 * This function reads the hostnames from the file pointed to by
 * filename and adds the machine names to the server access control list. 
 *
 * @return 0 on success, -1 on error
 */
int
rtx_auth_hosts (
		    char *filename,     /**< file containing auth info */
		    int verbose         /**< verbose level */
		    )
{
	int	i, j, len;
	char	buf[BUFSIZ], hostname[BUFSIZ];
	FILE	*fr;

	if (verbose)
		 rtx_message_routine ("rtcStuff_rtc_hosts: reading the RTC hosts file: %s\n", filename);

	/*
	 * Open the file pointed to by filename
	 */
	if ((fr=fopen(filename, "r")) == NULL) {
		rtx_message_routine ("rtcStuff_rtc_hosts: cannot open file %s\n", filename);
		return -1;
	}

	while (fgets(buf, BUFSIZ, fr) != NULL) {
		j = 0;
		len = strlen(buf);
		for (i=0; i<len; i++) {
			if (buf[i] == '#') {
				continue;
			}
			else if ((buf[i] == ' ') || (buf[i] == '\n')) {
				hostname[j] = '\0';
				if (j > 0) {
					if (verbose > 1)
						rtx_message_routine ("rtcStuff_rtc_hosts: adding %s to server list\n", hostname);
					if (rtcServerAuth(rtc_auth_host, hostname, 0)) {
						return (rtx_error ("rtcStuff_rtc_hosts: rtcServerAuth() failed: %s\n", rtcMessageGet()));
					}	
				}
				j = 0;
			}
			else {
				hostname[j] = buf[i];	
				j++;
				if (i == len-1) {
					if (verbose > 1)
						rtx_message_routine ("rtcStuff_rtc_hosts: adding %s to server list\n", hostname);
					if (rtcServerAuth(rtc_auth_host, hostname, 0)) {
						return (rtx_error ("rtcStuff_rtc_hosts: rtcServerAuth() failed: %s\n", rtcMessageGet()));
					}
				}
			}
		}
	}

	/*
	 * Close the hosts file
	 */
	fclose(fr);

	return 0;
}

/**
 * add users, groups and hosts to the list of acceptable clients
 *
 * This function does the RTC server authentication. It reads a file 
 * (pointed to by filename) that contains a list of the machines, users and
 * groups that are allowed to connect to the RTC server.
 *
 * Example file:
 *
 *	_HOST
 *	dline
 *	lame.field.csiro.au
 *	dropes.field.csiro.au
 *	_END
 *	_USER
 *	jmr
 *	pic
 *	_END
 *	_USER_GROUP
 *	autom
 *	_END
 *
 * @return 0 on success, -1 on error
 */
int
rtx_auth (
		  char *filename,     /**< file containing auth info */
		  int verbose         /**< verbose level */
		  )
{
	FILE	*fr;
	int isLine;
	int result;
	char line[100], name[100];
	enum { nothing, host, user, group } definition = nothing;

	/*
	 * Open the file pointed to by filename
	 */
	if ((fr=fopen(filename, "r")) == NULL) {
		return (rtx_error_errno ("rtcStuff_suthenticate: cannot open file %s\n", filename));
	}

	do {
		/* 
		 * Reads the next line 
		 */
		isLine = fscanf(fr,"%[^\n]\n", line);
		if( isLine == 1) {
			/* 
			 * Looks if the line is a comment 
			 */
			if(strncmp( line,"#",1) == 0) {
			}
			/* 
			 * Looks if the kind of current 
			 * definitions is changing 
			 */
			else if(sscanf(line, "_HOST%*[ :]")) {
				definition = host;
			}
			else if(sscanf(line, "_USER%*[ :]")) {
				definition = user;
			}
			else if(sscanf(line, "_USER_GROUP%*[ :]")) {
				definition = group;
			}
			else if(sscanf(line, "_END%*[ :]")) {
				definition = nothing;
			}
			else {
			/* Reads the next definition */
	
				switch( definition) {
				case host :
					if ((result = sscanf(line, "%s", name))) {
						rtcServerAuth(rtc_auth_host, &name,0);
					}
					if (verbose)
						rtx_message_routine  ("sscanf = %d; authorizing host %s", result, name);
					break;
				case user :
					if ((result = sscanf(line, "%s", name))) {
						rtcServerAuth(rtc_auth_user, &name,0);
					}
					if (verbose)
						rtx_message_routine  ("sscanf = %d; authorizing user %s", result, name);
					break;
				case group :
					if ((result = sscanf(line, "%s", name))) {
						rtcServerAuth(rtc_auth_group, &name,0);
					}
					if (verbose)
						rtx_message_routine  ("sscanf = %d; authorizing group %s", result, name);
					break;
				case nothing :
			    		break;
				default :
					break;
				}
			}
      		}
	} while( isLine > 0);

	/*
	 * Close the authentication file
	 */
	fclose(fr);

	return 0;
}

