/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 *
 * $Id: timer.h 2287 2007-12-24 04:11:30Z roy029 $
 *
 ***********************************************************************/
/**
 ********************************************************************
 *
 * \file timer.h
 * \brief A simple interface to POSIX.1b timers
 * \author Jonathan Roberts and Pavan Sikka
 *
 ********************************************************************
 */

#ifndef RTX_TIMER_H
#define RTX_TIMER_H

#include <time.h>
#include <signal.h>

#include <rtx/defines.h>
#include <rtx/thread.h>

RTX_CXX_OPEN

/**
 * two possible clock sources - CLOCK_REALTIME and user-defined CUSTOM
 */
typedef enum _rtx_timer_clock_id {
    RTX_TIMER_CLOCK_REALTIME = 0,
    RTX_TIMER_CLOCK_CUSTOM
} RtxTimerClockId;

/**
 * RTX timer handle
 */
typedef struct _rtx_timer {
    RtxTimerClockId clockId;       /**< clock source */
#ifndef DARWIN
    timer_t timer;                 /**< POSIX timer */
    struct itimerspec itimer;      /**< timer delay/period */
    int sigNum;                    /**< signal associated with timer */
    sigset_t tmrSig;
    void (* func) (void *);        /**< user-defined function to be run whenever
				        timer expires */
    void * arg;                    /**< argument passed to user-defined function */
    struct sigaction old;
#endif
} RtxTimer;

/**
 * RTX thread_with_timer handle
 */
typedef struct _rtx_timer_thread {
    RtxTimer * tmr;                /**< RTX timer */
    RtxThread * th;
    void (* func) (void *);
    void * arg;
    double initDelay;
    double period;
    RtxTimerClockId clockId;
    int done;
} RtxTimerThread;

#ifndef DOXYGEN_SHOULD_SKIP_THIS

int rtx_timer_sleep (double dt);
int rtx_timer_sleep_eintr (double dt);

double rtx_timer_get_clock_resolution (RtxTimerClockId clockId);

RtxTimer * rtx_timer_create (double initDelay, double period, 
			     void (* func) (void *), void * arg,
			     RtxTimerClockId clockId);
int rtx_timer_wait (RtxTimer * tmr);
int rtx_timer_destroy (RtxTimer * tmr);
int rtx_timer_set (RtxTimer * tmr, double initDelay, double period);
int rtx_timer_start (RtxTimer * tmr);
int rtx_timer_stop (RtxTimer * tmr);
RtxTimerThread * rtx_timer_create_thread (double initDelay, 
					  double period, 
					  int prio,
					  void (* func) (void *), 
					  void * arg,
					  RtxTimerClockId clockId);
int rtx_timer_destroy_thread (RtxTimerThread * tmrTh);
int rtx_timer_destroy_thread_sync (RtxTimerThread * tmrTh);

#endif /* DOXYGEN_SHOULD_SKIP_THIS */

RTX_CXX_CLOSE

#endif /* RTX_TIMER_H */
