/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 *
 * $Id: thread.h 3070 2008-05-15 04:22:31Z roy029 $
 *
 ***********************************************************************/
/**
 ********************************************************************
 *
 * \file thread.h
 * \brief Simple interface to POSIX threads
 * \author Jonathan Roberts
 *
 ********************************************************************
 */

#ifndef RTX_THREAD_H
#define RTX_THREAD_H

#include <pthread.h>
#include <semaphore.h>

#include <rtx/defines.h>
#include <rtx/sem.h>

RTX_CXX_OPEN

/**
 * The cancel mode enum
 */
typedef enum {
	RTX_THREAD_CANCEL_DISABLE,
	RTX_THREAD_CANCEL_ASYNCHRONOUS,
	RTX_THREAD_CANCEL_DEFERRED
} RtxThreadCancel;

/**
 * Ths scheduling policy enum
 */
typedef enum {
	RTX_THREAD_SCHED_OTHER = 0,
	RTX_THREAD_SCHED_FIFO = 1,
	RTX_THREAD_SCHED_RR = 2,
	RTX_THREAD_SCHED_INHERIT = 3
} RtxThreadSched;

/**
 * The RTX thread prio typedef
 */
typedef int RtxThreadPrio;
#define RTX_THREAD_PRIO_MIN (-1)
#define RTX_THREAD_PRIO_MAX (-2)

/**
 * The RTX thread structure
 */
typedef struct _rtx_thread {
	pthread_t       id;                  /*!< Thread ID */
	char            *name;               /*!< Name of the thread */
	RtxSem          sem;                 /*!< Pointer to semaphore */
	void            *(*func)(void *);    /*!< Pointer to function */
	void            *arg;                /*!< Argument to function */
	RtxThreadCancel	cancelType;          /*!< Cancel type */
	void            (*cancel_func)(void *);	/*!< Cancel function */
	void            *cancelArg;          /*!< Argument to cancel function */
	int             initErrors;          /*!< Initialisation errors */
	int             cancelled;           /*!< Cancelled flag */
	int             detached;            /*!< Detached flag */
	int             debug;               /*!< Debug flag */
	int             errorFlags;          /*!< Error module flags */
	void            *var;                /*!< User variable */
	void            *stackBottom;        /*!< Beginning of thread stack */
} RtxThread;

#ifndef DOXYGEN_SHOULD_SKIP_THIS

RtxThread *rtx_thread_create (const char *name, int debug, RtxThreadSched policy, 
		RtxThreadPrio prio, int stackSize, RtxThreadCancel cancelType,
		void *(*func)(void *), void *arg,
		void (*cancel_func)(void *), void *cancelArg);
int rtx_thread_destroy (RtxThread *thread);
int rtx_thread_destroy_sync (RtxThread *thread);
int rtx_thread_detach(RtxThread *thread);
int rtx_thread_join (RtxThread *thread);
int rtx_thread_setcancel(RtxThreadCancel cancelType);
void rtx_thread_testcancel(void);
int rtx_thread_cancelled(RtxThread *thread);
void rtx_thread_set_var(RtxThread *thread, void *var);
void rtx_thread_get_var(RtxThread *thread, void *var);
unsigned int rtx_thread_get_stack_size(void);

int rtx_thread_prio_to_system(RtxThreadPrio prio);
RtxThreadPrio rtx_thread_system_to_prio(int prio);
RtxThreadPrio rtx_thread_prio_get(void);
int rtx_thread_prio_set(RtxThreadPrio prio);

int rtx_thread_create_legacy(pthread_t *threadId, int threadPrio,
		       int stackSize, void *(*func)(void *), void *arg);
int rtx_thread_destroy_legacy(pthread_t *threadId);

#define rtx_thread_cleanup_push(f,a) pthread_cleanup_push(f,a)
#define rtx_thread_cleanup_pop(i) pthread_cleanup_pop(i)

#endif

RTX_CXX_CLOSE

#endif /* RTX_THREAD_H */
