/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 *
 * $Id: parse.h 2296 2008-01-03 05:54:51Z roy029 $
 *
 ***********************************************************************/
/**
 ********************************************************************
 *
 * \file parse.h
 * \brief parser module for C data structures
 * \author Pavan Sikka
 *
 ********************************************************************
 */

#ifndef RTX_PARSE_H
#define RTX_PARSE_H

#include <rtx/defines.h>
#include <rtx/hash.h>

RTX_CXX_OPEN

/*
 * Tokens recognized by the parser
 */

#define TOK_STRUCT          	1
#define TOK_CHAR            	2
#define TOK_SHORT           	3
#define TOK_INT             	4
#define TOK_LONG            	5
#define TOK_FLOAT           	6
#define TOK_DOUBLE          	7
#define TOK_LEFT_BRACE      	8
#define TOK_RIGHT_BRACE     	9
#define TOK_SEMICOLON       	10
#define TOK_LEFT_BRACKET    	11
#define TOK_RIGHT_BRACKET   	12
#define TOK_NAME            	13
#define TOK_NUMBER          	14
#define TOK_INT_NUMBER      	15
#define TOK_REAL_NUMBER     	16
#define TOK_DECL            	17
#define TOK_DECL_LIST       	18
#define TOK_TYPE            	19
#define TOK_VAR             	20
#define TOK_DOT                 21	
#define TOK_UNSIGNED            22
#define TOK_BYTE                23
#define TOK_BOOL                24
#define TOK_REAL                25
#define TOK_TYPEDEF             26
    
#define MAX_TOK_LEN         	256       /* max number of chars in a token */
#define MAX_STR_LEN         	256       /* max number of chars in a string */
#define MAX_NUM_ARRAY_DIMS      3         /* max number of dimensions for 
					   * multi-dimensional arrays */
#define MAX_SYM_TAB_SIZE        128       /* max size of a symbol table */

/*
 * VAR tags
 */
#define VAR_ATTR_PRINT              1
#define VAR_ATTR_TAG                2
#define VAR_ATTR_ENTIRE_ARRAY       4

/**
 * types recognized by the parser
 */
typedef enum _rtx_parse_type {
	rtx_struct_t = TOK_STRUCT, rtx_char_t, rtx_short_t, 
	rtx_int_t, rtx_long_t, rtx_float_t, rtx_double_t
} RtxParseType;

/**
 * token value union, used by RtxParseToken
 */
typedef union _rtx_parse_val {
    long     intVal;
    double   realVal;
} RtxParseVal;

/**
 * structure to hold a token being parsed
 */
typedef struct _rtx_parse_token {
    char tokStr[MAX_TOK_LEN];
    int  tokId;
    RtxParseVal  tokVal;
} RtxParseToken;

/**
 * a node in the parse-tree
 */
typedef struct _rtx_parse_var {
    char         name[MAX_TOK_LEN];        /**< var or type name */
    char         tag[MAX_TOK_LEN];         /**< tag name (for structs) */
    int          type;                     /**< type */
    unsigned int attribs;                  /**< used for tagging parse-trees */
    int          dim;                      /**< number of array dimensions */
    int          arrayDim[MAX_NUM_ARRAY_DIMS];     /**< size of each dimension */
    int          arrayIndex[MAX_NUM_ARRAY_DIMS];   /**< used for traversing parse-tree */
    int          elemSize;                 /**< size of one element */
    int          alignment;                /**< alignment of this node */
    int          offset;                   /**< offset of this node (relative to top) */
    int          size;                     /**< node size (bytes) */
    char         * matHdr;                 /**< used by ddxreadlof (MATLAB) */
    struct _rtx_parse_var  * subVar;       /**< parse sub-tree */
    struct _rtx_parse_var  * next;         /**< parse next-tree */
} RtxParseVar;

/**
 * an entry in the CONST table
 */
typedef struct _const_table_entry {
    char             name[MAX_TOK_LEN];
    int              value;
} RtxParseConstTableEntry;

/**
 * an entry in the TYPE table
 */
typedef struct _type_table_entry {
    char             name[MAX_TOK_LEN];
    int              typeNum;
    int              primitiveType;
    RtxParseVar      * t;
} RtxParseTypeTableEntry;

/**
 * an entry in the VAR table
 */
typedef struct _var_table_entry {
    char             name[MAX_TOK_LEN];
    RtxParseVar      * v;
    int              offset;
    int              headerOffset;
    int              varId;
    char             * decl;
} RtxParseVarTableEntry;

/**
 * structure to hold all the different tables for an architecture
 */
typedef struct _symbol_tables {
    rtx_hash_t constTable;
    rtx_hash_t typeTable;
    rtx_hash_t varTable;
    int typeNumber;
} RtxParseSymTabs;

#ifndef DOXYGEN_SHOULD_SKIP_THIS

/* functions */

RtxParseSymTabs * rtx_parse_init (void);
RtxParseSymTabs * rtx_parse_init_remote (int dataSize[], int dataAlign[]);
void rtx_parse_finish(RtxParseSymTabs * symtabs);

void rtx_parse_get_compiler_parms (int dataSize[], int dataAlign[]);

int rtx_parse_add_const_symbol (char * name, int val, rtx_hash_t symTab);
int rtx_parse_add_type_symbol (RtxParseVar * v, int primitiveType, rtx_hash_t symTab,
			       int * typeNumber);
int rtx_parse_add_var_symbol (RtxParseVar * v, int varId, int offset, int headerOffset,
			      char * varDecl, rtx_hash_t symTab);
int rtx_parse_del_const_symbol (char * name, rtx_hash_t symTab);
int rtx_parse_del_var_symbol (char * name, rtx_hash_t symTab);
int rtx_parse_del_type_symbol (char * name, rtx_hash_t symTab);

RtxParseTypeTableEntry * rtx_parse_lookup_type (int typeNumber, rtx_hash_t symTab);

int rtx_parse_cmp_types (RtxParseVar * v1, RtxParseVar * v2);
RtxParseVar * rtx_parse_copy_var (RtxParseVar * v);
int rtx_parse_expand_var_definition (RtxParseVar * v, rtx_hash_t symTab);
void rtx_parse_free_var (RtxParseVar * v);

RtxParseVar * rtx_parse (char * s, RtxParseSymTabs * symTabs);
int rtx_parse_decl (char * s, int * p, RtxParseToken * t, 
		    RtxParseVar * v, RtxParseSymTabs * symTabs);
RtxParseVar * rtx_parse_name_list (char * s, int * p, RtxParseToken * t,
				   RtxParseVar * v, 
				   RtxParseSymTabs * symTabs);

int rtx_parse_old_decl (char * s, int * p, RtxParseToken * t,
			RtxParseVar * v, RtxParseSymTabs * symTabs);

int rtx_parse_generate_var_decl (RtxParseVar * v, char * buf);
int rtx_parse_print_var (RtxParseVar * v, void * d, char * s);
int rtx_parse_scan_var (RtxParseVar * v, void * din, char * s,
			char ** lasts, int initCall);
void rtx_parse_print_decl (RtxParseVar * v, int offset);
void rtx_parse_print_const_table (rtx_hash_t stEntry);
void rtx_parse_print_type_table (rtx_hash_t stEntry);
void rtx_parse_print_var_table (rtx_hash_t stEntry);
int rtx_parse_translate_data (void * d, RtxParseVar * v, void * dRemote,
			      RtxParseVar * vRemote, int swapword);

#endif /* DOXYGEN_SHOULD_SKIP_THIS */

RTX_CXX_CLOSE

#endif /* RTX_PARSE_H */
