/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 *
 * $Id: logread.h 2946 2008-04-14 06:22:18Z rue006 $
 *
 ***********************************************************************/
/**
 ********************************************************************
 *
 * \file logread.h
 * \brief API for reading log files created by the log module
 * \author Pavan Sikka
 *
 ********************************************************************
 */

#ifndef RTX_LOGREAD_H
#define RTX_LOGREAD_H

#include <zlib.h>

#include <rtx/defines.h>
#include <rtx/sem.h>
#include <rtx/mutex.h>
#include <rtx/thread.h>
#include <rtx/time.h>
#include <rtx/parse.h>
#include <rtx/log.h>

RTX_CXX_OPEN

/**
 * This structure is required to convert data into a MATLAB binary data
 * file.
 */
typedef struct _rtx_logread_matlab_header {
        /*
         * data is stored in row order
         */
        union {
	        double  *d;
                float   *f;
                char    *p;
        }       p0;             /* pointer to matlab matrix (retriever) */
        int     len;            /* length of data array in bytes */
        int     rows;           /* number of rows in matrix */
        int     dim;            /* number of columns in matrix */
} RtxLogreadMatlabHeader;

/**
 * This structure holds all the information relevant to a variable
 * stored in the log file.
 */
typedef struct _rtx_logread_var {
    RtxParseVar * v;                /**< var tree */
    RtxLogItemHeader vh;            /**< var header */
    void * d;                       /**< var data */
    RtxParseVar * vRemote;          /**< var tree for logfile arch */
    RtxLogItemHeader vhRemote;      /**< var header for logfile arch */
    void * dRemote;                 /**< var data for logfile arch */
    RtxLogreadMatlabHeader * mht;   /**< MATLAB area */
    int numRecords;                 /**< number of records in logfile */
    struct _rtx_logread_var * next; /**< next var */
	void * userdata;                /**< pointer to store user data with the var.
									 * user is responsible for freeing this
									 * memory **/
} RtxLogreadVar;

/**
 * This structure holds all the information needed to process a logfile.
 */
typedef struct _rtx_logread {
    FILE            *logFileP;  /**< file pointer for logfile*/
    gzFile          gzLogFile;  /**< file pointer for gzipped logfile*/
    char            *logFile;   /**< logfile name */ 
    char            header[RTX_LOG_MAX_HEADER_SIZE]; /**< logfile header */ 
    int             headerLen;  /**< logfile header size */ 
    int             verbose;    /**< verbose flag */
    int             gzippedLogFile;  /**< flag to indicate gzipped logfile*/
    int				fileDescriptor;		/**< flag to indicate to read from a file descriptor (e.g. socket) instead of a FILE pointer*/
    int				fd;			/**< file descriptor for a socket or similar*/
    char			catalogName[256];
    int             max_id;     /**< number of vars in logfile */
    RtxLogreadVar       varc;	/* Somehow this cannot be a pointer */
    RtxParseVar     commentVar; /**< var for comments in logfile */
    char            comment[BUFSIZ];  /**< data for comments in logfile */
    RtxLogreadVar       *vars;  /**< list of vars in the logfile */
    RtxLogreadVar       **varmap;  /**< array of vars in logfile */
    int             swapword;   /**< byte-order info for arch */
    int             swapType;   /**< flag to indicate byte-swapping needed */
    int             oldLogFile; /**< flag to indicate old logfile */
    int             remoteData; /**< data translation needed */
    int             remoteAlign[8];  /**< data alignment on logfile arch */
    int             remoteSize[8];   /**< data size on logfile arch */ 
    int             localAlign[8];   /**< data alignemnt on own arch */
    int             localSize[8];    /**< data size on own arch */
    int             initOffset; /**< offset of first data record in logfile*/
    RtxTime         start;      /**< timestamp of first data record */
    int             numVars;    /**< number of vars in logfile */
    RtxParseSymTabs * symTabs;  /**< own symbol table */
    RtxParseSymTabs * remSymTabs;  /**< symbol table for logfile arch */
} RtxLogread;

/* functions */

RtxLogread * rtx_logread_init(char *filename, int verbose); 
RtxLogread * rtx_logread_init_fd(int fd, int verbose);
RtxLogreadVar * rtx_logread_tag_var (RtxLogread * l, char *name);
RtxLogreadVar * rtx_logread_next (RtxLogread * l);
RtxLogreadVar * rtx_logread_next_tagged (RtxLogread * l);
int rtx_logread_rewind(RtxLogread * l);
void rtx_logread_close (RtxLogread * l);

void rtx_logread_print (FILE * fp, RtxLogreadVar *v);
char ** rtx_logread_get_files (char *dir, int *nf);
double rtx_logread_time (RtxLogreadVar *v);
char * rtx_logread_stime (RtxLogreadVar *v);
double rtx_logread_rtime (RtxLogreadVar *v, RtxTime start);
long rtx_logread_get_filepos (FILE * fp);

RTX_CXX_CLOSE

#endif /* RTX_LOGREAD_H */


