/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 *
 * $Id: log.h 2287 2007-12-24 04:11:30Z roy029 $
 *
 ***********************************************************************/
/**
 ********************************************************************
 *
 * \file log.h
 * \brief data-logging functions
 * \author Pavan Sikka
 *
 ********************************************************************
 */

#ifndef RTX_LOG_H
#define RTX_LOG_H

#include <zlib.h>

#include <rtx/defines.h>
#include <rtx/sem.h>
#include <rtx/mutex.h>
#include <rtx/thread.h>
#include <rtx/time.h>
#include <rtx/parse.h>

RTX_CXX_OPEN

/* #define */

#define MAX_PATHNAME_LEN    128
#define MAX_FILENAME_LEN    64
#define MAX_WRITE_BUFS      5

#define RTX_LOG_HEADER_MAGIC_NUMBER     0x02030405
#define RTX_LOG_HEADER_EOH              "_EOH_\n"
#define RTX_LOG_MAX_HEADER_SIZE         (100 * BUFSIZ)

#define USE_GZIP_COMPRESSION

#ifdef USE_GZIP_COMPRESSION
#include <stdio.h>
#endif

typedef struct _rtx_log_item_header {
    int     id;             /* file record id */
    RtxTime ts;             /* time when variable data was last acquired */
    int     id_prev;
    int     spare;
} RtxLogItemHeader;

/* typedefs */

typedef struct _rtx_log_write_bufs {
    unsigned char     * buf[MAX_WRITE_BUFS];
    int               bufLen[MAX_WRITE_BUFS];
} RtxLogWriteBufs;

typedef struct _rtx_log_parms {
    int               timeLimit;   /* roll-over file after so many minutes */
    int               sizeLimit;   /* new file reqd after so many KBytes */
    int               numBufs;	   /* number of buffers in buffer Q */
    int               bufSize;	   /* size of each buffer, in bytes */
    int               noTimeLimit; /* disable time limit on log file */
    int               noSizeLimit; /* disable size limit on log file */
    int               verbose;	   /* local logwriteDebugging variable */
#ifdef USE_GZIP_COMPRESSION
    int               gzip;        /* gzip compression level. No compression if 0 */
#endif
    int               writerThreadPrio;
    unsigned char     * header;    /* human readable log file header */
    int               headerLen;
    int               appendCreateComment;
    char              * appName;
    char              myHostName[BUFSIZ];
    char              myUserName[L_cuserid];
    int               simpleLogFileName;
    char              loggerDir[MAX_PATHNAME_LEN];
    char              loggerFileName[MAX_FILENAME_LEN];
    char              loggerFileNamePrefix[MAX_FILENAME_LEN];
    char              loggerFileNameSuffix[MAX_FILENAME_LEN];
	char              indexFileName[MAX_FILENAME_LEN];
} RtxLogParms;

typedef struct _rtx_log_stats {
    int      bytesToBuffer;
    int      bytesToFile;
    int      filesCreated;
    int      filesClosed;
    int      openErrors;
    int      closeErrors;
    int      bufFullErrors;
    int      bufOverflow;
    int      writeErrors;
    int      fileRolloverFlag;
    int      fileLoggingOn;
} RtxLogStats;

typedef struct _rtx_log_buf {
    unsigned char     * buf;
    int               curPos;
    RtxMutex *        lock;
    int               full;
    int               rollover;
    time_t            ts;
} RtxLogBuf;

typedef struct _rtx_log {
    RtxLogBuf         * bufs;	/* log file buffers */
    RtxLogParms       parms;
    RtxLogStats       stats;
    int               fd;	/* logger file fd */
#ifdef USE_GZIP_COMPRESSION
	FILE*             gzPipe; /* pointer to the pipe to gzip */
#endif
    int               done;     /* flag to signal module shutdown */
    RtxThread         * writerThread;	/* identity of file-writer thread */
    RtxSem *          bufFull;	/* semaphore to signal full buffer */
    RtxMutex *        lock;   /* mutex for critical section
					       in file_write_var */
    int               curBuf;	/* current buffer for data */
    int               bufToWrite;	/* buffer from which to write file */
    time_t            curFileCreated;	/* time file being written was created */
    int               curFileSize;/* size of current file being written */
    int               bufWriteSuspended;
    char              pathname[MAX_PATHNAME_LEN+MAX_FILENAME_LEN+32];
    char              prevPathname[MAX_PATHNAME_LEN+MAX_FILENAME_LEN+32];
} RtxLog;

/* functions */

RtxLog * rtx_log_open (char * logDir, char * logFile, int timeLim,
		       int sizeLim, int nBufs, int szBufs,
		       int writerThreadPrio, int gzip, int verbose, 
		       unsigned char * header, int headerSize,
		       char * appName, int appendCreateComment, char *indexFile);
int rtx_log_close (RtxLog * logger, RtxLogStats * stats);

int rtx_log_write (RtxLog * logger, unsigned char * buf, int n);
int rtx_log_write_bufs (RtxLog * logger, RtxLogWriteBufs * bufs);
int rtx_log_write_item (RtxLog * logger, int id, RtxTime * tp,
			unsigned char * buf, int n);
			
unsigned char * rtx_log_build_header (char * headerBody, int * n);
#ifdef USE_GZIP_COMPRESSION
int rtx_log_set_gzip (RtxLog * logger, int gzip_compression);
#endif
int rtx_log_set_header (RtxLog * logger, unsigned char * header, int n);
int rtx_log_rollover_log (RtxLog * logger, unsigned char * header,
		      int headerLen, int newHeader);

RTX_CXX_CLOSE

#endif /* RTX_LOG_H */


