/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 *
 * $Id: inet.h 2287 2007-12-24 04:11:30Z roy029 $
 *
 ***********************************************************************/
/**
 ********************************************************************
 *
 * \file inet.h
 * \brief simple interface to the network
 * \author Pavan Sikka
 *
 * A simple interface to the BSD-style networking facilities
 * provided by most UNIX-like systems.
 *
 ********************************************************************
 */

#ifndef RTX_INET_H
#define RTX_INET_H

#include <rtx/defines.h>
#include <rtx/thread.h>
#include <rtx/list.h>
#include <rtx/sem.h>

RTX_CXX_OPEN

/**
 * argument for the listen() system call
 */
#define RTX_INET_MAX_NUM_CONNS           5

/**
 * maximum number of chars in a hostname
 */
#define RTX_INET_MAX_HOST_NAME_LEN       64

/**
 * maximum size of the RTX field to store the 
 * inet sockaddr struct
 */
#define RTX_INET_MAX_ADDR_LEN            128

  /**
   * Default multicast TTL value
   */
#define RTX_INET_DEFAULT_MULTICAST_TTL   5

/**
 * Connection types - TCP/UDP
 */
typedef enum _rtx_inet_conn_type {
    RTX_INET_UDP,                  /**< simple UDP socket */
    RTX_INET_UDP_CONNECTED,        /**< connected UDP socket */
    RTX_INET_UDP_MULTICAST,        /**< multicast UDP socket */
    RTX_INET_TCP_CLIENT,           /**< client TCP socket */
    RTX_INET_TCP_SERVER            /**< server TCP socket */
} RtxInetConnType;

/**
 * Connection end-point
 */
typedef struct _rtx_inet_endpoint {
    char ifname[RTX_INET_MAX_HOST_NAME_LEN];  /**< name */
    int portNum;                              /**< port number */
    int addrLen;                              /**< size of OS-specific network address */
    char addr[RTX_INET_MAX_ADDR_LEN];         /**< OS-specific network address */
} RtxInetEndpoint;

/**
 * An internet connection
 */
typedef struct _rtx_inet_conn {
    int sockfd;         /**< socket id */
    int multicastTtl;   /**< TTL value for multicast sockets */
    char hostname[RTX_INET_MAX_HOST_NAME_LEN]; /**< hostname */
    RtxInetEndpoint local;   /**< local endpoint */
    RtxInetEndpoint remote;  /**< remote endpoint */
} RtxInetConn;

/**
 * a TCP/IP client handle (for server)
 *
 * this handle is passed to the user function each time a new
 * client connects to the server.
 */
typedef struct _rtx_inet_tcp_client {
    RtxInetConn * sock;                /**< the socket for client comms */
    RtxThread * thread;                /**< the thread handling client comms */
    int done;                          /**< flag to indicate that the client has finished */
    struct _rtx_inet * inetServer;     /**< the inet handle */
} RtxInetTcpClient;

/**
 * the Inet handle
 */
typedef struct _rtx_inet {
    RtxInetConnType connType;               /**< connection type */
    RtxInetConn * sock;                     /**< comms socket to listen for clients */
    int done;                               /**< flag to indicate server is exiting */
    RtxThread * listenerThread;             /**< thread listening for client conns */
    RtxThread * cleanerThread;              /**< thread listening for client conns */
    RtxSem * cleanerSem;                    /**< semaphore for signalling cleaner thread */
    void * (* userfn) (void *, void *);     /**< user fn called to handle client requests */
    void * (* cleanupfn) (void *, void *);  /**< cleanup fn called when client exits */
    void * userarg;                         /**< argument passed to the user fn */
    RtxList * clients;                      /**< list of all connected clients */
} RtxInet;

#ifndef DOXYGEN_SHOULD_SKIP_THIS

RtxInet * rtx_inet_init (RtxInetConnType connType, 
			 const char * localHost, int localPort, 
			 const char * remoteHost, int remotePort, 
			 void * (* server) (void *, void *),
			 void * (* cleanup) (void *, void *),
			 void * arg);
int rtx_inet_done (RtxInet * p);

int rtx_inet_wait_data (RtxInetConn * p, double timeout);

int rtx_inet_read (RtxInetConn * p, char * buf, int n, 
		   RtxInetEndpoint * from);
int rtx_inet_readline (RtxInetConn * p, char * buf, int n, 
		       RtxInetEndpoint * from);
int rtx_inet_write (RtxInetConn * p, const char * buf, int n, 
		    RtxInetEndpoint * to);

int rtx_inet_set_loopback ( RtxInet * p, int loopback);
int rtx_inet_set_multicast_ttl (RtxInet * p, int ttl);
char * rtx_inet_print_net_addr (RtxInetEndpoint * p);
int rtx_inet_print_net_portnum (RtxInetEndpoint * p);
int rtx_inet_get_net_addr (const char * hostname, int portnum,
                           RtxInetEndpoint * p);

#endif /* DOXYGEN_SHOULD_SKIP_THIS */

RTX_CXX_CLOSE

#endif /* RTX_INET_H */


