/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 *
 * $Id: httpd.h 2296 2008-01-03 05:54:51Z roy029 $
 *
 ***********************************************************************/

#ifndef RTX_HTTPD_H
#define RTX_HTTPD_H

#include <rtx/defines.h>
#include <rtx/inet.h>
#include <rtx/hash.h>

RTX_CXX_OPEN


/** HTTP methods */
typedef enum {
    RTX_HTTPD_M_OPTIONS=0,
    RTX_HTTPD_M_GET,
    RTX_HTTPD_M_HEAD,
    RTX_HTTPD_M_PUT,
    RTX_HTTPD_M_POST,
    RTX_HTTPD_M_DELETE,
    RTX_HTTPD_M_TRACE,
    RTX_HTTPD_M_CONNECT
} RtxHttpdMethod;

/** Contains the information about this request */
typedef struct 
{
    RtxInetConn * sock;              /**< connection */
	int persistent;                  /**< do we want the client to close the conection on reception */
    RtxHttpdMethod method;           /**< the method to use */
    char * url;                      /**< url, % codes decoded */
    rtx_hash_t headers;               /**< headers map */
    rtx_hash_t args;                  /**< arguments map, from the URL in a get */
    rtx_hash_t post;                  /**< arguments map, from the content of a post */
    int http_major;                  /**< http major number */
    int http_minor;                  /**< http minor number */
    long int content_length;         /**< if this request contains data, this is 
									   its length. You have to read the data yourself 
									   with rtx_inet_read* functions. */
	/* private data, used to bufferize the transaction */
	unsigned long outputsize,outputused;
	char * outputbuf;
	void * user_data;                /**< whatever data has been registered in the server */
} RtxHttpdReq;


typedef struct _httpd_server
{
    int port;         /* port to listen on to */
    int debug;        /* sets the debuglevel, 0 for quiet */
	int done;         /* private, set to 1 on termination */
    int (*handler)(RtxHttpdReq *r); /* request handler */
	RtxInet * server; /* underlying inet server */
	void * user_data; /* some user data attached to the server */
} RtxHttpd;


/** Initialize Server
 *  Opens the socket, sets up data structures
 *  @param localhost do we need to be able to specify the interface, can be
 *  NULL for any.
 *  @param localhost   which interface to use, NULL for any 
 *  @param port   TCP port to use, 80 if zero.
 *  @param debug  Debug level
 *  @param handler Request handler, must be non null for the server to be
 *  usefull
 *  @return NULL on error
 */
RtxHttpd* rtx_httpd_server_init(const char * localhost, 
		int port, int debug, int (*handler)(RtxHttpdReq *r));


/**
 * Put data in the user_data field of the server. This data, will be availabe
 * subsequently in the RtxHttpdReq passed in the handler function
 * */
int rtx_httpd_server_set_user_data(RtxHttpd *s, void *data);


/** Destroys the server.
 *  Kills all connection threads (breaks currently active clients).
 *  and frees the allocated memory
 *  @param s Server struct
 *  @return 0 if success, negative on error
 */
int rtx_httpd_server_destroy(RtxHttpd *s);

/** Send status response line
 *  This must be called before adding any headers.
 *  @param r request
 *  @param status the http status code (200 for success)
 *  @param text more detailed description of an error, can be NULL
 *  @return 0 if success, negative on error
 */
int rtx_httpd_reply_send_status(RtxHttpdReq *req, 
		int status, const char * text);

/** Send header response line, must be called after send_status
 *  @param req request
 *  @param header header name to add (e.g. Content-Type)
 *  @param value header value to add (e.g. text/html)
 *  @return 0 if success, negative on error
 */
int rtx_httpd_reply_send_header(RtxHttpdReq *req, 
		const char * header, const char * value);

/** Send the collection of headers on the socket. This must not be 
 *  used in conjunction with send_body. After sending the headers, you're
 *  responsible for sending data on the socket. The "Content-Length" and
 *  "Connection" headers are NOT added to the reply, but an empty line is
 *  added. This means the body of the reply is the only thing you can send 
 *  on the socket after this call.
 *  @param req request
 *  @return 0 if success, negative on error
 */
int rtx_httpd_reply_flush_headers(RtxHttpdReq *req);



/** Send a response to the network.
 *  Must be called after sending status and, if necessary, headers
 *  which typically looks like this
 *     1. rtx_httpd_reply_send_status(req, 200, NULL);
 *     2. rtx_httpd_reply_send_header(req, "Content-Type", "text/html");
 *     3. rtx_httpd_reply_send_body(req, body, sizeof(body));
 *
 * @param req   request struct
 * @param body  body of the http message
 * @parma len   length of body
 * @return the number of bytes successfully written, -1 on error.
 */
unsigned int rtx_httpd_reply_send_body(RtxHttpdReq *r,
		const void * body, size_t len);

/** Send a full response to the network, and wait until everything is sent
 *  Must be called after sending status and, if necessary, headers
 *  which typically looks like this
 *     1. rtx_httpd_reply_send_status(req, 200, NULL);
 *     2. rtx_httpd_reply_send_header(req, "Content-Type", "text/html");
 *     3. rtx_httpd_reply_send_body(req, body, sizeof(body));
 *
 * @param req   request struct
 * @param body  body of the http message
 * @parma len   length of body
 * @parma timeout maximum time to wait before giving up, in seconds
 * @return 0 on success
 */
int rtx_httpd_reply_send_allbody(RtxHttpdReq *req,
		const void * body, size_t len, double timeout);

/** Write out an http error.
 *  @param req request struct
 *  @param errorcode http error code to send
 *  @param text Additional Error info (keep short)
 *  @return 0 if success, negative on error
 */
int rtx_httpd_send_error(RtxHttpdReq *r,
		int errorcode, const char * text);

/** Send a file as a reply to request r
 *  All the reply is handled, send_header should not be 
 *  sent before this function
 *  Handles file IO errors directly.
 *  @param req request struct
 *  @param path to the file to be sent
 *  @return 0 if success, negative on error
 */
int rtx_httpd_send_file(RtxHttpdReq *r, const char * path, const char * mimetype);

/** Default Handler returns error message */
int rtx_httpd_def_handler(RtxHttpdReq * r);


RTX_CXX_CLOSE


#endif /* RTX_HTTPD_H */
