/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 *
 * $Id: hash.h 2990 2008-04-24 04:38:10Z roy029 $
 *
 ***********************************************************************/
/**
 ********************************************************************
 *
 * \file hash.h
 * \brief Hash-table definitions
 * \author Peter Corke
 *
 ********************************************************************
 */

#ifndef RTX_HASH_H
#define RTX_HASH_H

#include <rtx/defines.h>
#include <rtx/error.h>

RTX_CXX_OPEN

/**
 * Default hash table capacity.
 */
#define RTX_HASH_DEFAULT_CAPACITY 32  

/**
 * A hash table.
 *
 * The hash table does not grow in capacity. 
 * All operations are protected by a read/write mutex. 
 */
typedef struct rtx_hash *rtx_hash_t;


/**
 * The set of function hooks required to implement a key type.
 */
typedef struct rtx_hash_key_hooks *rtx_hash_key_hooks_t;
struct rtx_hash_key_hooks {

  /**
   * Function prototype to copy keys.
   *
   * @param key The pointer to the key to copy.
   * @param error Place to store any errors.
   * @returns a copy of the key on success, NULL on failure.
   */
  void *(*copy)(void *key, rtx_error_t error);

  /**
   * Function prototype to free a key.
   *
   * @param key The key to free.
   */  
  void (*free)(void *key);

  /**
   * Function prototype to return a hash value of a key.
   *
   * @param key The key to return the hash value for.
   * @return The hash value of the key.
   */
  unsigned int (*hash)(void *key);

  /**
   * Function prototype to test the equality of keys.
   *
   * @param keyA the first key
   * @param keyB the second key
   * @returns Zero if the keys match, non-zero if they do not match.
   */
  int (*equals)(void *keyA, void *keyB);

  /**
   * Function prototype to push a "key not found" message onto the error.
   */
  void (*error)(void *key, rtx_error_t error);

};


/**
 * The set of function hooks required to implement a value type.
 */
typedef struct rtx_hash_value_hooks *rtx_hash_value_hooks_t;
struct rtx_hash_value_hooks {
  /**
   * Function prototype to copy values.
   *
   * @param value The value to copy.
   * @param error Place to store any errors.
   * @returns a copy of the key on success, NULL on failure.
   */
  void *(*copy)(void *value, rtx_error_t error);

  /**
   * Function prototype to free a value.
   *
   * @param value The value to free.
   */  
  void (*free)(void *value);

};

/**
 * Predefined set of function hooks to implement shallow string keys;
 * the keys are not copied but only referenced.
 */
extern rtx_hash_key_hooks_t rtx_hash_shallow_string_keys;

/**
 * Predefined set of function hooks to implement deep string keys;
 * the keys are copied and freed for the hash table.
 */
extern rtx_hash_key_hooks_t rtx_hash_deep_string_keys;

/**
 * Predefined set of function hooks to implement integer keys.
 */
extern rtx_hash_key_hooks_t rtx_hash_int_keys;

/**
 * Predefined set of function hooks to implement shallow values;
 * the values are not copied but only referenced.
 */
extern rtx_hash_value_hooks_t rtx_hash_shallow_values;

/**
 * Predefined set of function hooks to implement deep string values;
 * the string values are copied for the hash table.
 */
extern rtx_hash_value_hooks_t rtx_hash_deep_string_values;

/**
 * Function prototype for iterators.
 *
 * @param key The key being looked at.
 * @param value The value being looked at.
 * @param i The i'th iteration, starts at 1
 * @param n The last iteration
 * @param stop_iter Set to 1 to stop the iteration
 * @param context The user context.
 */
typedef void (*rtx_hash_iterator_f_t)(void *key,
				      void *value,
				      int i, int n,
				      int *stop_iter,
				      void *context);

/**
 * Create a new hash table, with generic keys.
 *
 * @param capacity the capacity of the hashtable, RTX_HASH_DEFAULT_CAPACITY if zero is passed.
 * @param key_hooks the function hooks to handle keys.
 * @param value_hooks the function hooks to handle values.
 * @param error Place to store details of any errors that may occur.
 * @return A hashtable pointer, or NULL if error.
 */
rtx_hash_t
rtx_hash_alloc(int capacity, 
	       rtx_hash_key_hooks_t key_hooks, rtx_hash_value_hooks_t value_hooks,
	       rtx_error_t error);

/**
 * Clears an existing hash table of all elements.
 *
 * @param ht the hash table to initialize.
 */
void
rtx_hash_clear(rtx_hash_t hash);


/** 
 * Free a hashtable.
 *
 * All storage associated with the hashtable and the hashtable
 * itself is freed.
 *
 * @param hash Hashtable to free.
 *
 */
void
rtx_hash_free(rtx_hash_t hash);


/**
 * Returns the number of elements in the hash table.
 *
 * @param ht hashtable.
 */
int
rtx_hash_used(rtx_hash_t hash);


/**
 * Find the value associated with the key.
 *
 * Returns a pointer to the stored value. Note that a value of NULL is ambiguous.
 *
 * @param hash A hashtable object.
 * @param key key of the entry to find.
 * @param error Place to store detailed error information.
 * @returns If the key is found in hash, returns the value, otherwise NULL.
 */
void *
rtx_hash_find(rtx_hash_t hash, void *key,
	      rtx_error_t error);

/**
 * Associate the key with value in the hash.
 *
 * @param hash A hashtable object.
 * @param key key to set value of.
 * @param value value pointer associated with entry.
 * @param error Place to store details of any errors that may occur.
 * @return -1 on error, 0 on success
 */
int
rtx_hash_set(rtx_hash_t hash, void *key, void *value, rtx_error_t error);


/**
 * Remove the key from the hash.
 *
 * @param hash A hashtable object.
 * @param key key of the entry to remove.
 * @param error Place to store details of any errors that may occur.
 * @return -1 if the key wasn't found, 0 on success
 */
int
rtx_hash_delete(rtx_hash_t hash, void *key, rtx_error_t error);


/**
 * Iterate through a hash table.
 *
 * @param hash the hashtable to iterate through
 * @param iterator function pointer of the iterate function.
 * @param context the user context passed to the iterate function.
 */
void
rtx_hash_iter(rtx_hash_t hash,  rtx_hash_iterator_f_t iterator, void *context);


RTX_CXX_CLOSE

#endif /* RTX_HASH_H */
