/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 *
 * $Id: getopt.h 3067 2008-05-15 03:41:30Z roy029 $
 *
 ***********************************************************************/
/**
 ********************************************************************
 *
 * \file getopt.h
 * \brief parsing command-line options
 * \author Peter Corke 
 * \author Pavan Sikka
 *
 ********************************************************************
 */

#ifndef RTX_GETOPT_H
#define RTX_GETOPT_H

#include <rtx/defines.h>

RTX_CXX_OPEN

/**
 * macro to call the rtx_getopt() function with build info
 */
#define RTX_GETOPT_CMD(opt, argc, argv, version, desc) \
    rtx_getopt_cmd (opt, argc, argv, __DATE__, __TIME__, version, desc)

/**
 * macro to call the rtx_getopt_print() function with build info
 */
#define RTX_GETOPT_PRINT(opt, prog, version, desc) \
    rtx_getopt_print (opt, prog, __DATE__, __TIME__, version, desc)

/**
 * macro to call the rtx_getopt_print_version() function with build info
 */
#define RTX_GETOPT_PRINT_VERSION(opt, prog, version, desc) \
    rtx_getopt_print_version (opt, prog, __DATE__, __TIME__, version, desc)

/**
 * Maximum number of arguments to an option
 */
#define RTX_GETOPT_MAX_NUM_OPTARGS     10

/**
 * RtxGetopt option argument types
 */
typedef enum _rtx_getopt_type {
    RTX_GETOPT_TYPE_NONE = 0,    /**< invalid */
    RTX_GETOPT_TYPE_CHAR,        /**< char */
    RTX_GETOPT_TYPE_INT,         /**< int */
    RTX_GETOPT_TYPE_ENUM,        /**< enum */
    RTX_GETOPT_TYPE_DOUBLE,      /**< double */
    RTX_GETOPT_TYPE_STR,         /**< char * */
    RTX_GETOPT_TYPE_SET,         /**< int, set to 1 */
    RTX_GETOPT_TYPE_INCR,        /**< int, add 1 */
    RTX_GETOPT_TYPE_CLEAR,       /**< int, set to 0 */
    RTX_GETOPT_TYPE_STRCAT,      /**< char **, concatenate multiple invocations */
    RTX_GETOPT_TYPE_FUNC         /**< function callback */
} RtxGetoptType;

/**
 * type names
 */
#define RTX_GETOPT_NONE     "none"
#define RTX_GETOPT_CHAR     "char"
#define RTX_GETOPT_INT      "int"
#define RTX_GETOPT_ENUM     "enum"
#define RTX_GETOPT_DOUBLE   "double"  
#define RTX_GETOPT_STR      "string"
#define RTX_GETOPT_SET      "set"
#define RTX_GETOPT_INCR     "increment"
#define RTX_GETOPT_CLEAR    "clear"
#define RTX_GETOPT_STRCAT   "strcat"
#define RTX_GETOPT_FUNC     "function"

/**
 * end the arg list for an option in the RtxGetopt struct
 */
#define RTX_GETOPT_END_ARG             {RTX_GETOPT_NONE, NULL, NULL}

/**
 * end the option list for the RtxGetopt struct
 */
#define RTX_GETOPT_END                 {NULL, NULL, {RTX_GETOPT_END_ARG}}

/**
 * option arg structure
 */
typedef struct _rtx_getopt_arg {
    const char * type;      /**< arg type */
    void * ptr;       /**< arg pointer */
    const char * name;      /**< arg name */
} RtxGetoptArg;
  
/**
 * Option structure
 * 
 * An application will typically declare an initialized array
 * of this type with one element per option
 */
typedef struct _rtx_getopt {
    const char * name;          /**< option name */
    const char * help;          /**< option help */
    RtxGetoptArg arg[RTX_GETOPT_MAX_NUM_OPTARGS];   /**< option args */
	unsigned int min_cmp_len; /**< the minimum number of char to compare the string with to avoid ambiguities */
} RtxGetopt;

#ifndef DOXYGEN_SHOULD_SKIP_THIS

extern int rtx_getopt_debug;

int rtx_getopt (RtxGetopt * opt, int argc, char * argv[],
		const char * date, const char * time, const char * version, 
		const char * desc, int * argFlag);
int rtx_getopt_cmd (RtxGetopt * opt, int argc, char * argv[],
		    const char * date, const char * time, const char * version, const char * desc);
void rtx_getopt_print (RtxGetopt * opt, const char * progName, const char * date, 
		       const char * time, const char * version, const char * desc);
void rtx_getopt_print_version (RtxGetopt * opt, char * progName, char * date, 
			       char * time, char * version, char * desc);
int rtx_getopt_get_num_strings (char ** p);
void rtx_getopt_disable_config (void);

  /* Functions to set/get values of built-in options.
   */
int rtx_getopt_get_verbose (int defaultVal);
int rtx_getopt_get_store (int defaultVal);
char * rtx_getopt_get_debug (char * defaultVal);
int rtx_getopt_get_priority (int defaultVal);
int rtx_getopt_get_export (int defaultVal);
double rtx_getopt_get_sample (double defaultVal);
char * rtx_getopt_get_config (char * defaultVal);
char ** rtx_getopt_get_D (char ** defaultVal);

  /* These functions should not be used; they may be removed at
   * any time.
   */
int rtx_getopt_get_int (char * name);
double rtx_getopt_get_double (char * name);
char * rtx_getopt_get_str (char * name);
char ** rtx_getopt_get_strcat (char * name);
int rtx_getopt_set_int (char * name, int val);
int rtx_getopt_set_double (char * name, double val);
int rtx_getopt_set_str (char * name, char * val);
int rtx_getopt_set_strcat (char * name, char ** val);

#endif /* DOXYGEN_SHOULD_SKIP_THIS */

RTX_CXX_CLOSE

#endif /* RTX_GETOPT_H */


