/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 *
 * $Id: filter.h 2287 2007-12-24 04:11:30Z roy029 $
 *
 ***********************************************************************/
/**
 ********************************************************************
 *
 * \file filter.h
 * \brief General purpose digital filter package
 * \author Peter Corke
 *
 ********************************************************************
 */

#ifndef RTX_FILTER_H
#define RTX_FILTER_H

#include <rtx/defines.h>

RTX_CXX_OPEN

typedef enum _rtx_filter_type {
        RtxFilterLinear,
        RtxFilterRateLimit,
        RtxFilterMaximum,
        RtxFilterMinimum,
} RtxFilterType;

typedef struct {
	RtxFilterType	type;	/*!< filter type */
	int		nn;	/*!< number of numerator coefficients */
	int		nd;	/*!< number of denominator coefficients */
	double		*N;	/*!< numerator coefficients */
	double		*D;	/*!< denominator coefficients */
	double		*u;	/*!< filter input history */
	double		*y;	/*!< filter output history */
	double		uprev;	/*!< last input value */
	double		ratelimit;	/*!< rate limit */
} RtxFilter;

RtxFilter * rtx_filter_init(int nn, int nd, ...);
RtxFilter * rtx_filter_init_from_file(char *filename);
RtxFilter * rtx_filter_nonlinear_init(RtxFilterType type, ...);
RtxFilter * rtx_filter_ratelimiter_init(double ratelimit);

void rtx_filter_done(RtxFilter *f);
double rtx_filter_step(RtxFilter *f, double u);
void rtx_filter_print(RtxFilter *f);
void rtx_filter_reset(RtxFilter *f);
void rtx_filter_reset2(RtxFilter *f, double y);
double rtx_filter_lasty(RtxFilter *f);
RtxFilter * rtx_filter_smooth1(double p);
RtxFilter * rtx_filter_smoothdiff(double dt, double p);

RtxFilter * rtx_filter_median_init(int winSize);
double rtx_filter_median_step(RtxFilter *f, double u);

/* released the memory allocated by the above function
 * f MUST be a malloc'ed pointer */
void rtx_filter_destroy(RtxFilter *f);

RTX_CXX_CLOSE

#endif /* RTX_FILTER_H */
