/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 *
 * $Id: export.h 2287 2007-12-24 04:11:30Z roy029 $
 *
 ***********************************************************************/
/**
 ********************************************************************
 *
 * \file export.h
 * \brief Data export module based on RTC
 * \author Peter Corke
 *
 * An application package layered over RTC. Allows variables inside a
 * multi-threaded program to be read/written by a tcl/tk client.
 *
 ********************************************************************
 */

#ifndef RTX_EXPORT_H
#define RTX_EXPORT_H

#include <string.h>

#include <rtx/defines.h>
#include <rtx/srpc.h>
#include <rtx/thread.h>

RTX_CXX_OPEN

/**
 * Main interface to exportvar package are the export macros:
 *   RTX_EXPORT(variable, type) where type is one of INT, DOUBLE,
 *      STRING, possibly * or'd with ARRAY(maxel) if its a vector
 *   RTX_EXPORT_CALLBACK(variable, type, func) is the same, but
 *      declares an optional callback function to be invoked each time
 *      the variable is read/written.  The function is of type
 *      RtxExportCallback.
 */

#define	RTX_EXPORT(name, type) \
    rtx_export_var_register(#name, &(name), type)
#define	RTX_EXPORTVEC(name, type, len) \
    rtx_export_var_register(#name, &(name), type | RTX_EXPORT_ARRAY(len))
#define	RTX_EXPORT2(name, var, type) \
    rtx_export_var_register(#name, &(var), type)
#define	RTX_EXPORT_CALLBACKFN(name, type, func) \
    rtx_export_var_register(#name, &(name), (type)|RTX_EXPORT_CALLBACK, func)

#define	RTX_EXPORT_CHAR			0x1        /**< char type */
#define	RTX_EXPORT_SHORT		0x2        /**< short type */
#define	RTX_EXPORT_INT			0x3        /**< int type */
#define	RTX_EXPORT_LONG		        0x4        /**< long type */
#define	RTX_EXPORT_FLOAT		0x5        /**< float type */
#define	RTX_EXPORT_DOUBLE		0x6        /**< double type */
#define	RTX_EXPORT_STRING		0x7        /**< string type */
#define	RTX_EXPORT_ARRAY(l)		((l<<16))
#define	RTX_EXPORT_READONLY		0x8
#define	RTX_EXPORT_POLL			0x10
#define	RTX_EXPORT_CALLBACK		0x20
#define	RTX_EXPORT_SPAN			0x40
#define	RTX_EXPORT_USERDATA		0x80

#define	RTX_EXPORT_TYPE_MASK		0x7
#define	RTX_EXPORT_ARRAY_LENGTH(v)	(v>>16)

enum RtxExportOp {
	RTX_EXPORT_READ,
	RTX_EXPORT_WRITE
};


/*
 * Callback function. Is invoked just prior to reading, or just after writing
 * a new value.  The type of operation is given by exportOp.
 */
typedef struct _rtx_export_sym RtxExportSym;
typedef  int (* RtxExportCallback) (RtxExportSym *s, int indx,
				      enum RtxExportOp op);
typedef  int (* RtxExportGlobalCallback) (RtxExportSym *s, int indx,
				      char *newval);
typedef  double (* RtxExportVirtualCallback) (RtxExportSym *s, void *arg);

/*
 * Symbol table data structure, holds info about variable and pointer to it
 */
struct _rtx_export_sym {
	struct _rtx_export_sym	*s_next;
	char			*s_name;
	int			type;
	unsigned int		dimension;
	union {
		void	*p;
		char	*cp;
		short	*sp;
		int	*ip;
		long	*lp;
		float	*fp;
		double	*dp;
		char	**s;
	}			p;
	RtxExportCallback		callback;
	union {
		double	d;
		int	i;
	} min;
	union {
		double	d;
		int	i;
	} max;
	union {
		double	d;
		int	i;
	} value;
	void		*userdata;
	RtxExportVirtualCallback		virtual_callback;
	void		*callback_arg;
	double		scale, offset;
	char		*units;
};


#define	RTX_HASH_SIZE	32

extern int	rtxExportDebug;

/*
 * Exports
 */
int	rtx_export_init (void);
int	rtx_export_done (void);
int	rtx_export_register (void);
int	rtx_export_var_register (char *name, void *p, int type, ...);
RtxThread *rtx_export_server_start(int prio, int rtcNumber, char *transport, char *authfile);
int	rtx_export_server_stop(RtxThread *thread);
RtxSrpc * rtx_export_get_srpc_handle (void);

/* new API */
int rtx_export_int(char *name, int *p, int len, int flag, char *units);
int rtx_export_string(char *name, char *p, int len, int flag);
int rtx_export_double(char *name, double *p, int len, int flag, char *units);
int rtx_export_float(char *name, float *p, int len, int flag, char *units);
int rtx_export_virtual(char *name, RtxExportVirtualCallback func, void *arg, int flag, char *units);
int rtx_export_double_linear(char *name, double *p, int len, int flag, char *units, double scale, double offset);
int rtx_export_float_linear(char *name, float *p, int len, int flag, char *units, double scale, double offset);
int rtx_export_double_degrees(char *name, double *p, int len, int flag);
int rtx_export_float_degrees(char *name, float *p, int len, int flag);

/* xmlrpc part */
RtxThread *rtx_export_xmlrpc_server_start(int prio,unsigned int port, char * cfgdir);
int rtx_export_xmlrpc_server_stop(RtxThread *thread);

RTX_CXX_CLOSE

#endif /* RTX_EXPORT_H */
