/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 *
 * $Id: error.h 3851 2008-08-25 02:38:40Z roy029 $
 *
 ***********************************************************************/

#ifndef RTX_ERROR_H
#define RTX_ERROR_H

#include <errno.h>
#include <stddef.h>

#include <rtx/defines.h>

RTX_CXX_OPEN

/**
 * Error handling structure.
 *
 * The rtx_error_t structure stores a stack of errors, similar to an
 * exception in other languages.
 */
typedef struct rtx_error *rtx_error_t;


/**
 * Create an error structure.
 *
 * This is usually the first thing done in a program.
 *
 * @returns The created error structure, or NULL on failure.
 */
rtx_error_t rtx_error_alloc(void) RTX_WARN_UNUSED_RESULT;


/**
 * Free an error structure
 *
 * Usually the last thing done in a program.
 *
 * @param error The error structure to be freed.
 */
void 
rtx_error_free(rtx_error_t error);


/**
 * Clear an error structure.
 *
 * Call this after an error has been dealt with.
 *
 * @param error The error structure to be cleared.
 */
void 
rtx_error_clear(rtx_error_t error);


/**
 * Return a textual form of the error.
 *
 * The returned result will be valid for the life of the error string.
 *
 * @returns The textual form of the error.
 */
char *
rtx_error_str(rtx_error_t error);


/**
 * Logs the error message and clears it.
 *
 * By default rtx_error_log will display the error message on
 * standard error. 
 *
 * If the environment variable RTX_ERROR is set to syslog, then
 * rtx_error_log will log the error to syslog. If RTX_ERROR is set to
 * message, then the message will be sent to an rtx message daemon.
 *
 * @param error The error to flush.
 */
void
rtx_error_log(rtx_error_t error);


/**
 * Pushes an error onto the error stack.
 *
 * Designed to be used via the RTX_ERROR_PUSH* macros.  Formatted
 * messages will be copied into private storage of the error context,
 * no memory will be allocated. If the stack has run out of room,
 * silently continues (thus not overriding the root cause).
 *
 * @param error The error structure to store the error in.
 * @param func The calling function.
 * @param file The source file of the calling function.
 * @param line The line number of the file of the calling function.
 * @param format The format of the error string.
 * @param ... The arguments that match the error string.
 */
void 
rtx_error_push(rtx_error_t error, 
	       const char *function, const char *file, size_t lineno,
	       const char *format, ...)

  RTX_FORMAT(printf, 5, 6);


/**
 * Pushes a custom message onto the error stack.
 *
 * Use this macro when giving a specific reason for failure.
 */
#define RTX_ERROR_PUSH_CUSTOM(error, format, args...) \
  rtx_error_push(error, __func__, __FILE__, __LINE__, format, ##args);

/**
 * Pushes a default failure message onto the error stack.
 *
 * Use this to build up the error stack when a called function has failed.
 */
#define RTX_ERROR_PUSH_DEFAULT(error) \
  rtx_error_push(error, __func__, __FILE__, __LINE__, "failed because of previous errors.");

/**
 * Pushes an error message about a failed system call, using errno.
 */
#define RTX_ERROR_PUSH_SYSCALL(error, function) \
  rtx_error_push(error, __func__, __FILE__, __LINE__, "%s() failed(%d) - %s", function, errno, strerror(errno));
  

#ifndef DOXYGEN_SHOULD_SKIP_THIS

#define	RTX_ERROR_EXIT_SHIFT    16
#define	RTX_ERROR_EXIT(v)	(4096 | (v<<RTX_ERROR_EXIT_SHIFT))

/**
 * Specify the destination for error messages to be output to.
 */
typedef enum _rtx_error_dest {
    RTX_ERROR_STDERR = 1,      /**< standard error */
    RTX_ERROR_STDOUT = 2,      /**< standard output */
    RTX_ERROR_MESSAGE = 4,     /**< message daemon */
    RTX_ERROR_SYSLOG = 8       /**< syslog daemon */
} RtxErrorDestination;


/**
 * flags that control the output of error messages
 */
typedef enum _rtx_error_flag {
    RTX_ERROR_OCCURS = 128,      /**< print error messages immediately */
    RTX_ERROR_DEBUG = 256        /**< print debugging info */
} RtxErrorFlag;


int	rtx_error_init(const char *name, int flags, void (*handler)(void));

int	rtx_error_traceback(void);
int     rtx_error_traceback_depth(int depth);

int	rtx_error_fn(const char *fname, int lineno, const char *fmt, ...);
int	rtx_error_errno_fn(const char *fname, int lineno, const char *fmt, ...);
int     rtx_error_errno2_fn(const char *fname, int lineno, int errnum, const char *fmt, ...);
void *	rtx_error_null_fn(const char *fname, int lineno, const char *fmt, ...);
void *	rtx_error_errno_null_fn(const char *fname, int lineno, const char *fmt, ...);

void	rtx_error_flush_fn(const char *fname, int lineno, const char *fmt, ...);
void	rtx_error_flush_depth_fn(const char *fname, int lineno, int depth, const char *fmt, ...);
void	rtx_error_errno_flush_fn(const char *fname, int lineno, const char *fmt, ...);
void	rtx_error_errno_flush_depth_fn(const char *fname, int lineno, int depth, 
				       const char *fmt, ...);

struct _rtx_err * rtx_error_status_get(void);
int	rtx_error_flags_get(void);

/*
 * The following macros use the older style vararg macros supported by GNU C
 * to accomodate the LynxOS 3.1 compiler. Once this constraint is gone, we should
 * implement these using the newer style.
 * ref: http://gcc.gnu.org/onlinedocs/cpp/Variadic-Macros.html
 */
#include <stdarg.h>
#define	rtx_error(fmt, args...)\
        rtx_error_fn(__FILE__, __LINE__, fmt , ## args)
#define	rtx_error_errno(fmt, args...)\
        rtx_error_errno_fn(__FILE__, __LINE__, fmt , ## args)
#define	rtx_error_errno2(errnum, fmt, args...)\
        rtx_error_errno2_fn(__FILE__, __LINE__, errnum, fmt , ## args)
#define	rtx_error_null(fmt, args...)\
        rtx_error_null_fn(__FILE__, __LINE__, fmt , ## args)
#define	rtx_error_errno_null(fmt, args...)\
        rtx_error_errno_null_fn(__FILE__, __LINE__, fmt , ## args)
#define	rtx_error_flush(fmt, args...)\
        rtx_error_flush_fn(__FILE__, __LINE__, fmt , ## args)
#define	rtx_error_flush_depth(depth, fmt , args...)\
        rtx_error_flush_fn(__FILE__, __LINE__, depth, fmt , ## args)
#define	rtx_error_errno_flush(fmt, args...)\
        rtx_error_errno_flush_fn(__FILE__, __LINE__, fmt , ## args)
#define	rtx_error_errno_flush_depth(fmt, args...)\
        rtx_error_errno_flush_fn(__FILE__, __LINE__, depth, fmt , ## args)

#endif /* Doxygen */

RTX_CXX_CLOSE

#endif /* RTX_ERROR_H */
