/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <rtx/serial.h>
#include <rtx/time.h>
#include <rtx/timer.h>
#include <rtx/getopt.h>

#include "svnversion.h"

#define DESCRIPTION "Writes a test pattern to the chosen serial device."

static char rcsid[] RTX_UNUSED = "$Id: serialwrite.c 2299 2008-01-05 13:03:22Z roy029 $";

int	baudRate = 38400;
char	*device = "/dev/ttyS0";
int	verbose = 0;
int	pktSize = 510;
double	delay = 0.5;
int	noBytes = 0;
int	flowOff = 1;

RtxGetopt myOpts[] = {
	{"flowOff", "Turn hardware flow control off (default is on)", 
		{
			{RTX_GETOPT_SET, &flowOff, ""},
			RTX_GETOPT_END_ARG
		}
	},
	{"noBytes", "Total number of bytes transmitted (bytes)", 
		{
			{RTX_GETOPT_INT, &noBytes, "number of bytes"},
			RTX_GETOPT_END_ARG
		}
	},
	{"delay", "Inter-packet delay (seconds)", 
		{
			{RTX_GETOPT_DOUBLE, &delay, "delay"},
			RTX_GETOPT_END_ARG
		}
	},
	{"pktSize", "Packet size (bytes)", 
		{
			{RTX_GETOPT_INT, &pktSize, "number of bytes"},
			RTX_GETOPT_END_ARG
		}
	},
	{"baudrate", "Serial baud rate", 
		{
			{RTX_GETOPT_INT, &baudRate, "baud rate"},
			RTX_GETOPT_END_ARG
		}
	},
	{"device", "Serial device", 
		{
			{RTX_GETOPT_STR, &device, "device"},
			RTX_GETOPT_END_ARG
		}
	},
	RTX_GETOPT_END
};

int
main(int argc, char **argv)
{
	unsigned char	buf[1020];
	int		baudRate = 38400;
	int		fd;
	int		i, j, n, offset = 0, count = 0;
	RtxTime		tPrev, tNow;
	double		dt;
	RtxSerialFlow	flow;

	RTX_GETOPT_CMD (myOpts, argc, argv, SVNVERSION, DESCRIPTION);
	verbose = rtx_getopt_get_verbose(0);

	if ((pktSize > 510) || (pktSize < 1)) {
		fprintf(stderr, "pktSize must be between 1 and 510\n");
		exit(1);
	}

	/* Create triangle-wave data 0 to 255 to 1 (510 elements) */
	for (i=0; i<256; i++)
		buf[i] = i;
	for (i=256,j=254; i<510; i++,j--)
		buf[i] = j;
	/* Double the buffer with another triangle-wave */
	for (i=510,j=0; i<1020; i++,j++)
		buf[i] = buf[j];

	/* Open serial port */
	fprintf(stderr, "Opening serial port %s...", device);
	if (flowOff)
		flow = RTX_SERIAL_FLOW_NONE;
	else
		flow = RTX_SERIAL_FLOW_HW;
	if ((fd = rtx_serial_open(device, baudRate, 8, 1,
			RTX_SERIAL_PARITY_NONE, flow,
			RTX_SERIAL_MODEM_OFF, 0, 1)) == -1) {
		fprintf(stderr, "rtx_serial_open() failed on %s\n", device);
		exit(1);
	}
	rtx_serial_flush_input(fd);
	rtx_serial_flush_output(fd);
	fprintf(stderr, "done\n");

	while (1) {
		if (noBytes && ((noBytes-count) < pktSize))
			pktSize = noBytes-count;

		/* Write bytes */
		if ((n = write(fd, (char*)&buf[offset], pktSize)) != pktSize)
			fprintf(stderr, "write() did not write the correct number of characters (%d instead of %d)\n", n, pktSize);
		rtx_timer_sleep(pktSize*10.0/baudRate+delay);

		offset += pktSize;
		if (offset >= 510)
			offset -= 510;
		
		count += pktSize;	
		if (noBytes && (count >= noBytes)) {
			fprintf(stderr, "written %d bytes\n", count);
			exit(1);
		}
		if (verbose) {
			/* Timing */
			rtx_time_get(&tNow);
			dt = rtx_time_subtract_to_double (&tNow, &tPrev);
			tPrev = tNow;
			if (dt > 0.0000) {	
				fprintf(stdout, "%f ", dt);
				fflush(stdout);
			}
		}
	}

	exit(0);
}
