/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <rtx/serial.h>
#include <rtx/timer.h>
#include <rtx/getopt.h>
#include <rtx/thread.h>
#include <rtx/main.h>

#include "svnversion.h"

#define DESCRIPTION "Reads data from the specified serial device and displays it in a variety of forms."

static char rcsid[] RTX_UNUSED = "$Id: serialspy.c 2299 2008-01-05 13:03:22Z roy029 $";

int		ascii = 0;
int		decimal = 0;
int		baudRate = 38400;
char		*device = "/dev/ttyS0";
RtxThread	*threadId;
unsigned char	buf[12];
int		fd;
int		noBytes = 0;
int		flowOff = 1;

RtxGetopt myOpts[] = {
	{"flowOff", "Turn hardware flow control off (default is on)", 
		{
			{RTX_GETOPT_SET, &flowOff, ""},
			RTX_GETOPT_END_ARG
		}
	},
	{"ascii", "Print output as ascii (default is hex)", 
		{
			{RTX_GETOPT_SET, &ascii, ""},
			RTX_GETOPT_END_ARG
		}
	},
	{"decimal", "Print output as decimal (default is hex)", 
		{
			{RTX_GETOPT_SET, &decimal, ""},
			RTX_GETOPT_END_ARG
		}
	},
	{"baudrate", "Serial baud rate", 
		{
			{RTX_GETOPT_INT, &baudRate, "baud rate"},
			RTX_GETOPT_END_ARG
		}
	},
	{"device", "Serial device", 
		{
			{RTX_GETOPT_STR, &device, "device"},
			RTX_GETOPT_END_ARG
		}
	},
	RTX_GETOPT_END
};

void
reading_thread()
{
	int	nread;

	rtx_serial_flush_input(fd);
	rtx_serial_flush_output(fd);

	fprintf(stderr, "\n");
	while (1) {
		/* Read byte */
		if ((nread = rtx_serial_read(fd, (char*)buf, 1)) != 1) {
			continue;
		} else {
			noBytes++;

			if (ascii)
				fprintf(stdout, "%c", buf[0]);
			else if (decimal)
				fprintf(stdout, "%03d ", (int)buf[0]);
			else
				fprintf(stdout, "0x%02x ", buf[0]);
			fflush(stdout);
		}
		fprintf(stderr, "\rnoBytes = %d", noBytes);
	}
}

int
main(int argc, char **argv)
{
	RtxSerialFlow	flow;

	RTX_GETOPT_CMD(myOpts, argc, argv, SVNVERSION, DESCRIPTION);

	/* Open serial port */
	fprintf(stderr, "Opening serial port %s...", device);
	if (flowOff)
		flow = RTX_SERIAL_FLOW_NONE;
	else
		flow = RTX_SERIAL_FLOW_HW;
	if ((fd = rtx_serial_open(device, baudRate, 8, 1,
			RTX_SERIAL_PARITY_NONE, flow,
			RTX_SERIAL_MODEM_OFF, 0, 1)) == -1) {
		fprintf(stderr, "rtx_serial_open() failed on %s\n", device);
		exit(1);
	}
	fprintf(stderr, "done\n");

	/* Launch reading thread */
	if ((threadId = rtx_thread_create(
			"reading_thread", 0,
			RTX_THREAD_SCHED_OTHER, 21, 0,
			RTX_THREAD_CANCEL_ASYNCHRONOUS,
			(void * (*)(void*))reading_thread, NULL,
			NULL, NULL)) == NULL) {
		fprintf(stderr, "rtx_thread_create() failed\n");
		exit(1);
	}

	/* Wait to be killed */
	rtx_main_wait_shutdown(90);

	/* Cleanup */
	rtx_thread_destroy(threadId);
	rtx_timer_sleep(0.2);
	rtx_serial_close(fd);

	fprintf(stderr, "read %d bytes (loss = %f)\n", noBytes, 100.0-100.0*noBytes/10000);
	exit(0);
}
