/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "rtx/main.h"
#include "rtx/message.h"
#include "rtx/error.h"
#include "rtx/signal.h"
#include "rtx/srpc.h"
#include "rtx/getopt.h"
#include "rtx/timer.h"

#include "svnversion.h"

#define DESCRIPTION "Process controller, ensures process is running by a heartbeat, restarts if it crashes."

static char rcsid[] RTX_UNUSED = "$Id: procctrl.c 2299 2008-01-05 13:03:22Z roy029 $";

static enum {STOPPED,RUNNING} g_status = STOPPED;
static int g_end = 0;
static int g_pid = -1;
static int g_argc = -1;
static char** g_argv = NULL;
RtxSrpc * srpc = NULL;

int dofork(int argc, char * argv[])
{
	int pid = fork();
	if (pid==0) {
		int i;
		char **rargv = (char**)malloc((argc+1)*sizeof(char*));
		sigset_t sset;
		for (i=0;i<argc;i++) {
			rargv[i] = argv[i];
		}
		rargv[argc] = NULL;
		if (srpc) {
			rtx_srpc_done(srpc);
		}
		sigemptyset(&sset);
		sigprocmask(SIG_SETMASK,&sset,NULL);
		//rtx_message("Running '%s'",argv[0]);
		execvp(argv[0],rargv);
		return rtx_error("Failed to fork child process");
	}
	//rtx_message("Successfully started '%s'\n",argv[0]);
	return pid;
}
		
void dokill(int pid)
{
	if (pid>0) {
		kill(pid, SIGTERM);
		//fprintf(stderr,"Killing process %d: %d\n",pid,ret);
		waitpid(pid,NULL,0);
		//fprintf(stderr,"Received sigchld %d: %d\n",pid,ret);
		g_status = STOPPED;
		g_pid = -1;
	}
}

int quitProcess(char * buf, int len, int argc, char * argv[])
{
	rtx_message("Killing process '%d'",g_pid);
	dokill(g_pid);
	snprintf(buf,len,"OK");
	g_end += 1;
	return 0;
}

int statusProcess(char * buf, int len, int argc, char * argv[])
{
	switch (g_status) {
		case RUNNING:
			snprintf(buf,len,"RUNNING");
			break;
		case STOPPED:
			snprintf(buf,len,"STOPPED");
			break;
		default:
			snprintf(buf,len,"UNKNOWN");
			break;
	}
	return 0;
}

int startProcess(char * buf, int len, int argc, char * argv[])
{
	g_pid = dofork(g_argc,g_argv);
	g_status = RUNNING;
	snprintf(buf,len,"OK");
	return 0;
}

int stopProcess(char * buf, int len, int argc, char * argv[])
{
	rtx_message("Killing process '%d'",g_pid);
	dokill(g_pid);
	snprintf(buf,len,"OK");
	return 0;
}

int restartProcess(char * buf, int len, int argc, char * argv[])
{
	dokill(g_pid);

	rtx_timer_sleep(1.0);

	g_pid = dofork(g_argc,g_argv);
	g_status = RUNNING;
	snprintf(buf,len,"OK %d",g_pid);
	return 0;
}


void sighdl(int n) 
{
	g_end += 1;
}

void sigchld(int n) 
{
	g_pid = -1;
	g_status = STOPPED;
}




static int verbose = 0;
static int port = 137;

static RtxGetopt myOpts[] = {
	{ "port", "set srpc port number",
		{ { RTX_GETOPT_INT, &port, "port"}, RTX_GETOPT_END_ARG } },
	RTX_GETOPT_END 
};


int main(int argc, char * argv[])
{
	RtxSrpc * srpc;
	int remargs = RTX_GETOPT_CMD( myOpts, argc, argv, NULL, DESCRIPTION);
	if(remargs  == -1 ) {
		RTX_GETOPT_PRINT( myOpts, argv[0], NULL, DESCRIPTION); 
		exit(-1); 
	}
	rtx_message_init("Rtx Process control",RTX_ERROR_STDERR|RTX_ERROR_MESSAGE);
	rtx_error_init("Rtx Process control",RTX_ERROR_STDERR|RTX_ERROR_MESSAGE,NULL);
	verbose = rtx_getopt_get_verbose(0);

	g_argv = argv+remargs;
	g_argc = argc-remargs;

	if (argc<1) {
		RTX_GETOPT_PRINT( myOpts, argv[0], NULL, DESCRIPTION); 
		exit(-1); 
	}

	signal(SIGCHLD,sigchld);
	signal(SIGINT,sighdl);
	srpc = rtx_srpc_init(port);
	if (!srpc) {
		rtx_error_flush("Failed to create srpc control");
		exit(-1);
	}
	rtx_srpc_register(srpc,"stop", stopProcess);
	rtx_srpc_register(srpc,"start", startProcess);
	rtx_srpc_register(srpc,"restart", restartProcess);
	rtx_srpc_register(srpc,"status", statusProcess);
	rtx_srpc_register(srpc,"quit", quitProcess);

	g_pid = dofork(g_argc,g_argv);
	g_status = RUNNING;

	while (!g_end) {
		rtx_timer_sleep(0.1);
	}

	rtx_srpc_done(srpc);

	dokill(g_pid);
	rtx_timer_sleep(1);

	return 0;
}



