/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file messaged.c
 * \brief The message daemon.
 * \author Jon Roberts
 *
 * The message daemon listens for messages coming in on a Posix.1b
 * message Q and then prints them out to file.
 *
 */

#include "config.h"

#include <unistd.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifndef __qnxrtp
#include <termio.h>
#else
#include <strings.h>
#include <termios.h>
#endif
#include <time.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <limits.h>
#include <signal.h>
#ifdef HAVE_MESSAGE_QUEUE
#include <mqueue.h>
#endif
#include <dirent.h>

#include "rtx/defines.h"
#include "rtx/main.h"
#include "rtx/getopt.h"
#include "rtx/command.h"
#include "rtx/error.h"
#include "rtx/timer.h"
#include "rtx/message.h"

#include "messaged-webserver.h"

#include "messaged.h"

static char rcsid[] RTX_UNUSED = "$Id: messaged.c 2274 2007-12-23 05:37:32Z roy029 $";



#define NO_MQS			1000
#define INIT_CHANGE_PERIOD	86400	/* 24 hours (in seconds) */
#define INIT_CHANGE_SIZE        1048576	/* 1 MB (in bytes) */
#define LOG_BUFFER_SIZE 30

#ifndef STDERR
#ifdef STDERR_FILENO
#define STDERR STDERR_FILENO
#else
#define STDERR 2
#endif
#endif

static int end = 0;
static pid_t    pid;

MESSAGED       *messaged;

/*
 * Forward defines
 */
static int      messaged_open();
static int      messaged_close();
static void     messaged_logger_thread();
static int      messaged_log_message(char *logString);
static int      lock_get(char *name);
static int      sortfuncinc(void **ap, void **bp);
static int      sortfuncdec(void **ap, void **bp);
static char     **messaged_get_files (int *nf);
#ifdef HAVE_MESSAGE_QUEUE
mqd_t mq_create(char *name, int nmessages, int msglen, void (*handler)() );
mqd_t mq_connect(char *name);
mqd_t mq_connect2(char *name);
int mq_disconnect(mqd_t mqdes);
int mq_destroy(char *name);
#else
int fifo_create (char *name);
int fifo_done (int fd, char *name);
int fifo_recv (int fd, RtxMessage * msg);
#endif

	static void
mesgd_sighandler (int arg)
{
	fprintf (stderr, "messaged: exiting ...\n");
	if (messaged_close())
		fprintf (stderr, "messaged: messaged_close() failed\n");

	exit(0);
}

static void messageLog(const char * message){

	rtx_csbuffer_push(messaged->log, message);
	logMessage_setTabMessage(messaged->logMess);
	logMessage_xmlConverter(messaged->logMess);
}

static int parse_cmd_line(MESSAGED *msg, int argc, char * argv[]) {
	char * period = "3600";
	char * age = "1m";
	RtxGetopt messageOpts[] = {
		{"changePeriod", "Age at which message file are saved",
			{
				{RTX_GETOPT_STR, &period, "int"},
				RTX_GETOPT_END_ARG
			}
		},
		{"changeSize", "Size at which message file are saved",
			{
				{RTX_GETOPT_STR, &age, "int"},
				RTX_GETOPT_END_ARG
			}
		},
		{"web", "Start a web server to access the message list",
			{
				{RTX_GETOPT_SET, &msg->webFlag, "bool"},
				RTX_GETOPT_END_ARG
			}
		},
		{"wport", "TCP port for the web server",
			{
				{RTX_GETOPT_INT, &msg->webPort, "int"},
				RTX_GETOPT_END_ARG
			}
		},
		{"dir", "Directory in which message file are stored",
			{
				{RTX_GETOPT_STR, &msg->dir, "path"},
				RTX_GETOPT_END_ARG
			}
		},
		{"archive", "Directory in which message file are archived",
			{
				{RTX_GETOPT_STR, &msg->archivedir, "path"},
				RTX_GETOPT_END_ARG
			}
		},
		{"filename", "Name of the message file",
			{
				{RTX_GETOPT_STR, &msg->fileName, "path"},
				RTX_GETOPT_END_ARG
			}
		},
		{"nofork", "Do not detach",
			{
				{RTX_GETOPT_SET, &msg->nofork, "bool"},
				RTX_GETOPT_END_ARG
			}
		},
		{"numfiles", "Maximum number of log files",
			{
				{RTX_GETOPT_INT, &msg->maxNumLogFiles, "int"},
				RTX_GETOPT_END_ARG
			}
		},
		RTX_GETOPT_END
	};
	char * messageHelpStr = "Message daemon";
	int errs,value;
	char unit;
	
	if ((errs = RTX_GETOPT_CMD (messageOpts, argc, argv, NULL, 
					messageHelpStr)) <= 0) {
		if (errs == -1)
			RTX_GETOPT_PRINT (messageOpts, argv[0], NULL, messageHelpStr);
		exit (1);
	}    
	if (msg->maxNumLogFiles < 1) {
		msg->maxNumLogFiles = 1;
		fprintf (stderr, "messaged: max num files < 1, using 1\n");
	}
	unit=0;value = msg->changePeriod;
	sscanf(period,"%d%c",&value,&unit);
	if (value < 0) {
		msg->changePeriod = -1;
	} else {
		switch (unit) {
			case 'm':
				msg->changePeriod = value*60;
				break;
			case 'h':
				msg->changePeriod = value*3600;
				break;
			case 'd':
				msg->changePeriod = value*3600*24;
				break;
			case 's':
			default:
				msg->changePeriod = value;
				break;
		}
	}
	unit=0;value = msg->changeSize;
	sscanf(period,"%d%c",&value,&unit);
	if (value < 0) {
		msg->changeSize = -1;
	} else {
		switch (unit) {
			case 'g':
			case 'G':
				msg->changeSize = value*1024*1024*1024;
				break;
			case 'm':
			case 'M':
				msg->changeSize = value*1024*1024;
				break;
			case 'k':
			case 'K':
				msg->changeSize = value*1024;
				break;
			case 'b':
			case 'B':
			default:
				msg->changeSize = value;
				break;
		}
	}

	return 0;
}


	int
main(int argc, char **argv)
{
	int             i;
	int		    confd;
	extern char    *optarg;
	char            fname[BUFSIZ * 2 + 1];

	signal (SIGTSTP, mesgd_sighandler);
	signal (SIGINT, mesgd_sighandler);
	signal (SIGQUIT, mesgd_sighandler);
	signal (SIGTERM, mesgd_sighandler);
#ifdef __Lynx__
	signal (SIGBRK, mesgd_sighandler);
#endif

	rtx_error_init(argv[0],RTX_ERROR_STDERR,0);
	rtx_message_init(argv[0],RTX_ERROR_STDERR);

	confd = STDERR;
	/*
	 * Allocate data structures
	 */
	if ((messaged = calloc(1, sizeof(MESSAGED))) == NULL) {
		fprintf (stderr, "messaged: calloc: no memory for MESSAGED struct\n");
		exit(2);
	}
	messaged->changePeriod = INIT_CHANGE_PERIOD;
	messaged->changeSize = INIT_CHANGE_SIZE;
	messaged->fileName = strdup(INIT_MESSAGED_FILENAME);
	messaged->dir = strdup(INIT_MESSAGED_DIR);
	messaged->archivedir = strdup(INIT_MESSAGED_DIR);
	messaged->webFlag = 0;
	messaged->webPort = 8087;
	messaged->nofork = 0;
	messaged->log = rtx_csbuffer_init(NULL,LOG_BUFFER_SIZE);
	messaged->logMess = logMessage_init(messaged->log);
	messaged->mutex = rtx_mutex_init(NULL, RTX_MUTEX_DEFAULT, 0);
	messaged->confd = confd;
	messaged->maxNumLogFiles = 100;

	parse_cmd_line(messaged,argc,argv);

	/* get the existing log files */
	messaged->files = messaged_get_files (&messaged->numLogFiles);
	fprintf (stderr, "messaged: found %d files\n", messaged->numLogFiles);
	for (i=0; i<messaged->numLogFiles; i++)
		fprintf (stderr, "%s\n", messaged->files[i]);
	fprintf (stderr, "\n");
	/* delete any excess log files */
	if (messaged->numLogFiles >= messaged->maxNumLogFiles) {
		for (i=messaged->maxNumLogFiles; i<messaged->numLogFiles; i++) {
			sprintf(fname, "%s/%s", messaged->archivedir, messaged->files[i]);
			unlink (fname);
		}
		qsort((void *) messaged->files, messaged->maxNumLogFiles, sizeof(char *),
				(int (*)(const void *, const void *)) sortfuncinc);
		messaged->curLogFileIndex = 0;
	} else {
		messaged->files = realloc (messaged->files, 
				messaged->maxNumLogFiles * sizeof (char *));
		if (messaged->files == NULL)
			messaged->curLogFileIndex = -1;
		else {
			messaged->curLogFileIndex = messaged->numLogFiles;
			for (i=messaged->curLogFileIndex; i<messaged->maxNumLogFiles; i++)
				messaged->files[i] = NULL;
		}
	}

	/*
	 * Get hostname
	 */
	if (gethostname(messaged->hostString, 128)) {
		fprintf (stderr, "messaged: gethostname() failed: %s\n", strerror(errno));
		exit(2);
	}
	setsid();
	chdir("/");
	umask(0);

	/*
	 * Open the message daemon
	 */
	fprintf (stderr, "messaged: starting\n");
	if (messaged_open()) {
		messaged_log_message("messaged: messaged_open() failed");
		messaged_close();
		exit(1);
	}



	/*
	 * Fork the process
	 */
	if (!messaged->nofork) {
		pid = fork();
		if (pid < 0) {
			fprintf (stderr, "messaged: fork() failed\n");
			exit(2);
		} else if (pid > 0) {
			exit(0);
		}
	}

	if(messaged->webFlag == 1){
		rtx_message("\nMessage daemon starting via Web-based interface");

		//start web server
		if (messaged_webServerStart(messaged,messaged->webPort)) {
			return rtx_error("Failed to start web server");
		}
	}

	while (!end) {
		messaged_logger_thread ();
	}

	if(messaged->webFlag == 1){
		//stop web server
		messaged_webServerStop(messaged);

		rtx_message("\nMessage daemon terminated");

		return 0;
	}
	rtx_mutex_destroy(messaged->mutex);

	return 0;
}

/*****************************************************************************
 *                             OPEN FUNCTION                                 *
 *****************************************************************************/

	static int
messaged_open_log_file(char *dir, char *fileName)
{
	char            string[BUFSIZ * 2 + 1];

	sprintf(string, "%s/%s", dir, fileName);

	/*
	 * Open messages log file and set buffer to zero length
	 */
	if ((messaged->logFile = fopen(string, "a")) == NULL) {
		fprintf (stderr, "messaged_open: cannot open file %s\n", string);
		return -1;
	}
	fseek(messaged->logFile, 0L, 2);
	messaged->currentSize = ftell(messaged->logFile);
	if (messaged->currentSize < 0)
		messaged->currentSize = 0;

	setbuf(messaged->logFile, NULL);
	if (chmod(string, 0666)) {
		fprintf (stderr, "messaged_open: chmod() failed, errno = %s\n", strerror(errno));
		return -1;
	}
	return 0;
}

	static int
messaged_open()
{
	int             oldmask;

	/*
	 * Check for other messages binaries running, if they are running then
	 * return.
	 */
	if (lock_get("/tmp/.messaged-lock") == -1) {
		fprintf (stderr, "messaged_open: lock file (/tmp/.messaged.lock) is locked, binary already running?\n");
		return -1;
	}
	/*
	 * Open messages log file
	 */
	if (messaged_open_log_file(messaged->dir, messaged->fileName)) {
		fprintf (stderr, "messaged_open: cannot open file %s/%s\n", messaged->dir, messaged->fileName);
		return -1;
	}
	/*
	 * Get current time
	 */
	messaged->logStart = time(NULL);

	oldmask = umask(0);
#ifdef HAVE_MESSAGE_QUEUE
	/*
	 * Create the message queue
	 */
	strcpy(messaged->mqInName, RTX_MESSAGE_MQ_NAME);
	if ((messaged->mqIn = mq_create(RTX_MESSAGE_MQ_NAME, NO_MQS, sizeof(RtxMessage), NULL)) == (mqd_t) - 1) {
		umask(oldmask);
		fprintf (stderr, "messaged_open: mq_create() failed on mqIn\n");
		return -1;
	}
#else
	if (fifo_create (RTX_MESSAGE_FIFO_PATH) == -1) {
		umask(oldmask);
		fprintf (stderr, "messaged_open: unixsock_create() failed\n");
		return -1;
	}
#endif
	umask(oldmask);

	return 0;
}

/*****************************************************************************
 *                            CLOSE FUNCTION                                 *
 *****************************************************************************/

	static int
messaged_close()
{
	int             error = 0;

	/*
	 * Close log file
	 */
	fclose(messaged->logFile);

#ifdef HAVE_MESSAGE_QUEUE
	/*
	 * Close the message queue
	 */
	if (mq_disconnect(messaged->mqIn)) {
		fprintf (stderr, "messaged_close: mq_disconnect() failed\n");
		error--;
	}
	/*
	 * Destroy message queue
	 */
	if (mq_destroy(RTX_MESSAGE_MQ_NAME)) {
		fprintf (stderr, "messaged_close: mq_unlink() failed\n");
		error--;
	}
#else
	if (fifo_done (messaged->fifofd, RTX_MESSAGE_FIFO_PATH) == -1) {
		fprintf (stderr, "messaged_close: unixsock_done() failed\n");
		error--;
	}
#endif
	if(messaged->webFlag == 1){
		//stop web server
		messaged_webServerStop(messaged);

		rtx_message("\nMessage daemon terminated");

		return 0;
	}
	/*
	 * Free memory
	 */
	free(messaged);

	return error;
}

/*****************************************************************************
 *                           LOGGER THREAD FUNCTION
 *****************************************************************************/

	static void
messaged_logger_thread()
{
	char            string[128],
					stringTime[128];
	char            logString[RTX_MESSAGE_SIZE + 34 + 128];
	int             result;

	/*
	 * Wait for a message
	 */
#ifdef HAVE_MESSAGE_QUEUE
	do {
		/*
		   result = mq_receive(messaged->mqIn, (char *) &messaged->msgIn, sizeof(RtxMessage), NULL);
		   */
		result = mq_receive(messaged->mqIn, (char *) &messaged->msgIn,
				8192, NULL);
	} while ((result < 0) && (errno == EINTR));
	if ((result < 0) && (errno != EINTR))
		fprintf (stderr, "messaged_logger_thread: mq_receive() failed: %s\n", strerror(errno));
#else
	do {
		result = fifo_recv(messaged->fifofd, &messaged->msgIn);
	} while ((result < 0) && (errno == EINTR));
	if ((result < 0) && (errno != EINTR))
		fprintf (stderr, "messaged_logger_thread: fifo_recv() failed: %s\n", strerror(errno));
#endif

	/*
	 * Put severity into logString Add hostname to log message
	 */
	switch (messaged->msgIn.type) {
		case RTX_MESSAGE_ERR:
			sprintf(logString, "R:%s ", &messaged->hostString[0]);
			break;
		case RTX_MESSAGE_WARNING:
			sprintf(logString, "W:%s ", &messaged->hostString[0]);
			break;
		case RTX_MESSAGE_SHUTDOWN:
			sprintf(logString, "P:%s ", &messaged->hostString[0]);
			break;
	}

	/*
	 * Put timestamp into logString
	 */
	strftime(stringTime, 128, "%b %d %H:%M:%S", localtime((const time_t *) &messaged->msgIn.timestamp.seconds));
	strcat(logString, stringTime);
	sprintf(string, ".%09d: ", (int) messaged->msgIn.timestamp.nanoSeconds);
	strcat(logString, string);

	/*
	 * Add message string to logString
	 */
	if (strlen(messaged->msgIn.string) > RTX_MESSAGE_SIZE) {
		fprintf (stderr, "messaged_logger_thread: message too long\n");
		return;
	}
	strcat(logString, messaged->msgIn.string);
	rtx_mutex_lock(messaged->mutex);
	messageLog(logString);
	rtx_mutex_unlock(messaged->mutex);

	if (messaged_log_message(logString))
		fprintf (stderr, "messaged_logger_thread: messaged_log_message() failed\n");

}

/******************************************************************************
 *			LOG FILE CHANGE FUNCTIONS
 ******************************************************************************/
	static void
messaged_change_file()
{
	char            string[128],
					new[2 * BUFSIZ + 128],
					old[2 * BUFSIZ + 128],
					cmd[2 * (2*BUFSIZ+128)+128];

	strftime(string, 128, "%Y%m%d-%H%M%S", localtime(&messaged->logStart));
	sprintf(old, "%s/%s", messaged->dir, messaged->fileName);
	sprintf(new, "%s/%s-%s", messaged->archivedir, string, messaged->fileName);

	/*
	 * Close existing log file
	 */
	fclose(messaged->logFile);

	/*
	 * Archive old log file
	 */
#if 0
	if (rename(old, new)) {
		perror ("messaged_change_file: rename() failed\n");
		exit(1);
	}
#else
	/* this it not very elegant, but it should do the job 
	 * rename will not work if we are sending the file to another 
	 * partition */
	sprintf(cmd,"cp -f \"%s\" \"%s\"",old,new);
	system(cmd);

	/* wipe the old file out, now that we have made a backup */
	unlink(old);
#endif
	/*
	 * Open a new log file
	 */
	if (messaged_open_log_file(messaged->dir, messaged->fileName)) {
		fprintf (stderr, "messaged_change_file: messaged_open_log_file() failed\n");
		exit(1);
	}
	/*
	 * Get the current time
	 */
	messaged->logStart = time(NULL);

	messaged->currentSize = 0;
	/* trim log files if reqd */
	if (messaged->curLogFileIndex != -1) {
		if (messaged->files[messaged->curLogFileIndex] != NULL) {
			unlink (messaged->files[messaged->curLogFileIndex]);
			free (messaged->files[messaged->curLogFileIndex]);
		}
		messaged->files[messaged->curLogFileIndex] = strdup (new);
		messaged->curLogFileIndex++;
		if (messaged->curLogFileIndex >= messaged->maxNumLogFiles)
			messaged->curLogFileIndex = 0;
	}
}

	static int
messaged_log_message(char *logString)
{
	int             len;

	/*
	 * Ensure there is a newline char
	 */
	len = strlen(logString);

	/*
	 * Write out message string to log file
	 */
	fputs(logString, messaged->logFile);
	if (logString[len - 1] != '\n') {
		fputs("\n", messaged->logFile);
	}

	messaged->currentSize += strlen(logString);

	if (((time(NULL) - messaged->logStart) > messaged->changePeriod) ||
			(messaged->currentSize > messaged->changeSize))
		messaged_change_file();

	return 0;
}

	static int
sortfuncinc(void **ap, void **bp)
{
	char           *a = (char *) *ap;
	char           *b = (char *) *bp;
	return strcmp(a, b);
}

	static int
sortfuncdec(void **ap, void **bp)
{
	char           *a = (char *) *ap;
	char           *b = (char *) *bp;
	return strcmp(b, a);
}

	static char **
messaged_get_files (int *nf)
{
	DIR            *d;
	struct dirent  *de;
	char          **names = NULL;
	int             nfiles = 0;
	char            exten[BUFSIZ];
	char            logno[BUFSIZ];
	char            secs[BUFSIZ];

	if ((d = opendir(messaged->archivedir)) == NULL)
		return (NULL);

	while ((de = readdir(d)) != NULL) {
		if (sscanf(de->d_name, "%[0-9]-%[0-9]-%s", logno, secs, exten) != 3)
			if (sscanf(de->d_name, "%[0-9]-%s", logno, exten) != 2)
				continue;
		if (strcmp(exten, messaged->fileName) != 0)
			continue;
		if (nfiles == 0)
			names = malloc(sizeof(char *));
		else
			names = realloc(names, (nfiles + 1) * sizeof(char *));
		if (names == NULL)
			return (NULL);
		names[nfiles] = strdup(de->d_name);
		nfiles++;
	}

	closedir(d);
	qsort((void *) names, nfiles, sizeof(char *),
			(int (*)(const void *, const void *)) sortfuncdec);

	*nf = nfiles;
	return names;
}

static int
lock_get (
		char *name       /**< filename of file to be locked */
		)
{
	int	lockFile;
	struct flock lock;
	int     result;

	if ((lockFile = open(name, O_RDWR|O_CREAT, 0666)) < 0) {
		return (-1);
	}

	lock.l_type = F_WRLCK;
	lock.l_start = 0;
	lock.l_whence = SEEK_SET;
	lock.l_len = 0;

	/* if lock cannot be obtained, fcntl returns with errno set to
	 * EACCESS or EAGAIN
	 */
	if ((result = fcntl (lockFile, F_SETLK, &lock)) == -1)
		return (-1);

	return (lockFile);
}

#ifdef HAVE_MESSAGE_QUEUE

/*
 *   mq.c
 *
 *  simpler interface to POSIX message queues
 *
 */

#define	MSG_EVENT       SIGRTMIN

/*
 * Create a new message queue.
 *
 * Optionally an asynchronous handler function can be specified which
 * will be called whenever a message is queued.
 */
	mqd_t
mq_create(char *name, int nmessages, int msglen, void (*handler)() )
{
	char	b[BUFSIZ];
	int	i, oldmask;
	mqd_t	mq;
	struct mq_attr attr, oattr, nattr;

	/* 
	 * Create message queue
	 */
	attr.mq_flags = 0;
	attr.mq_maxmsg = nmessages;	/* number of messages */
	attr.mq_msgsize = msglen;  	/* message size */
	if (strlen(name) > (BUFSIZ-5)) {
		fprintf (stderr, "mq_create: message queue name too long\n");
		return (mqd_t)-1;
	}
	sprintf(b, "/mq-%s", name);

	oldmask = umask(0);

	if ((i = mq_unlink(b)) < 0) {
		if (errno != ENOENT) {
			fprintf (stderr, "mq_create: mq_unlink() failed (%s) on mq %s", strerror(errno), b);	
			umask(oldmask);
			return (mqd_t)-1;
		}
	}
	mq = mq_open(b, O_RDWR|O_CREAT|O_EXCL, 0666, NULL);
	umask(oldmask);

	if ((int)mq < 0) {
		fprintf (stderr, "mq_create: mq_open failed (%s) on mq %s", strerror(errno), b);
		return (mqd_t)-1;
	}
	if (mq_getattr (mq, &oattr) == -1)
		fprintf (stderr, "mq_create: mq_getattr failed (%s) on mq %s",
				strerror(errno), b);
	if (mq_setattr (mq, &attr, NULL) == -1)
		fprintf (stderr, "mq_create: mq_setattr failed (%s) on mq %s",
				strerror(errno), b);
	if (mq_getattr (mq, &nattr) == -1)
		fprintf (stderr, "mq_create: mq_getattr failed (%s) on mq %s",
				strerror(errno), b);
#ifdef i486_linux
	fprintf (stderr, "mq_create: current linux glibc does not allow "
			"mq parms to be changed\n");
	fprintf (stderr, "mq_create: default q len = %ld, msg size = %ld\n",
			oattr.mq_maxmsg, oattr.mq_msgsize);
	fprintf (stderr, "mq_create: requested q len = %ld, msg size = %ld\n",
			attr.mq_maxmsg, attr.mq_msgsize);
	fprintf (stderr, "mq_create: set to q len = %ld, msg size = %ld\n",
			nattr.mq_maxmsg, nattr.mq_msgsize);
#endif

	if (handler != NULL) {
		struct sigaction	sa;
		struct sigevent		se;

		/* 
		 * Set up a signal handler for incoming message events 
		 */
		sa.sa_sigaction = handler;
		sigemptyset(&sa.sa_mask);
		sa.sa_flags = SA_SIGINFO;	/* POSIX real time signal */

		if (sigaction(MSG_EVENT, &sa, NULL) < 0) {
			fprintf (stderr, "mq_create: sigaction failed (%s)", strerror(errno));
			return (mqd_t)-1;
		}

		/* 
		 * Tell message system to deliver event on incoming message
		 */
		se.sigev_notify = SIGEV_SIGNAL;		/* send a signal */
		se.sigev_signo = MSG_EVENT;		/* send THIS signal */
		se.sigev_value.sival_ptr = (void *)mq;	/* with this data */
		if (mq_notify(mq, &se) < 0) {
			fprintf (stderr, "mq_create: mq_notify failed (%s) on mq %s", strerror(errno), b);
			return (mqd_t)-1;
		}
	}

	return mq;
}

/*
 * Connect to an existing message queue, return the queue id or error.
 */
	mqd_t
mq_connect(char *name)
{
	char	b[BUFSIZ];
	mqd_t	mq;
	int	oldmask;

	if (strlen(name) > (BUFSIZ-5)) {
		fprintf (stderr, "mq_connect: message queue name too long\n");
		return (mqd_t)-1;
	}
	sprintf(b, "/mq-%s", name);

	/* 
	 * Open the message queue 
	 */
	oldmask = umask(0);
	if ( (mq = mq_open(b, O_RDWR))  == (mqd_t)-1 ) {
		fprintf (stderr, "mq_connect: mq_open failed (%s) on mq %s\n", strerror(errno), b);
		umask(oldmask);
		return (mqd_t)-1;
	}
	umask(oldmask);

	return mq;
}

/*
 * Connect to an existing message queue, return the queue id or error.
 */
	mqd_t
mq_connect2(char *name)
{
	char	b[BUFSIZ];
	mqd_t	mq;
	int	oldmask;

	if (strlen(name) > (BUFSIZ-5)) {
		fprintf (stderr, "mq_connect2: message queue name too long\n");
		return (mqd_t)-1;
	}
	sprintf(b, "/mq-%s", name);

	/* 
	 * Open the message queue 
	 */
	oldmask = umask(0);
	if ( (mq = mq_open(b, O_RDWR))  == (mqd_t)-1 ) {
		fprintf (stderr, "mq_connect2: mq_open failed (%s) on mq %s\n", strerror(errno), b);
		umask(oldmask);
		return (mqd_t)-1;
	}
	umask(oldmask);

	return mq;
}

/*
 * Disconnect from a message queue
 */
	int
mq_disconnect(mqd_t mqdes)
{
	if (mq_close(mqdes)) {
		fprintf (stderr, "mq_disconnect: mq_close failed (%s)\n", strerror(errno));
		return -1;
	}

	return 0;
}

/*
 * Unlink from a message queue. The message queue will be destroyed
 * when the last attached thread calls this function
 */
	int
mq_destroy(char *name)
{
	char	b[BUFSIZ];
	int	result;

	if (strlen(name) > (BUFSIZ-5)) {
		fprintf (stderr, "mq_destroy: message queue name too long\n");
		return -1;
	}
	sprintf(b, "/mq-%s", name);

	if ((result = mq_unlink(b)) < 0)
		fprintf (stderr, "mq_destroy: mq_unlink() failed(%s) on mq %s\n", strerror(errno), b);

	return (result);
}

#else

/* use named pipes (fifo) */

int 
fifo_create (
		char *name
		)
{
	if (mkfifo (RTX_MESSAGE_FIFO_PATH, 0666) == -1) {
		if (errno != EEXIST) {
			fprintf (stderr, "unixsock_create: socket() failed(%s)\n", 
					strerror(errno));
			return (-1);
		}
	}
	if ((messaged->fifofd = open (RTX_MESSAGE_FIFO_PATH, O_RDWR)) == -1) {
		fprintf (stderr, "unixsock_create: socket() failed(%s)\n", 
				strerror(errno));
		return (-1);
	}
	return (0);
}

int
fifo_done (
		int fd,
		char * pathname
		)
{
	int errs = 0;

	if (close (fd) == -1) {
		fprintf (stderr, "unixsock_done: close() failed(%s)\n", 
				strerror(errno));
		errs++;
	}
	if (unlink (pathname) == -1) {
		fprintf (stderr, "unixsock_done: unlink() failed(%s)\n", 
				strerror(errno));
		errs++;
	}
	if (errs)
		return (-1);
	return (0);
}

int 
fifo_recv (
		int fd,
		RtxMessage * msg
		)
{
	int buflen;

	if ((buflen = read (fd, msg, sizeof (RtxMessage))) == -1) {
		fprintf (stderr, "unixsock_recv: recvfrom(%d) failed(%s)\n", 
				fd, strerror(errno));
		return (-1);
	}
	return (0);
}

#endif
