/*
 * This file is part of RTC, the Remote Tool Control library
 * CSIRO Division of Manufacturing Technology
 *
 * $Id: server.c 3077 2008-05-15 06:25:11Z roy029 $
 * Robin Kirkham, February 1997 (from PIRAT)
 *
 * server.c -- USER SERVER SIDE FUNCTIONS
 *
 * RPC (and hence, RTC) servers are non-reentrant, so there can be only one
 * per process (on Unix-like systems) or per system (on shared-memory systems).
 * On multi-threaded systems, the server can be one thread, but the user is
 * reponsible for bringing the thread into existance and having it enter 
 * rtcServer(), as well as mutex locking of data accessed by the stub functions
 */
#include "internal.h"

#undef RTC_NEWRPC


    /* the server debug flag */
RTC_BOOL rtcServerDebug = rtc_bool_false;

    /* flag to tell if server initialised */
HIDDEN int rtcServerAlready = 0;

    /* appease gcc messages */
typedef struct svc_req svc_req;


RTC_STATUS rtcServerInit(void) {
    /*
     * This should be called before any other RTC server function.
     * It ensures the sub-modules used by the server are initialised
     */
    if (rtcServerAlready++)
	return rtcFatal("Can't initialise server twice");
    rtc_eret( rtcAuthInit() );
    rtc_eret( rtcExecInit() );
    return rtc_ok;
}


RTC_STATUS rtcServerDone(void) {
    /*
     * This is called after the user has finished with RTC and has
     * (or wants to) get rid of the server. Not implemented yet
     */
    rtc_eret( rtcAuthDone() );
    rtc_eret( rtcExecDone() );
    return rtc_ok;
}


RTC_STATUS rtcServerAuth(RTC_AUTH tag, ...) {
    /*
     * Accepts a list of RTC_AUTH tags followed by values. The values are
     * added to the server authorisation lists. This function is just a 
     * wrapper for rtcAuthAdd()
     */
    va_list ap;
    va_start(ap, tag);
    rtc_eret( rtcAuthAddV(tag, ap) );
    return rtc_ok;
}


RTC_STATUS rtcServerRegister(char *name, RTC_EXEC function) {
    /*
     * Registers the user server stub function() with the given name.
     * The function can subsequently be remotely called. This is just
     * a wrapper for rtcExecRegister()
     */
    rtc_eret( rtcExecRegister(name, function) );
    return rtc_ok;
}


char *rtcServerBuffer(unsigned size) {
    /*
     * Allocate, or reallocate, a string return buffer of at least size
     * and return a pointer to it. A wrapper for rtcExecBuffer()
     */
    return rtcExecBuffer(size);
}


static void rtcServerProg(svc_req *rqstp, SVCXPRT *transp)
    /*
     * The RPC program, which has only one useful procedure.
     * This gets called whenever the server loop gets a request
     * for this service
     */
{
    rtcargs args;
    rtcret ret;

    /* the null procedure is not authenticated */
    if (rqstp->rq_proc == NULLPROC) {
        svc_sendreply(transp, (xdrproc_t)xdr_void, (char *)NULL);
	return;
    }

    /* get the authentication stuff */
    switch (rqstp->rq_cred.oa_flavor) {
    case AUTH_UNIX:
	if (! rtcAuthTest((authunix_parms *)rqstp->rq_clntcred)) {
	    svcerr_auth(transp, AUTH_BADCRED);
	    return;
	}
	break;

    default:
	svcerr_weakauth(transp);
	return;
    }


    /* now deal with the procedures -- only one, really */
    switch (rqstp->rq_proc) {

    case RTCCALL:

	/* decode argument structure */
	memset((void *)&args, 0, sizeof(args));
	if (!svc_getargs(transp, (xdrproc_t)xdr_rtcargs, (caddr_t) &args)) {
	    svcerr_decode(transp);
	    return;
	}

	if (rtcServerDebug)
	    rtcLog("    serial = %d\n", args.serial);

	/* call our RPC handler */
	ret.stat = rtcExec(args.arg.arg_len, args.arg.arg_val, &ret.str);
	if (rtcServerDebug)
	    rtcLog("    status = %d\n", ret.stat);

	/* return result */
	if (!svc_sendreply(transp, (xdrproc_t)xdr_rtcret, (caddr_t)&ret))
	    svcerr_systemerr(transp);

	/* free arguments */
	if (!svc_freeargs(transp, (xdrproc_t)xdr_rtcargs, (caddr_t) &args))
	    rtcLog("rtcServerProg: unable to free arguments\n");

	break;

    default:
	svcerr_noproc(transp);
	return;
    }

    return;
}


RTC_STATUS rtcServer(char *transports, unsigned program, char *name) {
    /*
     * This function is the server body, called either from main() in the
     * UNIX test server or from rtcServerTask() on vxWorks. Normally it
     * never returns.
     *
     * transports is a list of nettype names to establish the server on
     * (e.g. "udp tcp"); program is the RTC program number; name is a
     * name to use for debug messages/logs
     */
    int handles = 0;

    switch (rtcServerAlready) {
	case 0: return rtcFatal("Server not initialised");
	case 1: ++rtcServerAlready; break;
	default: return rtcFatal("One server per process only");
    }

    /* on vxWorks, mark task as using RPC */
#ifdef __vxworks__     
    rpcTaskInit();
#endif

    /* the RPC program number is our number plus an offset */
    program += RTCPROG;

    /* default transports string if missing or empty */
    if (!transports || !*transports)
	transports = "tcp udp";

#ifdef RTC_NEWRPC
    /*
     * On Solaris, use the "new" server creation functions, which
     * accept the nettype string, and do the portmapper (rpcbind)
     * stuff in one hit
     */

    /* unregister any previous invocations */
    svc_unreg(program, RTCVERS);

    if (! (handles = svc_create(rtcServerProg, program, RTCVERS, transports)))
	return rtcError("svc_create() failed");

#else
    /*
     * Otherwise, use the "old" RPC functions, and parse the 
     * transports string ourselves
     */

    /* unregister any previous invocations */
    pmap_unset(program, RTCVERS);

    /* create and register UDP program */
    if (strstr(transports, "udp")) {
	SVCXPRT *transp;
	if ((transp = svcudp_create(RPC_ANYSOCK)) == NULL)
	    return rtcError("svcudp_create() failed");
	if (!svc_register(transp, program, RTCVERS, rtcServerProg, IPPROTO_UDP))
	    return rtcError("svc_register() failed");
	++handles;
    }

    /* create and register TCP service */
    if (strstr(transports, "tcp")) {
	SVCXPRT *transp;
	if ((transp = svctcp_create(RPC_ANYSOCK, 0, 0)) == NULL) 
	    return rtcError("svctcp_create() failed");
	if (!svc_register(transp, program, RTCVERS, rtcServerProg, IPPROTO_TCP))
	    return rtcError("svc_register() failed");
	++handles;
    }
#endif

    /* check at least one handle was registered */
    if (! handles)
	return rtcError("No server handles created");

    /* run server */
    if (rtcServerDebug)
	rtcLog(
	    "Starting server:\n"
	    "    program = RTCPROG + %lu = %u\n"
	    "    transport%s = %s\n",
	    program - RTCPROG, program,
	    (handles > 1) ? "s" : "", transports
	);

    svc_run();
    return rtcFatal("svc_run() returned!");
}


RTC_STATUS rtcServer2(char *transports, unsigned program, char *name) {
    /*
     * This function is the server body, called either from main() in the
     * UNIX test server or from rtcServerTask() on vxWorks. Normally it
     * never returns.
     *
     * transports is a list of nettype names to establish the server on
     * (e.g. "udp tcp"); program is the RTC program number; name is a
     * name to use for debug messages/logs
     */
    int handles = 0;

    switch (rtcServerAlready) {
	case 0: return rtcFatal("Server not initialised");
	case 1: ++rtcServerAlready; break;
	default: return rtcFatal("One server per process only");
    }

    /* on vxWorks, mark task as using RPC */
#ifdef __vxworks__     
    rpcTaskInit();
#endif

    /* the RPC program number is our number plus an offset */
    program += RTCPROG;

    /* default transports string if missing or empty */
    if (!transports || !*transports)
	transports = "tcp udp";

#ifdef RTC_NEWRPC
    /*
     * On Solaris, use the "new" server creation functions, which
     * accept the nettype string, and do the portmapper (rpcbind)
     * stuff in one hit
     */

    /* unregister any previous invocations */
    svc_unreg(program, RTCVERS);

    if (! (handles = svc_create(rtcServerProg, program, RTCVERS, transports)))
	return rtcError("svc_create() failed");

#else
    /*
     * Otherwise, use the "old" RPC functions, and parse the 
     * transports string ourselves
     */

    /* unregister any previous invocations */
    pmap_unset(program, RTCVERS);

    /* create and register UDP program */
    if (strstr(transports, "udp")) {
	SVCXPRT *transp;
	if ((transp = svcudp_create(RPC_ANYSOCK)) == NULL)
	    return rtcError("svcudp_create() failed");
	if (!svc_register(transp, program, RTCVERS, rtcServerProg, IPPROTO_UDP))
	    return rtcError("svc_register() failed");
	++handles;
    }

    /* create and register TCP service */
    if (strstr(transports, "tcp")) {
	SVCXPRT *transp;
	if ((transp = svctcp_create(RPC_ANYSOCK, 0, 0)) == NULL) 
	    return rtcError("svctcp_create() failed");
	if (!svc_register(transp, program, RTCVERS, rtcServerProg, IPPROTO_TCP))
	    return rtcError("svc_register() failed");
	++handles;
    }
#endif

    /* check at least one handle was registered */
    if (! handles)
	return rtcError("No server handles created");

    /* run server */
    if (rtcServerDebug)
	rtcLog(
	    "Starting server:\n"
	    "    program = RTCPROG + %lu = %u\n"
	    "    transport%s = %s\n",
	    program - RTCPROG, program,
	    (handles > 1) ? "s" : "", transports
	);

    return rtc_ok;
}


RTC_STATUS rtcServerShow(void) {
    /*
     * Print out server information
     */
    rtcAuthShow();
    rtcExecShow();
    return rtc_ok;
}
