/*
 * This file is part of RTC, the Remote Tool Control library
 * CSIRO Division of Manufacturing Technology
 *
 * $Id: rtcclnt.c 363 2007-09-10 01:18:03Z roy029 $
 * Robin Kirkham, February 1997 (from PIRAT)
 *
 * rtcclnt.c -- interactive rtc client testing utility
 */
#include "internal.h"


    /* compiler atpeasement */
#ifndef VXWORKS
#define rtc__main main
#endif

    /* forward declarations */
int rtc__help(int argc, char *argv[]);
int rtc__tty(FILE *in, int argc, char *argv[]);

RTC rtc__rtc = NULL;
char *rtc__prog = "rtcclnt";
RTC_BOOL rtc__persevere = rtc_bool_true;


int rtc__yesno(char *resp) {
    /*
     * Returns 1 for various "yes"-like responses
     */
    int len = strlen(resp);
    if (
	(strncmp(resp, "yes", len) == 0) ||
	(strncmp(resp, "Yes", len) == 0) ||
	(strncmp(resp, "YES", len) == 0) ||
	(strncmp(resp, "1", 1) == 0)
    )
	return 1;
    return 0;
}


int rtc__call(int argc, char *argv[]) {
    /*
     * Do the RTC call
     */
    char *result;
    RTC_STATUS callStat;

    if ((callStat = rtcClientCall(rtc__rtc, argc - 1, argv + 1, &result)) != rtc_ok)
	printf("rtcClientCall() failed: %s\n", rtcMessageGet());
    else {
	if (!rtcClientDebug && !rtc__rtc->debug)
	    printf("%s\n", result);

    }
    if (callStat == rtc_ok)
        if (rtcClientResponseFree(&result))
	    printf("rtcClientResponseFree() failed: %s\n", rtcMessageGet());
    return 0;
}

int rtc__debug(int argc, char *argv[]) {
    /*
     * Toggle client side debugging
     */
    RTC_BOOL flag;
    rtcClientInterrogate(rtc__rtc, rtc_config_debug, &flag, 0);
    flag = (argc > 1) ? rtc__yesno(argv[1]) : !flag;
    rtcClientConfigure(rtc__rtc, rtc_config_debug, flag, 0);
    return 0;
}

int rtc__recreate(int argc, char *argv[]) {
    /*
     * Set/toggle recreate flag
     */
    RTC_BOOL flag;
    rtcClientInterrogate(rtc__rtc, rtc_config_recreate, &flag, 0);
    flag = (argc > 1) ? rtc__yesno(argv[1]) : !flag;
    rtcClientConfigure(rtc__rtc, rtc_config_recreate, flag, 0);
    return 0;
}

int rtc__server(int argc, char *argv[]) {
    /*
     * Set the server name
     */
    if (argc > 1)
	rtcClientConfigure(rtc__rtc, rtc_config_server, argv[1], 0);
    else
	printf("usage: server <hostname>\n");
    return 0;
}

int rtc__transport(int argc, char *argv[]) {
    /*
     * Set the transport name
     */
    if (argc > 1)
	rtcClientConfigure(rtc__rtc, rtc_config_transport, argv[1], 0);
    else
	printf("usage: transport udp|tcp|...\n");
    return 0;
}

int rtc__timeout(int argc, char *argv[]) {
    /*
     * Set the timeout period
     */
    if (argc > 1) {
	double period = atof(argv[1]) * 1000.0;
	if (period <= 0.0)
	    printf("timeout: bad period\n");
	else
	    rtcClientConfigure(rtc__rtc, rtc_config_timeout, (int)period, 0);
    } else
	printf("usage: timeout <period>\n");
    return 0;
}

int rtc__retry(int argc, char *argv[]) {
    /*
     * Set the retry period
     */
    if (argc > 1) {
	double period = atof(argv[1]) * 1000.0;
	if (period <= 0.0)
	    printf("retry: bad period\n");
	else
	    rtcClientConfigure(rtc__rtc, rtc_config_retry, (int)period, 0);
    } else
	printf("usage: retry <period>\n");
    return 0;
}

int rtc__program(int argc, char *argv[]) {
    /*
     * Set the program number
     */
    if (argc > 1) {
	int prog = atoi(argv[1]);
	if (prog < 0)
	    printf("retry: bad program number\n");
	else
	    rtcClientConfigure(rtc__rtc, rtc_config_program, prog, 0);
    } else
	printf("usage: program <program number>\n");
    return 0;
}

int rtc__show(int argc, char *argv[]) {
    /*
     * Show the client information
     */
    rtcClientShow(rtc__rtc);
    return 0;
}


int rtc__source(int argc, char *argv[]) {
    /*
     * Read commands from a file, by opening the file, and
     * recursively calling tty()
     */
    FILE *in;
    
    if (! argv[1] || !*argv[1]) 
	return rtcError("Need to specify filename");

    if (! (in = fopen(argv[1], "r")))
	return rtcError(NULL);

    return rtc__tty(in, argc, argv);
}

int rtc__shell(int argc, char *argv[]) {
    /*
     * Try to shell the given command as a UNIX command
     */
    static char buffer[BUFSIZ];
    int argk;

    /* we actually want the whole line not tokenised */
    *buffer = '\0';
    for (argk = 1; argk < argc; ++argk) {
	strcat(buffer, argv[argk]);
	strcat(buffer, " ");
    }
    return system(buffer);
}

int rtc__test(int argc, char *argv[]) {
    /*
     * Just print out arguments out
     */
    int argk;

    printf("Arguments: argc = %d\n", argc);
    for (argk = 0; argk < argc + 1; ++argk)
	printf(
	    "    argv[%02d] = %s%s%s\n", 
	    argk,
	    argv[argk] ? "\"" : "",
	    argv[argk] ? argv[argk] : "<null>",
	    argv[argk] ? "\"" : ""
	);
    return rtc_ok;
}

int rtc__comment(int argc, char *argv[]) {
    /*
     * Comments do nothing
     */
    return rtc_ok;
}

int rtc__quit(int argc, char *argv[]) {
    /*
     * Quit program
     */
    rtc__persevere = 0;
    return rtc_ok;
}


    /* array of functions and names */
static struct {
    char *command;
    int (* function)(int argc, char *argv[]);
    char *args, *help;
} rtc__info[] = {
    {
	"call", rtc__call,
	"<name> [<arg> ...]", "do remote call of <name> with <args>"
    }, {
	"server", rtc__server,
	"<server>", "set RTC server host to <server>"
    }, {
	"timeout", rtc__timeout,
	"<period>", "set timeout period (seconds)"
    }, {
	"retry", rtc__retry,
	"<period>", "set retry period (seconds)"
    }, {
	"recreate", rtc__recreate,
	"[y|n]", "toggle/set recreate flag"
    }, {
	"transport", rtc__transport,
	"[tcp|udp|...]", "set network transport type"
    }, {
	"program", rtc__program,
	"<program>", "set program number to <program>"
    }, {
	"debug", rtc__debug,
	"[y|n]", "toggle/set debug flag"
    }, {
	"show", rtc__show,
	"", "print RTC client information"
    }, {
	"source", rtc__source,
	"<file>", "read in/run RTC commands from <file>"
    }, {
	"shell", rtc__shell,
	"<command> [<arg> ...]", "run UNIX <command> with <args>"
    }, {
	"help", rtc__help,
	"", "print this message"
    }, {
	"#", rtc__comment,
	"", "lines beginning with # are ignored"
    }, {
	"quit", rtc__quit,
	"", "quit program"
    },
    { NULL, NULL, NULL }
};

int rtc__help(int argc, char *argv[]) {
    /*
     * Print the list of functions and help lines
     */
    int index;

    printf("Commands:\n");
    for (index = 0; rtc__info[index].command; ++index) {
	char buffer[100];
	sprintf(
	    buffer,
	    "%s %s",
	    rtc__info[index].command,
	    rtc__info[index].args
	);
	printf(
	    "    %-30s %s\n",
	    buffer,
	    rtc__info[index].help
	);
    }
    printf("Commands can be abbreviated\n");
    return rtc_ok;
}

int rtc__tty(FILE *in, int margc, char *margv[]) {
    /*
     * Tty interface. A mini-shell that accepts a range of commands
     * and calls a function, passing the arguments in the same way as
     * to main(). tty() expects to be called by main() with main()'s 
     * arguments as well
     */
    int index, nmatch, argc;
    int (* function)(int argc, char *argv[]) = NULL;
    char line[BUFSIZ];
    char *argv[40];

    while (rtc__persevere) {

	/* print prompt and get command line */
	if (in == stdin)
	    printf(
		"%s/%d> ",
		rtc__rtc->server,
		rtc__rtc->program
	    );
	if (fgets(line, sizeof(line), in) != line)
	    break;
	/* echo if we are reading a file */
	if (in != stdin)
	    fputs(line, stdout);
	    
	/* find first text item, if any */
	argv[0] = strtok(line, " \t\r\n");
	if (!argv[0] || !*argv[0])
	    continue;

	/* tokenize the rest of the line */
	for (argc = 1; ; ++argc)
	    if (! (argv[argc] = strtok(NULL, " \t\r\n")))
		break;

	/* search the list for a unique match */
	for (nmatch = index = 0; rtc__info[index].command; ++index)
	    if (strncmp(
		argv[0],
		rtc__info[index].command,
		strlen(argv[0])
	    ) == 0) {
		++nmatch;
		argv[0] = rtc__info[index].command;
		function = rtc__info[index].function;
	    }

	/* if more that one possible match, complain */
	if (nmatch > 1) {
	    printf("Ambiguous command `%s': Type `help' for help\n", argv[0]);
	    continue;
	}

	/* no matches at all */
	if (nmatch == 0) {
	    printf("No command `%s': Type `help' for help\n", argv[0]);
	    continue;
	}

	/* call the function, passing the full name and arguments */
	switch (function(argc, argv)) {
	case rtc_status_ok:
	    continue;

	case rtc_status_error:
	    fprintf(stderr,
		"%s: %s: %s\n",
		argv[0],
		argv[1],
		rtcMessageGet()
	    );
	    continue;

	case rtc_status_fatal:
	    fprintf(stderr, "%s: FATAL ERROR: EXITING\n", rtc__prog);
	    return 1;
	}
    }
    return rtc_ok;
}

void rtc__interrupt(int sig) {
    /*
     * SIGINT etc signal handler. Cleans up and exits
     */
    fprintf(stderr, "\n%s: signal %d: exiting\n", rtc__prog, sig);
    rtcClientDone();
    rtc__persevere = 0;
#ifndef VXWORKS
    exit(10);
#endif
}


int rtc__main(int argc, char *argv[]) {

    int stat;

    /* where is basename()? */
    if ((rtc__prog = strrchr(argv[0], '/'))) 
	++rtc__prog;
    else
	rtc__prog = argv[0];

    /* start up RTC and get a client handle */
    if (rtcClientInit()) {
	fprintf(
	    stderr,
	    "%s: rtcClientInit() failed: %s\n",
	    rtc__prog,
	    rtcMessageGet()
	);
	return 1;
    }

    /* there are no arguments */
    if (argc > 1) {
	fprintf(
	    stderr,
	    "%s: usage: %s\n",
	    rtc__prog,
	    rtc__prog
	);
	return 1;
    }

    /* start RTC */
    if (
	rtcClientCreate(
	    &rtc__rtc,
	    rtc_config_name, rtc__prog,
	    0)
    ) {
	fprintf(
	    stderr,
	    "%s: rtcClientCreate() failed: %s\n",
	    rtc__prog,
	    rtcMessageGet()
	);
	/* rtcClientDone(); */
	return 1;
    }


    /* catch termination signals to exit cleanly */
    signal(SIGHUP, rtc__interrupt);
    signal(SIGINT, rtc__interrupt);
    signal(SIGQUIT, rtc__interrupt);

    /* go to interactive mode */
    stat = rtc__tty(stdin, argc, argv);

    rtcClientDone();
    return stat;
}
