/*
 * This file is part of RTC, the Remote Tool Control library
 * CSIRO Division of Manufacturing Technology
 *
 * $Id: internal.h 363 2007-09-10 01:18:03Z roy029 $
 * Robin Kirkham, February 1997 (from PIRAT)
 *
 * internal.h -- interal header file
 *
 * This contains types and functions that are global but
 * are not used by user application programs
 */ 
#ifndef _internal_h_
#define _internal_h_

    /*
     * All the include files the RTC library needs are here, and the C files
     * just include this file, internal.h. While it slows things down, it
     * means only one copy of this nonsense is actually required
     */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#ifdef __vxworks__
#include <semLib.h>
#include <taskLib.h>
#include <logLib.h>
#include <rpcLib.h>
#include <nfsLib.h>
#include <errnoLib.h>
#include <groupLib.h>
#include <passwdLib.h>
#else
#include <netdb.h>
#include <grp.h>
#include <pwd.h>
#endif

#include <rpc/rpc.h>
#include <rpc/pmap_clnt.h>
#include <rpc/svc.h>
#ifdef __solaris__
#include <rpc/svc_soc.h>
#endif
#include "proto.h"

#include "rtc.h"


#define RTC_MAGIC 0x52544320		/* "RTC " */

typedef struct RTC_CLIENT {
    /*
     * The RTC client control block. There is one of these for each RTC
     * client; the RTC client handle is a pointer to one of these. It holds
     * the RPC CLIENT structure pointer, plus all the information required to
     * re-build it where necessary. All strings are dynamically allocated
     */
    unsigned magic;			/* magic number */
    CLIENT *client;			/* RPC client handle */

    char *name, *server;		/* "name" of handle and server host */
    unsigned program;			/* program number (offset) */
    char *transport;			/* transport string ("udp" etc) */
    unsigned timeout, retry;		/* timeout/retry period in ms */
    RTC_BOOL recreate, debug;		/* flags */

    unsigned serial;			/* call usage count */
    RTC_BOOL changed;			/* true if config details changed */
} RTC_CLIENT;

    /*
     * The mutex type is assumed to be a structure of some kind. The mutex
     * 'library' allocates and uses this type, using OS primitives; only
     * pointers to these are passed around
     */
typedef void RTC_MUTEX;


    /*
     * RTC list blocks. There is a simple linked-list "library" based
     * on these structures, which depend on the next element being first
     */
typedef struct RTC_LIST {
    struct RTC_LIST *next;
} RTC_LIST;

    /* list of numbers */
typedef struct RTC_LIST_INT {
    struct RTC_LIST_INT *next;
    int number;
} RTC_LIST_INT;

    /* list of names */
typedef struct RTC_LIST_NAME {
    struct RTC_LIST_NAME *next;
    char *name;
} RTC_LIST_NAME;

    /* list of registered stub functions */
typedef struct RTC_LIST_EXEC {
    struct RTC_LIST_EXEC *next;
    char *argv0;
    RTC_EXEC exec;
} RTC_LIST_EXEC;


    /* buffer/string allocation functions (buffer.c) */
extern RTC_STATUS
    rtcBufferAlloc(unsigned size, void **new),
    rtcBufferRealloc(void **old, unsigned size),
    rtcBufferStrdup(char *old, char **new);
extern void
    rtcBufferFree(void *block);

#define rtcBufferStalloc(type, new) \
    rtcBufferAlloc(sizeof(type), (void **)new)

    /* client debug flag (client.c) */
extern RTC_BOOL rtcClientDebug;

    /* server stub executor (exec.c) */
extern RTC_STATUS
    rtcExec(int argc, char* argv[], char **ret),
    rtcExecRegister(char *argv0, RTC_EXEC exec),
    rtcExecShow(void),
    rtcExecInit(void),
    rtcExecDone(void);
extern char
    *rtcExecBuffer(unsigned size);


    /* server authorisation lists (auth.c) */
typedef struct authunix_parms authunix_parms;

extern RTC_BOOL
    rtcAuthTest(authunix_parms *cred); 
extern RTC_STATUS
    rtcAuthAdd(RTC_AUTH tag, ...),
    rtcAuthAddV(RTC_AUTH tag, va_list ap),
    rtcAuthShow(void),
    rtcAuthInit(void),
    rtcAuthDone(void);


    /* linked list library (list.c) */
extern RTC_STATUS
    rtcListNew(RTC_LIST **list, unsigned size, RTC_LIST **new);
extern void
    rtcListDelete(RTC_LIST **list);

#define rtcListStalloc(list, type, new) \
    rtcListNew((RTC_LIST **)(list), sizeof(type), (RTC_LIST **)new)


    /* os-dependant module (mutex.c) */
extern RTC_STATUS
    rtcMutexCreate(RTC_MUTEX **mutex),
    rtcMutexGrab(RTC_MUTEX *mutex),
    rtcMutexDrop(RTC_MUTEX *mutex),
    rtcMutexFree(RTC_MUTEX *mutex);

    /* os-dependant module (return.c) */
extern RTC_STATUS
    rtcReturnError(char *message, char *file, unsigned lineno),
    rtcReturnFatal(char *message, char *file, unsigned lineno);
extern char
    *rtcReturnMessageGet(void);

    /* os-dependent module (message.c) */
extern RTC_STATUS
    rtcMessageSet(char *message);

    /* os-dependent module (id.c) */
extern RTC_STATUS
    rtcIdUserGet(char *user, int *uid),
    rtcIdGroupGet(char *group, int *gid);
extern int
    rtcIdGet(void);

    /* message logging library (log.c) */
#if (__GNUC__ == 2)
extern RTC_STATUS
    rtcLog(char *fmt, ...) __attribute__ ((format(printf, 1, 2)));
#else
extern RTC_STATUS
    rtcLog(char *fmt, ...);
#endif

    /* use HIDDEN for local objects */
#define HIDDEN static

    /*
     * This macro is used when generating errors, usually used with 
     * a return statement. The S argument should be a static string pointer.
     */
#define rtcError(S) rtcReturnError((S), __FILE__, __LINE__)

    /*
     * This macro is for generating FATAL errors; as for the above, but logs
     * a message indicating where is was detected, as well as printing S
     */
#define rtcFatal(S) rtcReturnFatal((S), __FILE__, __LINE__)

    /*
     * This macro evaluates S once (usually a function call) and returns from
     * the calling function if S is an error. This is used to pass up errors
     * at lower levels. If S indicates a fatal error, a tracing message is
     * logged as well
     */
#define rtc_eret(S) \
    switch ((S)) { \
    case rtc_status_ok: \
	break; \
    case rtc_status_error: \
	return rtc_status_error; \
    case rtc_status_fatal: \
	rtcLog( \
	    "RTC: called from: file %s, line %d\n", \
	    __FILE__, __LINE__); \
	return rtc_status_fatal; \
    }

#define rtc_fret(S) \
    switch ((S)) { \
    case rtc_status_ok: \
	break; \
    case rtc_status_error: \
	break; \
    case rtc_status_fatal: \
	rtcLog( \
	    "RTC: called from: file %s, line %d\n", \
	    __FILE__, __LINE__); \
	return rtc_status_fatal; \
    }

#endif
