/*
 * This file is part of RTC, the Remote Tool Control library
 * CSIRO Division of Manufacturing Technology
 *
 * $Id: exec.c 350 2003-05-12 02:45:56Z sik057 $
 * Robin Kirkham, February 1997 (from PIRAT)
 *
 * exec.c -- execute remote calls and return result
 */
#include "internal.h"


    /* the list of exec info blocks, and sem */
HIDDEN RTC_LIST_EXEC *rtcExecList = NULL;

    /* optional malloc()ed return value */
HIDDEN char *rtcExecBufferPointer = NULL;

    /* exec list semaphore */
HIDDEN RTC_MUTEX *rtcExecMutex; 


char *rtcExecBuffer(unsigned size)
    /*
     * This function can be called by user functions which need memory
     * for a return value or a scratchpad. A pointer to the block is
     * stored in rtcExecBufferPointer. If a buffer has already been allocated,
     * it re-sized to size. The buffer is deallocated after the call.
     */
{
     if (rtcExecBufferPointer)
	 rtcBufferRealloc((void **)&rtcExecBufferPointer, size);
     else
	 rtcBufferAlloc(size, (void **)&rtcExecBufferPointer);

     return rtcExecBufferPointer;
}


RTC_STATUS rtcExec(int argc, char* argv[], char **ret)
    /*
     * The executor. Looks up argv[0] in rtcExecList, and if a matching
     * entry is found, calls the specivied sub-executor function, passing
     * the list element and the arguments and return value pointer. The 
     * sub-executor actually calls the function and returns a status.
     * The status is passed on, unless the function was not found in the 
     * first place, in which case ERROR is returned
     */
{
    RTC_LIST_EXEC *this = NULL;
    int stat = RTCRETSTAT_OK;

    if (rtcServerDebug) {
	int argk;

	/* print out arguments ... */
	rtcLog("Exec:\n    call =");
	for (argk = 0; argk < argc; ++argk)
	    rtcLog(" %s", argv[argk]);
    }

    if (argc < 1 || !argv[0]) {
	/* not found: return error message instead */
	*ret = "Empty argument list passed";
	stat = RTCRETSTAT_ERROR;
    }
    rtc_eret( rtcMutexGrab(rtcExecMutex) );

    /* clear the last call's return buffer, if any */
    if (rtcExecBufferPointer)
	rtcBufferFree(rtcExecBufferPointer);
    rtcExecBufferPointer = NULL;

    /* search for argv[0] in the exec list */
    for (this = rtcExecList; this; this = this->next)
	if (strcmp(this->argv0, argv[0]) == 0) {

	    /* found: call the user function */
	    *ret = this->exec(argc, argv);
	    break;
	}

    if (! this) {
	/* not found: retrun error message instead */
	*ret = "Function not registered";
	stat = RTCRETSTAT_ERROR;
    }

    rtc_eret( rtcMutexDrop(rtcExecMutex) );

    if (rtcServerDebug)
	rtcLog("\n    response = %s\n", *ret);

    return stat;
}


RTC_STATUS rtcExecRegister(char *argv0, RTC_EXEC exec) {
    /*
     * Called by the user to register a user function with its name.
     * If the name is alredy registered, the new function is slotted
     * in; otherwise, a new exec control block is added to the start
     * of the list
     */
    RTC_LIST_EXEC *this;

    rtc_eret( rtcMutexGrab(rtcExecMutex) );

    /* search for argv0 in the exec list */
    for (this = rtcExecList; this; this = this->next)
	if (strcmp(this->argv0, argv0) == 0) {

	    /*
	     * If we find a structure with the same name,
	     * just change the function pointer and leave
	     */
	    this->exec = exec;

	    rtc_eret( rtcMutexDrop(rtcExecMutex) );
	    return rtc_ok;
	}

    /*
     * If we get here, the function (name) is new, so we allocate
     * a new control block and tack it on the front
     */

    rtc_eret( rtcListStalloc((RTC_LIST **)&rtcExecList, RTC_LIST_EXEC, &this) );
    rtc_eret( rtcBufferStrdup(argv0, &this->argv0) );
    this->exec = exec;

    rtc_eret( rtcMutexDrop(rtcExecMutex) );
    return rtc_ok;
}


RTC_STATUS rtcExecShow(void)
    /*
     * Print all the exec list information
     */
{
    RTC_LIST_EXEC *this;

    rtcLog("Server exec list:\n");

    rtc_eret( rtcMutexGrab(rtcExecMutex) );
    for (this = rtcExecList; this; this = this->next)
	rtcLog("    %s = 0x%08x\n", this->argv0, (unsigned)this->exec);

    rtc_eret( rtcMutexDrop(rtcExecMutex) );
    return rtc_ok;
}


static char *rtcExecPing(int argc, char *argv[])
    /*
     * A user function that returns the name it was invoked under.
     * This is always the first function registered.
     * Used for testing clients
     */
{
    return argv[0];
}
    

static char *rtcExecCatalogue(int argc, char *argv[])
    /*
     * A user function that returns a list of all registered functions,
     * including itself. This is generally the second function registered,
     * under the name "catalogue"
     */
{
    char *buffer, *point;
    RTC_LIST_EXEC *this;
    int size = 0;

    /* get length of return string required */
    for (this = rtcExecList; this; this = this->next)
	size += strlen(this->argv0) + 1;

    /* get return buffer (aready full of nulls) */
    point = buffer = rtcExecBuffer(size + 1);

    /* concatenate into buffer */
    for (this = rtcExecList; this; this = this->next) {
	strcpy(point, this->argv0);
	point += strlen(this->argv0);
	*point++ = ' ';
    }

    /* elide the dangling space */
    *--point = '\0';
	
    return buffer;
}


RTC_STATUS rtcExecInit(void)
    /*
     * First creates the protection semaphore (if required), clears the
     * exec list, and re-registers the standard user functions 
     */
{
    /* create the semaphore, if not created already */
    rtc_eret( rtcMutexCreate(&rtcExecMutex) );

    /* clear the exec list */
    rtc_eret( rtcMutexGrab(rtcExecMutex) );
    rtcListDelete((RTC_LIST **)&rtcExecList);
    rtc_eret( rtcMutexDrop(rtcExecMutex) );

    /* register the ping and catalogue functions */
    rtcExecRegister("catalogue", rtcExecCatalogue);
    rtcExecRegister("ping", rtcExecPing);

    return rtc_ok;
}

RTC_STATUS rtcExecDone(void) {
    /* 
     * Clean up the exec stuff
     */

    rtc_eret( rtcMutexGrab(rtcExecMutex) );
    rtcListDelete((RTC_LIST **)&rtcExecList);

    rtc_eret( rtcMutexFree(rtcExecMutex) );

    return rtc_ok;
}
