/*
 * This file is part of RTC, the Remote Tool Control library
 * CSIRO Division of Manufacturing Technology
 *
 * $Id: client.c 3119 2008-05-19 01:13:35Z roy029 $
 * Robin Kirkham, February 1997 (from PIRAT)
 *
 * client.c -- USER CLIENT SIDE FUNCTIONS
 *
 * These functions create, use, or destroy the RPC client handle within the
 * RTC control block. They can be called directly by user functions in C
 * program, or by the Tcl-level functions response to callbacks from the 
 * Tcl interpreter
 */
#include "internal.h"


RTC_BOOL rtcClientDebug = rtc_bool_false; 


RTC_STATUS rtcClientInit(void) {
    /*
     * Client-side initialisation function. This should be called once
     * per process (or system in shared-memory systems) before any other
     * RTC client-side functions
     */
#ifdef __vxworks__
    rpcTaskInit();
#endif
    return rtc_ok;
}


RTC_STATUS rtcClientDone(void) {
    /*
     * Client-side cleanup function. This should be called once
     * per process (or system in shared-memory systems) after all
     * RTC client-side activity has ceased
     */
    return rtc_ok;
}


static RTC_STATUS rtcClientConfigureV(RTC rtc, RTC_CONFIG tag, va_list ap) {
    /*
     * Configure the given RTC handle according to the zero-terminated list
     * of configuration tags and values pointed to by the stdarg pointer ap.
     * All string type configuration parameters are duplicated. If the 
     * new parameter indicates the RPC client data needs to be changed, the
     * changed flag is set. This forces the client handle to be recreated
     * on the next use of rtcClientCall() 
     */

    /* check the given structure */
    if (!rtc || !(rtc->magic == RTC_MAGIC))
	return rtcFatal("Bad client handle structure");

    while (tag != rtc_config_LAST) {
	switch (tag) {
	case rtc_config_name:
	    rtcBufferFree(rtc->name);
	    rtc_eret( rtcBufferStrdup(va_arg(ap, char*), &rtc->name) );
	    break;

	case rtc_config_server:
	    rtcBufferFree(rtc->server);
	    rtc_eret( rtcBufferStrdup(va_arg(ap, char*), &rtc->server) );
	    rtc->changed = rtc_bool_true;
	    break;

	case rtc_config_program:
	    rtc->program = va_arg(ap, RTC_BOOL);
	    rtc->changed = rtc_bool_true;
	    break;

	case rtc_config_transport:
	    rtcBufferFree(rtc->transport);
	    rtc_eret( rtcBufferStrdup(va_arg(ap, char*), &rtc->transport) );
	    rtc->changed = rtc_bool_true;
	    break;

	case rtc_config_timeout:
	    rtc->timeout = va_arg(ap, unsigned);
	    rtc->changed = rtc_bool_true;
	    break;

	case rtc_config_retry:
	    rtc->retry = va_arg(ap, unsigned);
	    rtc->changed = rtc_bool_true;
	    break;

	case rtc_config_recreate:
	    rtc->recreate = va_arg(ap, RTC_BOOL);
	    break;

	case rtc_config_debug:
	    rtc->debug = va_arg(ap, RTC_BOOL);
	    break;

	default:
	    return rtcError("Bad client configuration tag");
	}
	tag = va_arg(ap, RTC_CONFIG);
    }

    return rtc_ok;
}


RTC_STATUS rtcClientConfigure(RTC rtc, RTC_CONFIG tag, ...) {
    /*
     * Configure the given RTC handle according to the zero-terminated list
     * of configuration tags and values
     */
    va_list ap;
    va_start(ap, tag);
    rtc_eret( rtcClientConfigureV(rtc, tag, ap) );
    return rtc_ok;
}


RTC_STATUS rtcClientInterrogate(RTC rtc, RTC_CONFIG tag, ...) {
    /*
     * Opposite from rtcClientConfigure(). Used to get data out from
     * the RTC client handle. NOTE WELL: the string data coming out 
     * is malloc()'ed data, and should be freed when the caller has
     * finished with it. Each tag should be followed by a POINTER
     * to an object of the correct type
     */
    va_list ap;
    va_start(ap, tag);

    /* check the given structure */
    if (!rtc || !(rtc->magic == RTC_MAGIC))
	return rtcFatal("Bad client handle structure");

    while (tag != rtc_config_LAST) {

	/* get the data pointer */
	void *data = va_arg(ap, void*);
	if (! data) 
	    return rtcFatal("Bad data return pointer");

	switch (tag) {
	case rtc_config_name:
	    rtc_eret( rtcBufferStrdup(rtc->name, (char **)data) );
	    break;

	case rtc_config_server:
	    rtc_eret( rtcBufferStrdup(rtc->server, (char **)data) );
	    break;

	case rtc_config_program:
	    *(int *)data = rtc->program;
	    break;

	case rtc_config_transport:
	    rtc_eret( rtcBufferStrdup(rtc->transport, (char **)data) );
	    break;

	case rtc_config_timeout:
	    *(int *)data = rtc->timeout;
	    break;

	case rtc_config_retry:
	    *(int *)data = rtc->retry;
	    break;

	case rtc_config_recreate:
	    *(RTC_BOOL *)data = rtc->recreate;
	    break;

	case rtc_config_debug:
	    *(RTC_BOOL *)data = rtc->debug;
	    break;

	default:
	    return rtcError("Bad client configuration tag");
	}
	tag = va_arg(ap, RTC_CONFIG);
    }

    return rtc_ok;
}


RTC_STATUS rtcClientCreate(RTC *new, RTC_CONFIG tag, ...) {
    /*
     * Create a new RTC/RPC client handle. This handle is used to identify
     * the remote server, and is passed to subsequent calls to rtcClientCall(),
     * rtcClientConfigure(), and so on. The RTC handle is configured with
     * default values, but these may be altered by the zero-terminated
     * configuration list that follows (see rtcClientConfigure()). The handle
     * is written back to new
     */
    RTC_CLIENT *rtc;
    char hostname[MAXHOSTNAMELEN];
    va_list ap;
    va_start(ap, tag);

    /* create an RTC client structure */
    rtc_eret( rtcBufferStalloc(RTC_CLIENT, &rtc) );

    /* initialise handle to default values */
    gethostname(hostname, sizeof(hostname));
    rtc_eret( rtcBufferStrdup(hostname, &rtc->server) );
    rtc_eret( rtcBufferStrdup("rtc", &rtc->name) );
    rtc_eret( rtcBufferStrdup("udp", &rtc->transport) );

    rtc->magic = RTC_MAGIC;
    rtc->program = 0;
    rtc->timeout = 25000;
    rtc->retry = 1000;
    rtc->recreate = rtc_bool_false;
    rtc->debug = rtc_bool_false;

    rtc->serial = 0;
    rtc->changed = rtc_bool_true;

    /* configure the RTC handle */
    rtc_eret( rtcClientConfigureV(rtc, tag, ap) );

    /* return handle to user */
    if (! new)
	return rtcFatal("Bad client handle return pointer");
    *new = rtc;

    return rtc_ok;
}


HIDDEN RTC_STATUS rtcClientRebuild(RTC rtc) {
    /*
     * This is called by rtcClientCall() when it finds the rtc->changed or
     * rtc->recreate flags set. It brings the RPC client handle (if any) into
     * line with the data in the RTC handle, basically by destroying it and
     * building it from scratch. This will involve calling clnt_create(),
     * which will communicate with the portmapper
     */
    struct timeval timeout, retry;

    /* destroy existing RPC client, if any */
    if (rtc->client) {
	/* need to close socket? */
	auth_destroy(rtc->client->cl_auth);
	clnt_destroy(rtc->client);
	rtc->client = NULL;
    }

    /* now create a new client handle */
    if (
	(rtc->client = clnt_create(
	    rtc->server, rtc->program + RTCPROG, RTCVERS, rtc->transport
	)) == NULL
    ) 
	return rtcError(clnt_spcreateerror(rtc->name));

    /* set overall timeout period */
    timeout.tv_usec = (rtc->timeout % 1000) * 1000;
    timeout.tv_sec = rtc->timeout/1000;
    clnt_control(rtc->client, CLSET_TIMEOUT, (char *)&timeout);

    /* set retry period (useful for udp only) */
    retry.tv_usec = (rtc->retry % 1000) * 1000;
    retry.tv_sec = rtc->retry/1000;
    clnt_control(rtc->client, CLSET_RETRY_TIMEOUT, (char *)&retry);

#ifdef CLSET_FD_CLOSE
    /*
     * Set automatic socket closure on clnt_destroy.
     * This may be set already
     */
    clnt_control(rtc->client, CLSET_FD_CLOSE, NULL);
#endif

    /*
     * Create a UNIX auth_unix credential structure. The rtc server uses the 
     * machine/uid/gid info in the auth_unix to selectively accept or reject
     * requests
     */
    if (! (rtc->client->cl_auth = authunix_create_default()))
	return rtcFatal(NULL);

    if (rtc->debug || rtcClientDebug)
	rtcLog("recreated client handle for %s\n", rtc->name);
    
    rtc->changed = rtc_bool_false;
    return rtc_ok;
}


RTC_STATUS rtcClientCall(RTC rtc, int argc, char *argv[], char **resp) {
    /*
     * Perform an RTC call to the server identified by the client handle rtc,
     * passing the given argument list. The server returns the result (or
     * error string) which is written into the resp pointer
     */
    rtcargs args;
    rtcret result;
    enum clnt_stat stat;

    /* dummy timeout struct, ignored by clnt_call()  */
    static struct timeval UNUSED = { 25, 0 };

    /* check the given structure */
    if (!rtc || !(rtc->magic == RTC_MAGIC))
	return rtcFatal("Bad client handle structure");

    /* make sure there are enough arguments */
    if (argc < 1) 
	return rtcFatal("wrong # args");

    /* check return string */
    if (! resp)
	return rtcFatal("Bad response return string pointer");


    /* build RPC argument structure */
    args.arg.arg_len = argc;
    args.arg.arg_val = argv;
    args.serial = rtc->serial++;

    /* check if handle needs rebuilding */
    if (rtc->changed || rtc->recreate) {
	RTC_STATUS stat = rtcClientRebuild(rtc);
	if (stat) {
	    /* make sure we return the error string as the response */
	    *resp = rtcMessageGet();
	    return stat;
	}
    }

    if (rtc->debug || rtcClientDebug) {
	int argk;
	rtcLog("%s:\n    call = ", rtc->name);
	for (argk = 0; argk < argc; ++argk)
	    rtcLog("%s ", argv[argk]);
    }

    /* clear the return structure */
    memset((char *)&result, 0, sizeof(result));

    /* actually do the RPC call */
    stat = clnt_call(
	rtc->client, RTCCALL,
	(xdrproc_t) xdr_rtcargs, (caddr_t)&args,
	(xdrproc_t) xdr_rtcret, (caddr_t)&result,
	UNUSED
    );

    /* decode the status */
    if (stat == RPC_SUCCESS)
	*resp = result.str;
    else {
	/* duplicate the RPC error string and give it as reponse */
	rtcBufferStrdup(clnt_sperrno(stat), resp);

	/* displose of any response--is this necessary? */
	clnt_freeres(rtc->client, (xdrproc_t) xdr_rtcret, (caddr_t)&result);
    }

    if (rtc->debug || rtcClientDebug)
	rtcLog("\n    response = %s\n    status = %d\n", *resp, stat);

    return (stat == RPC_SUCCESS)
	? rtc_ok
	: rtcError(*resp);
}

RTC_STATUS rtcClientResponseFree(char **resp) {
    /*
     * Deallocate a response string. The XDR routine xdr_rtcret(), called
     * by clnt_call(), would have malloc()'ed this string (but not the full
     * rtcret structure). If there was an error, rtcClientCall() would have
     * duplicated the error string. In either case, it needs to be freed
     */

    /* check return string */
    if (! resp)
	return rtcFatal("Bad response return string pointer");

#ifdef WHY_NOT 
    /* I think this is what should happen, but it prangs vxWorks */
    xdr_free(xdr_string, (caddr_t)*resp);
#else
    /* this second-guesses XDR, but is right and works */
    free(*resp);
#endif

    /* zero the pointer, just to make sure */
    *resp = NULL;
    return rtc_ok;
}


RTC_STATUS rtcClientDestroy(RTC rtc) {
    /*
     * Destroy the RTC handle rtc, and deallocate all memory asociated
     * with it. The handle cannot be used again
     */

    /* check the given structure */
    if (!rtc || !(rtc->magic == RTC_MAGIC))
	return rtcFatal("Bad client handle structure");

    rtcBufferFree(rtc->name);
    rtcBufferFree(rtc->server);
    rtcBufferFree(rtc->transport);

    if (rtc->client) {

	/* we think this closes the socket as well */
	auth_destroy(rtc->client->cl_auth);
	clnt_destroy(rtc->client);
    }
    rtc->magic = 0;
    rtcBufferFree(rtc);
    return rtc_ok;
}


RTC_STATUS rtcClientShow(RTC rtc) {
    /*
     * Print out the details contained in the given RTC client handle
     */
    /* check the given structure */
    if (!rtc || !(rtc->magic == RTC_MAGIC))
	printf("Bad client handle structure");

    printf(
	"RTC = 0x%08x:\n"
	"    name = %s\n"
	"    server = %s\n"
	"    timeout = %u ms\n"
	"    retry = %u ms\n"
	"    recreate = %d\n"
	"    debug = %d\n"
	"    transport = %s\n"
	"    program = RTCPROG + %u = %u\n"
	"    serial = %d\n"
	"    changed = %d\n"
	"    magic = %x\n",
	(unsigned)rtc, rtc->name, rtc->server, rtc->timeout,
	rtc->retry, rtc->recreate, rtc->debug, rtc->transport,
	rtc->program, rtc->program + RTCPROG, rtc->serial,
	rtc->changed, rtc->magic
    );
    return rtc_ok;
}
