/*
 * This file is part of RTC, the Remote Tool Control library
 * CSIRO Division of Manufacturing Technology
 *
 * $Id: auth.c 350 2003-05-12 02:45:56Z sik057 $
 * Robin Kirkham, February 1997 (from PIRAT)
 *
 * auth.c -- server authenticator
 */
#include "internal.h"


    /* lists of acceptable hosts, uids, and gids */
HIDDEN RTC_LIST_NAME *rtcAuthHosts = NULL;
HIDDEN RTC_LIST_INT *rtcAuthUids = NULL;
HIDDEN RTC_LIST_INT *rtcAuthGids = NULL;

    /* auth list monitor */
HIDDEN RTC_MUTEX *rtcAuthMutex;

    /*
     * Functions to look through either the name type list or the
     * int type list, and return rtc_bool_true if the name or int is in
     * the list, or rtc_bool_false if not or the list is empty. The list 
     * protection semaphore is assumed taken
     */

HIDDEN __inline__ RTC_BOOL rtcAuthTestName(RTC_LIST_NAME *point, char *name) {
    /*
     * Examines each element in list starting at point, and compares
     * name against the name field; returns true if found, else false
     */
    if (!point)
	return rtc_bool_false;
    while (point) {
	if (point->name && strcmp(point->name, name) == 0)
	    return rtc_bool_true;
	point = point->next;
    }
    return rtc_bool_false;
}


HIDDEN __inline__ RTC_BOOL rtcAuthTestNumber(RTC_LIST_INT *point, int number) {
    /*
     * Examines each element in list starting at point, and compares
     * number against the number field; returns true if found, else false
     */
    if (!point)
	return rtc_bool_false;
    while (point) {
	if (point->number == number)
	    return rtc_bool_true;
	point = point->next;
    }
    return rtc_bool_false;
}


RTC_BOOL rtcAuthTest(authunix_parms *cred)
    /*
     * This is the access control function. It is called from the server
     * so see if a request should be executed or rejected (true or false).
     * The authunix_parms structure contains the client machine name, and
     * the uid, effective gid and gids list. This info is tested against
     * the lists. Access permitted if:
     *
     *    client machine name in machine list, AND
     *       client uid is in uid list, OR
     *       client effective gid is in gid list, OR
     *       any client gid is in gid list
     */
{
    int count;
    RTC_BOOL result, result_mach, result_uid, result_gid, result_gids;

    /* take the semaphore */
    rtc_eret( rtcMutexGrab(rtcAuthMutex) );

    /* test machine name, the uid and effective gid */
    result_mach = rtcAuthTestName(rtcAuthHosts, cred->aup_machname);
    result_uid = rtcAuthTestNumber(rtcAuthUids, cred->aup_uid);
    result_gids = rtcAuthTestNumber(rtcAuthGids, cred->aup_gid);

    result_gids = rtc_bool_false;
    /*
     * test each listed gid against the list, but only bother if
     * if it might actually change the result
     */
    if (result_mach && !result_uid && !result_gid && rtcAuthGids) {
	for (count = 0; count < cred->aup_len; ++count)
	    if (rtcAuthTestNumber(rtcAuthGids, cred->aup_gids[count])) {
		result_gids = rtc_bool_true;
		break;
	    }
    }
    rtc_eret( rtcMutexDrop(rtcAuthMutex) );

    /* the result is ... */
    result = result_mach && (result_uid || result_gid || result_gids);

    if (rtcServerDebug) {
	rtcLog(
	    "Request:\n    machine = %s\n    uid = %d, gid = %d\n    gids =",
	    cred->aup_machname,
	    (int)cred->aup_uid,
	    (int)cred->aup_gid
	);
	for (count = 0; count < cred->aup_len; ++count)
	    rtcLog(" %d", (int) cred->aup_gids[count]);
	rtcLog("\n    auth result = %s\n", result ? "OK" : "failed");
    }

    return result;
}


RTC_STATUS rtcAuthAddV(RTC_AUTH tag, va_list ap) {
    /*
     * Accept a zero-terminated list of RTC_AUTH tags followed by values.
     * The first tag is passed directly, and the following values and tags
     * are in the stdarg pointer ap. Adds the values to the appropriate
     * authorisation lists
     */
    RTC_LIST_NAME *new_name;
    RTC_LIST_INT *new_int;

    /* grab list protection semaphore */
    rtc_eret( rtcMutexGrab(rtcAuthMutex) );

    while (tag != rtc_auth_LAST) {
	switch (tag) {
	case rtc_auth_host:
	    rtc_eret( rtcListStalloc(&rtcAuthHosts, RTC_LIST_NAME, &new_name) );
	    rtc_eret( rtcBufferStrdup(va_arg(ap, char*), &new_name->name) );
	    break;

	case rtc_auth_uid:
	    rtc_eret( rtcListStalloc(&rtcAuthUids, RTC_LIST_INT, &new_int) );
	    new_int->number = va_arg(ap, int);
	    break;

	case rtc_auth_gid:
	    rtc_eret( rtcListStalloc(&rtcAuthGids, RTC_LIST_INT, &new_int) );
	    new_int->number = va_arg(ap, int);
	    break;

	case rtc_auth_user:
	    /* look up user in /etc/passwd or equivalent */
	    rtc_eret( rtcListStalloc(&rtcAuthUids, RTC_LIST_INT, &new_int) );
	    rtc_eret( rtcIdUserGet(va_arg(ap, char*), &new_int->number) );
	    break;

	case rtc_auth_group:
	    /* look up group in /etc/groups or equivalent */
	    rtc_eret( rtcListStalloc(&rtcAuthGids, RTC_LIST_INT, &new_int) );
	    rtc_eret( rtcIdGroupGet(va_arg(ap, char*), &new_int->number) );
	    break;

	default:
	    rtc_eret( rtcMutexDrop(rtcAuthMutex) );
	    return rtcFatal("Bad server authorisation tag");
	}

	/* get next tag */
	tag = va_arg(ap, RTC_AUTH);
    }

    /* drop list protection semaphore */
    rtc_eret( rtcMutexDrop(rtcAuthMutex) );
    return rtc_ok;
}


RTC_STATUS rtcAuthAdd(RTC_AUTH tag, ...) {
    /*
     * Accept a zero-terminated list of RTC_AUTH tags followed by values,
     * starting from tag. rDadd the values to the appropriate authorisation
     * lists. This just makes a stdarg pointer and calls rtcAuthAddV()
     */
    va_list ap;
    va_start(ap, tag);
    rtc_eret( rtcAuthAddV(tag, ap) );
    return rtc_ok;
}


RTC_STATUS rtcAuthShow(void) {
    /*
     * Print out the contents of the server auth lists
     */
    rtcLog("Server authorisation lists:\n");

    rtc_eret( rtcMutexGrab(rtcAuthMutex) );

    /* host list */
    if (rtcAuthHosts) {
	RTC_LIST_NAME *this;

	rtcLog("    hosts =");
	for (this = rtcAuthHosts; this; this = this->next)
	    rtcLog(" %s", this->name);
	rtcLog("\n");
    }
    
    /* UID list */
    if (rtcAuthUids) {
	RTC_LIST_INT *this;

	rtcLog("    uids =");
	for (this = rtcAuthUids; this; this = this->next)
	    rtcLog(" %d", this->number);
	rtcLog("\n");
    }

    /* GID list */
    if (rtcAuthGids) {
	RTC_LIST_INT *this;

	rtcLog("    gids =");
	for (this = rtcAuthGids; this; this = this->next)
	    rtcLog(" %d", this->number);
	rtcLog("\n");
    }

    rtc_eret( rtcMutexDrop(rtcAuthMutex) );
    return 0;
}


RTC_STATUS rtcAuthInit(void) {
   /*
    * Create the list protection semaphore, then clear out all the
    * authorisation lists. This is called by rtcInit() when a new server
    * is started. If default is true, it inserts the  machine/uid/gid
    * entries for the owner of the server process
    */
    char buffer[MAXHOSTNAMELEN];

    /* create semaphore for auth lists */
    rtc_eret( rtcMutexCreate(&rtcAuthMutex) );

    /* clear the authorisation lists */
    rtc_eret( rtcMutexGrab(rtcAuthMutex) );
    rtcListDelete((RTC_LIST **)&rtcAuthHosts);
    rtcListDelete((RTC_LIST **)&rtcAuthUids);
    rtcListDelete((RTC_LIST **)&rtcAuthGids);
    rtc_eret( rtcMutexDrop(rtcAuthMutex) );

    /* add the local host and server user */
    gethostname(buffer, sizeof(buffer));
    rtc_eret(
	rtcAuthAdd(
	    rtc_auth_host, buffer,
	    rtc_auth_uid, rtcIdGet(),
	    0
	)
    );
    return rtc_ok;
}


RTC_STATUS rtcAuthDone(void) {
   /*
    * Clean up after server exit
    */

    /* clear the authorisation lists */
    rtc_eret( rtcMutexGrab(rtcAuthMutex) );
    rtcListDelete((RTC_LIST **)&rtcAuthHosts);
    rtcListDelete((RTC_LIST **)&rtcAuthUids);
    rtcListDelete((RTC_LIST **)&rtcAuthGids);

    /* get rid of auth semaphore */
    rtc_eret( rtcMutexFree(rtcAuthMutex) );

    return rtc_ok;
}
