/*
 *****************************************************************************
 * CSIRO MANUFACTURING SCIENCE & TECHNOLOGY
 * QCAT, PO Box 883, Kenmore, Q 4068, Australia
 *
 *	$Id: plsddx.c 1660 2004-06-17 05:27:12Z jmr $
 * 
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *****************************************************************************
 */

/**
 *****************************************************************************
 * \file
 * \brief The Sick PLS DDX api C file
 * \author Jonathan Roberts
 *****************************************************************************
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <rtx/error.h>
#include <rtx/message.h>
#include <rtx/timer.h>

#include "plsddx.h"

/*
#define DEBUG
#define	TEST
*/

typedef struct _plsddx {
	char		*name;
	DDX_STORE_ID	*storeId;
	DDX_STORE_ITEM	*scanItem;
	DDX_STORE_ITEM	*cmdItem;
	DDX_STORE_ITEM	*modeItem;
} PlsDdx;

static PlsDdx	*plsddx;
static int	nPls = 0;

/**
 * This function initialises a PLS device with the plsddx module
 *
 * \return A handle for the PLS (non-negative int) on success, -1 on error
 */
int
plsddx_init(
	DDX_STORE_ID	*storeId,	/*!< Store ID of pls */
	char		*name		/*!< Name of the pls */
	)
{
	char	string[BUFSIZ];

	if (nPls == 0) {
		if ((plsddx = (PlsDdx *)realloc(NULL, sizeof(PlsDdx))) == NULL)
			return rtx_error("plsddx_init: no memory");
	} else {
		if ((plsddx = (PlsDdx *)realloc(plsddx, (nPls+1)*sizeof(PlsDdx))) == NULL)
			return rtx_error("plsddx_init: no memory");
	}

	plsddx[nPls].storeId = storeId;
	plsddx[nPls].name = strdup(name);

	if (DDX_STORE_REGISTER_CONST(storeId, PLS_MAX_NSEGS))
		return rtx_error("plsddx_init: DDX_STORE_REGISTER_CONST: failed on name: %s", name);
	if (DDX_STORE_REGISTER_TYPE(storeId, PlsScan))
		return rtx_error("plsddx_init: DDX_STORE_REGISTER_TYPE: failed on name: %s", name);

	if ((plsddx[nPls].scanItem = ddx_store_lookup_item(storeId, name, "PlsScan", sizeof(PlsScan))) == NULL)
		return rtx_error("plsddx_init: ddx_store_lookup_item: failed on name: %s", name);

	sprintf(string, "%sCmd", name);
	if ((plsddx[nPls].cmdItem = ddx_store_lookup_item(storeId, string, "int", sizeof(int))) == NULL)
		return rtx_error("plsddx_init: ddx_store_lookup_item: failed on name: %s", name);

	sprintf(string, "%sMode", name);
	if ((plsddx[nPls].modeItem = ddx_store_lookup_item(storeId, string, "int", sizeof(int))) == NULL)
		return rtx_error("plsddx_init: ddx_store_lookup_item: failed on name: %s", name);

	nPls++;

	return nPls-1;
}

/**
 * This function cleans up the plsddx module
 *
 * \return Zero on success, -1 on error
 */
int
plsddx_done()
{
	int	i, error = 0;

	for (i=0; i<nPls; i++) {
		if (ddx_store_done_item(plsddx[i].scanItem)) {
			rtx_error("plsddx_done: ddx_store_done_item: failed");
			error--;
		}

		if (ddx_store_done_item(plsddx[i].cmdItem)) {
			rtx_error("plsddx_done: ddx_store_done_item: failed");
			error--;
		}

		if (ddx_store_done_item(plsddx[i].modeItem)) {
			rtx_error("plsddx_done: ddx_store_done_item: failed");
			error--;
		}
	}

	free(plsddx);

	return error;
}

/**
 * This function reads a scan of data from the PLS
 *
 * \return Zero on success, -1 on error
 */
int
plsddx_read_scan(
	int 		id,		/*!< PLS id */
	PlsScan		*data,		/*!< Pointer to where to put data */
	RtxTime		*t,		/*!< Pointer to where to put time */
	double		stale,		/*!< Stale time if async */
	int		sync		/*!< Sync flag */
	)
{
	int	result;

	result = ddx_store_read(plsddx[id].scanItem, data, t, stale, sync);

	if (result == -1)
		return rtx_error("plsddx_read_scan: ddx_store_read: failed");

	return result;
}

/**
 * This function writes a scan of data to the store
 *
 * \return Zero on success, -1 on error
 */
int
plsddx_write_scan(
	int 		id,		/*!< PLS id */
	PlsScan		*data,		/*!< Pointer to data */
	RtxTime		*t		/*!< Pointer to where to put time */
	)
{
	if (ddx_store_write(plsddx[id].scanItem, data, t))
		return rtx_error("plsddx_write_scan: ddx_store_write: failed");

	return 0;
}

/**
 * This function reads the mode of a PLS
 *
 * \return Zero on success, 1 if data is stale, -1 on error
 */
int
plsddx_read_mode(
	int 		id,		/*!< PLS id */
	int		*mode,		/*!< Pointer to where to put mode */
	RtxTime		*t,		/*!< Pointer to where to put time */
	double		stale,		/*!< Stale time if async */
	int		sync		/*!< Sync flag */
	)
{
	int	result;

	if ((result = ddx_store_read(plsddx[id].modeItem, mode, t, stale, sync)) < 0)
		return rtx_error("plsddx_read_mode: ddx_store_read: failed");

	return result;
}

/**
 * This function wtites a command to a PLS
 *
 * \return Zero on success, -1 on error
 */
int
plsddx_write_cmd(
	int 		id,		/*!< PLS id */
	int		cmd		/*!< The command */
	)
{
	int	cmd_p;

	cmd_p = cmd;

	if (ddx_store_write(plsddx[id].cmdItem, &cmd_p, NULL))
		return rtx_error("plsddx_write_cmd: ddx_store_write: failed");

	return 0;
}


static int
plsddx_mode_change(
	int 		id,		/*!< PLS id */
	double		timeout,	/*!< Time out period (sec) */
	int		cmd,
	int		mode
	)
{
	int	modeNow;
	RtxTime	start, now;

	rtx_time_get(&start);

	if (plsddx_write_cmd(id, cmd))
		return rtx_error("plsddx_mode_change: plsddx_write_cmd: failed");

	while (rtx_time_get_delta(&start) < timeout) {
		if (plsddx_read_mode(id, &modeNow, &now, 0.0, 0) < 0)
			return rtx_error("plsddx_mode_change: plsddx_read_mode: failed");
		if (rtx_time_subtract_to_double(&now, &start) > 0.0) {
			if (modeNow == mode)
				return 0;
		}
		rtx_timer_sleep(0.1);
	}

	return -1;
}

/**
 * This function turns the PLS into idle mode and checks that it has done
 * so.
 *
 * \return Zero on success, -1 on error
 */
int
plsddx_idle(
	int 		id,		/*!< PLS id */
	double		timeout		/*!< Time out period (sec) */
	)
{
	return plsddx_mode_change(id, timeout, 0, 0x25);
}

/**
 * This function turns the PLS into continuous mode and checks that it has done
 * so.
 *
 * \return Zero on success, -1 on error
 */
int
plsddx_continuous(
	int 		id,		/*!< PLS id */
	double		timeout		/*!< Time out period (sec) */
	)
{
	return plsddx_mode_change(id, timeout, 1, 0x24);
}

/**
 * This function generates a scan from a list of cartesian points. 
 *
 * NOTE: the coordinate system for the cartesian data is as follows:
 *
 *         (northing)
 *             y
 *             ^
 *             |
 *             |
 *              ---> x (easting)
 *
 * Heading is positive clockwise! This is in line with GPS type co-ords.
 * A heading (theta) of zero lines up with the y-axis pointing north.
 * +pi/2 lines you up with the x-axis pointing east.
 * -pi/2 lines you up with the x-axis pointing west.
 *
 * The scan data produced is the standard in-house PlsScan format
 * which has the zero and pointing north, -pi/2 east and pi/2 west.
 *
 * The gap size is used to open up a scan if two cartesian points are
 * further apart than the the specified distance. Otherwise the points
 * are joined.
 *
 * Opened up points, or points with no data (ends of chains) are given
 * the range maxR.
 *
 * \return Zero on success, -1 on error
 */
int
plsddx_cartesian_to_scan(
	double		*x,	/*!< Pointer to vector of x-coords */
	double		*y,	/*!< Pointer to vector of y-coords */
	int		n,	/*!< Number of data points */
	double		theta,	/*!< Bearing of scanner to y-axis (rad) */
	double		gap,	/*!< Gap size */
	double		maxR,	/*!< Maximum range */
	PlsScan		*scan	/*!< Pointer to scan structure */
	)
{
	int	i, j, *order, *segment, *tmp, min, p0, p1, nSegs360;
	double	*bearing, *range, mSeg, mLine, cLine, xIntersec, yIntersec;
	double	d, dx, dy;
	float	*range360;

	if (n < 1) {
		for (i=0; i<scan->nSegs; i++)
			scan->range[i] = maxR;
		return 0;
	}

	nSegs360 = (int)(2.0*M_PI/scan->res+0.5);

	/* Allocate memory */
	if ((range360 = (float *)calloc(nSegs360, sizeof(float))) == NULL)
		return rtx_error("plsddx_cartesian_to_scan: no memory\n");
	if ((bearing = (double *)calloc(n, sizeof(double))) == NULL)
		return rtx_error("plsddx_cartesian_to_scan: no memory\n");
	if ((range = (double *)calloc(n, sizeof(double))) == NULL)
		return rtx_error("plsddx_cartesian_to_scan: no memory\n");
	if ((segment = (int *)calloc(n, sizeof(int))) == NULL)
		return rtx_error("plsddx_cartesian_to_scan: no memory\n");
	if ((tmp = (int *)calloc(n, sizeof(int))) == NULL)
		return rtx_error("plsddx_cartesian_to_scan: no memory\n");
	if ((order = (int *)calloc(n, sizeof(int))) == NULL)
		return rtx_error("plsddx_cartesian_to_scan: no memory\n");

	/* Clear scan data */
	for (i=0; i<scan->nSegs; i++)
		scan->range[i] = 0.0;

	/* Calculate range, bearing and segment of each point */
	for (i=0; i<n; i++) {
		bearing[i] = -atan2(x[i], y[i]);
		range[i] = sqrt(x[i]*x[i]+y[i]*y[i]);
		segment[i] = (int)((bearing[i]-scan->res/2.0)/scan->res)+nSegs360/2;
	}

#ifdef DEBUG
	for (i=0; i<n; i++) {
		fprintf(stderr, "%d\t%f\t%f\t%f\t%f\n", i, x[i], y[i], bearing[i], range[i]);
	}

	fprintf(stderr, "Segment: ");
	for (i=0; i<n; i++)
		fprintf(stderr, "%d ", segment[i]);
	fprintf(stderr, "\n");
#endif
	/* Order the segment data (lowest first) */
	for (j=0; j<n; j++)
		tmp[j] = segment[j];
	for (j=0; j<n; j++) {
		min = 10000;
		for (i=0; i<n; i++) {
			if (tmp[i] <= min) {
				min = tmp[i];
				order[j] = i;
			}
		}
		tmp[order[j]] = 10000;
	}
#ifdef DEBUG
	fprintf(stderr, "Order: ");
	for (i=0; i<n; i++)
		fprintf(stderr, "%d ", order[i]);
	fprintf(stderr, "\n");
#endif

	/* 
	 * Remove points with duplicate segment values (choose one with
	 * min range
	 */

	/* 
	 * We do not know what is before or after the data we see, so
	 * assume max range
	 */
	for (i=0; i<segment[order[0]]; i++)
		range360[i] = maxR;
	for (i=segment[order[n-1]]+1; i<nSegs360; i++)
		range360[i] = maxR;

	/*
	 * We first work with a full 360 deg scan, which is made up of
	 * segments of the same size as the final scan.
	 */
	for (i=0; i<n-1; i++) {
		p0 = order[i];
		p1 = order[i+1];

		/* Work out distnace between two points */
		dx = x[p1]-x[p0];
		dy = y[p1]-y[p0];
		d = sqrt(dx*dx+dy*dy);
		if (d > gap) {
			/* Set range to max range if there is a gap */
			range360[segment[p0]] = range[p0];
			for (j=segment[p0]+1; j<segment[p1]; j++) {
				range360[j] = maxR;
			}
		} else {
			/* Work out line between points */
			mLine = dy/dx;
			cLine = y[p0]-mLine*x[p0];

			range360[segment[p0]] = range[p0];
			for (j=segment[p0]+1; j<segment[p1]; j++) {
				/* Work out slope of segment */
				mSeg = -1.0/(tan(j*scan->res-M_PI));
	
				/* Find intersection of two lines */
				xIntersec = cLine/(mSeg-mLine);
				yIntersec = mSeg*xIntersec;
	
				/* Work out range */
				range360[j] = sqrt(xIntersec*xIntersec+yIntersec*yIntersec);
			}
		}
	}
	p1 = order[n-1];
	range360[segment[p1]] = range[p1];

	/* 
	 * Get nSeg segments of the full 360 degree scan that we need 
	 * Note that first seg (0) is at -pi (south) then segs count up 
	 * anti-clockwise
	 */
	p0 = (int)((scan->first-theta+M_PI)/scan->res+0.5);
	if (p0 <0)
		p0 += nSegs360;

#ifdef DEBUG
	fprintf(stderr, "first 360 seg of scan is: %d\n", p0);
#endif

	for (i=0; i<scan->nSegs; i++) {
		if (p0 >= nSegs360) {
			p0 = 0;
		}
		scan->range[i] = range360[p0];
		p0++;
	}

	/* Free memory */
	free(range360);
	free(bearing);
	free(range);
	free(segment);
	free(tmp);
	free(order);

	return 0;
}

#ifdef TEST
int
main()
{
	int		i, n = 16, N = 0;
	PlsScan		scan;
	double		*x, *y, theta, xpos, ypos, *X, *Y;

	/* Input points */
	if ((x = (double *)calloc(n, sizeof(double))) == NULL)
		return rtx_error("no memory\n");
	if ((y = (double *)calloc(n, sizeof(double))) == NULL)
		return rtx_error("no memory\n");
	if ((X = (double *)calloc(n, sizeof(double))) == NULL)
		return rtx_error("no memory\n");
	if ((Y = (double *)calloc(n, sizeof(double))) == NULL)
		return rtx_error("no memory\n");
	for (i=0; i<n; i++)
		fscanf(stdin, "%lf %lf\n", &x[i], &y[i]);

/*
	xpos = (491342.878749+491346.935994)/2.0;
	ypos = (6955132.970109+6955132.891019)/2.0;

	for (i=0; i<n; i++) {
		x[i] -= xpos;
		y[i] -= ypos;

		if (sqrt(x[i]*x[i]+y[i]*y[i]) < 10.0) {
			X[N] = x[i];
			Y[N] = y[i];
			N++;
		}
	}
*/
	scan.nSegs = 361;
	scan.res = 0.00872665;
	scan.first = -1.570796327;
	plsddx_cartesian_to_scan(x, y, n, 0.0, 2.5, 50.0, &scan);

	/* Output points */
	for (i=0; i<scan.nSegs; i++) {
		theta = i*scan.res+scan.first;
		fprintf(stdout, "%f %f\n", -scan.range[i]*sin(theta), scan.range[i]*cos(theta));
	}

	free(x);
	free(y);
	free(X);
	free(Y);

	exit(1);
}
#endif
