/******************************************************************************
 * \file
 * \brief The SICK PLS/LMS slow server
 * \author Jonathan Roberts
 * CSIRO MANUFACTURING SCIENCE & TECHNOLOGY
 * QCAT, PO Box 883, Kenmore, Q 4068, Australia
 *
 *	$Id: plsServer.c 3115 2008-05-19 00:25:09Z roy029 $
 * 
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *****************************************************************************/

static char *rcsid = "$Header$";

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <math.h>
#include <limits.h>
#include <signal.h>

#include <rtx/main.h>
#include <rtx/thread.h>
#include <rtx/message.h>
#include <rtx/error.h>
#include <rtx/time.h>
#include <rtx/mutex.h>
//#include <rtx/sync.h>
#include <rtx/sem.h>
#include <rtx/auth.h>
#include <rtx/srpc.h>

#include "pls.h"
#include "plsddx.h"

#define	INIT_MAIN_PRIO		RTX_THREAD_PRIO_MIN
#define	INIT_SRPC_PRIO		RTX_THREAD_PRIO_MIN
#define	INIT_KILLER_PRIO	RTX_THREAD_PRIO_MIN

#define INIT_BAUD_RATE		38400
#define	INIT_HIGH_SPEED		0
#define INIT_DEV		"com2"
#define INIT_VERBOSE		0
#define INIT_SRPC_PROG_NUM	52
#define	PI			3.141592654

PlsScan		plsData;
LmsScan		lmsData;

DDX_STORE_ID	*storeId;
DDX_STORE_ITEM	*storePlsItem;
DDX_STORE_ITEM	*storeLmsItem;
DDX_STORE_ITEM	*storeCmdItem;
DDX_STORE_ITEM	*storeModeItem;

char		*pname;
int		verbose = INIT_VERBOSE;
Pls		*pls;
RtxThread	*srpcThread;
int		srpcProgNum = INIT_SRPC_PROG_NUM;
int		storeOn = 0;
FILE		*fileBin;
int		logging = 0;
int		localDebug = 0;
RtxSem		*queueSem;
RtxMutex        *queueMut;
int		processing = 0;
int		spyOn = 0;
int		spySeg = 0;
RtxThread	*cmdThread;
RtxThread	*processingThread;
int             srpcPrio = INIT_SRPC_PRIO;
int             shutdownPrio = INIT_KILLER_PRIO;
int             mainPrio = INIT_MAIN_PRIO;

RtxSrpc          *srpcServerHandle;


#define MAX_QUEUE_LEN 16
struct {
	RtxTime		t;
	unsigned char	data[PLS_PACKET_MAX+4];
} buffer[MAX_QUEUE_LEN];
int qHead=0;
int qTail=0;
int qSize=0;

static void
range_proc(
	unsigned char	*buf,
	int		id
	)
{
  int	nsegs;

  if (rtx_mutex_lock (queueMut) == -1)
    rtx_error_null ("range_proc: rtx_mutex_lock() failed");
  /* check queue size */
  if (qSize >= 3) {
    //rtx_message_routine("range_proc: queue size %d",qSize);
  }
  if (qSize >= MAX_QUEUE_LEN) {
    rtx_message_routine("range_proc: queue overflow %d",qSize);
    qTail = (qTail+1)%MAX_QUEUE_LEN;
    qSize--;
    /* also need to decrement semaphore */
    rtx_sem_wait (queueSem); /* this shouldn't actually have to wait */
  }

  /*
   * Copy data into buffer for processing
   */
  nsegs = hex_14_to_dec(buf[0], buf[1]);
  memcpy(buffer[qHead].data, buf, 2*nsegs+2 + 2);
  buffer[qHead].t.seconds = pls->arrivalTime.seconds;
  buffer[qHead].t.nanoSeconds = pls->arrivalTime.nanoSeconds;

  /* increment head */
  qHead = (qHead+1)%MAX_QUEUE_LEN;
  qSize++;
  rtx_sem_post (queueSem);
  if (rtx_mutex_unlock (queueMut) == -1)
    rtx_error_null ("range_proc: rtx_mutex_unlock() failed");

  if (pls->lms) {
    lmsData.status = pls->status;
    lmsData.nSegs = nsegs;
    lmsData.first = pls->thetaMin*(PI/180.0);
    lmsData.res = pls->resolution*(PI/180.0);
  } else {
    plsData.status = pls->status;
    plsData.nSegs = nsegs;
    plsData.first = pls->thetaMin*(PI/180.0);
    plsData.res = pls->resolution*(PI/180.0);
  }

  /*
   * Broadcast sync
   */
  /*if (rtx_sync_broadcast(syncVar)) 
    rtx_error_flush("range_proc: rtx_sync_broadcast: failed");	
  */
}

static void
process_packet()
{
  RtxTime t;

	while (1) {
		/*
	 	 * Wait on semaphore
	 	 */
      if (verbose) rtx_message_routine("process_packet: waiting...");

		if (rtx_sem_wait(queueSem)) {
			rtx_error("process_packet: rtx_sem_wait: failed!");
			return;
		}

		/* enter critical section */
		if (rtx_mutex_lock (queueMut) == -1)
		  rtx_error_null ("process_packet: rtx_mutex_lock() failed");

		if (qSize <=0) {
		  rtx_error_null ("process_packet: queue should not be empty here!");
		  /* leave critical section */
		  if (rtx_mutex_unlock (queueMut) == -1)
		    rtx_error_null ("process_packet: rtx_mutex_unlock() failed");
		  return;
		}
		
		processing = 1;

		if (verbose)
			rtx_message_routine("process_packet: packet t=%d.%09d, qSize=%d",
					    buffer[qTail].t.seconds, buffer[qTail].t.nanoSeconds,
					    qSize);
		if (pls->lms) {
			if ((lmsData.infringe = pls_range_buf_to_m(pls, buffer[qTail].data, lmsData.range, lmsData.intensity))) {
				rtx_error("process_packet: pls_range_buf_to_m: failed");
				/* leave critical section */
				if (rtx_mutex_unlock (queueMut) == -1)
				  rtx_error_null ("process_packet: rtx_mutex_unlock() failed");
				return;
			}
		} else {
			if ((plsData.infringe = pls_range_buf_to_m(pls, buffer[qTail].data, plsData.range, NULL))) {
				rtx_error("process_packet: pls_range_buf_to_m: failed");
				/* leave critical section */
				if (rtx_mutex_unlock (queueMut) == -1)
				  rtx_error_null ("process_packet: rtx_mutex_unlock() failed");
				return;
			}
		}
		t = buffer[qTail].t;

        lmsData.intensity[396] = qHead;
        lmsData.intensity[397] = qTail;
        lmsData.intensity[398] = qSize;

		/* increment tail */
		qTail = (qTail+1)%MAX_QUEUE_LEN;
		qSize--;

		/* leave critical section */
		if (rtx_mutex_unlock (queueMut) == -1)
		  rtx_error_null ("process_packet: rtx_mutex_unlock() failed");

		if (logging)
			pls_data_to_file(pls, fileBin);	/* this doesn't log data from the queue, so be careful */

		/* Send data to the store */
		if (storeOn) {
			if (pls->lms) {
				if (ddx_store_write(storeLmsItem, &lmsData, &t)) {
					rtx_error("process_packet: ddx_store_write: failed");
					return;
				}
			} else {
				if (ddx_store_write(storePlsItem, &plsData, &t)) {
					rtx_error("process_packet: ddx_store_write: failed");
					return;
				}
			}
		}

		if (spyOn) {
			if (spySeg >= pls->samples) {
				fprintf(stderr, "Spying: there is no %d'th segment. Turning spying off!\n", spySeg);
				spyOn = 0;
			} else {
				if (pls->lms) {
					fprintf(stderr, "Spying: %d'th segment = %f (intensity = %d)\n", spySeg, lmsData.range[spySeg], lmsData.intensity[spySeg]);
				} else {
					fprintf(stderr, "Spying: %d'th segment = %f\n", spySeg, plsData.range[spySeg]);
				}
			}
		}

		processing = 0;
	}
}

static void
cmd_thread_func()
{
	int	cmd=-1;

	if (ddx_store_write(storeModeItem, &pls->mode, NULL)) {
		rtx_error("cmd_thread_func: ddx_store_write: failed");
		return;
	}

	while (1) {
		/*
		 * Do a blocking read on the store cmd item
		 */
		switch (ddx_store_read(storeCmdItem, &cmd, NULL, 0.0, 1)) {
        case 0:
          break;
        case 1:
          rtx_message_routine("cmd_thread_func: still waiting for %s, cmd=%d", storeCmdItem->varName, cmd);
          continue;
          break;
        default:
          rtx_error("cmd_thread_func: ddx_store_read: failed");
          return;
		}

        rtx_message_routine("cmd_thread_func: %s got command %d", pls->name, cmd);

		/*
		 * Process command
		 */
		switch (cmd) {
		case 0:
			if (pls_set_mode(pls, PLS_MODE_IDLE))
				rtx_message_warning("cmd_thread_func: couldn't put into idle mode");
			break;
		case 1:
			if (pls_set_mode(pls, PLS_MODE_CONTINUOUS))
				rtx_message_warning("cmd_thread_func: couldn't put into continuous mode");
			break;
		default:
			break;
		}

		/*
		 * Write mode to store
		 */
		if (ddx_store_write(storeModeItem, &pls->mode, NULL)) {
			rtx_error("cmd_thread_func: ddx_store_write: failed");
			return;
		}
	}
}

void
print_menu()
{
	fprintf(stderr, "\n");
	if (localDebug)
		fprintf(stderr, "------ PLS/LMS User Interface -------- local debug = %1d ----\n", pls->debug);
	else
		fprintf(stderr, "------ PLS/LMS User Interface ------- global debug = %1d ----\n", pls->debug);
	fprintf(stderr, "s - get status,    b - change baudrate, B - perm baudrate\n");
	fprintf(stderr, "i - idle mode,     c - continuous mode, m - monitor mode\n");
	fprintf(stderr, "w - change noSegs, v - debug set,       t - test sequence\n");
	fprintf(stderr, "z - start log,     x - stop log,        r - partial data\n");
	fprintf(stderr, "a - average data,  y - setup mode,      d - diagnostic mode\n");
	fprintf(stderr, "j - get one scan,  n - segment spy,     W - get warning field\n");
	fprintf(stderr, "g - toggle local/global debug messages, e - get error\n");
	if (pls->lms) {
	fprintf(stderr, "------ LMS Specific ---------------------------------------\n");
	fprintf(stderr, "k - set variant,   K - perm variant,    h - hardware status\n");
	fprintf(stderr, "u - set units,     o - laser on/off,    f - zero fields\n");
	fprintf(stderr, "I - intensity o/p  T - stop threshold   F - set availability (anti-dazzle)\n");
    fprintf(stderr, "R - set restart    O - get operating data\n");
	}
	fprintf(stderr, "--------------------------------------- q - quit ----------\n");
	fprintf(stderr, "Enter command: ");
}

/*****************************************************************************
 *                            SRPC FUNCTIONS
 *****************************************************************************/

int
get_capability(char * string, int len, int argc, char * argv[])
{
	pls_capability_string(pls, string);

	return 0;
}

int
get_range(char * string, int len, int argc, char * argv[])
{
	int		i;

	strcpy(string, "");
	for (i=0; i<pls->samples; i++) {
		if (pls->lms)
			sprintf(string, "%s %7.3f ", string, lmsData.range[i]);
		else
			sprintf(string, "%s %7.3f ", string, plsData.range[i]);
	}

	return 0;
}

int
get_status(char * string, int len, int argc, char *argv[])
{
	pls_get_status(pls, 1);
	pls_status_byte_print(pls);
	strcpy (string, "OK");
	return 0;
}

int
get_error(char * string, int len, int argc, char *argv[])
{
	pls_get_error(pls, 1);
	pls_status_byte_print(pls);
	strcpy (string, "OK");
	return 0;
}

int
set_cmd(char * string, int len, int argc, char *argv[])
{
	int	cmd;

	cmd = atoi(argv[1]);
	switch (cmd) {
	case 0:
		if (pls_set_mode(pls, PLS_MODE_IDLE)) {
			rtx_message_warning("set_cmd: couldn't put into idle mode");
			strcpy (string, "set_cmd: couldn't put into idle mode");
			return -1;
		}
		break;
	case 1:
		if (pls_set_mode(pls, PLS_MODE_CONTINUOUS)) {
			rtx_message_warning("set_cmd: couldn't put into continuous mode");
			strcpy (string, "set_cmd: couldn't put into continuous mode");
			return -1;
		}
		break;
	default:
		break;
	}

	strcpy (string, "OK");
	return 0;
}

int
main(int argc, char **argv)
{
	int		baudrate = INIT_BAUD_RATE;
	int		highSpeed = 0;
	int		embedded = 0;
	int		useNameFile = 0;
	unsigned int	noSegs = 180;
	unsigned char   set = 0;
	int             angle = 0;
	int             units = 0;
	int		continuous = 1;
	double          resolution = 0.0;
	int		threshold = 70;
    int     val;
    int     val2;
	int		lms = 0;
	int		id = 0;
	int		testing = 0;
	char		device[128];
	char		interface[128];
	int		errflg = 0;
	int		count, s;
	char		name[BUFSIZ] = "";
	char		nameFile[BUFSIZ] = "";
	extern char	*optarg;
	char		string[BUFSIZ], c = 's';
	unsigned int	fieldData[PLS_MAX_NSEGS];
	int		n;
	size_t		i;
	FILE		*fr;
	char 		*lineBuffer = NULL;
	char		serialNum[BUFSIZ];
    int     dont_give_up = 0;

	pname = argv[0];
	sprintf(device, "/dev/%s", INIT_DEV);

	while ((s = getopt(argc, argv, "ti:b:d:v:hr:p:n:lSIN:f")) != -1) {
		switch (s) {
		case 'I':
			continuous = 0;
			break;
		case 'S':
			storeOn = 1;
			break;
		case 'b':
			baudrate = atoi(optarg);
			break;
		case 'd':
			sprintf(interface, "%s", optarg);
			n = strlen(interface);
			interface[n-1] = '\0';
			if (!strcmp(interface, "pls")) {
				embedded = 1;	
			} else if (!strcmp(interface, "lms")) {
				lms = 1;
				embedded = 1;	
			} else if (!strcmp(interface, "hss")) {
				highSpeed = 1;
				sprintf(device, "/dev/%s", optarg);
			} else {
				interface[3] = '\0';
				if (!strcmp(interface, "tty")) {
					sprintf(device, "/dev/%s", optarg);
				} else {
					sprintf(device, "%s", optarg);
				}
			}

			break;
		case 'v':
			verbose = atoi(optarg);
			break;
		case 'r':
			srpcProgNum = atoi(optarg);
			break;
		case 'p':
			mainPrio = atoi(optarg);
			if (mainPrio < 0)
			        mainPrio = 0;
			if (mainPrio == 0) {
			        srpcPrio = mainPrio;
				shutdownPrio = 0;
			}
			break;
		case 'i':
			id = atoi(optarg);
			break;
		case 'n':
			strcpy(name, optarg);
			break;
		case 'N':
			strcpy(nameFile, optarg);
			useNameFile = 1;
			break;
		case 'l':
			lms = 1;
			break;
		case 't':
			testing = 1;
			break;
        case 'f':
            dont_give_up = 1;
            break;
		case 'h':
		case '?':
			errflg++;
		}
	}

	if (errflg) {
	  printf("\nUsage:\t%s {switches}\n", pname);
	  printf("\t-h\t\tHelp message\n");
	  printf("\t-t\t\tRun in interactive test mode\n");
	  printf("\t-f\t\tDon't give up when opening device\n");
	  printf("\t-l\t\tDevice is an LMS\n");
	  printf("\t-S\t\tSend data to DDX store\n");
	  printf("\t-N\t\tName of config file (see below)\n");
	  printf("\t-I\t\tStart up in idle mode\n");
	  printf("\t-b rate\t\tSet baud rate (default = %d)\n", baudrate);
	  printf("\t-d device\tSet device port (default = %s)\n",INIT_DEV);
	  printf("\t-i id\t\tSet PLS/LMS id number (default = %d)\n", id);
	  printf("\t-n name\t\tSet name of device\n");
	  printf("\t-p prio\t\tSet base priority (default = %d)\n", mainPrio);
	  printf("\t-v debug\tSet debug level (default = %d)\n", verbose);
      printf("\t-r srpcNum\tSet SRPC server number (default = %d)\n\n", srpcProgNum);
      printf("The device port is the filename of the device in the /dev directory\n");
      printf("For example, to access PLS/LMS on USB, you can use ttyUSB0\n\n");

      printf("Use the -N option to use a config file.\n");
      printf("This file should contain the serial number followed by names for the store.\n");
      printf("An example config file would look like:\n\n");
      printf("99010237\trear\n99010232\tfront\t# test comment\n99010234\tleft\n\n");

      strtok(strdup(rcsid), " "); /* Needed for some reason */
      fprintf(stderr, "Source  : %s\n", strtok(NULL," "));
      fprintf(stderr, "Version : %s ", strtok(NULL, " "));
      fprintf(stderr, "%s ", strtok(NULL, " "));
      fprintf(stderr, "%s ", strtok(NULL, " "));
      fprintf(stderr, "%s\n", strtok(NULL, " "));
      fprintf(stderr, "Build   : %s %s\n\n", __TIME__,__DATE__);
	  exit(2);
	}

	/*
	 * Initialise error and message modules
	 */
	rtx_message_init("pls", /*RTX_ERROR_MESSAGE |*/ RTX_ERROR_STDERR);
	if (rtx_error_init("pls",/* RTX_ERROR_MESSAGE |*/ RTX_ERROR_STDERR, NULL)) {
		rtx_message_warning("rtx_error_init: failed");
		exit(1);
	}

	/*
	 * Create sync
	 */
	/*
	 if ((syncVar = rtx_sync_init(NULL, RTX_MUTEX_DEFAULT, 0, 
	  RTX_MUTEX_DEFAULT)) == NULL) {
		rtx_error_flush("rtx_sync_init: failed");
		exit(1);
	 }
	*/

	/* 
	 * Initalize queue semaphore and mutex
	 */
	if ((queueSem = rtx_sem_init(NULL, 0, 0)) == NULL) {
		rtx_error_flush("rtx_sem_init: failed");
		exit(1);
	}
	if ((queueMut = rtx_mutex_init(NULL, RTX_MUTEX_DEFAULT,0)) == NULL) {
		rtx_error_flush("rtx_mutex_init: failed");
		exit(1);
	}
	


	/*
	 * If no name specified, create a default
	 */
	if (!strcmp(name, "")) {
		if (lms)
			sprintf(name, "lms%d", id);
		else
			sprintf(name, "pls%d", id);
	}

	/*
	 * Open connection to the PLS/LMS
	 */
	if (verbose)
		rtx_message_routine("opening %s on %s at %d baud", name, device, baudrate);
	if ((pls = pls_open(device, baudrate, highSpeed, verbose, mainPrio, id, name, lms, dont_give_up)) == NULL) {
		rtx_error_flush("pls_open: failed");
		exit(1);
	}
	if (verbose)
		rtx_message_routine("serial number: %s", pls->serialNum);
	/* 
	 * Get name from file. Lookup the serial number.
	 */
	if (useNameFile) {
		if ((fr=fopen(nameFile, "r")) == NULL) {
			rtx_error_flush("cannot open %s file", nameFile);
			exit(1);
		}

		/* Read file a line at a time */
		i = 80;
		while (getline(&lineBuffer, &i, fr) > 0) {
			sscanf(lineBuffer, "%s%s", serialNum, string);
			if (!strcmp(serialNum, pls->serialNum)) {
				strcpy(name, string);
			}
		}
		free(lineBuffer);

		fclose(fr);      
	}
	if (verbose)
		rtx_message_routine("store name: %s", name);

	/*
	 * Initialising STORE stuff
	 */
	if (storeOn) {
		if (ddx_client_init(verbose)) {
			rtx_error_flush("ddx_client_init() failed");
			exit(1);
		}
		if ((storeId = ddx_store_open(NULL, 0, 2)) == NULL) {
			rtx_error_flush("ddx_store_open: failed");
			exit(1);
		}
		if (DDX_STORE_REGISTER_CONST(storeId, PLS_MAX_NSEGS)) {
			rtx_error_flush("DDX_STORE_REGISTER_CONST: failed");
			exit(1);
		}
		if (DDX_STORE_REGISTER_TYPE(storeId, PlsScan)) {
			rtx_error_flush("DDX_STORE_REGISTER_TYPE: failed");
			exit(1);
		}
		if (DDX_STORE_REGISTER_TYPE(storeId, LmsScan)) {
			rtx_error_flush("DDX_STORE_REGISTER_TYPE: failed");
			exit(1);
		}
		if (lms) {
			if ((storeLmsItem = ddx_store_lookup_item(storeId, name, "LmsScan", sizeof(LmsScan))) == NULL) {
				rtx_error_flush("ddx_store_lookup_item: failed");
				exit(1);
			}
		} else {
			if ((storePlsItem = ddx_store_lookup_item(storeId, name, "PlsScan", sizeof(PlsScan))) == NULL) {
				rtx_error_flush("ddx_store_lookup_item: failed");
				exit(1);
			}
		}
		sprintf(string, "%sCmd", name);
		if ((storeCmdItem = ddx_store_lookup_item(storeId, string, "int", sizeof(int))) == NULL) {
			rtx_error_flush("ddx_store_lookup_item: failed");
			exit(1);
		}
		sprintf(string, "%sMode", name);
		if ((storeModeItem = ddx_store_lookup_item(storeId, string, "int", sizeof(int))) == NULL) {
			rtx_error_flush("ddx_store_lookup_item: failed");
			exit(1);
		}
	}

	/*
	 * Launch processing thread
	 */
	if ((processingThread = rtx_thread_create(
			"process_packet", verbose,
			RTX_THREAD_SCHED_OTHER, mainPrio, 0,
			RTX_THREAD_CANCEL_ASYNCHRONOUS,
			(void * (*)(void*))process_packet, NULL,
			NULL, NULL)) == NULL) {
		rtx_error_flush("rtx_thread_create: failed");
		exit(1);
	}

	/*
	 * Setup SRPC server
	 */
	if ((srpcServerHandle = rtx_srpc_init (srpcProgNum)) == NULL) {
		rtx_error_flush("rtx_srpc_init: failed");
		exit(1);
	}
	rtx_srpc_register(srpcServerHandle, "get_capability", get_capability);
	rtx_srpc_register(srpcServerHandle, "get_range", get_range);
	rtx_srpc_register(srpcServerHandle, "get_status",  get_status);
	rtx_srpc_register(srpcServerHandle, "get_error",  get_error);
	rtx_srpc_register(srpcServerHandle, "set_cmd",  set_cmd);

	/*
	 * Set callback function
	 */
	pls_set_callback(pls, range_proc);

	/*
	 * Put into CONTINUOUS MODE
	 */
	if (!testing && continuous) {
		if (pls_set_mode(pls, PLS_MODE_CONTINUOUS)) {
			rtx_error_flush("couldn't change mode");
			exit(1);
		}
	}

	/*
	 * Launch the command thread
	 */
	if (storeOn) {
		if ((cmdThread = rtx_thread_create(
				"cmd_thread_func", verbose,
				RTX_THREAD_SCHED_OTHER, mainPrio, 0,
				RTX_THREAD_CANCEL_ASYNCHRONOUS,
				(void * (*)(void*))cmd_thread_func, NULL,
				NULL, NULL)) == NULL) {
			rtx_error_flush("rtx_thread_create: failed");
			exit(1);
		}
	}

       	rtx_message_routine("%s Pid=%d, storeName=%s", pls->name, getpid(), name);

	if (testing) {
		print_menu();
		while (c != 'q') {
			c = fgetc(stdin);
			switch (c) {
			case 'g':
				if (localDebug) {
					rtx_message_global();
					localDebug = 0;
				}
				else {
					rtx_message_local();
					localDebug = 1;
				}
				break;
			case 'h':
				pls_get_lms_hw_configuration(pls, 1);
				break;
			case 's':
				pls_get_status(pls, 1);
				pls_status_byte_print(pls);
				break;
			case 'e':
				pls_get_error(pls, 1);
				pls_status_byte_print(pls);
				break;
			case 'b':
				fprintf(stderr, "Rate - [1] 9600, [2] 19200, [3] 38400, [4] 58800, [5] 111000, [6] 200000, [7] 250000, [8] 334000, [9] 500000?:");
				c = fgetc(stdin);
				c = fgetc(stdin);
				switch (c) {
				case '1':
					baudrate = 9600;
					break;
				case '2':
					baudrate = 19200;
					break;
				case '3':
					baudrate = 38400;
					break;
				case '4':
					baudrate = 58800;
					break;
				case '5':
					baudrate = 111000;
					break;
				case '6':
					baudrate = 200000;
					break;
				case '7':
					baudrate = 250000;
					break;
				case '8':
					baudrate = 334000;
					break;
				case '9':
					baudrate = 500000;
					break;
				default:
					fprintf(stderr, "Choice must be between 1 and 9\n");
					break;
				}
				if (pls_set_baudrate(pls, baudrate)) {
					fprintf(stderr, "couldn't change baudrate\n");
				}
				break;
			case 'v':
				fprintf(stderr, "Debug flag - [0] no messages, [1] basic, [2] detailed, [3] v detailed, [4] mega detailed, [5] special?:");
				c = fgetc(stdin);
				c = fgetc(stdin);
				switch (c) {
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
					sprintf(string, "%c", c);
					pls->debug = atoi(string);
					break;
				case '5':
					pls->debug = 99;
					break;
				default:
					fprintf(stderr, "Choice must be 0, 1, 2, 3 or 4!\n");
					break;
				}
				break;
			case 'm':
				if (pls_set_mode(pls, PLS_MODE_MON1)) {
					fprintf(stderr, "couldn't change mode\n");
				}
				break;
			case 'c':
				if (pls_set_mode(pls, PLS_MODE_CONTINUOUS)) {
					fprintf(stderr, "couldn't change mode\n");
				}
				noSegs = 361;
				break;
			case 'i':
				if (pls_set_mode(pls, PLS_MODE_IDLE)) {
					fprintf(stderr, "couldn't change mode\n");
				}
				break;
			case 'k':
				fprintf(stderr, "Angle range - [1] 180, [2] 100?:");
				c = fgetc(stdin);
				c = fgetc(stdin);
				switch (c) {
				case '1':
					angle = 180;
					break;
				case '2':
					angle = 100;
					break;
				default:
					fprintf(stderr, "Choice must be 1 or 2!\n");
					break;
				}
	
				fprintf(stderr, "Resolution - [1] 1.0, [2] 0.5, [3] 0.25?:");
				c = fgetc(stdin);
				c = fgetc(stdin);
				switch (c) {
				case '1':
					resolution = 1.0;
					break;
				case '2':
					resolution = 0.5;
					break;
				case '3':
					resolution = 0.25;
					break;
				default:
					fprintf(stderr, "Choice must be 1, 2 or 3!\n");
					break;
				}
	
				if (pls_set_variant(pls, angle, resolution)) {
					fprintf(stderr, "can't set variant\n");
				}
				break;
			case 'n':
				if (spyOn) {
					spyOn = 0;
					fprintf(stderr, "Segment spying turned off\n");
				} else {
					fprintf(stderr, "Segment spying turned on\n");
					fprintf(stderr, "Enter spy segment?:");
					fscanf(stdin, "%d", &spySeg);
					spyOn = 1;
				}
				break;	
			case 'K':
				if (pls_set_perm_baudrate(pls, 0x02)) {
					fprintf(stderr, "can't set perm variant\n");
				} else {
					fprintf(stderr, "Perm variant set to %d fov, %f seg\n", pls->fov, pls->resolution);
				}
				break;
			case 'B':
				fprintf(stderr, "Rate - [1] 9600, [2] current %d?:", pls->baudRate);
				c = fgetc(stdin);
				c = fgetc(stdin);
				switch (c) {
				case '1':
					set = 0x00;
					break;
				case '2':
					set = 0x01;
					break;
				default:
					fprintf(stderr, "Choice must be 1 or 2!\n");
					break;
				}
				if (pls_set_perm_baudrate(pls, set)) {
					fprintf(stderr, "can't set perm baudrate\n");
				}
				break;
			case 'o':
				if (pls->laserOn) {
					if (pls_set_lms_laser(pls, 0))
						fprintf(stderr, "Cannot turn laser off\n");
					else
						fprintf(stderr, "Laser transmitter is now OFF");
				} else {
					fprintf(stderr, "This can take up to 16 seconds. Please wait...");
					if (pls_set_lms_laser(pls, 1))
						fprintf(stderr, "cannot turn laser on\n");
					else
						fprintf(stderr, "laser transmitter is now ON");
				}
				break;
			case 'f':
				fprintf(stderr, "This can take up to 10 seconds. Please wait...");
				if (pls_set_lms_zero_fields(pls))
					fprintf(stderr, "can't zero fields\n");
				else
					fprintf(stderr, "done\n");
				break;
			case 'I':
				fprintf(stderr, "This can take up to 10 seconds. Please wait...");
				if(pls_set_lms_setting_measured(pls, 0x01))
					fprintf(stderr, "can't change setting\n");
				else
					fprintf(stderr, "done\n");
				break;
			case 'F':
				fprintf(stderr, "Enter availability value (5 is good)?:");
				fscanf(stdin, "%d", &val);
				fprintf(stderr, "Setting availability to %d\n", val);
				fprintf(stderr, "This can take up to 10 seconds. Please wait...");
				if (pls_set_lms_availability(pls, (unsigned char)val))
					fprintf(stderr, "can't set availability\n");
				else
					fprintf(stderr, "done\n");
				break;
            case 'R':
                fprintf(stderr, "Definition of restart modes:\n");
                fprintf(stderr, "00h:   Restart when restart button is actuated\n");
                fprintf(stderr, "01h:   Restart after a set time\n");
                fprintf(stderr, "       No restart block (default setting)\n");
                fprintf(stderr, "02h:\n");
                fprintf(stderr, "03h:   Restart button switches field set, restart after a set time\n");
                fprintf(stderr, "04h:   Restart button switches field set, no restart block\n");
                fprintf(stderr, "05h:   LMS 2xx operates as a slave, restart after a set time\n");
                fprintf(stderr, "06h:   LMS 2xx operates as a slave, immediate restart\n");
                fprintf(stderr, "Bit 5:\n");
                fprintf(stderr, "0:     No motor flap (default setting)\n");
                fprintf(stderr, "1:     Use motor flap (output A controls the motor flap, output A cannot be used for\n");
                fprintf(stderr, "       evaluating the field)\n");
                fprintf(stderr, "Bit 6:\n");
                fprintf(stderr, "0:     No master (default setting)\n");
                fprintf(stderr, "1:     Master (output C outputs the synchronisation cycle)\n");
                fprintf(stderr, "Bit 7:\n");
                fprintf(stderr, "0:     Time basis for restart 1 s (default setting)\n");
                fprintf(stderr, "1:     Time basis for restart 1/10 s\n");
                fprintf(stderr, "Enter restart mode (2=off, 1=after set time)?:\n");
                fscanf(stdin, "%d", &val);
                if (val != 2) {
                  fprintf(stderr, "Enter restart time (0-255)?:");
                  fscanf(stdin, "%d", &val2);
                } else {
                  val2 = 02;    /* default setting */
                }
				fprintf(stderr, "Setting restart to (%d,%d)\n", val,val2);
				fprintf(stderr, "This can take up to 10 seconds. Please wait...");
				if (pls_set_lms_restart(pls, (unsigned char)val, (unsigned char)val2))
					fprintf(stderr, "can't set restart\n");
				else
					fprintf(stderr, "done\n");
				break;
            case 'O':
              if (pls_get_lms_operating_data(pls, 1))
                fprintf(stderr, "can't get operating data\n");
              break;
			case 'T':
				fprintf(stderr, "Enter threshold in mV (normal is 70)?:");
				fscanf(stdin, "%d", &threshold);
				fprintf(stderr, "Setting threshold to %d mV\n", threshold);
				fprintf(stderr, "This can take up to 10 seconds. Please wait...");
				if (pls_set_lms_stop_threshold(pls, threshold))
					fprintf(stderr, "can't set stop threshold\n");
				else
					fprintf(stderr, "done\n");
				break;
			case 'u':
				fprintf(stderr, "Units - [1] cm, [2] mm, [3] dm?:");
				c = fgetc(stdin);
				c = fgetc(stdin);
				switch (c) {
				case '1':
					units = PLS_UNIT_CM;
					break;
				case '2':
					units = PLS_UNIT_MM;
					break;
				case '3':
					units = PLS_UNIT_DM;
					break;
				default:
					fprintf(stderr, "Choice must be 1, 2 or 3!\n");
					break;
				}
				fprintf(stderr, "This can take up to 10 seconds. Please wait...");
				if (pls_set_lms_units(pls, units))
					fprintf(stderr, "can't set units\n");
				else
					fprintf(stderr, "done\n");
				break;
			case 'W':
				if (pls_get_warning_field(pls, &noSegs, fieldData)) {
					fprintf(stderr, "can't get warning field\n");
				} else {
					fprintf(stderr, "Enter warning field data file to write?:");
					fscanf(stdin, "%s", string);
					fprintf(stderr, "Writing %s data file\n", string);
					if (pls_file_write_warning_field(string, noSegs, fieldData))
						fprintf(stderr, "couldn't write warning field data\n");
				}
				break;
			case 'w':
				fprintf(stderr, "Options:\n[1] from file\n[2] 180 deg fan 0m\n[3] 180 deg fan 1m\n[4] 180 deg fan 2m\n[5] 60 deg fan 1.2m\nEnter option:");
				c = fgetc(stdin);
				c = fgetc(stdin);
				switch (c) {
				case '5':
					for (i=0; i<60; i++)
						fieldData[i] = 0;
					for (i=60; i<120; i++)
						fieldData[i] = 120;
					for (i=120; i<181; i++)
						fieldData[i] = 0;
					noSegs = 180;
					break;
				case '4':
					for (i=0; i<181; i++)
						fieldData[i] = 200;
					noSegs = 180;
					break;
				case '3':
					for (i=0; i<181; i++)
						fieldData[i] = 100;
					noSegs = 180;
					break;
				case '2':
					for (i=0; i<181; i++)
						fieldData[i] = 0;
					noSegs = 180;
					break;
				case '1':
					fprintf(stderr, "Enter warning field data file to read?:");
					fscanf(stdin, "%s", string);
					fprintf(stderr, "Loading %s data file\n", string);
					if (pls_file_read_warning_field(string, &noSegs, fieldData))
						fprintf(stderr, "couldn't read warning field data\n");
					break;
				default:
					fprintf(stderr, "Invalid choice!\n");
					break;
				}

				fprintf(stderr, "Sending to PLS...\n");
				if (pls_set_warning_field(pls, noSegs, fieldData)) {
					fprintf(stderr, "can't set warning field\n");
				}
				break;
			case 't':
				if (lms)
					fprintf(stderr, "TESTING LMS...will take approx. 20 seconds...\n");
				else
					fprintf(stderr, "TESTING PLS...will take approx. 20 seconds...\n");
				count = 0;
				if (pls_set_mode(pls, PLS_MODE_CONTINUOUS)) {
					count++;
					fprintf(stderr, "pls_set_mode to CONTINUOUS failed\n");
				}
				sleep(1);
				if (pls_set_mode(pls, PLS_MODE_IDLE)) {
					count++;
					fprintf(stderr, "pls_set_mode to IDLE failed\n");
				}
				sleep(1);
				if (pls_set_baudrate(pls, 9600)) {
					count++;
					fprintf(stderr, "pls_set_baudrate to 9600 failed\n");
				}
				sleep(1);
				if (pls_set_baudrate(pls, baudrate)) {
					count++;
					fprintf(stderr, "pls_set_baudrate to %d failed\n", baudrate);
				}
				sleep(1);
				if (pls_set_perm_baudrate(pls, 1)) {
					count++;
					fprintf(stderr, "pls_set_perm_baudrate failed\n");
				}
				sleep(1);
				if (pls_set_mode(pls, PLS_MODE_IDLE)) {
					count++;
					fprintf(stderr, "pls_set_mode to IDLE failed\n");
				}
				if (count == 0)
					fprintf(stderr, "SYSTEM IS FULLY OPERATIONAL !!!\n");
				else
					fprintf(stderr, "SYSTEM IS ILL, with a total of %d failures\nplstest: Suggest you investigate. Turn verbose onto 3 and run this test again.\n", count);
				break;
			case 'z':
				switch (pls->mode) {
					case PLS_MODE_MON1:
					case PLS_MODE_IN_FIELD:
					case PLS_MODE_CONTINUOUS:
						break;
					default:
						fprintf(stderr, "not in an output mode (current mode = 0x%x). Put it into either CONTINUOUS (c) or MONITOR (m) mode.\n", pls->mode);
						continue;
				}
				if ((fileBin=pls_open_log_file(pls, "")) == NULL) {
					fprintf(stderr, "pls_open_log_file() failed\n");
					exit(2);
				}
				logging = 1;
				break;
			case 'x':
				if (logging == 1) {
					logging = 0; 
					fclose(fileBin);
				}
				break;	
			case 'r':
				if (pls_get_partial_range(pls, 90, 181)) {
					fprintf(stderr, "can't get partial range\n");
				}
				break;
			case 'a':
				fprintf(stderr, "Enter number of scans (2 to 250):");
				fscanf(stdin, "%s", string);
				if (pls_get_average_range(pls, atoi(string))) {
					fprintf(stderr, "can't get average range\n");
				}
				break;
			case 'd':
				if (pls_set_mode(pls, PLS_MODE_DIAGNOSTIC)) {
					fprintf(stderr, "couldn't change mode\n");
				}
				break;	
			case 'y':
				if (pls_set_mode(pls, PLS_MODE_SETUP)) {
					fprintf(stderr, "couldn't change mode\n");
				}
				break;	
			case 'j':
				if (pls_get_range(pls)) {
					fprintf(stderr, "couldn't get scan\n");
				}
				break;
			case 'q':
				c = 'q';
				break;
			default:
				print_menu();
				break;
			}
		}
	} else {
		if (rtx_main_wait_shutdown(shutdownPrio)) {
			rtx_error_flush("rtx_main_wait_shutdown: failed");
			exit(1);
		}
	}

	if (verbose)
		rtx_message_routine("shutting down ...");

	/*
	 * Close the PLS/LMS
	 */
	if (pls_close(pls)) {
		rtx_error_flush("pls_close: failed");
		exit(1);
	}

	/*
	 * Destroy the processing thread
	 */
	if (rtx_thread_destroy_sync(processingThread)) {
		rtx_error_flush("rtx_thread_destroy_sync: failed");
		exit(1);
	}

	/*
	 * Destroy sync
	 */
/* COMMENT
	if (rtx_sync_destroy(syncVar)) {
		rtx_error_flush("rtx_sync_destroy: failed");
		exit(1);
	}
*/



	/*
	 * Cleanup store stuff
	 */
	if (storeOn) {
		if (verbose)
			rtx_message_routine("cleaning up DDX store stuff");
		if (lms)
			ddx_store_done_item(storeLmsItem);
		else
			ddx_store_done_item(storePlsItem);
		ddx_store_done_item(storeCmdItem);
		ddx_store_done_item(storeModeItem);
		ddx_store_close(storeId);
		ddx_client_done();
	}

	/* 
	 * Destroy queue sem and mutex
	 */
	if (rtx_sem_destroy(queueSem)) {
	  rtx_error_flush("rtx_sem_destroy: failed");
	  exit(1);
	}
	if (rtx_mutex_destroy(queueMut)) {
	  rtx_error_flush("rtx_mutex_destroy: failed");
	  exit(1);
	}

       	rtx_message_routine("%s exited", pls->name);

	exit(0);
}

