/*****************************************************************************
 * /file
 * /brief This is a simple program to convert the binary file produced by
 * the pls_data_to_file() function to an ASCII file
 * /author   Jonathan Roberts jmr@brb.dmt.csiro.au
 *      CSIRO ICT Centre Robotics Team 
 *      QCAT, PO Box 883, Kenmore, Q 4068, Australia
 *
 * Copyright (c) CSIRO 2004
 */
static char *rcsid = "$Header$";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <termio.h>
#include <time.h>
#include <sys/types.h>
#include <errno.h>
#include <limits.h>
#include <time.h>
                                                                                
#define TIMER_TO_DOUBLE(t) ((double)t.tv_sec+(double)(t.tv_nsec*1e-9))

#define    PLS_PACKET_MAX    808
#define hex_16_to_dec(hex1, hex2)  ((unsigned int)((hex2<<8)+hex1))
#define hex_13_to_dec(hex1, hex2)  ((unsigned int)(((hex2 & 0x1f) <<8 ) + hex1))
#define hex_top3_to_dec(hex1)  ((unsigned int) ((hex1 & 0xe0)>>5))


void
pls_int_swap(int *data)
{
    char    t, *p = (char *)data;
    t = p[0]; p[0] = p[3]; p[3] = t; /* swap bytes 0 & 3 */
    t = p[1]; p[1] = p[2]; p[2] = t; /* swap bytes 1 & 2 */
}

void
pls_double_swap(double *data)
{
    int    *p;
    int    t;
    p = (int *)data;
    pls_int_swap(p);
    pls_int_swap(p+1);
    t = p[0];
    p[0] = p[1];
    p[1] = t;
}

/**
 * This function converts a binary data file generated by pls_data_to_file()
 * to a nice human readable ASCII file.
 */

static int
pls_binToAscii(
	FILE *fileBin, 
	FILE *fileAscii, 
	char timeType, 
	int nodder,
	int intensity)
{
    int             i, id, noSegs, magic, swapping = 0;
    unsigned char   buf[PLS_PACKET_MAX+4], *p;
    time_t          timeSec;
    struct timespec timeStamp;
    struct timespec to, td;
    char            timeString[128];
    double          pos;

    fread(&magic, sizeof(int), 1, fileBin);
    if (magic != 0x12345678) {
        fprintf(stderr, "pls_binToAscii: byte swapping required\n");
        swapping++;
        }

    while ( fread(&id, sizeof(int), 1, fileBin) ) {
        fread(&timeStamp.tv_sec, sizeof(time_t), 1, fileBin);
        fread(&timeStamp.tv_nsec, sizeof(long), 1, fileBin);
        fread(&noSegs, sizeof(int), 1, fileBin);

        if (swapping) {
            pls_int_swap(&id);
            pls_int_swap((int *)&timeStamp.tv_sec);
            pls_int_swap((int *)&timeStamp.tv_nsec);
            pls_int_swap(&noSegs);
            }

        fread(&buf, 2*noSegs*sizeof(unsigned char), 1, fileBin);

        if (nodder) {
           fread(&pos, sizeof(double), 1, fileBin);
           if (swapping) pls_double_swap(&pos);
           }

        /*
        * Print to file fileAscii
        */

        switch (timeType) {

          case 'n':	/* Normal Time */
            fprintf(fileAscii, "%d ", id);
            fprintf(fileAscii, "%d ", (int)timeStamp.tv_sec);
            fprintf(fileAscii, "%d ", (int)timeStamp.tv_nsec);
            fprintf(fileAscii, "%d ", noSegs);
            break;

          case 'f':
            fprintf(fileAscii, "%d ", id);
            fprintf(fileAscii, "%d.", (int)timeStamp.tv_sec); 
	    fprintf(fileAscii, "%09d ", (int)timeStamp.tv_nsec);
            fprintf(fileAscii, "%d ", noSegs);
	    break;

          case 'l':	/* Local Time String */
            timeSec = (time_t)timeStamp.tv_sec;
            strftime(timeString, 128, "%Y%m%d%H%M%S", localtime(&timeSec));
            fprintf(fileAscii, "%d ", id);
            fprintf(fileAscii, "%s ", timeString);
            fprintf(fileAscii, "%d ", (int)timeStamp.tv_nsec);
            fprintf(fileAscii, "%d ", noSegs);
            break;

	  case 'a': case 'A': /* Absolute Time */
            fprintf(fileAscii, "%g ", TIMER_TO_DOUBLE(timeStamp));
	    fprintf(fileAscii, "%d ", id);
	    fprintf(fileAscii, "%d ", noSegs);
            break;

	  case 'r': case 'R': /* Relative Time */
	    fprintf(stderr, "Relative Time Not working Yet\n");
            fprintf(fileAscii, "%g ", TIMER_TO_DOUBLE(timeStamp));
	    fprintf(fileAscii, "%d ", id);
	    fprintf(fileAscii, "%d ", noSegs);
            break;

	  case 'd': case 'D': /* Difference Time */ 
	    
	    if( to.tv_sec ) {
              td.tv_nsec = timeStamp.tv_nsec - to.tv_nsec;
              td.tv_sec  = timeStamp.tv_sec  - to.tv_sec;
              if (td.tv_nsec < 0) { td.tv_nsec += 1000000000; td.tv_sec--; }
	      }

	    to.tv_nsec = timeStamp.tv_nsec;
	    to.tv_sec  = timeStamp.tv_sec;

            fprintf(fileAscii, "%g ", TIMER_TO_DOUBLE(td));
	    fprintf(fileAscii, "%d ", id);
	    fprintf(fileAscii, "%d ", noSegs);
	    break;

          default: 
            fprintf(stderr,"Unrecognised timeType (%c)\n", timeType);
            return -1;
            break;
       }	/* End of Switch */

       /*
       * Convert the raw PLS data into range data in cm
       */

       for (p=&buf[0],i=0; i<noSegs; i++, p+=2) {
           fprintf(fileAscii, "%d ", hex_13_to_dec(p[0], p[1]));
           if (intensity) fprintf(fileAscii, "%d ", hex_top3_to_dec(p[1]));
       }

       if (nodder)
            fprintf(fileAscii, "%f", pos);
                fprintf(fileAscii, "\n");

    } /* End of While Loop */

    return 0;
}

int
main(int argc, char **argv)
{
    int  i, errflag = 0;
    int  quiet = 0;
    char timeType = 'n';	/* Default */
    int  intensity = 0;
    int  nodder = 0;
 
    for(i=1; i<argc; i++)
      {
      if( argv[i][0] != '-' ) errflag++;
      else switch( argv[i][1] )
        {
        case 'f': timeType  = argv[++i][0]; break;
        case 'n': nodder    = 1; break;
        case 'q': quiet     = 1; break;
        case 'i': intensity = 1; break;
        default : errflag++; 
        }    
    }

    if (errflag) {
      fprintf(stderr, "\nUsage:%s [switches] < file.bin > file.pls\n", argv[0]);
      fprintf(stderr, "\t-q\t\tQuiet mode\n");
      fprintf(stderr, "\t-h\t\tPrints this help message and exits\n");
      fprintf(stderr, "\t-f format\tTime output format (%c)\n", timeType);
      fprintf(stderr, "\t   Normal\t(id sec nsec nseg ...)\n");
      fprintf(stderr, "\t   Float\t(id sec.nsec nseg ...)\n");
      fprintf(stderr, "\t   Local\t(id timestring.nsec nseg ...)\n");
      fprintf(stderr, "\t   Absolute\t\t(FloatTime id nseg ...)\n");
      fprintf(stderr, "\t   Relative\t\t(FloatTime id nseg ...)\n");
      fprintf(stderr, "\t   Difference\t\t(FloatTime id nseg ...)\n\n");

      strtok(strdup(rcsid), " "); /* Needed for some reason */
      fprintf(stderr, "Source  : %s\n", strtok(NULL," "));
      fprintf(stderr, "Version : %s ", strtok(NULL, " "));
      fprintf(stderr, "%s ", strtok(NULL, " "));
      fprintf(stderr, "%s ", strtok(NULL, " "));
      fprintf(stderr, "%s\n", strtok(NULL, " "));
      fprintf(stderr, "Build   : %s %s\n\n", __TIME__,__DATE__);

      exit(2);
      }

    if (!quiet) fprintf(stderr, "Converting Binary file to ASCII file...");
    pls_binToAscii(stdin, stdout, timeType, nodder, intensity);
    if (!quiet) fprintf(stderr, "Done.\n");
    exit(1);
}
