/*
 *****************************************************************************
 * CSIRO MANUFACTURING SCIENCE & TECHNOLOGY
 * QCAT, PO Box 883, Kenmore, Q 4068, Australia
 *
 *	$Id: pls.h 1684 2007-05-01 05:45:27Z bos057 $
 * 
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *****************************************************************************
 */

/**
 *****************************************************************************
 * \file
 * \brief The SICK PLS/LMS header file
 * \author Jonathan Roberts
 *****************************************************************************
 */

#include <unistd.h>

#include <rtx/time.h>
#include <rtx/mutex.h>
#include <rtx/thread.h>
#include <rtx/sem.h>
#ifdef USE_HSS
#include "hss.h"
#endif

#define STX		0x02
#define ACK		0x06
#define NAK		0x15

/*
 * Messages to the sensor
 */
#define	INIT_TGM	0x10	/* Initialisation and reset */
#define BM_TGM		0x20	/* Select or change operating mode */
#define MWANF_TGM	0x30	/* Measured value request */
#define SSANF_TGM	0x31	/* Sensor status request */
#define	ERRANF_TGM	0x32	/* Error telegram request */
#define	TSTANF_TGM	0x33	/* Test in diagnostic mode request */
#define	MEMANF_TGM	0x34	/* Memory dump request */
#define	ODCANF_TGM	0x35	/* Operating data counter request */
#define MMWANF_TGM	0x36	/* Averaged measured values request */
#define MWPANF_TGM	0x37	/* Average measured values partial-range req */
#define	BSFKFG_TGM	0x40	/* Configure the safe protection field */
#define	BSFCONF_TGM	0x41	/* Confirmation of the configured safe */
				/* protection field */
#define	PWCHG_TGM	0x42	/* Change the password for setting up the */
				/* protection field */
#define KSFKFG_TGM	0x43	/* Configure the warning field */
#define	WANL_TGM	0x44	/* Define the restart behaviour */
#define	SFANF_TGM	0x45	/* Configured protection field request */
#define	LMOD_TGM	0x46	/* Teach mode for SF configuration */
#define	WANLCFRM_TGM	0x47	/* Confirm restart definition */
#define	AGL_TGM		0x50	/* Perform calibration */
#define	ANLDEF_TGM	0x60	/* Define startup test */
#define	ANLCFRM_TGM	0x61	/* Confirm startup test */
#define	OUTPUTDEF_TGM	0x62	/* Define switching output WF or dirt */
#define	RSPTIMEDEF_TGM	0x63	/* Define the number of multiple evals. */
#define	WSFTYPDEF_TGM	0x65	/* Define WF type */
#define BRPERMDEF_TGM	0x66	/* Define permanent baud rate */
#define	ADRDEF_TGM	0x67	/* Define sensors address */
#define	LASER_TGM	0x68	/* Activate/deactivate laser transmitter */
#define	VARDEF_TGM	0x3b	/* Define LMS variant */
#define	LMSKFGANF_TGM	0x74	/* LMS hardware configuration request */
#define	LMSKFGDEF_TGM	0x77	/* Define LMS hardware configuration */

/*
 * Messages from the sensor
 */
#define PWON_TGM	0x90	/* Message after power on */
#define INIT_ACK_TGM	0x91	/* Confirm SW reset telegram */
#define NACK_TGM	0x92	/* Not acknowledge (eg. invlaid op. change) */
#define BMACK_TGM	0xa0	/* Response to operating mode change */
#define MW_TGM		0xb0	/* Response to measured value request */
#define SS_TGM		0xb1	/* Sensor status */
#define ERR_TGM		0xb2	/* Error message/Test message */
#define MEMRD_TGM	0xb4	/* Memory dump on request (diag. mode only) */
#define ODC_TGM     0xb5        /* operating data counters */
#define MMW_TGM		0xb6	/* Send average measured values */
#define MWP_TGM		0xb7	/* Send measured value partial-range */
#define	VARDEFACK_TGM	0xbb	/* Response to a variant change */
#define BSFACK_TGM	0xc0	/* Response to SF configuration */
#define BSFCFRM_TGM	0xc1	/* Response to SF confirmation */
#define	PWACK_TGM	0xc2	/* Confirm new password */
#define KSFACK_TGM	0xc3	/* Response to WF configuration */
#define WANLACK_TGM	0xc4	/* Response to restart definition */
#define SFDAT_TGM	0xc5	/* Response to protection field request */
#define LMODACK_TGM	0xc6	/* Response in teach mode */
#define	AGLACK_TGM	0xd0	/* Response to calibration */
#define ANLACK_TGM	0xe0	/* Response to start test definition */
#define OUTPUTACK_TGM	0xe2	/* Response to initial definition */
#define RSPTIMEACK_TGM	0xe4	/* Response to multiple evaluation definition */
#define	WSFTYPACK_TGM	0xe5	/* Response to WF type definition */
#define BRPERMACK_TGM	0xe6	/* Response to permanent baud rate definition */
#define	ADRACK_TGM	0xe7	/* Response to sensor address definition */
#define	LASERACK_TGM	0xe8	/* Response to activate/deactivate laser */
#define	LMSKFG_TGM	0xf4	/* LMS hardware configuartion */
#define	LMSKFGACK_TGM	0xf7	/* Response to LMS configuration definition */
#define FAKE_TGM	0xff

#define PLS_READER_PRIO		20
#define PLS_MAX_STR_LEN		128
#define PLS_ID			0x80
#define	PLS_PACKET_MAX		810
#define PLS_TTY_VTIME		1
#define PLS_MAX_NO_ACK		2
#define PLS_TRIES_MAX		3
#define PLS_MAX_REPLY_WAIT	12	/* Seconds */
#define PLS_MAX_NSEGS		401

#define PLS_PASSWORD		"SICK_PLS"
#define LMS_PASSWORD		"SICK_LMS"

/**
 * The PLS unit enum
 */
typedef enum _pls_unit {
	PLS_UNIT_CM	= 0,
	PLS_UNIT_MM	= 1,
	PLS_UNIT_DM	= 2
} PlsUnit;

/**
 * The PLS mode enum
 */
typedef enum _pls_mode {
	PLS_MODE_SETUP			= 0x00,
	PLS_MODE_DIAGNOSTIC		= 0x10,
	PLS_MODE_MON1			= 0x20,
	PLS_MODE_IN_FIELD		= 0x21,
	PLS_MODE_CONTINUOUS		= 0x24,
	PLS_MODE_IDLE			= 0x25,
	PLS_MODE_PASSWORD_TEST		= 0x30,
	PLS_MODE_B38400			= 0x40,
	PLS_MODE_B19200			= 0x41,
	PLS_MODE_B9600		 	= 0x42,
	PLS_MODE_B58800			= 0x43,
	PLS_MODE_B111000		= 0x44,
	PLS_MODE_B200000		= 0x45,
	PLS_MODE_B250000		= 0x46,
	PLS_MODE_B334000		= 0x47,
	PLS_MODE_B500000		= 0x48,
	PLS_MODE_POWER_ON		= 0x21,
	PLS_MODE_CONTINUOUS_OVERRIDE	= 0xfd,
	PLS_MODE_INIT	 	 	= 0xfe,	
	PLS_MODE_UNKNOWN	 	= 0xff	
} PlsMode;

/**
 * The LMS hardware configuration structure
 */
typedef struct _lms_hw {
	int	blanking;
	int	stopThreshold;
	char	availability;
	char	settingMeasured;
	char	unitsMeasured;
	char	temporaryField;
	char	subtractive;
	char	multipleEvaluation;
	char	restart;
	char	restartTime;
	char	restartTest;
	char	cAObject;
	char	cAPositive;
	char	cANegative;
	char	cAStarting;
	char	cAStopping;
	char	cBObject;
	char	cBPositive;
	char	cBNegative;
	char	cBStarting;
	char	cBStopping;
	char	cCObject;
	char	cCPositive;
	char	cCNegative;
	char	cCStarting;
	char	cCStopping;
	char	pixel;
	char	single;
	int	zeroPoint;
} LmsHw;

/**
 * The Pls type
 */
typedef struct _pls {
	int		fd;
	int		id;		/*!< Local ID given passed by pls_open */
	int		lms;		/*!< =0 if unit is a PLS, =1 if LMS */
	LmsHw		lmsHw;		/*!< LMS hardware configuration */
	int		laserOn;	/*!< =0 if laser off, =1 if laser on */
	char		name[BUFSIZ];	/*!< Name used to aid debugging */
	PlsMode		mode;		/*!< Current mode of PLS */
	unsigned int	baudRate;	/*!< Baudrate of PLS */
	int		highSpeed;	/*!< =0 low speed , =1 high speed */
	char		device[PLS_MAX_STR_LEN];/*!< com port, e.g. com1 */
	int		firmWare;	/*!< e.g. =7 if firmware 3.07 */
	char		serialNum[9];	/*!< Serial number (production code) */
	int		status;		/*!< PLS status byte, >2 is bad */
	int		dirt;		/*!< =0 if no dirt, =1 if dirt */
	int		noContact;
	int		debug;		/*!< Debug flag */
    int     hoursRunning;         /*!< number of hours running  */
    int     switchOnCount;        /*!< number of switch on's */
	unsigned int	samples;	/*!< Effective resolution */
	unsigned int	first;		/*!< Number of first segment */
	unsigned int	last;		/*!< Number of last segment */
	PlsUnit		unit;		/*!< Unit of range data */
	int		fov;		/*!< Field-of-view=180 or 100 deg */
	double		resolution;	/*!< Angular res=0.25, 0.5 or 1.0 deg */
	double		thetaMin;	/*!< First angular reading deg */
	double		thetaMax;	/*!< Last angular reading deg */
	double		rMax;		/*!< Maximum range (m) */
	int		checkPktLen;
	RtxMutex	*mutex;
	RtxThread	*readerThread;
	RtxThread	*stopHangThread;
	RtxThread	*monitorContinuousThread;
	int		lastAck;
	RtxSem		*replySem;
	int		replyWaiting;
	unsigned char	reply[PLS_PACKET_MAX+4];/*!< The reply buffer */
	unsigned int	replyLen;		/*!< Length of the reply */
	unsigned char	*rangeData;		/*!< Ptr to the range data */
	unsigned int	maxReplyWait;
	void		(*range_proc)(unsigned char *b, int id);
	void		(*header_proc)(int id);
	RtxTime		arrivalTime;
	int		readerPrio;	/*!< Priority of the reader thread */
} Pls;

#define hex_16_to_dec(hex1, hex2)  ((unsigned int)((hex2<<8)+hex1))
#define hex_14_to_dec(hex1, hex2)  ((unsigned int)(((hex2 & 0x3f) <<8 ) + hex1)) 
#define hex_13_to_dec(hex1, hex2)  ((unsigned int)(((hex2 & 0x1f) <<8 ) + hex1)) 
Pls * pls_open(char *dev, unsigned int baudrate, int highSpeed, int debug, 
	int readerPrio, int id, char *name, int lms, int dont_give_up);
int pls_close(Pls *pls);
void pls_set_callback(Pls *pls, void (*p)(unsigned char *b, int id) );
void pls_set_header_callback(Pls *pls, void (*p)(int id) );
int pls_set_mode(Pls *pls, PlsMode mode);
int pls_set_baudrate(Pls *pls, unsigned int baudrate);
int pls_set_perm_baudrate(Pls *pls, unsigned char perm);
int pls_get_warning_field(Pls *pls, unsigned int *noSegs, unsigned int *data);
int pls_set_warning_field(Pls *pls, unsigned int noSegs, unsigned int *data);
int pls_file_write_warning_field(char *filename, unsigned int noSegs, unsigned int *data);
int pls_file_read_warning_field(char *filename, unsigned int *noSegs, unsigned int *data);
PlsMode pls_get_status(Pls *pls, int printMessage);
PlsMode pls_get_error(Pls *pls, int printMessage);
void pls_status_byte_print(Pls *pls);
int pls_set_variant(Pls *pls, int angle, double resolution);
int pls_get_range(Pls *pls);
int pls_get_partial_range(Pls *pls, unsigned int first, unsigned int last);
int pls_get_average_range(Pls *pls, unsigned int nscans);
int pls_get_lms_hw_configuration(Pls *pls, int printMessage);
int pls_set_lms_hw_configuration(Pls *pls, LmsHw *config);
void pls_print_lms_hw_configuration(Pls *pls, FILE *fd);
int pls_get_lms_operating_data(Pls *pls, int printMessage);
int pls_set_lms_availability(Pls *pls, unsigned char availability);
int pls_set_lms_restart(Pls *pls, unsigned char restartMode, unsigned char restartTime);
int pls_set_lms_stop_threshold(Pls *pls, int threshold);
int pls_set_lms_units(Pls *pls, PlsUnit unit);
int pls_set_lms_zero_fields(Pls *pls);
int pls_set_lms_setting_measured(Pls *pls, unsigned char setting);
int pls_set_lms_laser(Pls *pls, unsigned char state);
FILE *pls_open_log_file(Pls *pls, char *filename);
void pls_data_to_file(Pls *pls, FILE *fw);
int pls_binToAscii_name(char *nameBin, char *nameAscii);
int pls_binToAscii(FILE *fileBin, FILE *fileAscii, int timeType);
void pls_intToHex(unsigned int number, unsigned char *data);
void pls_int_swap(int *data);
unsigned int pls_crc16_build(unsigned char *data, unsigned int len);
int pls_read(int fd, unsigned char *data, unsigned int pktlen);
int pls_range_buf_to_m(Pls *pls, unsigned char *buf, float *range, unsigned char *state);
int pls_range_cm_to_mm(unsigned char *data, int nsegs, int *range);
int pls_range_cm_to_m(unsigned char *data, int nsegs, float *range, unsigned char *intensity);
int pls_range_mm_to_m(unsigned char *data, int nsegs, float *range, unsigned char *intensity);
void pls_capability_string(Pls *pls, char *string);
void pls_range_string(Pls *pls, int nsegs, int *data, char *string);
