/*
 *****************************************************************************
 * CSIRO MANUFACTURING SCIENCE & TECHNOLOGY
 * QCAT, PO Box 883, Kenmore, Q 4068, Australia
 *
 *	$Id: pls.c 1684 2007-05-01 05:45:27Z bos057 $
 * 
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *****************************************************************************
 */

/**
 *****************************************************************************
 * \file
 * \brief The SICK PLS/LMS C file
 * \author Jonathan Roberts
 *****************************************************************************
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <sys/types.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <limits.h>
#include <sys/ioctl.h>

#include <rtx/time.h>
#include <rtx/timer.h>
#include <rtx/thread.h>
#include <rtx/mutex.h>
#include <rtx/serial.h>
#include <rtx/error.h>
#include <rtx/message.h>

#include "pls.h"

#define	PI		3.141592654
#define	SERIAL_FILE	"pls_serial_data"

/* 
 * Forward defines 
 */
static int pls_open_dev(Pls *pls, int highSpeed);
static int pls_send_packet(Pls *pls, unsigned char *request, unsigned int len);
void pls_reader(Pls *pls);
static int pls_wait_for_reply(Pls *pls, char *func_name);
static void pls_stop_hang(Pls *pls);
static void pls_monitor_continuous_mode(Pls *pls);

extern char *plsErrorCodes[];

/**
 * This function opens a connection to a PLS.
 *
 * \return A pointer to a Pls structure on success, NULL on error
 */
Pls *
pls_open(
	char 		*dev,
	unsigned int	baudrate,
	int		highSpeed,
	int		debug,
	int		readerPrio,
	int		id,
	char		*name,
	int		lms,
    int     dont_give_up
	)
{
	Pls		*pls;
	unsigned char	command[PLS_PACKET_MAX];
	int		result, i;
	int		makeIdle = 1;

	if (debug) {
		rtx_message_routine("pls_open: opening PLS/LMS on dev = %s, baudrate = %d, readerPrio = %d, id = %d, name = %s", dev, baudrate, readerPrio, id, name);
		rtx_message_routine("pls_open: This may take 10 seconds or so ...");
	}

	/* 
	 * Allocate a data structure for the PLS sensor 
	 */
	if ((pls = calloc(1, sizeof(Pls))) == NULL)
		return rtx_error_null("pls_open: calloc: no memory");

	/*
	 * Initialise varibales
	 */
	pls->id = id;
	if (lms)
		pls->lms = 1;

	if (debug) {
		if (pls->lms)
			rtx_message_routine("pls_open: device is an LMS");
		else
			rtx_message_routine("pls_open: device is a PLS");
	}

	strcpy(pls->device, dev);
	strcpy(pls->name, name);
	pls->mode = PLS_MODE_UNKNOWN;
	pls->debug = debug;
	pls->lastAck = NAK;
	pls->checkPktLen = 1;
	pls->status = 0;
	pls->dirt = 0;
	pls->noContact = 0;
	pls->replyWaiting = 0;
	pls->maxReplyWait = PLS_MAX_REPLY_WAIT;
	pls->baudRate = baudrate;
	pls->mode = PLS_MODE_INIT;
	pls->readerPrio = readerPrio;
	pls->samples = 0;
	pls->range_proc = NULL;
	pls->header_proc = NULL;
    pls->hoursRunning = -1;
    pls->switchOnCount = -1;

	/* 
	 * Create a semaphore that will be used to indicate a reply received 
	 */
	if ((pls->replySem = rtx_sem_init(NULL, 0, 0)) == NULL) {
		rtx_error("pls_open: rtx_sem_init: failed");
		free (pls);
		return NULL;
	}

	/* 
	 * Open the RS232 port for PLS 
	 */
	if (pls_open_dev(pls, highSpeed)) {
		rtx_error("pls_open: pls_open_dev: failed");
		free (pls);
		return NULL;
	}

	/* 
	 * At the end of this function, we get the status from the PLS/LMS.
	 * To do this we must make sure that the PLS/LMS isn't outputting
	 * any data. This can be done by putting the PLS/LMS into IDLE
	 * mode.
	 *
	 * Note: the reason we have to do this is because the PLS/LMS
	 *       will stop transmitting its range packets when a request
	 *       from the computer is received. It will just stop mid-packet!
	 */

	if (makeIdle) {
		/* 
	 	 * Construct the IDLE mode change telegram
	 	 */
		command[0] = STX;
		command[1] = 0x00;
		pls_intToHex(2, &command[2]);
		command[4] = BM_TGM;
		command[5] = PLS_MODE_IDLE;
		pls_intToHex(pls_crc16_build(command, 6), &command[6]);

		/*
		 * Send packet
		 */
		if (debug)
			rtx_message_routine("pls_open: attempting to put PLS into IDLE mode");
		for (i=0; i<5; i++) {
			write(pls->fd, command, 8);
			rtx_timer_sleep(0.2);
		}

		/*
		 * Wait a bit to allow time for any range data to stop pouring
 		 * from the PLS. Takes a max of PLS_MAX_REPLY_WAIT according
		 * to the PLS manual.
		 */
		rtx_timer_sleep(1.0);

		/*
		 * Flush RS232 port. This makes sure that all the crappy
		 * data is gone and we will not see it!
		 */
		if (!highSpeed) {
			if (rtx_serial_flush_input(pls->fd)) {
				rtx_error("pls_open: rtx_serial_flush_input: failed");
				free (pls);
				return NULL;
			}
#ifdef USE_HSS
		} else {
		        if (ioctl (pls->fd, HSS_FLUSH_BUFFER, NULL) == -1) {
			        rtx_error_errno ("pls_open: ioctlHSS_FLUSH_BUFFER failed");
				free (pls);
				return (NULL);
			}
#endif
		}
	}

	/*
	 * Enable the receiver
	 */
	if (!highSpeed) {
		if (rtx_serial_enable_receiver(pls->fd) == -1) {
			rtx_error("pls_open: rtx_serial_enable_receiver: failed");
			free (pls);
			return NULL;
		}
	}

	/*
	 * Create mutex
	 */
	if ((pls->mutex = rtx_mutex_init(NULL, RTX_MUTEX_DEFAULT, 0)) == NULL) {
		rtx_error("pls_open: rtx_mutex_init: failed");
		free (pls);
		return NULL;
	}

	/* 
	 * Launch a reading thread 
	 */
	if (debug)
		rtx_message_routine("pls_open: launching reading thread");
	if ((pls->readerThread = rtx_thread_create(
			"pls_reader", debug, 
			RTX_THREAD_SCHED_OTHER, readerPrio, 0,
			RTX_THREAD_CANCEL_ASYNCHRONOUS, 
			(void * (*)(void*))pls_reader, pls,
			NULL, NULL)) == NULL) { 
		rtx_error("pls_open: rtx_thread_create: failed");
		free (pls);
		return NULL;
	}

	/*
	 * Launch an anti-hang thread
	 */
	if (debug)
		rtx_message_routine("pls_open: launching anti-hanging thread");
	if ((pls->stopHangThread = rtx_thread_create(
			"pls_stop_hang", debug, 
			RTX_THREAD_SCHED_OTHER, readerPrio, 0,
			RTX_THREAD_CANCEL_ASYNCHRONOUS, 
			(void * (*)(void*))pls_stop_hang, pls,
			NULL, NULL)) == NULL) { 
		rtx_error("pls_open: rtx_thread_create: failed");
		free (pls);
		return NULL;
	}

	/*
	 * Wait for reader thread to achieve sync with PLS output (if there
	 * is PLS output)
	 */

	/*
	 * Get status
	 */	
	if (debug)
		rtx_message_routine("pls_open: getting PLS status");
	if ((result = pls_get_status(pls, debug)) < 0) {
		rtx_error_flush("pls_get_status: failed");
	} else { 
		pls->mode = result;
	}
	if (pls->mode == PLS_MODE_INIT) {
      rtx_error("pls_open: can't contact PLS. The device is either not conected properly or is not set at %d Baud as assumed.", pls->baudRate);
     if (dont_give_up) {
        while (pls->mode == PLS_MODE_INIT) {
          rtx_message_routine("pls_open: waiting a bit (5 sec)");
          rtx_timer_sleep(5.0);
          if (debug)
            rtx_message_routine("pls_open: getting PLS status");
          if ((result = pls_get_status(pls, debug)) < 0) {
            rtx_error_flush("pls_get_status: failed");
          } else { 
            pls->mode = result;
          }
        }
      } else {
		free (pls);
		return NULL;
      }
	} 

	if (pls->lms) {
      /*
       * Get the LMS hardware configuartion
       */
      if (pls_get_lms_hw_configuration(pls, 0)) {
         rtx_error("pls_open: pls_get_lms_hw_configuration: failed");
         return NULL;
      }
    }
    
	/*
	 * Launch the continuous monitor thread
	 */
	if (debug)
		rtx_message_routine("pls_open: launching monitor continuous mode thread");
	if ((pls->monitorContinuousThread = rtx_thread_create(
			"pls_monitor_continuous_mode", debug, 
			RTX_THREAD_SCHED_OTHER, readerPrio, 0,
			RTX_THREAD_CANCEL_ASYNCHRONOUS, 
			(void * (*)(void*))pls_monitor_continuous_mode, pls,
			NULL, NULL)) == NULL) { 
		rtx_error("pls_open: rtx_thread_create: failed");
		free (pls);
		return NULL;
	}

	if (debug)
		rtx_message_routine("pls_open: opened");

	return pls;
}

/**
 * This function opens the RS232 com port for the PLS device. 
 *
 * \return Zero on success and -1 on error
 *
 */
static int
pls_open_dev(
	Pls	*pls,		/*!< Pointer to Pls structure */
	int	highSpeed	/*!< Highspeed serial interface flag */
	)
{
	int             fd;
	int		parity;
#ifdef USE_HSS
        HSS_COMM_PARMS	hss_conf;
#else
#endif

	if (pls->debug)
		rtx_message_routine("pls_open_dev: opening %s", pls->device);

#ifdef USE_HSS
	if (highSpeed) {
		do {
			fd = open(pls->device, O_RDWR);
		} while ((fd <0) && (errno == EINTR));
		if ((fd < 0) && (errno != EINTR))
			return rtx_error_errno("pls_open_dev: open error. Canot open %s: %s\n", pls->device);

		pls->fd = fd;
		pls->highSpeed = 1;

                hss_conf.dataBits = HSS_DATA_BITS_8;
                hss_conf.stopBits = HSS_STOP_BITS_1;
		if (pls->lms) {
                	hss_conf.parity = HSS_PARITY_NONE;
		} else {
                	hss_conf.parity = HSS_PARITY_EVEN;
		}
		switch (pls->baudRate) {
		case 9600:
                        hss_conf.baudRate = HSS_BAUDRATE_9600;
			break;
		case 19200:
                        hss_conf.baudRate = HSS_BAUDRATE_19200;
			break;
		case 38400:
                        hss_conf.baudRate = HSS_BAUDRATE_38400;
			break;
		case 58800:
                        hss_conf.baudRate = HSS_BAUDRATE_58800;
			break;
		case 111000:
                        hss_conf.baudRate = HSS_BAUDRATE_111000;
			break;
		case 200000:
                        hss_conf.baudRate = HSS_BAUDRATE_200000;
			break;
		case 250000:
                        hss_conf.baudRate = HSS_BAUDRATE_250000;
			break;
		case 334000:
                        hss_conf.baudRate = HSS_BAUDRATE_334000;
			break;
		case 500000:
                        hss_conf.baudRate = HSS_BAUDRATE_500000;
			break;
		}
                if (ioctl(fd, HSS_SET_COMM_PARMS, (char *)(&hss_conf)))
			return rtx_error("pls_open_dev: hss ioctl() error");
	} else {
#endif
		/* 
	 	 * Set up RS232 port for PLS 
	 	 */
		if (pls->lms) {
			parity = RTX_SERIAL_PARITY_NONE;
		} else {
			parity = RTX_SERIAL_PARITY_EVEN;
		}

		if ((fd = rtx_serial_open_with_prio (pls->device, pls->baudRate, 8, 1, parity, RTX_SERIAL_FLOW_NONE, RTX_SERIAL_MODEM_OFF, 0, PLS_TTY_VTIME, pls->readerPrio)) == -1)
			return rtx_error("pls_open_dev: rtx_serial_open_with_prio: failed");

		/*
		 * Disable the reciever 
		 */
		if (pls->debug)
			rtx_message_routine("pls_open_dev: disabling receiver");
		if (rtx_serial_disable_receiver(fd))
			return rtx_error("pls_open_dev: rtx_serial_disable_receiver: failed");
		else {
			if (pls->debug)
				rtx_message_routine("pls_open_dev: receiver disabled");
		}

		pls->fd = fd;
#ifdef USE_HSS
	}
#endif

	if (pls->debug)
		rtx_message_routine("pls_open_dev: opened %s", pls->device);

	return 0;
}

/**
 * This function closes the PLS. All associated threads and memory are 
 * destoryed and cleaned up.
 *
 * \return Zero on success, -1 on error
 */
int
pls_close(
	Pls	*pls		/*!< Pointer to Pls structure */
	)
{
	int	debug, error = 0;

	if (pls->debug)
		rtx_message_routine("pls_close: closing %s PLS", pls->name);

	/*
	 * Kill the continuous mode monitoring thread
	 */
	if (pls->debug)
		rtx_message_routine("pls_close: killing monitor continuous mode thread");
	if (rtx_thread_destroy_sync(pls->monitorContinuousThread)) {
                rtx_error("pls_close: rtx_thread_destroy_sync: failed");
                error--;
        }

	/*
	 * Kill the anti-hang thread
	 */
	if (pls->debug)
		rtx_message_routine("pls_close: killing anti-hang thread");
	if (rtx_thread_destroy_sync(pls->stopHangThread)) {
                rtx_error("pls_close: rtx_thread_destroy_sync: failed");
                error--;
        }

	/*
	 * Kill reader thread
	 */
	if (pls->debug)
		rtx_message_routine("pls_close: killing reader thread");
	if (rtx_thread_destroy_sync(pls->readerThread)) {
                rtx_error("pls_close: rtx_thread_destroy_sync: failed");
                error--;
        }

	/*
	 * Destroy the mutex
	 */
	if (pls->debug)
		rtx_message_routine("pls_close: destroying mutex");
	if (rtx_mutex_destroy(pls->mutex)) {
                rtx_error("pls_close: rtx_mutex_destroy: failed");
                error--;
        }

	/*
	 * Destroy reply semaphore
	 */
	if (pls->debug)
		rtx_message_routine("pls_close: reply semaphore");
	if (rtx_sem_destroy(pls->replySem)) {
                rtx_error("pls_close: rtx_sem_destroy: failed");
                error--;
        }

	/*
	 * Close the RS232 port for PLS. 
	 */
	if (pls->debug)
		rtx_message_routine("pls_close: closing RS232 port");
	if (close(pls->fd)) {
                rtx_error_errno("pls_close: pls_close_dev: failed");
		error--;
        }

	/*
	 * Free the memory
	 */
	debug = pls->debug;
	free(pls);

	if (debug)
		rtx_message_routine("pls_close: closed PLS");

	return error;
}

/*****************************************************************************
 *                         PLS READER THREAD
 *****************************************************************************/

/**
 * This thread continuously reads/parses the output from the PLS
 *
 * \return Nothing
 */
void
pls_reader(
	Pls	*pls		/*!< Pointer to Pls structure */
	)
{
	int		nread, debug, valid;
	unsigned int	crc_read, crc_computed, pktlen;
	unsigned char	buf[PLS_PACKET_MAX+4];
	unsigned int	timeOutCount = 0;
	int		sync = 0, firstSync = 0, first = 1;
	FILE		*fw = NULL;
	static char	string[1024];

	/*
	 * Read PLS data forever 
	 */
	for (;;) {
		debug = pls->debug;

		nread = rtx_serial_read_vtimeout(pls->fd, (char *)&buf[0], 1);
		if (nread < 0) {
			if (debug > 2)
				rtx_error_flush("(%d: rtx_serial_read_vtimeout: failed", pls->id);
			continue;
		} else if (nread == 0) {
			/* PLS is either not there or in an non-ouput mode */
			timeOutCount++;
			if (debug > 3)
				rtx_message_warning("%d: read timeout", pls->id);
			continue;       /* probably timed out */
		} else if (nread == 1) {
			timeOutCount = 0;
		} else {
			rtx_error_flush("%d:  wrong number (%d) of bytes read\n", pls->id, nread);
			continue;
		}

		/*
		 * Special debug mode that simply prints bytes recieved
		 */
		if (debug == 99) {
			if (first) {
				if ((fw=fopen(SERIAL_FILE, "w")) == NULL) {
					rtx_message_warning("cannot open %s file", SERIAL_FILE);      
				}
				first = 0;
			}
			fprintf(fw, "0x%02x\n", buf[0]);		
			continue;
		}

		switch (buf[0]) {
		case ACK:
			if (debug > 1)
				rtx_message_routine("%d: ACK received", pls->id);
			pls->lastAck = ACK;
			break;
		case NAK:
			if (debug > 1)
				rtx_message_routine("%d: NAK received", pls->id);
			pls->lastAck = NAK;
			break;
		case FAKE_TGM:
			if (sync)
				rtx_error_flush("received FAKE_TGM from anit-hang thread, so PLS didn't reply when it should have!");
			break;
		case STX:
			/*
			 * Message coming in, get a timestamp ASAP
			 */
			rtx_time_get(&pls->arrivalTime);

			if (pls->header_proc != NULL)
				(*pls->header_proc)(pls->id);

			if (debug > 1)
				rtx_message_routine("%d: STX received", pls->id);

			/*
			 * Packet coming in, read rest of header 
			 *
			 *	buf[0]	STX (read already)
			 *	buf[1]	PLS ID
			 *	buf[2]	length, low byte
			 *	buf[3]	length, high byte
			 *
			 * Note that the packet length is the number of
			 * bytes after the length field, and includes
			 * the packet type byte.
			 */
			if ((nread = rtx_serial_read_vtimeout(pls->fd, (char *)&buf[1], 1)) != 1) {
				rtx_error_flush("%d: read error on header, should have read %d bytes, but read %d bytes", pls->id, 1, nread);
				break;
			}

			if (buf[1] != (0x80 + pls->id)) {
				if (sync || pls->debug)
					rtx_error_flush("%d: wrong ID OR wrong baudrate set OR garbage (%02x)", pls->id, buf[1]);
				break;
			}

			if ((nread = rtx_serial_read_vtimeout(pls->fd, (char *)&buf[2], 2)) != 2) {
				rtx_error_flush("%d: read error on header, should have read %d bytes, but read %d bytes", pls->id, 2, nread);
				break;
			}

			/*
			 * The packet length is the message length.
			 * This does not include the 4 byte header and the CRC
			 * trailer, so add 2 to read the CRC.
			 */
			pktlen = hex_16_to_dec(buf[2], buf[3])+2;
			if ((pktlen > PLS_PACKET_MAX) || (pktlen < 2)) {
				if (sync)
					rtx_error_flush("%d: length is %d, this is out of range (max is %d, min is 4)", pls->id, pktlen, PLS_PACKET_MAX);
				break;
			}
			if (debug > 2)
				rtx_message_routine("pktlen: %d", pktlen);
			
			/*
			 * Check for a valid packet length
			 */
			if (pls->checkPktLen) {
				valid = 0;
				switch (pktlen) {
				case 3:
				case 4:
				case 5:
				case 6:
				case 7:
				case 9:
				case 24:
                case 25:        /* for power on response */
				case 26:
				case 36:
				case 37:
				case 38:
				case 39:	// Added by Elliot - for 5 series
				case 42:
				case 66:
				case 96:
				case 124:
				case 127:
				case 154:
				case 156:
				case 186:
				case 188:
				case 208:
				case 366:
				case 368:
				case 370: /* for real-time indecies */
				case 408:
				case 410: /* for real-time indecies */
				case 728:
				case 730: /* for real-time indecies */
				case 808:
				case 810: /* for real-time indecies */
					valid = 1;
					break;
				default:
					if (debug)
						rtx_error_flush("%d: invalid length = %d", pls->id, pktlen);
                       	         valid = 0;
					break;
				}
	
				if (!valid) {
					if (sync)
						rtx_error_flush("%d: length is %d, this is not an allowed length", pls->id, pktlen);
					break;
				}
			}

			/*
			 * Read in the rest of the packet
			 *	buf[4]	packet type
			 *      data ....
			 *      buf[4+pktlen-3] = status
			 *      buf[4+pktlen-2] = CRC1
			 *      buf[4+pktlen-1] = CRC2
			 */
			if (debug > 1)
				rtx_message_routine("%d: about to read %d bytes", pls->id, pktlen);
			if ((nread = rtx_serial_read_vtimeout(pls->fd, (char *)&buf[4], pktlen)) != pktlen) {
				if (sync)
					rtx_error_flush("%d: read error on packet body, should have read %d bytes, but read %d bytes", pls->id, pktlen, nread);
				break;
			}
			if (debug > 2)
				rtx_message_routine("%d: read %d bytes", pls->id, pktlen);
			
			crc_read =  hex_16_to_dec(buf[4+pktlen-2], buf[4+pktlen-1]);
			crc_computed = pls_crc16_build(&buf[0], 4+pktlen-2);
		
			if (crc_read != crc_computed) {
				if (sync) {
                    rtx_message_routine("%d: lost sync", pls->id);
					rtx_error_flush("%d: CRC failure, lost sync", pls->id);
					sync = 0;
				}
				break;
			} else {
				if (!sync)
					if (debug || firstSync)
						rtx_message_routine("%d: got sync", pls->id);
				sync = 1;
				firstSync = 1;
			}

			rtx_mutex_lock(pls->mutex);
                        	pls->replyLen = pktlen;
			rtx_mutex_unlock(pls->mutex);

			/*
			 * The header so far has been put in buf[],
			 * and for a range packet (MW_TGM) will stay there.
			 *
			 * For a response packet the packet from the type
			 * byte onward will be put into pls->reply.
			 *
			 */
			if (buf[4] != MW_TGM) {
				rtx_mutex_lock(pls->mutex);
				memcpy((char *)&pls->reply[0], (char *)&buf[4], pktlen-2);
				rtx_mutex_unlock(pls->mutex);
			}

			/*
			 * If debugging then print out packet type and length
			 */
			if (debug > 2) {
				rtx_message_routine("R: type=%02x len=%d\n", buf[4], pktlen);
			}

			/*
			 * Copy the PLS status byte into the pls struct
			 * and update dirt byte if necessary
			 */
			rtx_mutex_lock(pls->mutex);
			pls->status = buf[4+pktlen-3];
			if (pls->status & 0x80)
				pls->dirt++;
			else
				pls->dirt = 0;
			rtx_mutex_unlock(pls->mutex);

			switch (buf[4]) {
			case MW_TGM:
				if (debug)
					rtx_message_routine("%d: received range packet", pls->id);
				rtx_mutex_lock(pls->mutex);
					pls->first = 1;
					pls->last = 361;
					pls->samples = hex_14_to_dec(buf[5], buf[6]); 
					pls->rangeData = &buf[7];
				rtx_mutex_unlock(pls->mutex);
				if (debug > 1)
					rtx_message_routine("%d: recieved range data (nsegs=%d) and raised the semaphore", pls->id, pls->samples);
				/* for a range packet, call the handler */
				if (pls->range_proc != NULL)
					(*pls->range_proc)(&buf[5], pls->id);
				break;
			case MWP_TGM:
				if (debug)
					rtx_message_routine("%d: received range packet", pls->id);
				rtx_mutex_lock(pls->mutex);
					pls->first = hex_16_to_dec(buf[5], buf[6]);	
					pls->last = hex_16_to_dec(buf[7], buf[8]);	
					pls->samples = hex_14_to_dec(buf[9], buf[10]);	
					pls->rangeData = &buf[11];
				rtx_mutex_unlock(pls->mutex);
				/* Post reply semaphore */
				if (rtx_sem_post(pls->replySem)) {
					rtx_error_flush("%d: rtx_sem_post: failed", pls->id);
					break;
				}
				if (debug > 1)
					rtx_message_routine("%d: recieved partial range data and raised the semaphore", pls->id);
				if (pls->range_proc != NULL)
					(*pls->range_proc)(&buf[9], pls->id);
				break;
			case MMW_TGM:
				if (debug)
					rtx_message_routine("%d: received range packet", pls->id);
				rtx_mutex_lock(pls->mutex);
					pls->first = 1;
					pls->last = 361;
					pls->samples = hex_14_to_dec(buf[7], buf[8]);	
					pls->rangeData = &buf[9];
				rtx_mutex_unlock(pls->mutex);
				/* Post reply semaphore */
				if (rtx_sem_post(pls->replySem)) {
					rtx_error_flush("%d: rtx_sem_post: failed", pls->id);
					break;
				}
				if (debug > 1)
					rtx_message_routine("%d: recieved partial range data and raised the semaphore", pls->id);
				if (pls->range_proc != NULL)
					(*pls->range_proc)(&buf[7], pls->id);
				break;
			case PWON_TGM:
				if (debug)
					rtx_message_routine("%d: Power On detected", pls->id);
				rtx_mutex_lock(pls->mutex);
					pls->mode = PLS_MODE_POWER_ON;
				rtx_mutex_unlock(pls->mutex);
				break;
			default:
				if (debug > 2) {
					int	i;
			
					sprintf(string, "%d: Reply: ", pls->id);
					for (i=0; i<pktlen; i++)
						sprintf(string, "%s %02x ", string, buf[4+i]);
					rtx_message_routine("%s", string);
				}
				/* Post reply semaphore */
				if (rtx_sem_post(pls->replySem)) {
					rtx_error_flush("%d: rtx_sem_post: failed", pls->id);
					break;
				}
				if (debug > 1)
					rtx_message_routine("%d: recieved reply [0x%x] and raised the semaphore", pls->id, buf[4]);
				if (buf[4] == NACK_TGM)
					rtx_error_flush("%d: the requested function cannot be performed on this device", pls->id); 
				break;
			}
			break;

		default:
			if (debug > 2)
				rtx_message_warning("%d: G: %02x ", pls->id, buf[0]);
			break;
		}
	}
}

/**
 * This function changes the mode of the PLS. The values and functions
 * of the PlsMode mode variable may be found in the pls.h header file.
 *
 * \return Zero on success, -1 on error
 */
int
pls_set_mode(
	Pls	*pls,
	PlsMode	mode
	)
{
	unsigned int	pktlen;
	unsigned char	request[40], reply[2];
	unsigned int	tries = 0;
	int		debug;

	debug = pls->debug;

	/*
	 * Check to see if we are already in the requested mode.
	 * If so then return.
	 */
	if ((pls->mode == PLS_MODE_CONTINUOUS) && (mode == PLS_MODE_CONTINUOUS_OVERRIDE)) {
		mode = PLS_MODE_CONTINUOUS;
	} else if (pls->mode == mode) {
		if (debug) 
			rtx_message_routine("pls_set_mode: requested mode is current");
		return 0;
	}

	if (debug)
		rtx_message_routine("pls_set_mode: about to change mode to 0x%02x", mode);

	/*
	 * Construct the request packet.
	 */
	request[0] = BM_TGM;
	request[1] = mode;	
	if ( (mode == PLS_MODE_SETUP) || (mode == PLS_MODE_PASSWORD_TEST) ) {
		if (pls->lms)
			strcpy((char *)&request[2], LMS_PASSWORD);
		else
			strcpy((char *)&request[2], PLS_PASSWORD);
                pktlen = 10;
	}
	else
		pktlen = 2;

	for (;;) {
		/*
		 * Check to see if it time to give up. if so return.
		 */
		if (tries > PLS_TRIES_MAX) {
			if (debug)
				return rtx_error("pls_set_mode: can't set mode, have tried %d times without success", tries);
		}

		/*
	 	 * Send the packet to the PLS, if it fails continue
	 	 */
		switch (pls_send_packet(pls, request, pktlen)) {
		case 99:
			return 0;
			break;
		case 0:
			break;
		default:
			if (debug)
				rtx_message_warning("pls_set_mode: can't set mode, error sending packet");
			tries++;
			continue;
		}

		/*
		 * Wait for reply and check for a successfull mode change
		 */
		if (pls_wait_for_reply(pls, "pls_set_mode"))
			return rtx_error("pls_set_mode: pls_wait_for_reply: failed");

		/*
		 * Decode reply
		 */
		rtx_mutex_lock(pls->mutex);
			reply[0] = pls->reply[0];
			reply[1] = pls->reply[1];
		rtx_mutex_unlock(pls->mutex);
		switch (reply[0]) {
		case BMACK_TGM:
			switch (reply[1]) {
			case 0x00:
				/* 
				 * Wait for PLS_MAX_REPLY_WAIT secs as this
				 * is how long it might actually take for
				 * the change to take place, even though a 
				 * replay message has been received.
				rtx_timer_sleep(PLS_MAX_REPLY_WAIT);
				 */
				rtx_mutex_lock(pls->mutex);
					pls->mode = mode;
				rtx_mutex_unlock(pls->mutex);
				if (debug)
					rtx_message_routine("pls_set_mode: mode changed to 0x%02x", mode);
				return 0;
				break;
			case 0x01:
				rtx_error("pls_set_mode: mode change not possible - password incorrect");
				return -1;
			case 0x02:
				rtx_error("pls_set_mode: mode change not possible - sensor fault"); 
				return -1;
			default:
				rtx_message_warning("pls_set_mode: invalid response");
				tries++;
				continue;
			}
			break;
		case NACK_TGM:
			rtx_error("pls_set_mode: not acknowledge, e.g. invalid operating mode change");		
			return -1;
			break;
		default:
			if (debug > 1) 
				rtx_message_warning("pls_set_mode: reply type [0x%x] incorrect", reply[0]);
			tries++;
			break;
		}
	}
}

/**
 * This function is used to set the so-called `variant type' of an LMS.
 * This is actually the angular resoultion and the field-of-view.
 *
 * \return Zero on success, -1 on error
 */
int
pls_set_variant(
	Pls	*pls,
	int	fov,
	double	resolution
	)
{
	unsigned int	pktlen;
	unsigned char	request[40], reply[2];
	unsigned int	tries = 0;
	int		resolutionInt;
	int		i, debug, fovResponse, resolutionResponse;

	debug = pls->debug;

	if (!pls->lms)
		return rtx_error("pls_set_varient: can only do this on an LMS");
		
	/*
	 * Check for valid request
	 */
	if ((fov != 180) && (fov != 100))
		return rtx_error("pls_set_varient: invalid fov request = %d", fov);
	resolutionInt = (int)(100*resolution);
	if ((resolutionInt != 100) && (resolutionInt != 50) && (resolutionInt != 25))
		return rtx_error("pls_set_varient: invalid resolution request = %f", resolution);

	if (debug)
		rtx_message_routine("pls_set_varient: about to change varient to fov = %d and resolution = %d(%f)", fov, resolutionInt, resolution);

	/*
	 * Check to see if we are already the requested variant
	 * If so then return.
	 */
	if ((pls->fov == fov) && (pls->resolution == resolution)) {
		if (debug) 
			rtx_message_routine("pls_set_variant: requested variant is current");
		return 0;
	}

	/*
	 * Construct the request packet.
	 */
	request[0] = VARDEF_TGM;
	pls_intToHex(fov, &request[1]);
	pls_intToHex(resolutionInt, &request[3]);
	pktlen = 5;

	for (;;) {
		/*
		 * Check to see if it time to give up. if so return.
		 */
		if (tries > PLS_TRIES_MAX)
			return rtx_error("pls_set_variant: can't set variant, have tried %d times without success", tries);

		/*
	 	 * Send the packet to the PLS, if it fails continue
	 	 */
		if (pls_send_packet(pls, request, pktlen)) {
			if (debug)
				rtx_message_routine("pls_set_variant: can't set variant, error sending packet");
			tries++;
			continue;
		}

		/*
		 * Wait for reply and check for a successfull variant change
		 */
		if (pls_wait_for_reply(pls, "pls_set_variant"))
			return rtx_error("pls_set_variant: pls_wait_for_reply: failed");

		/*
		 * Decode reply
		 */
		rtx_mutex_lock(pls->mutex);
			for (i=0; i<6; i++)
				reply[i] = pls->reply[i];
		rtx_mutex_unlock(pls->mutex);
		switch (reply[0]) {
		case VARDEFACK_TGM:
			switch (reply[1]) {
			case 0x00:
				rtx_error("pls_set_variant: change aborted, previous variant remains active");
				return -1;
				break;
			case 0x01:
				if (debug)
					rtx_message_routine("pls_set_variant: change accepted");
				break;
			default:
				rtx_error("pls_set_variant: unrecognised status");
				return -1;
				break;
			}

			fovResponse = hex_16_to_dec(reply[2], reply[3]);
			resolutionResponse = hex_16_to_dec(reply[4], reply[5]);
			if (fovResponse != fov) {
				rtx_message_warning("pls_set_variant: invalid fov response = %d", fovResponse);
				tries++;
				continue;
			}
			if (resolutionResponse != resolutionInt) {
				rtx_message_warning("pls_set_variant: invalid resolution response = %d", resolutionResponse);
				tries++;
				continue;
			}
			pls->fov = fov;
			pls->resolution = resolution;
			return 0;
			break;
		case NACK_TGM:
			rtx_message_warning("pls_set_variant: not acknowledge, e.g. invalid variant change");		
			return -1;
			break;
		default:
			if (debug > 1) 
				rtx_message_warning("pls_set_variant: reply type [0x%x] incorrect", reply[0]);
			tries++;
			break;
		}
	}
}

/**
 * This function changes the baudrate of the PLS, and the com port of the
 * host computer. Possible low speed baudrates are 9600, 19200 and 38400
 *
 * \return Zero on success, -1 on error
 */
int
pls_set_baudrate(
	Pls		*pls,
	unsigned int	baudrate
	)
{
	unsigned char 	mode = 0x00;
#ifdef USE_HSS
        HSS_COMM_PARMS	hss_conf;
#endif

	/*
	 * Check to see if requested baudrate is current, if so return.
	 */
	rtx_mutex_lock(pls->mutex);
	if (pls->baudRate == baudrate) {
		if (pls->debug)
			rtx_message_routine("pls_set_baudrate: requested baudrate is current");
		rtx_mutex_unlock(pls->mutex);
		return 0;
	}
	rtx_mutex_unlock(pls->mutex);

	/*
	 * I can't get 180 segments at 38400 baud for some unknown reason?
	 * Seems to be a firmware problem with the PLS.
	 */
	rtx_mutex_lock(pls->mutex);
	if ( (pls->samples == 180) && (baudrate == 38400) ) {
		if (pls->debug)
			rtx_error("pls_set_baudrate: 180 segs at 38400 doesn't seem to work? You will have to change no. of segments to run at 38400 baud.");
		rtx_mutex_unlock(pls->mutex);
		return -1;
	}
	rtx_mutex_unlock(pls->mutex);

	if (pls->debug)
		rtx_message_routine("pls_set_baudrate: about to change baudrate to %d", baudrate);

	/* 
	 * Put the PLS into IDLE mode before changing the baudrate
	 */
	rtx_mutex_lock(pls->mutex);
	if ( (pls->mode != PLS_MODE_IDLE) && (pls->mode != PLS_MODE_SETUP) ) {
		rtx_mutex_unlock(pls->mutex);
		if (pls_set_mode(pls, PLS_MODE_IDLE)) {
			if (pls->debug)
				rtx_error("pls_set_baudrate: can't get into IDLE mode and so can't change baudrate");
			return -1;
		}
	}
	rtx_mutex_unlock(pls->mutex);

	/* 
	 * Put the PLS into SETUP mode before changing the baudrate
	 */
	rtx_mutex_lock(pls->mutex);
	if ( pls->mode != PLS_MODE_SETUP ) {
		rtx_mutex_unlock(pls->mutex);
		if (pls_set_mode(pls, PLS_MODE_SETUP))
			return rtx_error("pls_set_baudrate: can't get into SETUP mode and so can't change baudrate");
	}
	rtx_mutex_unlock(pls->mutex);

       	switch (baudrate) {
       	case 9600:
#ifdef USE_HSS
		if (pls->highSpeed)
                 	hss_conf.baudRate = HSS_BAUDRATE_9600;
#endif
                mode = PLS_MODE_B9600;
                break;
        case 19200:
#ifdef USE_HSS
		if (pls->highSpeed)
                       	hss_conf.baudRate = HSS_BAUDRATE_19200;
#endif
               	mode = PLS_MODE_B19200;
               	break;
        case 38400:
#ifdef USE_HSS
		if (pls->highSpeed)
                       	hss_conf.baudRate = HSS_BAUDRATE_38400;
#endif
               	mode = PLS_MODE_B38400;
               	break;
	case 58800:
#ifdef USE_HSS
		if (pls->highSpeed)
               		hss_conf.baudRate = HSS_BAUDRATE_58800;
#endif
		mode = PLS_MODE_B58800;
		break;
	case 111000:
#ifdef USE_HSS
		if (pls->highSpeed)
               		hss_conf.baudRate = HSS_BAUDRATE_111000;
#endif
		mode = PLS_MODE_B111000;
		break;
	case 200000:
#ifdef USE_HSS
		if (pls->highSpeed)
               		hss_conf.baudRate = HSS_BAUDRATE_200000;
#endif
		mode = PLS_MODE_B200000;
		break;
	case 250000:
#ifdef USE_HSS
		if (pls->highSpeed)
               		hss_conf.baudRate = HSS_BAUDRATE_250000;
#endif
		mode = PLS_MODE_B250000;
		break;
	case 334000:
#ifdef USE_HSS
		if (pls->highSpeed)
               		hss_conf.baudRate = HSS_BAUDRATE_334000;
#endif
		mode = PLS_MODE_B334000;
		break;
	case 500000:
#ifdef USE_HSS
		if (pls->highSpeed)
         		hss_conf.baudRate = HSS_BAUDRATE_500000;
#endif
		mode = PLS_MODE_B500000;
		break;
        default:
        	return rtx_error("pls_set_baudrate: unsupported baud rate");
        }

	/*
	 * Send change baudrate command to PLS
	 */
	if (pls_set_mode(pls, mode))
		return rtx_error("pls_set_baudrate: can't change baudrate");

	/* 
	 * Now change the port speed 
	 */
#ifdef USE_HSS
	if (pls->highSpeed) {
        	hss_conf.dataBits = HSS_DATA_BITS_8;
      	  	hss_conf.stopBits = HSS_STOP_BITS_1;
       		hss_conf.parity = HSS_PARITY_EVEN;

                if (ioctl(pls->fd, HSS_SET_COMM_PARMS, (char *)(&hss_conf)))
			return rtx_error("pls_set_baudrate: hss ioctl() error");
	} else {
#endif
		rtx_mutex_lock(pls->mutex);
		if (rtx_serial_set_baudrate (pls->fd, baudrate, baudrate)) {	
			rtx_mutex_unlock(pls->mutex);
			return rtx_error("pls_set_baudrate: rtx_serial_set_baudrate: failed");
		}
		rtx_mutex_unlock(pls->mutex);
#ifdef USE_HSS
	}
#endif

	if (pls->debug)
		rtx_message_routine("pls_set_baudrate: baudrate now %d", baudrate);

	rtx_mutex_lock(pls->mutex);
		pls->baudRate = baudrate;
	rtx_mutex_unlock(pls->mutex);

	return 0;
}

/**
 * This function lets the current sensor baudrate be permanent (i.e.
 * remembered after Power Off), or temporary (i.e. it will default back to
 * 9600 after Power Off
 *
 * \return Zero on success, -1 on error
 */
int
pls_set_perm_baudrate(
	Pls		*pls,
	unsigned char 	perm
	)
{
	unsigned int	pktlen;
	unsigned char	request[PLS_PACKET_MAX+4], reply[2];
	unsigned int	tries = 0;

	/* 
	 * Put the PLS into IDLE mode before setting/unsetting the
	 * permanent baudrate
	 */
	if (pls_set_mode(pls, PLS_MODE_IDLE))
		return rtx_error("pls_set_perm_baudrate: can't get into IDLE mode and so can't set/unset permanent baudrate");

	/* 
	 * Put the PLS into SETUP mode before setting/unsetting the
	 * permanent baudrate
	 */
	if (pls_set_mode(pls, PLS_MODE_SETUP))
		return rtx_error("pls_set_perm_baudrate: can't get into SETUP mode and so can't set/unset permanent baudrate");

	/*
	 * Construct the request packet
	 */
	request[0] = BRPERMDEF_TGM;
	request[1] = perm;
	pktlen = 2;

	for (;;) {
		/*
		 * Check to see if it time to give up. if so return.
		 */
		if (tries >= PLS_TRIES_MAX)
			return rtx_error("pls_set_perm_baudrate: can't set/unset permanent baudrate, have tried %d times without success", tries);

		/*
		 * Send the packet to the PLS
		 */
		if (pls_send_packet(pls, request, pktlen)) {
			rtx_message_warning("pls_set_permanent_baudrate: can't set/unset permanent baudrate, error sending packet");
			tries++;
			continue;
		}

		/*
		 * Wait for reply and check that the setting was successfull
		 */
		if (pls_wait_for_reply(pls, "pls_set_permanent_baudrate"))
			return rtx_error("pls_set_permanent_baudrate: pls_wait_for_reply: failed");

		rtx_mutex_lock(pls->mutex);
			reply[0] = pls->reply[0];
			reply[1] = pls->reply[1];
			reply[2] = pls->reply[2];
		rtx_mutex_unlock(pls->mutex);
		switch (reply[0]) {
		case BRPERMACK_TGM:
			switch (reply[1]) {
			case 0x00:
				if (pls->debug)
					rtx_message_warning("pls_set_perm_baudrate: perm. baudrate/variant not accepted");
				tries++;
				continue;
			case 0x01:
				if (pls->debug)
					rtx_message_warning("pls_set_perm_baudrate: perm. baudrate/variant accepted");
				break;
			default:
				if (pls->debug)
					rtx_message_warning("pls_set_perm_baudrate: unrecognised reply");
				tries++;
				continue;
			}
			switch (reply[2]) {
			case 0x00:
				if (pls->debug)
					rtx_message_routine("pls_set_perm_baudrate: baudrate is set to 9600 after POWER ON");
				return 0;
			case 0x01:
				if (pls->debug)
					rtx_message_routine("pls_set_perm_baudrate: baudrate is not changed after POWER ON");
				return 0;
			case 0x02:
				if (pls->debug)
					rtx_message_routine("pls_set_perm_baudrate: perm. LMS variant type is not changed after POWER ON");
				return 0;
			default:
				if (pls->debug)
					rtx_message_routine("pls_set_perm_baudrate: unrecognised reply");
				tries++;
				continue;
			}
			break;
		case NACK_TGM:
			if (pls->debug)
				rtx_message_warning("pls_set_perm_baudrate: not acknowledge, e.g. invalid operating mode change");		
			tries++;
			continue;	
		default:
			if (pls->debug)
				rtx_message_warning("pls_set_perm_baudrate: reply type 0x%02x incorrect", reply[0]);
			tries++;
			break;
		}
	}
}
	
/**
 * This function gets the warning field.
 *
 * \return Zero on success, -1 on error
 */
int
pls_get_warning_field(
	Pls		*pls,		/* Pointer to PLS structure */
	unsigned int	*noSegs,	/* Number of segments in the field */
	unsigned int	*data		/* Warning field data (cm) */
	)
{
	unsigned char	request[2], reply[PLS_PACKET_MAX];
	unsigned int	i, j, tries = 0;

	request[0] = SFANF_TGM;
	request[1] = 0x01;
	
	for (;;) {
		/*
		 * Check to see if it time to give up. if so return.
		 */
		if (tries >= PLS_TRIES_MAX)
			return rtx_error("pls_get_warning_field: can't get protection field, have tried %d times without success", tries);

		if (pls->debug)
			rtx_message_routine("pls_get_warning_field: requesting protection field configuartion");

		/*
		 * Send the packet to the PLS
		 */	
		if (pls_send_packet(pls, request, 2)) {
			rtx_message_warning("pls_get_warning_field: pls_send_packet: failed");
			tries++;
			continue;
		}

		/*
		 * Wait for reply and print out config info
		 */
		if (pls_wait_for_reply(pls, "pls_get_warning_field"))
			return rtx_error("pls_get_warning_field: pls_wait_for_reply: failed");

		/*
		 * Copy the reply data to local buffer
		 */
		rtx_mutex_lock(pls->mutex);
			memcpy((char *)&reply[0], (char *)&pls->reply[0], PLS_PACKET_MAX);
		rtx_mutex_unlock(pls->mutex);

		/*
		 * Process the reply
		 */
		switch (reply[0]) {
		case SFDAT_TGM: 
			if (reply[1] == 0x03) {
				*noSegs = reply[2];
				for (i=0,j=3; i<(*noSegs+1); i++,j+=2) {
					data[i] = hex_16_to_dec(reply[j], reply[j+1]);
					rtx_message_routine("pls_get_warning_field: %d: %d", i, data[i]);
				}
			}
			rtx_message_routine("pls_get_warning_field: field type = 0x%02x, n = %d", reply[1], reply[2]);
			return 0;
		case NACK_TGM:
			rtx_error("pls_get_warning_field: not acknowledge, e.g. invalid request");		
			return -1;
		default:
			rtx_message_warning("pls_get_warning_field: reply type 0x%02x incorrect", reply[0]);
			tries++;
			break;
		}
	}

	return -1;
}

/**
 * This function writes a warning field file
 *
 * \return Zero to success, -1 on error
 */
int
pls_file_write_warning_field(
	char		*filename,
	unsigned int	noSegs,
	unsigned int	*data
	)
{
	FILE	*fw;
	int	i;

	if ((fw=fopen(filename, "w")) == NULL)
		return rtx_error("pls_file_write_warning_field: cannot open %s file", filename);      

	/*
	 * Write noSegs followed by field data in cm
	 */
	fprintf(fw, "%d\n", noSegs);
	for (i=0; i<noSegs+1; i++)
		fprintf(fw, "%d\n", data[i]);

	fclose(fw);

	return 0;	
}

/**
 * This function reads a warning field file
 *
 * \return Zero to success, -1 on error
 */
int
pls_file_read_warning_field(
	char		*filename,
	unsigned int	*noSegs,
	unsigned int	*data
	)
{
	FILE	*fr;
	int	i, val;

	if ((fr=fopen(filename, "r")) == NULL)
		return rtx_error("pls_file_read_warning_field: cannot open %s file", filename);      

	/*
	 * Read noSegs followed by field data in cm
	 */
	fscanf(fr, "%d\n", &val);
	*noSegs = val;
	for (i=0; i<(*noSegs)+1; i++) {
		fscanf(fr, "%d\n", &val);
		data[i] = val;
	}

	fclose(fr);

	return 0;	
}

/**
 * This function is incomplete. It should be passed an array containing
 * warning field data. It currently sets the warning field to be a 40cm 
 * radius semi-circle.
 *
 * Note: this function is also used to set the sensors angular resolution,
 *       given by the noSegs.
 *
 * \return Zero on success, -1 on error
 */
int
pls_set_warning_field(
	Pls		*pls,
	unsigned int	noSegs,
	unsigned int	*data	/* Warning field data (cm) */
	)
{
	unsigned int	i, j, pktlen;
	unsigned char	request[PLS_PACKET_MAX];
	unsigned int	tries = 0;

	switch (noSegs) {
	case 9:
	case 10:
	case 15:
	case 18:
	case 30:
	case 45:
	case 90:
	case 180:
		break;
	default:
		return rtx_error("pls_set_warning_field: invalid number of segments");
		break;
	}

	/*
	 * Check for an impossible request. This occurs if 180 segments are
	 * transferred at 9600 baud. Ths PLS will hang if this is attempted!
	 */
	rtx_mutex_lock(pls->mutex);
	if ( (noSegs == 180) && (pls->baudRate == 9600) ) {
		rtx_mutex_unlock(pls->mutex);
		return rtx_error("pls_set_warning_field: can't transfer 180 segments at 9600 baud. 90 is max at this baudrate");
	}

	/*
	 * I can't get 180 segments at 38400 baud for some unknown reason?
	 * Seems to be a firmware problem with the PLS.
	 */
/*
	if ( (noSegs == 180) && (pls->baudRate == 38400) ) {
		rtx_mutex_unlock(pls->mutex);
		return rtx_error("pls_set_warning_field: 180 segs at 38400 doesn't seem to work?");
	}
*/
	rtx_mutex_unlock(pls->mutex);

	/* 
	 * Put the PLS into IDLE mode before changing the warning field
	 */
	rtx_mutex_lock(pls->mutex);
	if ( (pls->mode != PLS_MODE_IDLE) && (pls->mode != PLS_MODE_SETUP) ) {
		rtx_mutex_unlock(pls->mutex);
		if (pls_set_mode(pls, PLS_MODE_IDLE)) {
			return rtx_error("pls_set_warning_field: can't get into IDLE mode and so can't change warning field");
		}
	}

	/* 
	 * Put the PLS into SETUP mode before changing the warning field
	 */
	if (pls->mode != PLS_MODE_SETUP) {
		rtx_mutex_unlock(pls->mutex);
		if (pls_set_mode(pls, PLS_MODE_SETUP)) {
			return rtx_error("pls_set_warning_field: can't get into SETUP mode and so can't change warning field");
		}
	}
	rtx_mutex_unlock(pls->mutex);

	/*
	 * Construct the request packet
	 */
	request[0] = KSFKFG_TGM;
	request[1] = (unsigned char)noSegs;
      
	/*
	 * Set warning field
	 */
	for (i=2,j=0; i<=2*(noSegs+1); i+=2,j++) 
		pls_intToHex(data[j], &request[i]);

	pktlen = 2*(noSegs+1)+2;

	if (pls->debug)
		rtx_message_routine("pls_set_warning_field: setting warning field");

	for (;;) {
		/*
		 * Check to see if it time to give up. if so return.
		 */
		if (tries >= PLS_TRIES_MAX)
			return rtx_error("pls_set_warning_field: can't set warning_field, have tried %d times without success", tries);

		/*
		 * Send the packet to the PLS
		 */
		if (pls_send_packet(pls, request, pktlen)) {
			rtx_message_warning("pls_set_warning_field: can't set warning field, error sending packet");
			tries++;
			continue;
		}

		/*
		 * Wait for reply and check that the mode change was successfull
		 */
		if (pls_wait_for_reply(pls, "pls_set_warning_field"))
			return rtx_error("pls_set_waring_field: pls_wait_for_reply() failed");

		rtx_mutex_lock(pls->mutex);
		switch (pls->reply[0]) {
		case KSFACK_TGM:
			switch (pls->reply[1]) {
				case 0x00:
					rtx_error("pls_set_warning_field: failed to change warning field");
				tries++;
				break;
			case 0x01:
				pls->samples = noSegs;
				rtx_mutex_unlock(pls->mutex);
                                if (pls->debug)
					rtx_message_routine("pls_set_waring_field: number of segments = %d", pls->samples);
                                return 0;
			default:
				if (pls->debug) 
					rtx_message_warning("pls_set_warning_field: invalid reply");
				tries++;
				break;
			}
			break;
		case NACK_TGM:
			if (pls->debug)
				rtx_message_warning("pls_set_warning_field: not acknowledge, e.g. invalid operating mode change");		
			tries++;
			break;	
		default:
			if (pls->debug)
				rtx_message_warning("pls_set_warning_field: reply type 0x%02x incorrect", pls->reply[0]);
			tries++;
			break;
		}
		rtx_mutex_unlock(pls->mutex);
	}
}

/**
 * This function is incomplete and should be added to at a later date.
 * It currently prints: SW Version, Defect Status and Production Code
 * and baudrate.
 *
 * \return PLS mode on success (non negative), -1 on error
 */
PlsMode
pls_get_status(
	Pls	*pls,
	int	printMessage
	)
{
	unsigned char	request, reply[125];
	unsigned int	i, j;
	int		result;
	char		string[512], string2[128];
	unsigned int	tries = 0;
	PlsMode		mode;

	request = SSANF_TGM;
	
	/*
	 * Get current mode
	 */
	mode = pls->mode;

	for (;;) {
		/*
		 * Check to see if it time to give up. if so return.
		 */
		if (tries >= PLS_TRIES_MAX)
			return rtx_error("pls_get_status: can't get status, have tried %d times without success", tries);

		if (pls->debug)
			rtx_message_routine("pls_get_status: requesting PLS status");

		/*
		 * Send the packet to the PLS
		 */	
		switch ((result = pls_send_packet(pls, &request, 1))) {
		case 99:
			return 0;
			break;
		case 0:
			break;
		default:
			rtx_message_warning("pls_get_status: pls_send_packet: failed: %d", result);
			tries++;
			continue;
		}

		/*
		 * Wait for reply and print out status
		 */
		if (pls_wait_for_reply(pls, "pls_get_status"))
			return rtx_error("pls_get_status: pls_wait_for_reply: failed");

		/*
		 * Copy the reply data to local buffer
		 */
		rtx_mutex_lock(pls->mutex);
			memcpy((char *)&reply[0], (char *)&pls->reply[0], 125);
		rtx_mutex_unlock(pls->mutex);

		/*
		 * Process the reply
		 */
		switch (reply[0]) {
		case SS_TGM: 
			sprintf(string, "%c%c", reply[5], reply[6]);
			rtx_mutex_lock(pls->mutex);

			pls->firmWare = atoi(string);
			sprintf(string, "pls_get_status: dev = %s, SW version = ", pls->device);
			for (i=1;i<7;i++)
				string2[i-1] = reply[i];
			string2[6] = '\0';
			strcat(string, string2);
			strcat(string, ", ");
			/* Make mode meaningful */
			pls->mode = reply[8];
			switch (reply[8]) {
			case 0x00:
				sprintf(string2, "Mode = SETUP ");
				break;
			case 0x25:
				sprintf(string2, "Mode = IDLE ");
				break;
			case 0x24:
				sprintf(string2, "Mode = CONT ");
				break;
			case 0x20:
				sprintf(string2, "Mode = MON1 ");
				break;
			case 0xff:
				sprintf(string2, "Mode = UNKNOWN ");
				break;
			default:
				sprintf(string2, "Mode = 0x%02x ", reply[8]);
				break;
			}
			strcat(string, string2);
			if (reply[9] > 0)
				sprintf(string2, "Defective = Yes, ");
			else
				sprintf(string2, "Defective = No, ");
			strcat(string, string2);
			strcat(string, "Production Code = ");
			for (i=10,j=0; i<18; i++,j++) {
				string2[j] = reply[i];
				pls->serialNum[j] = reply[i];
			}
			string2[8] = '\0';
			pls->serialNum[8] = '\0';
			strcat(string, string2);
			/* Current baudrate */

			switch(reply[116]) {
			case 0x01: // Added by Elliot - problem between 5 and 6 series
				pls->baudRate = 500000;
				break;
			case 0x02:
				pls->baudRate = 334000;
				break;
			case 0x03:
				pls->baudRate = 250000;
				break;
			case 0x04:
				pls->baudRate = 200000;
				break;
			case 0x08:
				pls->baudRate = 111000;
				break;
			case 0x10:
				pls->baudRate = 58800;
				break;
			case 0x19:
				pls->baudRate = 38400;
				break;
			case 0x33:
				pls->baudRate = 19200;
				break;
			case 0x67:
				pls->baudRate = 9600;
				break;
			default:
				pls->baudRate = 0;
				rtx_message_warning("pls_get_status: unrecognized baudrate reply = 0x%02x%02x", reply[117], reply[116]);
				break;
			}
			sprintf(string2, ", current baudrate = %d", pls->baudRate);
			strcat(string, string2);
			/* Perm baudrate */
			if (reply[119] == 0x00)
				sprintf(string2, ", perm baudrate = 9600");
			else
				sprintf(string2, ", perm baudrate = current");
			strcat(string, string2);

			/* Units */
			if (pls->lms) {	
				switch(reply[122]) {
				case 0:
					pls->unit = PLS_UNIT_CM;
					sprintf(string2, ", unit = cm");
					break;
				case 1:
					pls->unit = PLS_UNIT_MM;
					sprintf(string2, ", unit = mm");
					break;
				case 2:
					pls->unit = PLS_UNIT_DM;
					sprintf(string2, ", unit = dm");
					break;
				default:
					rtx_message_warning("pls_get_status: unrecognized unit = 0x%02x", reply[122]);
					break;
				}
			} else {
				pls->unit = PLS_UNIT_CM;
				sprintf(string2, ", unit = cm");
			}
			strcat(string, string2);

			if (pls->lms) {	
				switch (pls->unit) {
				case PLS_UNIT_CM:
					pls->rMax = 81.85;
					break;
				case PLS_UNIT_MM:
					pls->rMax = 8.185;
					break;
				case PLS_UNIT_DM:
					pls->rMax = 150.0;
					break;
				}
			} else {
				pls->rMax = 50.0;
			}

			/* Angle and resolution */
			if (pls->lms) {	
				pls->fov = hex_16_to_dec(reply[107], reply[108]);				
				pls->resolution = hex_16_to_dec(reply[109], reply[110])/100.0;				
			} else {
				pls->resolution = 0.5;
				pls->fov = 180;
			}
			if (pls->fov == 100.0) {
				pls->thetaMax = 50.0;
				pls->thetaMin = -50.0;
			} else {
				pls->thetaMax = 90.0;
				pls->thetaMin = -90.0;
			}
			pls->samples = (int)(pls->fov/pls->resolution)+1;
			sprintf(string2, ", fov = %d, res = %f", pls->fov, pls->resolution);
			strcat(string, string2);

			/* Laser state */
			if (pls->lms) {
				pls->laserOn = (int)reply[123];
			} else {
				pls->laserOn = 1;
			}
			sprintf(string2, ", laserOn = %d", pls->laserOn);
			strcat(string, string2);

			if (printMessage)
				rtx_message_routine(string);
			rtx_mutex_unlock(pls->mutex);
			return reply[8];
			break;
		case NACK_TGM:
			rtx_message_warning("pls_get_status: not acknowledge, e.g. invalid operating mode change");		
			return mode;
		default:
			rtx_message_warning("pls_get_status: reply type 0x%02x incorrect", reply[0]);
			tries++;
			break;
		}
	}
}

/**
 * This function prints the LMS hardware configuration to file fd
 *
 * \return Nothing
 */
void
pls_print_lms_hw_configuration(
	Pls	*pls,
	FILE	*fd
	)
{
	fprintf(fd, "LMS hardware configuration --------------------------\n"); 

	fprintf(fd, "blanking = %d ", pls->lmsHw.blanking);
	fprintf(fd, "stopThreshold = %d ", pls->lmsHw.stopThreshold);
	fprintf(fd, "availability = 0x%02x ", pls->lmsHw.availability);
	fprintf(fd, "settingMeasured = 0x%02x\n", pls->lmsHw.settingMeasured);
	fprintf(fd, "unitsMeasured = 0x%02x ", pls->lmsHw.unitsMeasured);
	fprintf(fd, "temporaryField = 0x%02x ", pls->lmsHw.temporaryField);
	fprintf(fd, "subtractive  = 0x%02x\n", pls->lmsHw.subtractive);
	fprintf(fd, "multipleEvaluation = 0x%02x ", pls->lmsHw.multipleEvaluation);

	fprintf(fd, "restart = 0x%02x ", pls->lmsHw.restart);
	fprintf(fd, "restartTime = 0x%02x\n", pls->lmsHw.restartTime);
	fprintf(fd, "restartTest = 0x%02x", pls->lmsHw.restartTest);

	fprintf(fd, "pixel  = 0x%02x ", pls->lmsHw.pixel);
	fprintf(fd, "single = 0x%02x ", pls->lmsHw.single);
	fprintf(fd, "zeroPoint = %d\n", pls->lmsHw.zeroPoint);

	fprintf(fd, "cAObject = 0x%02x ", pls->lmsHw.cAObject);
	fprintf(fd, "cAPositive = 0x%02x ", pls->lmsHw.cAPositive);
	fprintf(fd, "cANegative = 0x%02x ", pls->lmsHw.cANegative);
	fprintf(fd, "cAStart = 0x%02x ", pls->lmsHw.cAStarting);
	fprintf(fd, "cAStop = 0x%02x\n", pls->lmsHw.cAStopping);

	fprintf(fd, "cBObject = 0x%02x ", pls->lmsHw.cBObject);
	fprintf(fd, "cBPositive = 0x%02x ", pls->lmsHw.cBPositive);
	fprintf(fd, "cBNegative = 0x%02x ", pls->lmsHw.cBNegative);
	fprintf(fd, "cBStart = 0x%02x ", pls->lmsHw.cBStarting);
	fprintf(fd, "cBStop = 0x%02x\n", pls->lmsHw.cBStopping);

	fprintf(fd, "cCObject = 0x%02x ", pls->lmsHw.cCObject);
	fprintf(fd, "cCPositive = 0x%02x ", pls->lmsHw.cCPositive);
	fprintf(fd, "cCNegative = 0x%02x ", pls->lmsHw.cCNegative);
	fprintf(fd, "cCStart = 0x%02x ", pls->lmsHw.cCStarting);
	fprintf(fd, "cCStop = 0x%02x\n", pls->lmsHw.cCStopping);
}

static void
pls_lms_hw_config_read(
	Pls		*pls,
	unsigned char	*reply
	)
{
	rtx_mutex_lock(pls->mutex);
	pls->lmsHw.blanking = hex_16_to_dec(reply[0], reply[1]);
	pls->lmsHw.stopThreshold = hex_16_to_dec(reply[2], reply[3]);
	pls->lmsHw.availability = reply[4];
	pls->lmsHw.settingMeasured = reply[5];
	pls->lmsHw.unitsMeasured = reply[6];
	pls->lmsHw.temporaryField = reply[7];
	pls->lmsHw.subtractive = reply[8];
	pls->lmsHw.multipleEvaluation = reply[9];
	pls->lmsHw.restart = reply[10];
	pls->lmsHw.restartTime = reply[11];
	pls->lmsHw.restartTest = reply[12];
	pls->lmsHw.cAObject = reply[13];
	pls->lmsHw.cAPositive = reply[14];
	pls->lmsHw.cANegative = reply[15];
	pls->lmsHw.cAStarting = reply[16];
	pls->lmsHw.cAStopping = reply[17];
	pls->lmsHw.cBObject = reply[18];
	pls->lmsHw.cBPositive = reply[19];
	pls->lmsHw.cBNegative = reply[20];
	pls->lmsHw.cBStarting = reply[21];
	pls->lmsHw.cBStopping = reply[22];
	pls->lmsHw.cCObject = reply[23];
	pls->lmsHw.cCPositive = reply[24];
	pls->lmsHw.cCNegative = reply[25];
	pls->lmsHw.cCStarting = reply[26];
	pls->lmsHw.cCStopping = reply[27];
	pls->lmsHw.pixel = reply[28];
	pls->lmsHw.single = reply[29];
	pls->lmsHw.zeroPoint = hex_16_to_dec(reply[30], reply[31]);
	rtx_mutex_unlock(pls->mutex);
}

/**
 * This function gets the LMS hardware configuration.
 *
 * \return Zero on success, -1 on error
 */
int
pls_get_lms_hw_configuration(
	Pls	*pls,
	int	printMessage
	)
{
	unsigned char	request, reply[40];
	unsigned int	tries = 0;

	if (!pls->lms)
		return rtx_error("pls_get_lms_hw_configuration: this function is only valid with an LMS type 6 device");

	request = LMSKFGANF_TGM;
	
	for (;;) {
		/*
		 * Check to see if it time to give up. if so return.
		 */
		if (tries >= PLS_TRIES_MAX)
			return rtx_error("pls_get_lms_hw_configuration: can't get LMS configuration, have tried %d times without success", tries);

		if (pls->debug)
			rtx_message_routine("pls_get_lms_hw_configuration: requesting LMS configuartion");

		/*
		 * Send the packet to the LMS
		 */	
		if (pls_send_packet(pls, &request, 1)) {
			rtx_message_warning("pls_get_lms_hw_configuration: pls_send_packet: failed");
			tries++;
			continue;
		}

		/*
		 * Wait for reply and print out config info
		 */
		if (pls_wait_for_reply(pls, "pls_get_lms_hw_configuration"))
			return rtx_error("pls_get_lms_hw_configuration: pls_wait_for_reply: failed");

		/*
		 * Copy the reply data to local buffer
		 */
		rtx_mutex_lock(pls->mutex);
			memcpy((char *)&reply[0], (char *)&pls->reply[0], 40);
		rtx_mutex_unlock(pls->mutex);

		/*
		 * Process the reply
		 */
		switch (reply[0]) {
		case LMSKFG_TGM: 
			pls_lms_hw_config_read(pls, &reply[1]);
			if (printMessage)
				pls_print_lms_hw_configuration(pls, stderr);
			return 0;
		case NACK_TGM:
			rtx_error("pls_get_lms_hw_configuration: not acknowledge, e.g. invalid request");		
			return -1;
		default:
			rtx_message_warning("pls_get_lms_hw_configuration: reply type 0x%02x incorrect", reply[0]);
			tries++;
			break;
		}
	}

	return -1;
}

/**
 * This function gets the LMS operating data counters
 *
 * \return Zero on success, -1 on error
 */
int
pls_get_lms_operating_data(
	Pls	*pls,
	int	printMessage
	)
{
	unsigned char	request, reply[5];
	unsigned int	tries = 0;

	if (!pls->lms)
		return rtx_error("pls_get_lms_hw_configuration: this function is only valid with an LMS type 6 device");

	request = ODCANF_TGM;
	
	for (;;) {
		/*
		 * Check to see if it time to give up. if so return.
		 */
		if (tries >= PLS_TRIES_MAX)
			return rtx_error("pls_get_lms_operating_data: can't get LMS operating data, have tried %d times without success", tries);

		if (pls->debug)
			rtx_message_routine("pls_get_lms_operating_data: requesting operating data");

		/*
		 * Send the packet to the LMS
		 */	
		if (pls_send_packet(pls, &request, 1)) {
			rtx_message_warning("pls_get_lms_operating_data: pls_send_packet: failed");
			tries++;
			continue;
		}

		/*
		 * Wait for reply and print out config info
		 */
		if (pls_wait_for_reply(pls, "pls_get_lms_operating_data"))
			return rtx_error("pls_get_lms_operating_data: pls_wait_for_reply: failed");

		/*
		 * Copy the reply data to local buffer
		 */
		rtx_mutex_lock(pls->mutex);
			memcpy((char *)&reply[0], (char *)&pls->reply[0], 5);
		rtx_mutex_unlock(pls->mutex);

		/*
		 * Process the reply
		 */
		switch (reply[0]) {
		case ODC_TGM: 
          pls->hoursRunning = hex_16_to_dec(reply[1], reply[2]);
          pls->switchOnCount = hex_16_to_dec(reply[3], reply[4]);
          if (printMessage) {
            fprintf(stderr, "hours running = %d, num switch on's = %d\n", 
                    pls->hoursRunning, pls->switchOnCount);
          }
			return 0;
		case NACK_TGM:
			rtx_error("pls_get_lms_operating_data: not acknowledge, e.g. invalid request");		
			return -1;
		default:
			rtx_message_warning("pls_get_lms_operating_data: reply type 0x%02x incorrect", reply[0]);
			tries++;
			break;
		}
	}

	return -1;
}


/**
 * This function sets the hardware configuartion of an LMS
 *
 * \return Zero on success, -1 on error
 */
int
pls_set_lms_hw_configuration(
	Pls	*pls,
	LmsHw	*config
	)
{
	unsigned int	pktlen;
	unsigned char	request[45], reply[2];
	unsigned int	tries = 0;

	if (!pls->lms)
		return rtx_error("pls_set_lms_hw_configuration: this function is only valid with an LMS type 6 device");

	/* 
	 * Put the LMS into SETUP mode before setting the configuration
	 */
	if (pls_set_mode(pls, PLS_MODE_SETUP))
		return rtx_error("pls_set_lms_hw_configuration: can't get into SETUP mode and so can't set LMS hardware configuartion");

	/*
	 * Construct the request packet
	 */
	request[0] = LMSKFGDEF_TGM;
	pls_intToHex(config->blanking, &request[1]);
	pls_intToHex(config->stopThreshold, &request[3]);
	request[5] = config->availability;
	request[6] = config->settingMeasured;
	request[7] = config->unitsMeasured;
	request[8] = config->temporaryField;
	request[9] = config->subtractive;
	request[10] = config->multipleEvaluation;
	request[11] = config->restart;
	request[12] = config->restartTime;
	request[13] = config->restartTest;
	request[14] = config->cAObject;
	request[15] = config->cAPositive;
	request[16] = config->cANegative;
	request[17] = config->cAStarting;
	request[18] = config->cAStopping;
	request[19] = config->cBObject;
	request[20] = config->cBPositive;
	request[21] = config->cBNegative;
	request[22] = config->cBStarting;
	request[23] = config->cBStopping;
	request[24] = config->cCObject;
	request[25] = config->cCPositive;
	request[26] = config->cCNegative;
	request[27] = config->cCStarting;
	request[28] = config->cCStopping;
	request[29] = config->pixel;
	request[30] = config->single;
	pls_intToHex(config->zeroPoint, &request[31]);
	pktlen = 33;

	for (;;) {
		/*
		 * Check to see if it time to give up. if so return.
		 */
		if (tries >= PLS_TRIES_MAX)
			return rtx_error("pls_set_lms_hw_configuration: can't set LMS hardware configuartion, have tried %d times without success", tries);

		/*
		 * Send the packet to the LMS
		 */
		if (pls_send_packet(pls, request, pktlen)) {
			if (pls->debug)
				rtx_message_warning("pls_set_lms_hw_configuration: can't set LMS hardware configuartion, error sending packet");
			tries++;
			continue;
		}

		/* 
		 * Must change the anti-hang timeout length since changing
		 * hardware configuartion can take up to 10 seconds
		 */
		if (pls->debug)
			rtx_message_routine("pls_set_lms_hw_configuration: can take up to 10 seconds to change LMS hardware configuartion. Please be patient 8-)");
		rtx_mutex_lock(pls->mutex);
			pls->maxReplyWait = 10;
		rtx_mutex_unlock(pls->mutex);

		/*
		 * Wait for reply and check that the setting was successfull
		 */
		if (pls_wait_for_reply(pls, "pls_set_lms_hw_configuration"))
			return rtx_error("pls_set_lms_hw_configuration: pls_wait_for_reply: failed");

		/*
		 * Change anti-hang timeout length back to default value
		 */
		rtx_mutex_lock(pls->mutex);
			pls->maxReplyWait = PLS_MAX_REPLY_WAIT;
		rtx_mutex_unlock(pls->mutex);

		rtx_mutex_lock(pls->mutex);
			reply[0] = pls->reply[0];
			reply[1] = pls->reply[1];
		rtx_mutex_unlock(pls->mutex);

		switch (reply[0]) {
		case LMSKFGACK_TGM:
			switch (reply[1]) {
			case 0x00:
				if (pls->debug)
					rtx_message_warning("pls_set_lms_hw_configuration: LMS hardware configuartion definition not accepted\n");
				tries++;
				continue;
			case 0x01:
				if (pls->debug)
					rtx_message_routine("pls_set_lms_hw_configuration: LMS hardware configuartion definition accepted");
				break;
			default:
				if (pls->debug)
					rtx_message_warning("pls_set_perm_baudrate: unrecognised reply");
				tries++;
				continue;
			}
			pls_lms_hw_config_read(pls, &reply[2]);
			return 0;
			break;
		case NACK_TGM:
			if (pls->debug)
				rtx_message_warning("pls_set_lms_hw_configuration: not acknowledge, invalid request");		
			tries++;
			continue;	
		default:
			if (pls->debug)
				rtx_message_warning("pls_set_lms_hw_configuration: reply type 0x%02x incorrect", reply[0]);
			tries++;
			break;
		}
	}

	return -1;
}

/**
 * This function sets the stop threshold of the LMS. Note that first it gets 
 * the hardware configuartion and hence takes a bit of time.
 *
 * \return Zero on success, -1 on error
 */
int
pls_set_lms_stop_threshold(
	Pls	*pls, 
	int	threshold
	)
{
	LmsHw	config;

	if (!pls->lms)
		return rtx_error("pls_set_lms_stop_threshold: this function is only valid with an LMS type 6 device");

	/*
	 * Get the LMS hardware configuartion
	 */
	if (pls_get_lms_hw_configuration(pls, 0))
		return rtx_error("pls_set_lms_stop_threshold: pls_get_lms_hw_configuration: failed");

	/*
	 * Copy the current config to a temp struct
	 */
	config = pls->lmsHw;

	/*
	 * Change the stopThreshold field in the temp struct
	 */
	config.stopThreshold = threshold;
	
	/*
	 * Set new hardware configuartion
	 */
	if (pls_set_lms_hw_configuration(pls, &config))
		return rtx_error("pls_set_lms_stop_threshold: pls_set_lms_hw_configuration: failed");

	return 0;
}

/**
 * This function sets the units of the LMS. Note that first it gets the
 * hardware configuartion and hence takes a bit of time.
 *
 * \return Zero on success, -1 on error
 */
int
pls_set_lms_units(
	Pls	*pls,
	PlsUnit	unit
	)
{
	LmsHw	config;

	if (!pls->lms)
		return rtx_error("pls_set_lms_units: this function is only valid with an LMS type 6 device");

	/*
	 * Get the LMS hardware configuartion
	 */
	if (pls_get_lms_hw_configuration(pls, 0))
		return rtx_error("pls_set_lms_units: pls_get_lms_hw_configuration: failed");

	/*
	 * Copy the current config to a temp struct
	 */
	config = pls->lmsHw;

	/*
	 * Change the unit field in the temp struct
	 */
	config.unitsMeasured = unit;
	
	/*
	 * Set new hardware configuartion
	 */
	if (pls_set_lms_hw_configuration(pls, &config))
		return rtx_error("pls_set_lms_units: pls_set_lms_hw_configuration() failed");

	return 0;
}

/**
 * This function sets the availabilty level of the LMS. This used to be called
 * "fog correction".
 *
 * Note that first it gets the hardware configuartion and hence takes a bit 
 * of time.
 *
 * \return Zero on success, -1 on error
 */
int
pls_set_lms_availability(
	Pls		*pls,
	unsigned char	availability
	)
{
	LmsHw	config;

	if (!pls->lms)
		return rtx_error("pls_set_lms_availability: this function is only valid with an LMS type 6 device");

	/*
	 * Get the LMS hardware configuartion
	 */
	if (pls_get_lms_hw_configuration(pls, 0))
		return rtx_error("pls_set_lms_availability: pls_get_lms_hw_configuration: failed");

	/*
	 * Copy the current config to a temp struct
	 */
	config = pls->lmsHw;

	/*
	 * Change the unit field in the temp struct
	 */
	config.availability = availability;
	
	/*
	 * Set new hardware configuartion
	 */
	if (pls_set_lms_hw_configuration(pls, &config))
		return rtx_error("pls_set_lms_availability: pls_set_lms_hw_configuration() failed");

	return 0;
}


/**
 * This function sets the restart mode and time of the LMS. 
 *
 * Note that first it gets the hardware configuartion and hence takes a bit 
 * of time.
 *
 * \return Zero on success, -1 on error
 */
int
pls_set_lms_restart(
	Pls		*pls,
	unsigned char	restartMode,
    unsigned char   restartTime
	)
{
	LmsHw	config;

	if (!pls->lms)
		return rtx_error("pls_set_lms_restart: this function is only valid with an LMS type 6 device");

	/*
	 * Get the LMS hardware configuartion
	 */
	if (pls_get_lms_hw_configuration(pls, 0))
		return rtx_error("pls_set_lms_restart: pls_get_lms_hw_configuration: failed");

	/*
	 * Copy the current config to a temp struct
	 */
	config = pls->lmsHw;

	/*
	 * Change the fields in the temp struct
	 */
    config.restart = restartMode;
    config.restartTime = restartTime;
	
	/*
	 * Set new hardware configuartion
	 */
	if (pls_set_lms_hw_configuration(pls, &config))
		return rtx_error("pls_set_lms_restart: pls_set_lms_hw_configuration() failed");

	return 0;
}


/**
 * This function sets the LMS hardware config "setting measured" flag
 * which allows you to get intensity.
 *
 * \return Zero on success, -1 on error
 */
int
pls_set_lms_setting_measured(
	Pls		*pls,
	unsigned char	setting
	)
{
	LmsHw	config;

	if (!pls->lms)
		return rtx_error("pls_set_lms_setting_measured: this function is only valid with an LMS type 6 device");

	/*
	 * Get the LMS hardware configuartion
	 */
	if (pls_get_lms_hw_configuration(pls, 0))
		return rtx_error("pls_set_lms_setting_measured: pls_get_lms_hw_configuration: failed");

	/*
	 * Copy the current config to a temp struct
	 */
	config = pls->lmsHw;

	/*
	 * Change the setting measured value in the struct
	 */
	config.settingMeasured = setting;
	
	/*
	 * Set new hardware configuartion
	 */
	if (pls_set_lms_hw_configuration(pls, &config))
		return rtx_error("pls_set_lms_setting_measured: pls_set_lms_hw_configuration: failed");

	return 0;
}

/**
 * This function zeros the LMS fields A, B and C. Note that first it gets the
 * hardware configuartion and hence takes a bit of time.
 *
 * \return Zero on success, -1 on error
 */
int
pls_set_lms_zero_fields(
	Pls	*pls
	)
{
	LmsHw	config;

	if (!pls->lms)
		return rtx_error("pls_set_lms_zero_fields: this function is only valid with an LMS type 6 device");

	/*
	 * Get the LMS hardware configuartion
	 */
	if (pls_get_lms_hw_configuration(pls, 0))
		return rtx_error("pls_set_zero_fields: pls_get_lms_hw_configuration: failed");

	/*
	 * Copy the current config to a temp struct
	 */
	config = pls->lmsHw;

	/*
	 * Change the unit field in the temp struct
	 */
	config.cAObject = 0;
	config.cAPositive = 0;
	config.cANegative = 0;
	config.cAStarting = 0;
	config.cAStopping = 0;

	config.cBObject = 0;
	config.cBPositive = 0;
	config.cBNegative = 0;
	config.cBStarting = 0;
	config.cBStopping = 0;

	config.cCObject = 0;
	config.cCPositive = 0;
	config.cCNegative = 0;
	config.cCStarting = 0;
	config.cCStopping = 0;
	
	/*
	 * Set new hardware configuartion
	 */
	if (pls_set_lms_hw_configuration(pls, &config))
		return rtx_error("pls_set_zero_fields: pls_set_lms_hw_configuration: failed");

	return 0;
}

/**
 * This function activates/deactivates the LMS laser
 *
 * \return Zero on success, -1 on error
 */
int
pls_set_lms_laser(
	Pls		*pls, 
	unsigned char	state
	)
{
	unsigned char	request[2], reply[10];
	unsigned int	tries = 0;

	if (!pls->lms)
		return rtx_error("pls_set_lms_laser: this function is only valid with an LMS type 6 device");

	/*
	 * Construct request packet
	 */
	request[0] = LASER_TGM;
	request[1] = state;
	
	for (;;) {
		/*
		 * Check to see if it time to give up. if so return.
		 */
		if (tries >= PLS_TRIES_MAX)
			return rtx_error("pls_set_lms_laser: can't set LMS laser, have tried %d times without success", tries);

		if (pls->debug) {
			if (state)
				rtx_message_routine("pls_set_lms_laser: turning LMS laser ON");
			else
				rtx_message_routine("pls_set_lms_laser: turning LMS laser OFF");
		}

		/*
		 * Send the packet to the LMS
		 */	
		if (pls_send_packet(pls, request, 2)) {
			rtx_message_warning("pls_set_lms_laser: pls_send_packet: failed");
			tries++;
			continue;
		}

		/* 
		 * Must change the anti-hang timeout length since turning on
		 * the laser can take up to 16 seconds
		 */
		if (state) {
			if (pls->debug)
				rtx_message_routine("pls_set_lms_laser: can take up to 16 seconds to turn on laser. Please be patient.");
			rtx_mutex_lock(pls->mutex);
				pls->maxReplyWait = 16;
			rtx_mutex_unlock(pls->mutex);
		}

		/*
		 * Wait for reply
		 */
		if (pls_wait_for_reply(pls, "pls_set_lms_laser"))
			return rtx_error("pls_set_lms_laser: pls_wait_for_reply: failed");

		/*
		 * Change anti-hang timeout length back to default value
		 */
		if (state) {
			rtx_mutex_lock(pls->mutex);
				pls->maxReplyWait = PLS_MAX_REPLY_WAIT;
			rtx_mutex_unlock(pls->mutex);
		}

		/*
		 * Copy the reply data to local buffer
		 */
		rtx_mutex_lock(pls->mutex);
			memcpy((char *)&reply[0], (char *)&pls->reply[0], 10);
		rtx_mutex_unlock(pls->mutex);

		/*
		 * Process the reply
		 */
		switch (reply[0]) {
		case LASERACK_TGM: 
			switch (reply[1]) {
			case 0x00:
				rtx_message_warning("pls_set_lms_laser: change of laser NOT accepted");
				tries++;
				continue;
				break;
			case 0x01:
				if (pls->debug)
					rtx_message_routine("pls_set_lms_laser: change of laser accepted");
				break;
			default:
				rtx_message_warning("pls_set_lms_laser: unrecognized reply [0x%02x]", reply[1]);
				tries++;
				continue;
			}
			switch (reply[2]) {
			case 0x00:
				pls->laserOn = 0;
				if (pls->debug)
					rtx_message_routine("pls_set_lms_laser: laser deactivated");
				break;
			case 0x01:
				pls->laserOn = 1;
				if (pls->debug)
					rtx_message_routine("pls_set_lms_laser: laser activated");
				break;
			default:
				rtx_message_warning("pls_set_lms_laser: unrecognized reply [0x%02x]", reply[1]);
				tries++;
				continue;
			}
			return 0;
		case NACK_TGM:
			rtx_error("pls_set_lms_laser: not acknowledge, e.g. invalid request");		
			return -1;
		default:
			rtx_message_warning("pls_set_lms_laser: reply type 0x%02x incorrect", reply[0]);
			tries++;
			break;
		}
	}

	return -1;
}

/**
 * This function prints information about the error status of the PLS. It
 * should be called if the status byte returned by any other telegram 
 * indicates an error or a warning
 *
 * \return PLS mode on success, -1 on error
 */
PlsMode
pls_get_error(
	Pls	*pls,
	int	printMessage
	)
{
	unsigned char	request, reply[125];
        unsigned int    replyLen;
	unsigned int	i;
	unsigned int	tries = 0;
	char		*errorType;
	unsigned char	errorNum, errorOld = 0;
	PlsMode		mode;

	request = ERRANF_TGM;
	
	/*
	 * Get current mode
	 */
	mode = pls->mode;

	for (;;) {
		/*
		 * Check to see if it time to give up. if so return.
		 */
		if (tries >= PLS_TRIES_MAX)
			return rtx_error("pls_get_status: can't get status, have tried %d times without success", tries);

		if (pls->debug)
			rtx_message_routine("pls_get_error: requesting PLS status");

		/*
		 * Error packets can be any length so disable the packet
		 * length checker in the reader thread
		 */
		pls->checkPktLen = 0;

		/*
		 * Send the packet to the PLS
		 */	
		if (pls_send_packet(pls, &request, 1)) {
			rtx_message_warning("pls_get_error: pls_send_packet: failed");
			tries++;
			continue;
		}

		/*
		 * Wait for reply and print out status
		 */
		if (pls_wait_for_reply(pls, "pls_get_error"))
			return rtx_error("pls_get_status: pls_wait_for_reply: failed");

		/*
		 * Re-enable packet length checker in reader thread
		 */
		pls->checkPktLen = 1;

		/*
		 * Copy the reply data to local buffer
		 */
		rtx_mutex_lock(pls->mutex);
			memcpy((char *)&reply[0], (char *)&pls->reply[0], 125);
                        replyLen = pls->replyLen;
		rtx_mutex_unlock(pls->mutex);

		/*
		 * Process the reply
		 */
                rtx_message_routine("pls_get_error: %d warnings/errors", replyLen/2-1);
                if (replyLen > 728) {
                        rtx_message_warning ("pls_get_error: reply length %d > array size", replyLen);
                        replyLen = 124;
		}
		switch (reply[0]) {
		case ERR_TGM: 
			if (printMessage) {
				fprintf(stderr, "Error status\n");
				fprintf(stderr, "------------\n");
				fprintf(stderr, "Number\tType\n");
			}

   		        for (i=1; i<replyLen-2; i+=2) {
				switch (reply[i] & 0x0f) {
				case 0:
					errorType = strdup("No error");
					break;
				case 1:
					errorType = strdup("Information");
					break;
				case 2:
					errorType = strdup("Warning");
					break;
				case 3:
					errorType = strdup("Error");
					break;
				case 4:
					errorType = strdup("Fatal error");
					break;
				default:
					errorType = strdup("Unrecognized");
					break;
				}
				if (pls->lms)
					errorOld = (reply[i] & 0x80);
				errorNum = reply[i+1];

				if (pls->lms) {
					rtx_message_routine ("pls_get_error: %s, error number = %d, %s - old (irrelevant) = %d",
							     errorType, errorNum, plsErrorCodes[errorNum], errorOld); 

					if (printMessage) {
						fprintf(stderr, "%d\t%s", errorNum, errorType); 
						fprintf(stderr, " %s", plsErrorCodes[errorNum]);
						if (errorOld)
							fprintf(stderr, " - old (irrelevant)\n");
						else
							fprintf(stderr, " - current\n");
					}
				} else {
					rtx_message_routine ("pls_get_error: %s, error number = %d %s", 
							     errorType, errorNum, plsErrorCodes[errorNum]); 
					if (printMessage)
						fprintf(stderr, "%d\t%s", errorNum, errorType); 
						fprintf(stderr, " %s\n", plsErrorCodes[errorNum]);
				}
				free(errorType);
			}
			return reply[replyLen-1];
			break;
		case NACK_TGM:
			rtx_message_warning("pls_get_error: not acknowledge, e.g. Invalid operating mode change");		
			return mode;
		default:
			rtx_message_warning("pls_get_error: reply type 0x%02x incorrect", reply[0]);
			tries++;
			break;
		}
	}
}

/**
 * This function prints some info regarding the PLS status byte to the
 * error channel
 *
 * \return Nothing
 */
void
pls_status_byte_print(
	Pls	*pls
	)
{
	rtx_mutex_lock(pls->mutex);

	if (pls->status & 0x01)
		rtx_message_routine("pls_status_byte_print: Information bit set on PLS %d.", pls->id);

	if (pls->status & 0x02)
		rtx_message_routine("pls_status_byte_print: Warning bit set on PLS %d.", pls->id);

	if (pls->status & 0x03)
		rtx_message_routine("pls_status_byte_print: Error bit set on PLS %d.", pls->id);

	if (pls->status & 0x04)
		rtx_message_routine("pls_status_byte_print: Fatal error bit set on PLS %d.", pls->id);

	if (pls->status & 0x80)
		rtx_message_routine("pls_status_byte_print: Dirt bit set. There must be some kind of crap on PLS %d's front panel", pls->id);

	rtx_mutex_unlock(pls->mutex);
}

/**
 * This function sends a request to the PLS for a section of range data
 * between points first and last. This may be useful if the area of interest
 * is in a known fixed area. The data is returned at 0.5deg resolution only. 
 *
 * Note: this function is only supported by PLS's firmware 3.00 to 3.05, and
 *       has been dropped by SICK from more recent models
 *
 * \return Zero on success, -1 on error
 */
int
pls_get_partial_range(
	Pls		*pls,
	unsigned int	first,
	unsigned int	last
	)
{
	unsigned int	pktlen;
	unsigned char	request[PLS_PACKET_MAX];
	unsigned int	tries = 0;

	rtx_mutex_lock(pls->mutex);
	if (pls->firmWare > 5) {
		rtx_mutex_unlock(pls->mutex);
		return rtx_error("pls_get_partial_range: this function has been dropped by SICK from this version of firmware (supported upto and including 3.05).");
	}
	rtx_mutex_unlock(pls->mutex);

	if ( (first < 1) || (last > 361) )
		return rtx_error("pls_get_partial_range: invalid range request, values must be between 1 and 361");

	if (last < first)
		return rtx_error("pls_get_partial_range: last must be greater than or equal to first");

	/*
	 * Construct the request packet
	 */
	request[0] = MWPANF_TGM;
	pls_intToHex(first, &request[1]);
	pls_intToHex(last, &request[3]);
	pktlen = 5;

	if (pls->debug)
		rtx_message_warning("pls_get_partial_range: getting partial range data between %d and %d", first, last);

	for (;;) {
		/*
		 * Check to see if it time to give up. if so return.
		 */
		if (tries >= PLS_TRIES_MAX)
			return rtx_error("pls_get_partial_range: can't get partial range data, have tried %d times without success", tries);

		/*
		 * Send the packet to the PLS
		 */
		if (pls_send_packet(pls, request, pktlen)) {
			if (pls->debug)
				rtx_message_warning("pls_get_partial_range: can't get partial range data, error sending packet");
			tries++;
			continue;
		}

		/*
		 * Wait for reply
		 */
		if (pls_wait_for_reply(pls, "pls_get_partial_range"))
			return rtx_error("pls_get_partial_range: pls_wait_for_reply() failed");

		/*
		 * Note: the reader deals with the reply by calling the
		 * range_proc()
		 */	
		return 0;
	}
}

/**
 * This function sends a request to the PLS for measured values averaged of
 * N scans
 *
 * Note: this function is only supported by PLS's firmware 3.00 to 3.05, and
 *       has been dropped by SICK from more recent models
 *
 * \return Zero on success, -1 on error
 */
int
pls_get_average_range(
	Pls		*pls,
	unsigned int	nscans
	)
{
	unsigned int	pktlen;
	unsigned char	request[2];
	unsigned int	tries = 0;

	rtx_mutex_lock(pls->mutex);
	if (pls->firmWare > 5) {
		rtx_mutex_unlock(pls->mutex);
		return rtx_error("pls_get_partial_range: this function has been dropped by SICK from this version of firmware (supported upto and including 3.05).");
	}
	rtx_mutex_unlock(pls->mutex);

	if ( (nscans < 2) || (nscans > 250) )
		return rtx_error("pls_get_average_range: invalid range request, nscans must be between 2 and 250");

	/*
	 * Construct the request packet
	 */
	request[0] = MMWANF_TGM;
	request[1] = nscans;
	pktlen = 2;

	if (pls->debug)
		rtx_message_routine("pls_get_average_range: getting average range data averaged over %d scans", nscans);

	for (;;) {
		/*
		 * Check to see if it time to give up. if so return.
		 */
		if (tries >= PLS_TRIES_MAX)
			return rtx_error("pls_get_average_range: can't get average range data, have tried %d times without success", tries);

		/*
		 * Send the packet to the PLS
		 */
		if (pls_send_packet(pls, request, pktlen)) {
			if (pls->debug)
				rtx_message_warning("pls_get_average_range: can't get average range data, error sending packet\n");
			tries++;
			continue;
		}

		/* 
		 * Must change the anti-hang timeout length since averaging
		 * can take up to 10 seconds for 250 scans
		 */
		rtx_mutex_lock(pls->mutex);
			pls->maxReplyWait = ((int)(nscans/25))+1;
		rtx_mutex_unlock(pls->mutex);

		/*
		 * Wait for reply
		 */
		if (pls_wait_for_reply(pls, "pls_get_average_range"))
			return rtx_error("pls_get_average_range: pls_wait_for_reply() failed");

		/*
		 * Change anti-hang timeout length back to default value
		 */
		rtx_mutex_lock(pls->mutex);
			pls->maxReplyWait = PLS_MAX_REPLY_WAIT;
		rtx_mutex_unlock(pls->mutex);

		/*
		 * Note: the reader deals with the reply by calling the
		 * range_proc()
		 */	
		return 0;
	}
}

/**
 * This function sends a request to the PLS for all measured values
 *
 * \return Zero on success, -1 on error
 */
int
pls_get_range(
	Pls	*pls
	)
{
	unsigned int	pktlen;
	unsigned char	request[2];

	/*
	 * Construct the request packet
	 */
	request[0] = MWANF_TGM;
	request[1] = 0x01;
	pktlen = 2;

	if (pls->debug)
		rtx_message_routine("pls_get_range: getting range data");

	/*
	 * Send the packet to the PLS
	 */
	if (pls_send_packet(pls, request, pktlen))
		return rtx_error("pls_get_range: can't get range data, error sending packet");

	/*
	 * Note: the reader deals with the reply by calling the
	 * range_proc()
	 */	
	return 0;
}

/**
 * This function is passed the name of the range proc you want called
 * every time a range packet is received from the PLS. The range data
 * is stored in the buffer pointed to by b.
 * 
 * b[0]     no. range data (N), low byte
 * b[1]     no. range data (N), high byte
 * b[2]     first range data, low byte 	
 * b[3]     first range data, high byte 	
 *  ...			...
 * b[2*N]   last range data, low byte
 * b[2*N+1] last range data, high byte
 *
 * Note: the top 3 bits (MSB) of the range high byte contain status info
 * and should be ignored to get range data. 
 *
 * \return Nothing
 */
void
pls_set_callback(
	Pls	*pls,
	void	(*p)(unsigned char *b, int id)
	)
{
	rtx_mutex_lock(pls->mutex);
		pls->range_proc = p;
	rtx_mutex_unlock(pls->mutex);
}

/**
 * This function is passed the name of the header proc you want called
 * every time a range packet header is received from the PLS.
 *
 * \return Nothing
 */
void
pls_set_header_callback(
	Pls	*pls,
	void 	(*p)(int id)
	)
{
	rtx_mutex_lock(pls->mutex);
		pls->header_proc = p;
	rtx_mutex_unlock(pls->mutex);

}

/**
 * This function converts the range data from cm to mm values and deals 
 * with dazzle, warning field and protcetion field violations.
 *
 * \return	0 if no errors or infringments, 
 *		1 if a segment if dazzled, 
 *		2 if a segment violates the warning field, 
 *		3 if a segment violates the protection field,
 *		-1 on error
 */
int
pls_range_cm_to_mm(
	unsigned char	*data,
	int		nsegs,
	int		*range
	)
{
	unsigned char	*p;
	int		n, i, state = 0;

	n = hex_14_to_dec(data[0], data[1]);
	if (nsegs != n)
		return rtx_error("pls_range_cm_to_mm: invalid nsegs = %d", nsegs);

	for (p=&data[2],i=0; i<nsegs; i++, p+=2) {
		range[i] = 10*hex_13_to_dec(p[0], p[1]);
		if (p[1] & 0x20) {
			state = 1;
			range[i] = 0;
		} else if (p[1] & 0x40) {
			state = 2;
		} else if (p[1] & 0x80) {
			state = 3;	
		}
	}

	return state;
}

/**
 * This function converts the range data from cm to m values and deals 
 * with dazzle, warning field and protcetion field violations.
 *
 * \return	0 if no errors or infringments, 
 *		1 if a segment if dazzled, 
 *		2 if a segment violates the warning field, 
 *		3 if a segment violates the protection field,
 *		-1 on error
 */
int
pls_range_cm_to_m(
	unsigned char	*data,
	int		nsegs,
	float		*range,
	unsigned char	*intensity
	)
{
	unsigned char	*p;
	int		n, i, state = 0;

	n = hex_14_to_dec(data[0], data[1]);
	if (nsegs != n)
		return rtx_error("pls_range_cm_to_m: invalid nsegs = %d", nsegs);

	for (p=&data[2],i=0; i<nsegs; i++, p+=2) {
		range[i] = hex_13_to_dec(p[0], p[1])/100.0;
		if (p[1] & 0x20) {
			state = 1;
			range[i] = 0.0;
		} else if (p[1] & 0x40) {
			state = 2;
		} else if (p[1] & 0x80) {
			state = 3;	
		}
	}

	return state;
}

/**
 * This function converts the range data from mm to m values and deals 
 * with dazzle, warning field and protcetion field violations.
 *
 * \return	0 if no errors or infringments, 
 *		1 if a segment if dazzled, 
 *		2 if a segment violates the warning field, 
 *		3 if a segment violates the protection field,
 *		-1 on error
 */
int
pls_range_mm_to_m(
	unsigned char	*data,
	int		nsegs,
	float		*range,
	unsigned char	*intensity
	)
{
	unsigned char	*p;
	int		n, i, state = 0;

	n = hex_14_to_dec(data[0], data[1]);
	if (nsegs != n)
		return rtx_error("pls_range_mm_to_m: invalid nsegs = %d", nsegs);

	for (p=&data[2],i=0; i<nsegs; i++, p+=2) {
		range[i] = hex_13_to_dec(p[0], p[1])/1000.0;
		if (p[1] & 0x20) {
			state = 1;
			range[i] = 0.0;
		} else if (p[1] & 0x40) {
			state = 2;
		} else if (p[1] & 0x80) {
			state = 3;	
		}
	}

	return state;
}

/**
 * This function converts an LMS/PLS's range data string into real data
 *
 * \return Zero on success, -1 on error
 */
int
pls_range_buf_to_m(
	Pls		*pls,
	unsigned char	*buf,
	float		*range,
	unsigned char	*state
	)
{
	double		factor = 0.01;
	int		i, nsegs;
	unsigned char	*p;

	if (pls->lms) {
		switch (pls->unit) {
		case PLS_UNIT_CM:
			factor = 0.01;
			break;
		case PLS_UNIT_MM:
			factor = 0.001;
			break;
		case PLS_UNIT_DM:
			factor = 0.1;
			break;
		}
	} else {
		factor = 0.01;
	}

	nsegs = hex_14_to_dec(buf[0], buf[1]);
	for (p=&buf[2],i=0; i<nsegs; i++, p+=2) {
		range[i] = hex_13_to_dec(p[0], p[1])*factor;
		if (state != NULL) {
			state[i] = (p[1] & 0xe0)>>5;
		}
	}
    /* check the second bit to see if real-time indices are used */
	if (pls->lmsHw.availability & 0x02) { 
	  /* decode the real-time indices */
	  state[399] = p[0];	/* scan index */
	  state[400] = p[1];	/* telegram index */
	}

	return 0;
}

/**
 * This function fills a string with the LMS/PLS's capability.
 * Used for RTC servers.
 *
 * \return Nothing
 */
void
pls_capability_string(
	Pls	*pls,
	char	*string
	)
{
	sprintf(string, "%s %f %f %f %d %f %f %f %f",
		pls->name,
		pls->thetaMin*(PI/180.0),
		pls->thetaMax*(PI/180.0),
		pls->resolution*(PI/180.0),
		pls->samples,
		0.0,
		0.0,
		0.0,
		pls->rMax
		);
}

/**
 * This function fills a string with the LMS/PLS's range data
 *
 * \return Nothing
 */
void
pls_range_string(
	Pls	*pls,
	int	nsegs,
	int	*data,
	char	*string
	)
{
	double	dataM, factor = 0.01;
	int	i;

	if (pls->lms) {
		switch (pls->unit) {
		case PLS_UNIT_CM:
			factor = 0.01;
			break;
		case PLS_UNIT_MM:
			factor = 0.001;
			break;
		case PLS_UNIT_DM:
			factor = 0.1;
			break;
		}
	} else {
		factor = 0.01;
	}

	strcpy(string, "");
	for (i=0; i<nsegs; i++) {
		dataM = data[i]*factor;
		sprintf(string, "%s %7.3f ", string, dataM);
	}
}

/**
 * This function opens a file for raw PLS data logging
 *
 * \return Pointer to an open file on success, NULL on error
 */
FILE *
pls_open_log_file(
	Pls	*pls,
	char	*filename
	)
{
	char	string[BUFSIZ+25];
	int	tmp;
	FILE	*fw;
	time_t	now;

	if (!strcmp(filename, "")) {
		/* Generate a name */
		now = time(NULL);
                strftime(string, BUFSIZ+25, "%Y%m%d%H%M%S", localtime(&now));
                strcat(string, "-");  
                strcat(string, pls->name);  
	}
	else {
		if (strlen(filename) > BUFSIZ)
			return rtx_error_null("pls_open_log_file: filename too long");
		strcpy(string, filename);
	}

	if (pls->debug)
		rtx_message_routine("pls_open_log_file: opening %s logFile", string);

	if ((fw=fopen(string, "w")) == NULL)
		return rtx_error_null("pls_open_log_file: canot open %s file", string);      

	/*
	 * Write a magic number at the begining of the 
	 * binary data file to aid other reading programs
	 * foil the endian swap problem
	 */
	tmp = 0x12345678;
	if (fwrite(&tmp, sizeof(int), 1, fw) != 1)
		return rtx_error_null("pls_open_log_file: fwrite: failed");

	return fw;	
}

/**
 * This function saves range data to the file pointed to by fw.
 * The data is stored in a binary format.
 *
 * \return Nothing
 */
void
pls_data_to_file(
	Pls	*pls,
	FILE	*fw
	)
{
	rtx_mutex_lock(pls->mutex);
	fwrite(&pls->id, sizeof(int), 1, fw);
	fwrite(&pls->arrivalTime.seconds, sizeof(long), 1, fw);
	fwrite(&pls->arrivalTime.nanoSeconds, sizeof(long), 1, fw);
	fwrite(&pls->samples, sizeof(int), 1, fw);
	fwrite(pls->rangeData, 2*(pls->samples*sizeof(unsigned char)), 1, fw);
	rtx_mutex_unlock(pls->mutex);
}

/**
 * Convert an int to a two byte unsigned char
 *
 * \return Nothing
 */
void
pls_intToHex(
	unsigned int	number,
	unsigned char	*data
	)
{
        data[0] = number & 0x00ff;      
        data[1] = (number & 0xff00) >> 8;       
}

/**
 * Swap bytes in an int
 *
 * \return Nothing
 */
void
pls_int_swap(
	int *data
	)
{
	char	t, *p = (char *)data;

	/* swap bytes 0 & 3 */
	t = p[0];
	p[0] = p[3];
	p[3] = t;
 
	/* swap bytes 1 & 2 */
	t = p[1];
	p[1] = p[2];
	p[2] = t;
}

/** 
 * Compute and return the checksum value for buffer 
 *
 * \return Checksum
 */
unsigned int
pls_crc16_build(
	unsigned char	*data,
	unsigned int	len
	)
{
        unsigned int crc16, crc_data, i;

	crc16 = 0;
	crc_data = 0;
        for (i=0; i<len; i++) {
                crc_data <<= 8;
                crc_data |= (unsigned int)data[i]; 
                crc16 <<= 1;
                if ((crc16 & 0x10000) > 0)
                                crc16 ^= 0x8005;
                crc16 ^= crc_data;
                crc16 &= 0xffff;
        }

        return (crc16);
}

/****************************************************************************
                        Static support functions
 ****************************************************************************/

static int
pls_send_packet(
	Pls		*pls,
	unsigned char	*request,
	unsigned int 	len
	)
{
        unsigned char	command[1024];
	char		string[1024];
        unsigned int	i;
	int 		result;
	unsigned int	noAckCount = 0;

	/*
	 * Construct command telegram 
	 */
        command[0] = STX;
        command[1] = 0x00;
        pls_intToHex(len, &command[2]);
        for (i=0; i<len; i++)
                command[4+i] = request[i];

        pls_intToHex(pls_crc16_build(command, 4+len), &command[4+len]);

	pls->lastAck = NAK;

	/* 
	 * Send command telegram to PLS 
	 */
	if (pls->debug == 99) {
		/* Print packet to message file */
		sprintf(string, "pls_send_packet: W: ");
                for (i=0;i<(len+6); i++)
                       	sprintf(string, "%s %02x ", string, command[i]);
                rtx_message_routine("%s\n", string);

		/* Send packet */
		do {
                        result = write(pls->fd, command, 6+len);
                } while ((result < 0) && (errno == EINTR));
                if ((result < 0) && (errno != EINTR))
                        rtx_error_errno("pls_send_packet: write: failed");
		return 99;
	}

	while (pls->lastAck != ACK) {
      if (noAckCount >= PLS_MAX_NO_ACK)
        return rtx_error("pls_send_packet: no ACK received, tried %d times", noAckCount);
      if (pls->debug > 2) {
        sprintf(string, "pls_send_packet: W: ");
        for (i=0;i<(len+6); i++)
          sprintf(string, "%s %02x ", string, command[i]);
        rtx_message_routine("%s\n", string);
      }
      do {
        result = write(pls->fd, command, 6+len);
      } while ((result < 0) && (errno == EINTR));
      if ((result < 0) && (errno != EINTR))
        rtx_error_errno("pls_send_packet: write: failed");
      else {
        switch (pls->baudRate) {
        case 9600:
          rtx_timer_sleep(1.4);
          break;
        case 19200:
          rtx_timer_sleep(1.0);
          break;
        case 38400:
          rtx_timer_sleep(0.4);
          break;
        default:
          rtx_timer_sleep(0.1);
        }
        noAckCount++;
      }
	}

	pls->lastAck = NAK;

	return 0;
}

/**
 * This function waits for a reply and returns
 *
 * \return Zero on success, -1 on error
 */
static int
pls_wait_for_reply(
	Pls	*pls,
	char	*func_name
	)
{
	if (pls->debug > 1)
		rtx_message_routine("%s: waiting for reply semaphore", func_name);

	pls->replyWaiting = 1;

	if (rtx_sem_wait(pls->replySem))
		return rtx_error("pls_wait_for_reply: rtx_sem_wait: failed");

	pls->replyWaiting = 0;

	if (pls->debug > 1)
		rtx_message_routine("%s: seen reply semaphore", func_name);

	return 0;
}

/*****************************************************************************
 *                          STOP HANG FUNCTIONS
 *****************************************************************************/

/**
 * This thread continuously monitors the reply waiting flag and issues a
 * fake reply if waiting has hung (hung state is defined as no reply for
 * pls->maxReplyWait seconds
 *
 * \return Nothing
 */
static void
pls_stop_hang(
	Pls *pls
	)
{
	while (1) {
		if (pls->replyWaiting) {
			/*
			 * Wait for pls->maxReplyWait seconds before
			 * deciding that everything has hung
			 */	
			rtx_timer_sleep(pls->maxReplyWait);
			if (pls->replyWaiting) {
				pls->reply[0] = FAKE_TGM;
				if (rtx_sem_post(pls->replySem)) {
					rtx_error_flush("rtx_sem_post() failed");
				}
				rtx_error_flush("raised reply semaphore to clear hung state");
			}
		}
		/* 
		 * Sleep for a bit, this function will only be used
		 * when waiting for a reply, ie. not when reading range data
		 */
		rtx_timer_sleep(0.05);
	}
}

/**
 * This thread continuously monitors the output of the laser if the unit is in
 * continuous mode to see if the data stops unexpectedly (e.g. if the power
 * is lost). It will then try and put the unit back into continuous mode.
 *
 * \return Nothing
 */
static void
pls_monitor_continuous_mode(
	Pls *pls
	)
{
  double	age,timeSinceLastFail;
	int	count = 0, done = 0;
	double	checkInterval = 1.0;
    RtxTime	lastFailTime;
    rtx_time_get( &lastFailTime );

	while (1) {
		rtx_timer_sleep(checkInterval);
		if (pls->mode == PLS_MODE_CONTINUOUS) {
			/* Calculate age of laser data */
			age = rtx_time_get_delta(&pls->arrivalTime);

            if (pls->debug > 0) {
              rtx_message_routine("pls_monitor_continuous_mode: age = %g", age);
            }

			/* If too old then assume the laser has stopped */
			if (age > checkInterval) {
				rtx_error_flush("pls_monitor_continuous_mode: laser %s has stopped outputting", pls->name);

                timeSinceLastFail = rtx_time_get_delta(&lastFailTime);
                if (timeSinceLastFail < 5*checkInterval) {
                  /* we have just recently failed so try turning the laser on first */
                  rtx_message_routine("pls_monitor_continuous_mode: only %f seconds since last fail, trying to turn on laser.",
                                      timeSinceLastFail);
                  pls_set_lms_laser(pls, 1);
                }
				
                done  = 0; 
                count = 0;
				while ( !done && pls->mode == PLS_MODE_CONTINUOUS ) {

					/* Put back into continuous mode */
					rtx_message_routine("pls_monitor_continuous_mode: try and put into CONTINUOUS mode");
					if (pls_set_mode(pls, PLS_MODE_CONTINUOUS_OVERRIDE)) {
						rtx_error_flush("pls_monitor_continuous_mode: couldn't change mode. Wait 10 seconds.\n");
						rtx_timer_sleep(10.0);
						count++;
					} else {
						rtx_message_routine("pls_monitor_continuous_mode: laser %s back in CONTINUOUS mode", pls->name);
						done = 1;
						count = 0;
                        rtx_time_get( &lastFailTime );
					}
                    if (count > 5) {
						rtx_message_routine("pls_monitor_continuous_mode: tried too many times, attempting to turn on laser and will wait longer (30 seconds).\n");
                        pls_set_lms_laser(pls, 1);
                        rtx_timer_sleep(30.0);
                        count = 0;
                    }
				}
			}
		}
	}
}



char *plsErrorCodes[] = {
   "00 Internal CPU tests",
   "01 CRC-EPROM test",
   "02 EEPROM test",
   "03 Cyclic register RAM test",
   "04 Cyclic RAM test",
   "05 Evaluation; dazzle evaluation (constant flow, comparator, monoflop)",
   "06 Peak comparator test",
   "07 Evaluation; stop comparator threshold",
   "08 Not used",
   "09 Evaluation of signal amplitude; comparator",
   "10 Not used",
   "11 Not used",
   "12 Time Interval Counter; cycle frequency (PLL, VCO)",
   "13 Time Interval Counter; function at max. frequency (overflow * 1 LSB)",
   "14 Time Interval Counter; logical function",
   "15 TDC initialization & railing function test",
   "16 Not used",
   "17 Front window pollution evaluation; oil channel 1",
   "18 Front window pollution evaluation; polution channel 1",
   "19 Front window pollution evaluation; polution channel 2",
   "20 Front window pollution evaluation; oil channel 2",
   "21 Front window pollution evaluation; reference channel 1",
   "22 Front window pollution evaluation; reference channel 2",
   "23 Reference measurement; internal error",
   "24 Reference measurement; determination of TIC offset",
   "25 Reference measurement; determination of transmitter and receiver performance",
   "26 AD/DA transformer",
   "27 Switch-off path; field infringement field A (OUT 1)",
   "28 Switch-off path; field infringement field B (OUT 2)",
   "29 Motor revolutions",
   "30 System control; measurement number to code write increments do not correlate",
   "31 Sensor dazzle detected",
   "32 Not used",
   "33 Not used",
   "34 Not used",
   "35 Not used",
   "36 Calibration; reference target",
   "37 Calibration; front window pollution",
   "38 Not used",
   "39 Timeout; code write impulse or on TDC calibration",
   "40 Not used",
   "41 Not used",
   "42 Not used",
   "43 Not used",
   "44 Ref. target load measurement implausible",
   "45 1 measurement missed",
   "46 1 scan missed",
   "47 Reference target: load/impulse width value implausible",
   "48 Calibration laser power",
   "49 Laser power (outside 50 - 140%)",
   "50 Initialization TDC M0 channel 0 & 1",
   "51 DA/AD test -- stop branch",
   "52 DA/AD test -- peak branch",
   "53 Error on writing in FLASH",
   "54 Pollution channel measurement without active transmitter",
   "55 No two different angles detected on laser power calibration",
   "56 Watchdog (Hardware) defective",
   "57 No zero index signal available",
   "58 Slave cannot synchronise itself to the master cycle during initialisation",
   "59 Synchronisation lost during running operation",
   "60 Synchronisation cycle from master missing",
   "61 Hardware is not suitable for synchronisation (slave operation mode)",
   "62 DIP switch for laser pulse is in wrong position",
   "63 Not used",
   "64 Not used",
   "65 Not used",
   "66 Not used",
   "67 Not used",
   "68 Not used",
   "69 Not used",
   "70 Not used",
   "71 Not used",
   "72 Not used",
   "73 Not used",
   "74 Not used",
   "75 Not used",
   "76 Not used",
   "77 Not used",
   "78 Not used",
   "79 Not used",
   "80 Reference target: >40 values invalid through filter between current and measured reference target values.",
   "81 Reference target: >60 values invalid through filter between current and measured reference target values.",
   "82 Reference target: >80 values invalid through filter between current and measured reference target values.",
   "83 Reference target: >100 values invalid through filter between current and measured reference target values.",
   "84 Reference target: >20 values in averaged reference target table invalid through filter",
   "85 Reference target: gaps in reference target table",
   "86 Reference target: smallest impulse width too small",
   "87 Reference target: largest impulse width too large",
   "88 Reference target: impulse width spectrum (largest-smallest imp. width) too large",
   "89 Reference target: Reference target erroneous, Ref. table less than 2 cycles update",
   "90 Reference target: calculated bend in the impulse width implausible",
   "91 Reference target: reflectivity measurement cannot be calibrated",
   "92 Reference target: teach-in mode not completed",
   "93 Not used",
   "94 Not used",
   "95 Not used",
   "96 Not used",
   "97 Not used",
   "98 Not used",
   "99 Not used",
   "100 Not used",
   "101 Not used",
   "102 Not used",
   "103 Not used",
   "104 Not used",
   "105 Not used",
   "106 Not used",
   "107 Not used",
   "108 Not used",
   "109 Not used",
   "110 Not used",
   "111 Not used",
   "112 Not used",
   "113 Not used",
   "114 Not used",
   "115 Not used",
   "116 Not used",
   "117 Not used",
   "118 Not used",
   "119 Not used",
   "120 Not used",
   "121 Not used",
   "122 Not used",
   "123 Not used",
   "124 Out of memory: measurement routine",
   "125 Out of memory: reference target routine",
   "126 Out of memory: reference target angle table",
   "127 Out of memory: reference target value telegram preparation",
   0};
