/** \file
*   \brief Include file for Pip library
* 
* Revision History
- Pip 1.0   21 Jun 96 Particle Image processing Library
- Pip 1.1    8 Aug 96 Include Histogram Smoothing
- Pip 1.2   30 Sep 96 Seperate Source Directories 
- Pip 1.3   11 Oct 96 Include 24bit display
- Pip 1.4   29 Nov 96 Return Stat and Moment Structures
- Pip 1.5    7 Feb 97 Include Histogram Data in Image
- Pip 1.6    4 Mar 97 Prefix functions with PIP
- Pip 1.6    6 Mar 97 Improve Generalization 
- Pip 1.7   21 Mar 97 Introduce Histogram Structure (Phist)
- Pip 1.8    2 Jul 97 Add Name to Histogram Structure
- Pip 1.9   15 Jul 97 Add Chain Structure (Pchain)
- Pip 1.10  16 Jul 97 Add Shape Structure (Pshape)
- Pip 1.11   1 Oct 97 Binary Absorb and Sort / Change Pimage structure
- Pip 1.12   3 Oct 97 Add Spatial Structure
- Pip 2.00   7 Nov 97 Make stand-alone utilities
- Pip 2.01  13 Nov 97 Use Strings rather than Definitions
- Pip 2.02  20 Jun 98 Use Popen for other conversions
- Pip 2.03  26 Jan 99 Tidy up Names
- Pip 2.04  20 Jan 00 Improve Covar Support
- Pip 2.05  25 Oct 00 Change Cmap  - load in two passes.
- Pip 2.06  30 Jan 01 Merge Files and include Doxygen support
- Pip 3.00  31 Jan 01 --
- Pip 3.01  28 Feb 01 Include Timespec into Pimage Arrays 
\author       Elliot Duff esd@cat.csiro.au
*/

#ifndef _PIP_H_    /* prevent accidental re-inclusions */
#define _PIP_H_

#ifdef  __cplusplus
extern "C" {
#endif

typedef unsigned char pixel;	/*!< The base unit of PIP */

#define PIP_VERSION 		"3.00 Doxygen support"
#define PIPOK     0		/*!< Return state of most functions if OK */
#define PIPFAIL   -1		/*!< Return state of most functions if FAIL */
#define OFF       (pixel) 0	/*!< Pixel is defined to be OFF */
#define MARK      (pixel) 128	/*!< Pixel has been marked for processing */
#define ON        (pixel) 255	/*!< Pixel is defined to be ON */

/**
* Image Data structure 
*/

typedef struct { 
	char 	*name; 		/*!< Pointer to Name of Image */
	pixel 	*data, 		/*!< Data Pointer - the actual image */
	  	*mask;		/*!< Mask Pointer - mask of the image */
	int     sec, nsec;  
	double 	scale;		/*!< To scale image up to double */
	int 	nc,		/*!< Number of Columns (Width of Image) */
		nr,		/*!< Number of Rows (Height of Image) */
		np,		/*!< Number of pixels in image */
		ng,		/*!< Number of Grey Levels (Depth of Image) */
		c,		/*!< Current column  */
		c1, 		/*!< ROI column lowerbound */
		c2, 		/*!< ROI column upperbound */
		r,		/*!< Current row  */
		r1, 		/*!< ROI row lowerbound */
		r2; 		/*!< ROI row lowerbound */
	} Pimage;

/**
* Histogram Data Structure 
*/

typedef struct { 
	char 	name[80];	/*!< Name of Histogram */
	int 	data[256],  	/*!< The actual histogram data */
	    	mid, 
	    	min, 
	    	max, 
	    	pop; 
        int pos1, max1, var1, pop1;
        int pos2, max2, var2, pop2; } Phist;

/**
* Chain Data Structure
*/

typedef struct { 
	int 	n,		/*!< Number of links in chain */
		*x,		/*!< X position of link */
		*y; 		/*!< Y position of link */
		} Pchain;

/**
* Label data structure - used by PIP_centroid
*/

typedef struct { 
	int 	n; 		/*! Number of labelled blobs */
	double  m[256],		/*! Number of pixels (mass) in blob */
		x[256],		/*! X position of blob centroid */
		y[256]; 	/*! Y position of blob centroid */
	int     w[256],		/*! Width of blob in pixels */
		h[256],		/*! Width of blob in pixels */
		c1[256],	/*! Min Column in pixels */
		c2[256],	/*! Max Column in pixels */
		r1[256],	/*! Min Row in pixels */
		r2[256];	/*! Max Row in pixels */
		} Plabel;

/**
* Moment Data Structure 
*/

typedef struct { 
  double area,		/*!< Area of Blob */
	 xm,		/*!< X Position of Blob Centroid */
	 ym,		/*!< Y Position of Blob Centroid */
	 sx,		/*!< X Sigma */
	 sy,		/*!< X Sigma */
	 major,		/*!< Length of Major Axis  */
	 minor,		/*!< Length of Minor Axis */
	 theta,		/*!< Angle of Major Axis */
	 eccen; 	/*!< Eccentricity (major/minor) */
	 } Pmoment;

typedef struct { double min,max,avg,sum,perm,area; int n,*x,*y; } Pferet;
typedef struct { int num; double *x,*y,a,b,dev,perm,area; } Pfractal;
typedef struct { int num; double *data,min,max,avg,var,std,skw,kur; } Psignal;
typedef struct { double width,length,scale; } Pfibre;
typedef struct { int nf; double *f; char **s; } Ptexture;
typedef struct { double num,avg,dev,var,std,skw,kur; } Pstat; 

#define PIP_EDGE_NONE	0x0	/*!< No Blob has touches the edge */
#define PIP_EDGE_NORTH	0x1	/*!< A Blob has touched the North edge */
#define PIP_EDGE_SOUTH	0x2	/*!< A Blob has touched the South edge */
#define PIP_EDGE_EAST	0x4	/*!< A Blob has touched the East edge */
#define PIP_EDGE_WEST	0x8	/*!< A Blob has touched the WEST edge */
#define PIP_EDGE_ALL	0xf	/*!< A Blob has touched all edges */

typedef enum _Pedge {  
	PIP_EDGE_DETECT, 	/*!< Only detect edges */
	PIP_EDGE_REMOVE, 	/*!< Remove Edges (set to OFF) */
	PIP_EDGE_EXTEND 	/*!< Extend Edges (copy from neighbour) */
	} Pedge;

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#ifndef _PIP_EXTERN		/* Define in local.h */
#define _PIP_EXTERN extern
#endif
#endif

_PIP_EXTERN int pip_debug;	/*!< Flag to control writting debug files */
_PIP_EXTERN int pip_verbose;	/*!< Flag to control verbosity of functions */
_PIP_EXTERN int pip_silent;	/*!< Flag to control PIPcomment() */
_PIP_EXTERN int pip_clean;	/*!< Flag to control PIP_save() */

/*
** File & Misc Utilities
** Some Macros for shortening names
*/

#ifndef DOXYGEN_SHOULD_SKIP_THIS
void PIP_version();
int  PIP_load(Pimage *, char *);
int  PIP_save(Pimage *, char *);
int  PIP_loadRGB(Pimage *rgb[], char *);
int  PIP_saveRGB(Pimage *rgb[], char *);

Pimage *PIP_new();
int  PIP_roi(Pimage *, int, int, int, int);
int  PIP_roi_edge(Pimage *, int);
int  PIP_roi_mask(Pimage *);
void PIP_free(Pimage *);

int  PIP_init(Pimage *, pixel *, int, int, char *);
int  PIP_make(Pimage *, int, int, int, char *);
int  PIP_mask(Pimage *);
int  PIP_copy(Pimage *, Pimage *);
int  PIP_swapmask(Pimage *);
int  PIP_copymask(Pimage *, Pimage *);
int  PIP_info(Pimage *);

int  PIP_cat(Pimage *, Pimage *, char);
int  PIP_pad(Pimage *, char, int, pixel);
int  PIP_crop(Pimage *);
int  PIP_zoom(Pimage *, int);
int  PIP_stretch(Pimage *, Pimage *);
int  PIP_shrink(Pimage *, Pimage *);
int  PIP_double(Pimage *, Pimage *);
int  PIP_halve(Pimage *);
int  PIP_flip(Pimage *, char);
int  PIP_spin(Pimage *, char);
int  PIP_split(Pimage *, int);
int  PIP_frame(Pimage *, pixel);
int  PIP_field(Pimage *, int, Pimage *);
int  PIP_untear(Pimage *, int, int);
int  PIP_deinterlace(Pimage *);
int  PIP_deinterlace2(Pimage *, Pimage *, Pimage *);

int  PIP_RGBtoRGBI(Pimage *rgb[]);
/*int  PIP_RGBtoHSI(Pimage *rgb[]);*/
int  PIP_RGBtoHSV(Pimage *rgb[]);
int  PIP_HSVtoRGB(Pimage *rgb[]);
int  PIP_RGBtoHLS(Pimage *rgb[]);
int  PIP_RGBtoYUV(Pimage *rgb[]);
int  PIP_RGBtoRGY( Pimage *rgb[]);
int  PIP_YUVtoRGB(Pimage *rgb[]);
int  PIP_RGBtoYIQ(Pimage *rgb[]);
int  PIP_YIQtoRGB(Pimage *rgb[]);
int  PIP_RGBtoYCrCb_original(Pimage *rgb[]);
int  PIP_YCrCbtoRGB_original(Pimage *rgb[]);
int  PIP_RGBtoYCrCb(Pimage *rgb[]);
int  PIP_YCrCbtoRGB(Pimage *rgb[]);

int  PIP_loadcmap(char *cmap);
int  PIP_colourize(Pimage *rgb[], Pimage *);
int  PIP_overlay(Pimage *rgb[], Pimage *);
int  PIP_overcmap(Pimage *rgb[], Pimage *, Pimage *);

int  PIP_histogram(Pimage *, Pimage *);
int  **PIP_covar(Pimage *, int);
int  **PIP_matrix(int);
int  PIP_free_matrix(int **, int);
int  PIP_covar2(Pimage *, int **);
int  PIP_bivariant(Pimage *, Pimage *, Pimage *);

int  PIP_covar_to_image(int **, int, Pimage *);
int  PIP_unary(Pimage *, char, int);
int  PIP_stats(Pimage *, char);
int  PIP_maths(Pimage *, char, Pimage *);
int  PIP_linear(Pimage *, float, float, float, int, int, int);
int  PIP_remove(Pimage *, int *);
int  PIP_cumsum(Pimage *);
int  PIP_lookup(Pimage *, Pimage *, Pimage *, Pimage *);
int  PIP_thresh2(Pimage *, Pimage *, Pimage *, int, int, int, int);

/*   These are no longer need and should not be used */
int  PIP_save_image_cmap(char *, Pimage *, char *);
int  PIP_save_image_hist(char *, Phist *);

/* 
** Grey Image Filters & Utilities
*/

int PIP_mean(Pimage *, int, int);
int PIP_rank(Pimage *, char *, int, int);
int PIP_rankk(Pimage *, char *, Pimage *);
int PIP_rankm(Pimage *, Pimage *, int, int, int);
int PIP_convolve(Pimage *, char *);
int PIP_gradient(Pimage *, char *, char *);
int PIP_equalize(Pimage *, int, int);
int PIP_enhance(Pimage *, int, int);
int PIP_expand(Pimage *, int, int);
int PIP_smooth(Pimage *, char *, int, int);
int PIP_backlight(Pimage *, char *);
int PIP_gwatershed(Pimage *, int, int);

/* 
** Image Segmentation
*/

int PIP_segment_darea(Pimage *, Phist *);
int PIP_segment_kmeans(Pimage *, Phist *);
int PIP_segment_minima(Pimage *, Phist *, int);
int PIP_segment_simple(Pimage *, Phist *);

/* 
** Binary Image Utilities 
*/


int PIP_fill(Pimage *, int);
int PIP_seed(Pimage *, int, int);
int PIP_edge(Pimage *, Pedge);
int PIP_erode(Pimage *, int);
int PIP_dilate(Pimage *, int);
int PIP_dilate2(Pimage *);
int PIP_dilate3(Pimage *);
int PIP_final(Pimage *);

int PIP_label(Pimage *, int, int, int);
int PIP_label_inline(Pimage *, int, int, int);
int PIP_labelr(Pimage *, int);
int PIP_labelf(Pimage *);
int PIP_ilabel(Pimage *);
int PIP_absorb(Pimage *, int, int);
int PIP_holes(Pimage *, Pimage *, int, int);
int PIP_sort(Pimage *);

int PIP_chain(Pimage *, Pchain *);
int PIP_chain8(Pimage *, Pchain *);
double PIP_chain_perimeter(Pchain *);
int PIP_outline(Pimage *, Pimage *);
int PIP_border(Pimage *, Pimage *);
int PIP_unborder(Pimage *);
int PIP_thinning(Pimage *, Pimage *);	
int PIP_thin(Pimage *, Pimage *);	

int PIP_distance(Pimage *, Pimage *);	
int PIP_distance2(Pimage *, Pimage *);	
int PIP_mediaxis(Pimage *, Pimage *); 
int PIP_skeleton(Pimage *, Pimage *);
int PIP_watershed(Pimage *, Pimage *, float);
int PIP_watershed_merge(Pimage *, Pimage *, int);
int PIP_watershed_split(Pimage *, Pimage *, int, int);
int PIP_backbone(Pimage *, Pimage *);

/*
* Shape Analysis Utilities 
*/

int	  PIP_centroid(Pimage *, Plabel *);
Pmoment  *PIP_shape_moment(Pimage *, int);
Psignal  *PIP_shape_curves(Pchain *, int);
Psignal  *PIP_shape_radius(Pchain *, int, int);
Pferet   *PIP_shape_feret(Pimage *, int);
Pfibre   *PIP_shape_fibre(Pimage *, Pimage *);
Pfractal *PIP_shape_fractal(Pchain *);

/* 
** Texture Utilities 
*/

Ptexture  *PIP_texture_spatial(Pimage *, int, int); 
Ptexture  *PIP_texture_spectra(Pimage *);
Ptexture  *PIP_texture_statistics(Pimage *, Pimage *, int);
Pstat     *PIP_texture_stat(Pimage *, Pimage *, int);

int PIP_texture_fractal(Pimage *); 
int PIP_texture_law();

/* 
* Drawing Utilities 
*/

int PIP_DrawPixel(Pimage *, int, int, int);
int PIP_DrawLine(Pimage *, int, int, int, int, int);
int PIP_DrawBox(Pimage *, int, int, int, int, int);
int PIP_FillBox(Pimage *, int, int, int, int, int);
int PIP_DrawEllipse(Pimage *, int, int, int, int, int);

int PIP_Ellipse(int, int, int, int *);
int PIP_Line(int, int, int, int *);

int    PIP_listset(Pimage *, int, int, int *, int, int);
int   *PIP_listget(Pimage *, int, int, int *, int);
double PIP_listsum(Pimage *, int, int, int *, int);
Pimage *PIP_list2image(int *, int, int);
#endif

#ifdef  __cplusplus
}
#endif

#endif /* _PIP_H_ */
