/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/

/** \file
\brief Extract textural information from image
*/

#include "local.h"

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#define RADIX 	   2.0
#define EPSI 	   0.000000001
#define SIGN(x,y)  ((y)<0?-fabs(x):fabs(x))
#define SWAP(a,b)  {y=(a);(a)=(b);(b)=y;}
#define NF	   12 	/* 12 textures */
#endif

/**
Calculate Haralick Textural Features
*/

Ptexture *
PIP_texture_spatial(Pimage *image, int angle, int d)
    {
    int i, j, n;
    int NC, ng, np;
    pixel *ip; 
    int col, c, c1, c2, nc;
    int row, r, r1, r2, nr;

    Ptexture *S;

    double max, mean, sum, sqr, tmp, val;
    double mean_px, sdev_px, sums_px; 
    double f0, f1, f2, f3, f4, f5, f6, f7, f8, f9, f10, f11;
    double *P, *Px, *Py, *Pxpy, *Pxmy;

    if((ip = image->data) == NULL) NULern;

    ng = image->ng+1; 
    NC = image->nc;
    c1 = image->c1; c2 = image->c2; nc = c2 - c1; 
    r1 = image->r1; r2 = image->r2; nr = r2 - r1; 

    printf("\tTexture Spatial (angle:%d distance:%d) : ", angle, d);
    printf("Window (%d,%d) Depth (%d)\n", nr,nc,ng);

    if((P = calloc(ng*ng, sizeof(double))) == NULL) MEMern;

    switch( angle )
      {
      case   0: r = 0; c = d; c2 -= d; np = 2*(nr)*(nc-1); break;
      case  45: r = d; c =-d; r2 -= d; np = 2*(nr-1)*(nc-1); break;
      case  90: r = d; c = 0; r2 -= d; np = 2*(nr-1)*(nc); break;
      case 135: r = d; c = d; r2 -= d; c2 -= d; np = 2*(nr-1)*(nc-1); break;
      default: PIPern("Unknown Angle");
      }

    for(row=r1; row<r2-r; row++) 
    for(col=c1; col<c2-c; col++) { 
      i = ip[row*NC+col]; 
      j = ip[(row+r)*NC+(col+d)]; 
      if( i && j ) { P[i*ng+j]++; P[j*ng+i]++; }
      }

    /* 
    ** Normalize Matrix and find max
    */

    max = 0.0;
    for(i=0; i<ng*ng; i++) P[i] /= (double) np;
    for(i=0; i<ng*ng; i++) if( P[i] > max ) max = P[i];

    /* 
    ** Spatial Statistics
    */

    mean = 0;
    mean_px = 0;
    sums_px = 0;

    if((Px = calloc(ng, sizeof(double))) == NULL) MEMern;
    if((Py = calloc(ng, sizeof(double))) == NULL) MEMern;
    if((Pxpy = calloc(ng*2+1, sizeof(double))) == NULL) MEMern;
    if((Pxmy = calloc(ng*2+1, sizeof(double))) == NULL) MEMern;

    for(i=0; i<ng; ++i) 
    for(j=0; j<ng; ++j) 
        {
	val = P[i*ng+j];
        mean += i*val;
        Pxpy[i+j+2] += val;
        Pxmy[abs(i-j)] += val;
        Px[i] += val;
        Py[j] += val;
        }

    for(i = 0; i < ng; ++i) { mean_px += Px[i]*i; sums_px += Px[i]*i*i; }
    sdev_px = sqrt (sums_px - (mean_px * mean_px));

    /* comment("Mean %f mean X %f sdev X %f\n", mean, mean_px, sdev_px); */

    f0  = angle;
    f1  = 0; f2  = 0; f3  = 0; f4  = 0; f5  = 0; f6  = 0; 
    f7  = 0; f8  = 0; f9  = 0; f10 = 0; f11 = 0; 

    for(i=0; i<ng; ++i) 
    for(j=0; j<ng; ++j) 
        {
	val = P[i*ng+j];
        f1 += val * val;
        f3 += i*j*val;
        f4 += (i+1-mean) * (i+1-mean) * val;
        f5 += val / (1+(i-j)*(i-j));
        f9 -= val * log10 (val + EPSI);
        }

    /* F2 Contrast ***/

    for(n=0; n<ng; ++n) { sum = 0;
        for(i=0; i<ng; ++i) for(j=0; j<ng; ++j)
	  if( (i-j) == n || (j-i) == n) sum += P[i*ng+j];
        f2 += n * n * sum; }

    for(i=2; i<=2*ng; ++i) f6 += i * Pxpy[i];
    for(i=2; i<=2*ng; ++i) f8 -= Pxpy[i] * log10 (Pxpy[i] + EPSI);
    for(i=2; i<=2*ng; ++i) f7 += (i - f8) * (i - f8) * Pxpy[i];

    /* F10 Difference Variance ***/

    tmp = ng * ng; 
    sum = 0; for(i=0; i<ng; ++i) sum += Pxmy[i]; 
    sqr = 0; for(i=0; i<ng; ++i) sqr += Pxmy[i] * Pxmy[i]; 
    f10 = ((tmp * sqr) - (sum * sum)) / (tmp * tmp);
    for(i=0; i<ng; ++i) f11 -= Pxmy[i] * log10 (Pxmy[i] + EPSI);
    f3 = (f3 - mean_px * mean_px) / (sdev_px * sdev_px);

    /* Load Output Structure */

    if((S = malloc(sizeof(Ptexture))) == NULL) MEMern;
    S->nf = NF;
    if((S->f = (double *) malloc(NF*sizeof(double))) == NULL) MEMern;
    if((S->s = (char **) malloc(NF*sizeof(char *))) == NULL) MEMern;

    S->f[0]  = f0;  S->s[0]  = "Angle              ";
    S->f[1]  = f1;  S->s[1]  = "Angular 2nd Mom    "; 
    S->f[2]  = f2;  S->s[2]  = "Contrast           "; 
    S->f[3]  = f3;  S->s[3]  = "Correlation        "; 
    S->f[4]  = f4;  S->s[4]  = "Variance           "; 
    S->f[5]  = f5;  S->s[5]  = "Inverse Diff Mom   "; 
    S->f[6]  = f6;  S->s[6]  = "Sum Average        "; 
    S->f[7]  = f7;  S->s[7]  = "Sum Variance       "; 
    S->f[8]  = f8;  S->s[8]  = "Sum Entropy        "; 
    S->f[9]  = f9;  S->s[9]  = "Entropy            "; 
    S->f[10] = f10; S->s[10] = "Difference Variance"; 
    S->f[11] = f11; S->s[11] = "Difference Entropy "; 

    /*** Meas of Correlation-1 ***/
    /*** Meas of Correlation-2 ***/
    /*** Max Correlation Coeff ***/

    if( pip_verbose ) 
      for(j=0; j<NF; j++) 
	PIPcomment("%s %.3g\n", S->s[j],S->f[j]); 

    free(Pxpy); free(Pxmy);
    free(P); free(Px); free(Py);
    return(S);
    }

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#define NS 6560
#undef NF
#define NF 7
#define DEBUG	"spectra.dat"
#endif

/**
* Generate Texture energy spectra
*
* Texture features based on texture spectrum by Dong-Chen He and Li Wang [He91]
* The basic concept is that a texure image can be considered as a set of 
* essentiall small units termed texture units, which characterize the local 
* texture units.
*
* Texture Feaures
*
* - Black and white symmetry : 
*
* BWS values are normalized form 0 to 100 and measure the symmetry between the
* left and right part in the texture spectrum with the centre of axis of 
* symmetry at position at 3280. A high BWS value reveals the phenomenon that if
* we invert the intensity values 
*/

Ptexture *PIP_texture_spectra(Pimage *image)
    {
    Ptexture *TE;	/* Return Textural Energies */

    /* int a, b, c, d, e, f, g, h; */
    int i, j, m, n, k, row, col, val;

    int *N[8];		/* Texture Unit Number */
    int *S[8];		/* Texture spectra */
    int *HM, *VM;	/* Horizontal & Vertical Measure */
    int *D1, *D2;	/* Diagonal Measure (DM1 & DM2) */

    int E[10], V[10];
    int W[20] = { 0, 1, 3, 9, 27, 81, 243, 729, 2187, 
		     1, 3, 9, 27, 81, 243, 729, 2187 };
    /* int P[3][3][3] = { { {2,1,1}, {1,1,0}, {1,0,1} },
		       { {1,1,0}, {1,2,1}, {0,1,1} },
		       { {1,0,1}, {0,1,1}, {1,1,2} } }; */

    double tmp, sum, dif, bws, gs, dd;
    double mvs, mhs, md1, md2;

    pixel *ip;
    int nc = image->nc;
    int c1 = image->c1; int c2 = image->c2;
    int r1 = image->r1; int r2 = image->r2;
    int NP = (r2-r1) * (c2-c1);

    PIPcomment("\tSurface Texture Energies : ");
    if((ip = image->data) == NULL) NULern;
    if( NP == 0 ) PIPern("Zero Window");

    /* Allocate Memory for the Spectra and 8 Channels */

    if((HM = (int *) calloc(NS, sizeof(int) )) == NULL) MEMern;
    if((VM = (int *) calloc(NS, sizeof(int) )) == NULL) MEMern;
    if((D1 = (int *) calloc(NS, sizeof(int) )) == NULL) MEMern;
    if((D2 = (int *) calloc(NS, sizeof(int) )) == NULL) MEMern;

    for(i=0; i<8; i++) {
      if((N[i] = (int *) calloc(NP, sizeof(int))) == NULL) MEMern;
      if((S[i] = (int *) calloc(NS+1, sizeof(int))) == NULL) MEMern;
      }

    if( pip_verbose ) printf("\n\t\tGenerate Texture Units\n");

    k = 0;
    for(row = r1; row<(r2-1); row++)
    for(col = c1; col<(c2-1); col++)
      {
      i = (row-1)*nc + col; V[1] = ip[i-1]; V[2] = ip[i]; V[3] = ip[i+1];
      i = (row)*nc   + col; V[8] = ip[i-1]; V[0] = ip[i]; V[4] = ip[i+1];
      i = (row+1)*nc + col; V[7] = ip[i-1]; V[6] = ip[i]; V[5] = ip[i+1];

      for(i=1; i<=8; i++) 
	{
	if( V[i] < V[0] ) E[i] = 0; 
	else { if( V[i] > V[0] ) E[i] = 2; else E[i] = 1; }
	}

      for(i=0; i<8; i++)
	{
        val = 0;
        for(j=1; j<=8; j++) val += E[j]*W[j+i];
        N[i][k] = val;
	}

      k++;

      /*
      ** To be fixed 
      a = E[1]; b = E[2]; c = E[3]; d = E[4]; 
      e = E[5]; f = E[6]; g = E[7]; h = E[8];

      HM[i] = P[a][b][c] * P[f][g][h];
      VM[i] = P[a][d][f] * P[c][e][h];
      D1[i] = P[d][a][b] * P[g][h][e];
      D2[i] = P[b][c][e] * P[d][f][g];
      */
      }

    if( pip_verbose ) PIPcomment("\t\tGenerate Spectra\n");

    for(i=0; i<8; i++)
      for(j=0; j<NP; j++) 
	S[i][N[i][j]]++;

    if( pip_debug )
      {
      FILE *fp;
      fp = fopen(DEBUG,"w");
      for(j=0; j<NS; j++) 
	{
	fprintf(fp, "%d ",j);
	for(i=0; i<8; i++) fprintf(fp, "%d ", S[i][j]);
	fprintf(fp, "\n");
	}
      fclose(fp);
      }

    /* 
    ** Calculate the sum of each spectra
    ** They should all be the same 
    */

    for(i=0; i<8; i++)
      {
      sum = 0;
      for(j=0; j<NS; j++) sum += S[i][j];
      if( pip_verbose ) PIPcomment("\t\tSpectra %d Sum %f\n", i, sum);
      }

    /* 
    ** Black-White Symmetry 
    */

    bws = 0.0;
    for(j=0; j<3279; j++) bws += (double) (S[0][j] - S[0][3281+j]);
    bws = (1.0 - bws/sum)*100; 

    /* 
    ** Geometric Symmetry 
    ** (0->3) same as (1->4) 
    */

    gs = 0.0;
    for(j=0; j<4; j++) 
      {		
      tmp = 0;
      for(i=0; i<6560; i++) { dif = ABS(S[j][i]-S[j+4][i]); tmp += ABS(dif); }
      gs += tmp / (2.0 * sum);
      }
    gs = (1.0 - gs/4.0) * 100;

    /* 
    ** Degree of Direction 
    */

    dd = 0.0;
    for(m=0; m<3; m++)
    for(n=m+1; n<4; n++) 
      { 
      tmp = 0;
      for(i=0; i<6560; i++) { dif = (S[m][i] - S[n][i]); tmp += ABS(dif); }
      dd += tmp / (2.0 * sum);
      }
    dd = (1.0 - dd/6.0) * 100;

    /* 
    ** Orientational features 
    */

    mhs = 0.0;
    mvs = 0.0;
    md1 = 0.0;		/* First Diagonal */
    md2 = 0.0;		/* Second Diagonal */

    /*
    ** To be fixed 

    for(i=0; i<6560; i++) 
      {
      PIPcomment("Got here");
      mhs += (S[0][i] * HM[i]);
      mvs += (S[0][i] * VM[i]);
      md1 += (S[0][i] * D1[i]);
      md2 += (S[0][i] * D2[i]);
      }
    */

    /* 
    ** Store Data in Global vector 
    */

    if((TE = (Ptexture *) malloc( sizeof(Ptexture) )) == NULL) MEMern;
    TE->nf = NF;
    if((TE->f = (double *) malloc(NF*sizeof(double))) == NULL) MEMern;
    if((TE->s = (char **) malloc(NF*sizeof(char *))) == NULL) MEMern;

    TE->f[0] = bws; TE->s[0] = "Black and White Symmetry";
    TE->f[1] = gs;  TE->s[1] = "Geometric Symmetry      ";
    TE->f[2] = dd;  TE->s[2] = "Degree of Direction     ";
    TE->f[3] = mvs; TE->s[3] = "Verical Orientation     ";
    TE->f[4] = mhs; TE->s[4] = "Horizontal Orientation  ";
    TE->f[5] = md1; TE->s[5] = "First Diagonal          ";
    TE->f[6] = md2; TE->s[6] = "Second Diagonal         ";

    if( pip_verbose )
       for(i=0; i<NF; i++) 
	 PIPcomment("%d %s %f\n", i, TE->s[i], TE->f[i]);

    /*
    ** Free Memory
    */

    for(i=0; i<8; i++) { free(S[i]); free(N[i]); }
    free(HM); free(VM); free(D1); free(D2);

    PIPcomment("\n");
    return(TE);
    }

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#define NF 7
#endif

/**
* Description	Normal texture statistics
* Using Region of Interest and Mask.
*/

Ptexture *PIP_texture_statistics(Pimage *image, Pimage *mask, int label)
    { 
    int i;
    double s, p;
    double num, avg, dev;
    double var, std, skw, kur;

    Ptexture *S;

    int col, c1, c2, nc;
    int row, r1, r2, nr;

    int NC = image->nc;
    c1 = image->c1; c2 = image->c2; nc = c2 - c1;
    r1 = image->r1; r2 = image->r2; nr = r2 - r1;

    if( pip_verbose ) PIPcomment("\nSurface Statistics (Mask:%d) ",label);
    
    avg = 0.0; dev = 0.0; num = 0.0; 
    var = 0.0; skw = 0.0; kur = 0.0;

    for(row=r1; row<r2; row++)
    for(col=c1; col<c2; col++) 
      {
      i = row*NC+col;
      if( mask->data[i] == label ) 
        {
        avg += (double) image->data[i];
        num += 1.0; 
        }
      }

    avg /= num;

    for(row=r1; row<r2; row++)
    for(col=c1; col<c2; col++) 
      {
      i = row*NC+col;
      if( mask->data[i] == label ) 
        {
        s = (double) image->data[i] - avg ;
        dev += fabs(s);
        var += (p = s*s);
        skw += (p *= s);
        kur += (p *= s);
        }
      }
    
    dev /= num;		/* Average Deviation */
    var /= (num-1);	/* Variance */
    std = sqrt(var);	/* Standard Deviation */

    if( var ) skw /= (num*var*std);
    if( var ) kur = kur/(num*var*var) - 3.0;

    /*
    ** Create Output Structure
    */

    if( (S = (Ptexture *) malloc( sizeof(Ptexture) )) == NULL ) MEMern;
    if((S->f = (double *) malloc(NF*sizeof(double))) == NULL) MEMern;
    if((S->s = (char **) malloc(NF*sizeof(char *))) == NULL) MEMern;
    S->nf = NF;
    
    S->s[0] = "Number     "; S->f[0] = num;  
    S->s[1] = "Average    "; S->f[1] = avg; 
    S->s[2] = "Avg Dev    "; S->f[2] = dev; 
    S->s[3] = "Variance   "; S->f[3] = var; 
    S->s[4] = "Deviation  "; S->f[4] = std; 
    S->s[5] = "Skewness   "; S->f[5] = skw; 
    S->s[6] = "Kurtosis   "; S->f[6] = kur; 

    if( pip_verbose ) 
      for(i=0; i<NF; i++) 
       PIPcomment("%s %f\n", S->s[i], S->f[i]); 

    return(S);
    }

/**
* Generate texture statistics
*/

Pstat *PIP_texture_stat(Pimage *image, Pimage *mask, int label)
    { 
    int i;
    double s, p;
    double num, avg, dev;
    double var, std, skw, kur;
    Pstat *stat;
    int	np = image->np;

    if( pip_verbose ) PIPcomment("\nSurface Statistics (Mask:%d) ",label);
    
    avg = 0.0; dev = 0.0; num = 0.0; 
    var = 0.0; skw = 0.0; kur = 0.0;

    for(i=0; i<np; i++)
     if( mask->data[i] == label ) 
      {
      avg += (double) image->data[i];
      num += 1.0; 
      }

    avg /= num;

    for(i=0; i<np; i++)
     if( mask->data[i] == label ) 
      {
      s = (double) image->data[i] - avg ;
      dev += fabs(s);
      var += (p = s*s);
      skw += (p *= s);
      kur += (p *= s);
      }
    
    dev /= num;		/* Average Deviation */
    var /= (num-1);	/* Variance */
    std = sqrt(var);	/* Standard Deviation */

    if( var ) skw /= (num*var*std);
    if( var ) kur = kur/(num*var*var) - 3.0;

    if( pip_verbose ) {
      PIPcomment("Number     %f\n", num); 
      PIPcomment("Average    %f\n", avg); 
      PIPcomment("Avg Dev    %f\n", dev); 
      PIPcomment("Variance   %f\n", var); 
      PIPcomment("Deviation  %f\n", std); 
      PIPcomment("Skewness   %f\n", skw); 
      PIPcomment("Kurtosis   %f\n", kur); 
      }

    if( (stat = (Pstat *) malloc( sizeof(Pstat) )) == NULL ) MEMern;
    stat->num = num;
    stat->avg = avg;
    stat->dev = dev;
    stat->var = var;
    stat->std = std;
    stat->skw = skw;
    stat->kur = kur;
    return(stat);
    }
