/*********************************************************************
*
* CSIRO Automation
* Queensland Centre for Advanced Technologies
* PO Box 883, Kenmore, QLD 4069, Australia
* www.cat.csiro.au/cmst
*
* Copyright (c) CSIRO Manufacturing Science & Technology
*
*********************************************************************/

/** \file 
\brief Shape Analysis
*/

#include "local.h"

/** 
Calculate centroids of labelled image
@param image pointer to binary image
@param label pointer to label data structure
*/

int PIP_centroid(Pimage *image, Plabel *label)
    {
    int row, col, i;
    int nc = image->nc;
    int	nr = image->nr;
    pixel *ip;
    
    PIPcomment("Centroid\n");
    if((ip = image->data) == NULL ) NULerr;

    for(i=0; i<256; i++) { 
      label->m[i] = 0.0; 
      label->x[i] = 0.0; label->y[i] = 0.0; 
      label->c1[i] = nc; label->c2[i] = 0;
      label->r1[i] = nr; label->r2[i] = 0;
      label->w[i] = 0;   label->h[i] = 0;
      }

    for(row = 0; row < nr; row++)
    for(col = 0; col < nc; col++)
      {
      i = ip[row*nc + col];
      label->m[i] += 1.0; 
      label->x[i] += (double)col; 
      label->y[i] += (double)row;
      if( col < label->c1[i] ) label->c1[i] = col;
      if( col > label->c2[i] ) label->c2[i] = col;
      if( row < label->r1[i] ) label->r1[i] = row;
      if( row > label->r2[i] ) label->r2[i] = row;
      }

    for(i=0; i<256; i++) {
      if( label->m[i] == 0 ) break;
      label->x[i] = label->x[i] / label->m[i]; 
      label->y[i] = label->y[i] / label->m[i]; 
      label->w[i] = label->c2[i] - label->c1[i];
      label->h[i] = label->r2[i] - label->r1[i];
      }

    return PIPOK;
    }

int signal_stat(Psignal *signal);

/** 
Calculate angle of curvature around outline
This functions calculates curvature statistics based upon \a N pixel
look ahead. The shape of the particle is defined by the chain code (Pchain).
This routine generates the curvature for every point along the perimeter
of the particle, i.e. it is outputted as a Psignal. Statistical analysis
of a signal can be performed with \fn signal_stat(Psignal *) routine.

Curvature analysis is defined in terms of the angle changes that occur
along the outline profile of an image by moving three adjacent points located
on the border. Two lines are drawn through the points located at the border.
Two lines are drawn through the points and intercept one another to form
an angle. The angle is measured from line 1 to line 2 such that it is positive
in the clockwise direction &amp;mdash; which thus represents a convex shape.
By plotting the results of all such measurements around the image, a curvature
profile is generated. Because of the change in optical magnification it
is necessary to normalize the analysis. Trial and error experiments have
showed that a suitable mean change of angle is 10 degrees. The forward
and backward points are thus calculated from FB = perimeter points / 36.
Thus a convex shape is depicted when the distribution is above the horizontal
axis. Quantitative informations is derived from the statistical analysis
of the curvature. 
*/

Psignal *
PIP_shape_curves(Pchain *chain, int divisor)
    {
    int    i, j, k;
    int    n, *x, *y;
    double xd, yd, th, tb, tf;
    double *data;
    Psignal *curves;

    PIPcomment("\tShape Curves (%d)", divisor);

    n = chain->n;
    x = chain->x;
    y = chain->y;

    if( n == 0 ) PIPern("Null Chaincode");
    if( divisor == 0 ) PIPern("Null Divisor");
    if( (data = (double *) malloc( sizeof(double) * n )) == NULL ) MEMern;
    if( (curves = (Psignal *) malloc( sizeof(Psignal) )) == NULL ) MEMern;

    j = n/divisor;
    for(i=0; i<n; i++)
      {
      k = i - j; 		/* Backward Angle (tb) */
      if( k < 0 ) k += n;
      xd = (double)(x[i]-x[k]);
      yd = (double)(y[i]-y[k]);
      tb = atan2(yd,xd);
      if( tb < 0.0 ) tb += PI2; 

      k = i + j; 		/* Forward Angle (tf) */
      if( k >= n ) k -= n;
      xd = (double)(x[k]-x[i]);
      yd = (double)(y[k]-y[i]);
      tf = atan2(yd,xd);
      if( tf < 0.0 ) tf += PI2;
  
      k = i - j; 		/* Combined Angle  */
      if( k < 0 ) k += n;
      xd = (double)x[k];
      yd = (double)y[k];

      th = tf-tb;
      if( th > PI ) th -= PI2;
      if( th < -PI ) th += PI2; 
      data[i] = th;
      }

    /*
    ** Load Curve Signal and return
    */

    curves->num = n;
    curves->data = data;
    signal_stat(curves);
    return curves;
    }

int 
signal_stat(Psignal *signal)
    {
    int i; 
    double num, *data;

    double avg = 0.0;
    double std = 0.0;
    double max = 0.0;
    double min = 100000000.00;
    double var = 0.00;
    double skw = 0.0;
    double kur = 0.0;

    num = signal->num;
    data = signal->data;

    for(i=0; i<num; i++)
      {
      if( max < data[i] ) max = data[i];
      if( min > data[i] ) min = data[i];
      avg += data[i];
      }

    avg /= num;
    for(i=0; i<num; i++) var += POWER2(data[i] - avg);

    var /= num;
    std = sqrt(var);

    for(i=0; i<num; i++)
      {
      skw += pow(((data[i] - avg)/std), 3);
      kur += pow(((data[i] - avg)/std), 4);
      }

    skw /= num;
    kur /= num;

    if( pip_verbose ) {
      PIPcomment("\t\tMin	   \t%f\n", min);
      PIPcomment("\t\tMax	   \t%f\n", max);
      PIPcomment("\t\tAverage  \t%f\n", avg);
      PIPcomment("\t\tVariance \t%f\n", var);
      PIPcomment("\t\tDeviation\t%f\n", std);
      PIPcomment("\t\tSkewness \t%f\n", skw);
      PIPcomment("\t\tKurtosis \t%f\n", kur);
      }
    else  {
      PIPcomment(" Min %.2f", min);
      PIPcomment(" Max %.2f", max);
      }

    signal->min = min;
    signal->max = max;
    signal->avg = avg;
    signal->std = std;
    signal->var = var;
    signal->skw = skw;
    signal->kur = kur;
    return PIPOK;
    }

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#define NF		20	/* Maximum Ferrets */
#define DEBUG_FILE 	"feret.pip_debug"
#endif

static int ymin_col[NF], ymin_row[NF];
static int ymax_col[NF], ymax_row[NF];
static int xmin_col[NF], xmin_row[NF];
static int xmax_col[NF], xmax_row[NF];
 
/**
* Determine min anx max feret length
* 
* This function performs feret analysis of the blob, in <I>N</I> directions
* The length of a feature can be obtained using the rotation of coordinates.
* As the particle is rotated, each point on the perimeter of the particle
* is compared to a minium and maximum values for that particular angle, and
* the extreme vales saved. After the process is complete, the rotating angle
* whose minimum and maximum values give the greatest difference is found.
* The difference is taken to be the length of the particle and is called
* the <B>Feret's Length</B>. By analogy the minimum Feret's diameter is called
* the <B>Feret's Breath</B>. The Feret length corresponds the maximum extent
* of the particle, but if the particle is bent of concave, the breath is
* not a particular good measure of how wide the particle is. The Feret's
* breath is sometimes referred to as the letterbox width, because it represents
* the smallest keyhole that the particle could be pushed through. The length
* of feret at 90 Degrees to the maximum feret is called the <B>Orthogonal
* Feret. </B>This can be used to measure the extent of the particle. </P>
* 
* - The Convex perimeter  Pc = 2tan (Pi/2.nf ) SUM Feret Diameter
* - The Convex Area Ac = Pi/4 (Mean Feret Diameter) <SUP>2</SUP>
* - The Fullness Ratio Fr = Sqrt( A/Ap)
*/

Pferet *
PIP_shape_feret(Pimage *image, int nf)
    {
    int i, j, row, col;
    double x, xdif, xmin, xmax;
    double y, ydif, ymin, ymax;
    double theta, sinth, costh;
    pixel *ip;
    int *X, *Y;
    Pferet *feret;

    int nc = image->nc;
    int	nr = image->nr;
    int	np = image->np; 

    int	c1 = image->c1;
    int	c2 = image->c2;
    int	r1 = image->r1;
    int	r2 = image->r2;

    double max = 0.0;
    double min = 10000000;
    double avg = 0.0; /* Average of xdif and ydif */
    double sum = 0.0; /* Sum of max of xdif and ydif */

    ip = image->data;

    PIPcomment("\tShape Feret (%d) ", nf);
    if( nf >= NF ) PIPern("You have too many ferets"); 
    if( ip == NULL ) PIPern("Null Input Image"); 
    if((feret = ( Pferet * ) malloc ( sizeof(Pferet))) == NULL ) MEMern;
    if( c2 == 0 ) c2 = nc; if( r2 == 0 ) r2 = nr;
    if( pip_verbose ) PIPcomment("ROI %d %d %d %d\n", c1, c2, r1, r2);

    for(i=0; i<nf; i++)
      {
      ymin = xmin = np;
      ymax = xmax = -np;
      theta = PIO2 * (double ) i / (double)nf; 
      sinth = sin(theta);
      costh = cos(theta);

      for(row = r1; row < r2; row++)
      for(col = c1; col < c2; col++)
	if( ip[row * nc + col] ) 
          {
	  x = costh*col - sinth*row;
  	  y = sinth*col + costh*row;
          if(xmin > x) { xmin = x; xmin_col[i] = col; xmin_row[i] = row; }
          if(xmax < x) { xmax = x; xmax_col[i] = col; xmax_row[i] = row; }
          if(ymin > y) { ymin = y; ymin_col[i] = col; ymin_row[i] = row; }
          if(ymax < y) { ymax = y; ymax_col[i] = col; ymax_row[i] = row; }
	  }

      xdif = xmax-xmin;
      ydif = ymax-ymin;

      /*
      if( pip_verbose ) PIPcomment("\t\tAngle %.2f X %.2f Y %.2f\n",
	theta*RADIANS,xdif,ydif); 
	*/

      avg += xdif + ydif;
      if( xdif > ydif ) sum += xdif; else sum += ydif;
      if( max < xdif ) { max = xdif; }
      if( max < ydif ) { max = ydif; }
      if( min > xdif ) { min = xdif; }
      if( min > ydif ) { min = ydif; }
      }

    /* Generate Taut String */

    if((X = malloc(nf*4*sizeof(int))) == NULL) MEMern;
    if((Y = malloc(nf*4*sizeof(int))) == NULL) MEMern;

    j = 0;
    for(i=1; i<nf; i++,j++) { X[j] = ymax_col[i]; Y[j] = ymax_row[i]; }
    for(i=1; i<nf; i++,j++) { X[j] = xmax_col[i]; Y[j] = xmax_row[i]; }
    for(i=1; i<nf; i++,j++) { X[j] = ymin_col[i]; Y[j] = ymin_row[i]; }
    for(i=1; i<nf; i++,j++) { X[j] = xmin_col[i]; Y[j] = xmin_row[i]; }

    if( pip_debug )
      {
      FILE *fp; fp = fopen(DEBUG_FILE,"w");
      if( fp == NULL ) PIPern("Cannot Open pip_debug file");
      for(i=0; i<j; i++) fprintf(fp, "\t\tTstring %d (%d,%d)\n", i,X[i],Y[i]);
      fclose(fp);
      }

    feret->n = j;
    feret->x = X;
    feret->y = Y;
    feret->min = min;
    feret->max = max;
    feret->sum = sum;
    feret->avg = avg/(double)(2*nf);
    feret->perm = 2.0 * tan(PI/(double)(2*nf)) * feret->sum;
    feret->area = (PI/4.0) * POWER2(feret->avg);

    if( !pip_verbose ) 
      {
      PIPcomment("Min %.0f ", min);
      PIPcomment("Max %.0f\n", max);
      }
    else
      {
      PIPcomment("\t\tMaximum\t\t%f\n", feret->max);
      PIPcomment("\t\tMinimum\t\t%f\n", feret->min);
      PIPcomment("\t\tAverage\t\t%f\n", feret->avg);
      PIPcomment("\t\tSum    \t\t%f\n", feret->sum);
      PIPcomment("\t\tConvex Perm\t%f\n", feret->perm);
      PIPcomment("\t\tConvex Area\t%f\n", feret->area);
      }

    return feret;
    }


/**
* Calculate fibre length and width
* This function calculates the fibre ratio from the backbone and 
* the distance transform The fibre length can be estimated from 
*/

Pfibre *PIP_shape_fibre(Pimage *back, Pimage *dist)
    {
    int i, length, width;
    pixel *ip, *op;
    int	np = back->np;
    Pfibre *fibre;

    PIPcomment("\tShape Fibre : ");
    if( (ip = back->data) == NULL ) PIPern("Null Backbone Image");
    if( (op = dist->data) == NULL ) PIPern("Null Distance Image");
    if( dist->scale == 0.0 ) PIPern("Null Scale of Distance Image");

    width = 0;
    length = 0; 
    for(i=0; i<np; i++)
      if( ip[i] > 0 ) { length++; width += op[i]; }

    fibre = (Pfibre *) malloc ( sizeof( Pfibre ));
    if( fibre == NULL ) MEMern;

    fibre->scale  = (double) dist->scale;
    fibre->length = (double ) length;
    fibre->width  = fibre->scale * (double) width / (double) length;

    PIPcomment("Length (%.0f) ", fibre->length);
    PIPcomment("Width (%.0f)  ", fibre->width);
    PIPcomment("Scale (%.2f)\n", fibre->scale);
    return fibre;
    }

static void 
medfit(double *x,double *y, int ndata,double *a,double *b, double *abdev);

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#define MAXLENGTH	200
#define CHAINDEV	20
#endif

/**
* Fractal Analysis of Chain
* In this routine the fractal dimension of the perimeter is based upon
* the boundary walk method. The outside perimeter is defined by the chain
* code (Pimage). For a given step size, we measure the number of steps to
* walk the perimeter (ie the distance). We increase the step size, and repeat
* the walk around the perimeter. If the particle has a fractal dimesion greater
* than 1 then the perimeter will decrease with an increase in step size.
* If we plot perimeter against step size, then the slope of the line is related
* to the fractal dimension of the perimeter. In this case, a robust linear
* fit is performed to fit the data to a straight line. This routine returns
* the slope of the line and the y intercept. 
*/

Pfractal *
PIP_shape_fractal(Pchain *chain)
    {
    long   i, j, k, l;
    int    n, *x, *y, ns;
    double xd, yd, rd, sd;
    double hypot(double, double);

    Pfractal *fractal;
    double  area, perm, *step, *dist;
    double  a, b, dev;

    PIPcomment("\tShape Fractal ");

    n = chain->n;
    x = chain->x;
    y = chain->y;

    if( n == 0 || x == NULL || y == NULL ) PIPern("Invalid ChainCode");

    /* 
    ** Calculate Area and Length of ChainCode 
    */

    area = 0.0;
    perm = 0.0;
    for(i = 1; i < n; i++) 
      {
      if( x[i] == 0 ) break;
      xd = (double) x[i] - (double) x[i-1];
      yd = (double) y[i] - (double) y[i-1];
      area += x[i]*y[i-1] - x[i-1]*y[i]; 
      perm += hypot(xd,yd);
      }

    PIPcomment("Perm %.2f ", perm);
    PIPcomment("Area %.2f ", area); 

    /* 
    ** Calculate Fractal Dimension 
    */

    if( n < CHAINDEV ) PIPern("Chaincode is too short");
    if((step = (double *) malloc( sizeof(double) * 1000)) == NULL ) MEMern;
    if((dist = (double *) malloc( sizeof(double) * 1000)) == NULL ) MEMern;

    /* J is the Chain Step */
    /* K is the Starting point */
    /* I is the Position on the Chain */

    l = 0;

    for(j=2; j<n/CHAINDEV && l<MAXLENGTH; j++)
      {
      for(k=0; k<j; k++)
        {
	ns = 0;
	rd = 0;
	for(i=j+k; i<n; i=i+j)
	  {
	  ns++;
	  xd = (double)(x[i]-x[i-j]);
	  yd = (double)(y[i]-y[i-j]);
	  rd += hypot(xd,yd);
	  }

	/* Plus Whatever is left over */
	xd = (double)(x[n-1]-x[i-j]);
	yd = (double)(y[n-1]-y[i-j]);
	rd += hypot(xd,yd);
	sd = rd/(double)ns;

	step[l] = (double) log(sd);	/* Average Step Size */
	dist[l] = (double) log(rd);	/* Perimeter Distance */
	l++;
        }
      }

   medfit(step,dist,l,&a,&b,&dev);

   if( pip_verbose ) {
     PIPcomment("\n\t\tIntersect \t%f\n", a);
     PIPcomment("\t\tSlope     \t%f\n", b);
     PIPcomment("\t\tDeviation \t%f\n", dev);
     }
   else { PIPcomment("Dim %.3f\n",  1.0-b); }

   if( (fractal = (Pfractal *) malloc( sizeof(Pfractal) )) == NULL ) MEMern;

   fractal->x = step; 
   fractal->y = dist;
   fractal->a = a;
   fractal->b = b;
   fractal->num = l;
   fractal->dev = dev;
   fractal->perm = perm;
   fractal->area = area;
   return fractal;
   }

/*
** MEDFIT : taken from Numerical Recipes 
*/

#include <math.h>

static int ndatat=0;    	
static double *xt = 0, *yt = 0;
static double aa = 0.0, abdevt=0.0;   

static void  free_vector(double *v, int nl,int nh);
static void  sort(int n, double ra[]);
static double rofunc(double b);
static double *vector(int nl, int nh);

static void 
medfit(double *x, double *y, int ndata, double *a, double *b, double *abdev)
    {
    int j;
    double bb,b1,b2,del,f,f1,f2,sigb,temp;
    double sx=0.0,sy=0.0,sxy=0.0,sxx=0.0,chisq=0.0;

    ndatat=ndata;
    xt=x;
    yt=y;

    for (j=1;j<=ndata;j++) 
      {
      sx += x[j];
      sy += y[j];
      sxy += x[j]*y[j];
      sxx += x[j]*x[j];
      }

    del = ndata*sxx-sx*sx;
    aa  = (sxx*sy-sx*sxy)/del;
    bb  = (ndata*sxy-sx*sy)/del;
    for(j=1; j<=ndata; j++) chisq += (temp=y[j]-(aa+bb*x[j]),temp*temp);

    sigb = sqrt(chisq/del);
    b1   = bb;
    f1   = rofunc(b1);
    b2   = bb+((f1 > 0.0) ? fabs(3.0*sigb) : -fabs(3.0*sigb));
    f2   = rofunc(b2);

    while (f1*f2 > 0.0) 
      {
      bb=2.0*b2-b1;
      b1=b2;
      f1=f2;
      b2=bb;
      f2=rofunc(b2);
      }

    sigb = 0.01*sigb;
    while (fabs(b2-b1) > sigb) 
      {
      bb=0.5*(b1+b2);
      if (bb == b1 || bb == b2) break;
      f=rofunc(bb);
      if (f*f1 >= 0.0) { f1=f; b1=bb; } 
      else { f2=f; b2=bb; }
      }

    *a = aa;
    *b = bb;
    *abdev = abdevt/ndata;
    }

static double rofunc(double b)
    {
    int j,n1,nmh,nml;
    double *arr,d,sum=0.0;

    arr=vector(1,ndatat);
    n1=ndatat+1;
    nml=n1/2;
    nmh=n1-nml;
    for (j=1;j<=ndatat;j++) arr[j]=yt[j]-b*xt[j];
    sort(ndatat,arr);
    aa=0.5*(arr[nml]+arr[nmh]);
    abdevt=0.0;
    for (j=1;j<=ndatat;j++) {
        d=yt[j]-(b*xt[j]+aa);
        abdevt += fabs(d);
        sum += d > 0.0 ? xt[j] : -xt[j];
    }
    free_vector(arr,1,ndatat);
    return sum;
}

static void sort(int n, double ra[])
    {
    int l,j,ir,i;
    double rra;

    l=(n >> 1)+1;
    ir=n;
    for(;;) {
        if (l > 1) rra=ra[--l];
        else {
            rra=ra[ir];
            ra[ir]=ra[1];
            if (--ir == 1) { ra[1]=rra; return; }
        }
        i=l;
        j=l << 1;
        while (j <= ir) {
            if (j < ir && ra[j] < ra[j+1]) ++j;
            if (rra < ra[j]) { ra[i]=ra[j]; j += (i=j); }
            else j=ir+1;
        }
        ra[i]=rra;
    }
    }

static void free_vector(double *v, int nl,int nh)
    {
    free((char*) (v+nl));
    }

static double *vector(int nl, int nh)
    {
    double *v;
    v = (double *) malloc((unsigned) (nh-nl+1)*sizeof(double));
    if(!v) MEMern;
    return v-nl;
    }

static  int invariant, central;
static  double  SMX, SMY, M1, M2, M3, M4, M5, M6, M7,
	m00,  m01,  m10,  m11,  m20,  m02,  m21,  m12,  m03,  m30,
	mu00, mu10, mu01, mu20, mu02, mu11, mu30, mu03, mu21, mu12,
	nu00, nu10, nu01, nu20, nu02, nu11, nu30, nu03, nu21, nu12;        

/**
* Moment Analysis

The normal moments of any function \a I(i,j) are defined by
<P><I>m<SUB>pq</SUB>= Sum<SUB>i</SUB><SUP>nr</SUP> Sum <SUB>j</SUB><SUP>nc</SUP>
i <SUP>p </SUP>j <SUP>q </SUP>I(i,j)</I></P>

If <I>I(i,j)</I> is a binary image, where <I>i,j</I> is the position
of each pixel, then </P>

<UL>
<P><I>M = m<SUB>00<BR>
</SUB>x = m<SUB>01 </SUB>/<SUB> </SUB>m<SUB>00</SUB> ; I<SUB>x </SUB>=
m<SUB>02<BR>
</SUB>y = m<SUB>10 </SUB>/ m<SUB>00</SUB> ; I<SUB>y</SUB> = m<SUB>20</SUB></I></P>
</UL>

<P>where <I>M </I>is simply the number of pixels in the object (ie its
&quot;mass&quot;) and <I>(x,y)</I> is the position of the centroid, and
<I>Ix </I>and <I>Iy </I>the moment of inertia in the <I>x</I> and <I>y
</I>direction respectively. If the coordinate system is shifted so that
the origin coincides with the centroids, the the new set of moments are
invariant under translation. </P>

<UL>
<P><I>mu<SUB>pq </SUB>= Sum<SUB>i</SUB><SUP>nr</SUP> Sum<SUB>j</SUB><SUP>nc</SUP>(i-x)
<SUP>p</SUP>(j-y) <SUP>q </SUP>I(i,j)</I></P>
</UL>

<P>Normallly we would need to solve this equation to derive the moment
about the centroid, but to simplify matters we can use the parallel axis
theorem</P>

<UL>
<P><I>mu<SUB>02</SUB> = m<SUB>02</SUB> - Mx<SUP>2</SUP> = I<SUB>x</SUB>'<BR>
mu<SUB>20</SUB> = m<SUB>20</SUB> - My<SUP>2</SUP> = I<SUB>y</SUB>'<BR>
mu<SUB>11 </SUB>= m<SUB>11 </SUB>- Mxy</I></P>
</UL>

<P>From these values we can calculate the angle of the major axis to the
horizontal</P>

<UL>
<P><I>@= 0.5 arctan( 2mu<SUB>11 </SUB>/ (mu<SUB>20</SUB> - mu<SUB>02</SUB>))</I></P>
</UL>

<P>and the eccentricity </P>

<UL>
<P><I>e = sqrt( e1/e2)<BR>
e1 = mu<SUB>02</SUB>(cos@) <SUP>2</SUP> + mu<SUB>20</SUB>(sin@) <SUP>2</SUP>
- mu<SUB>11</SUB>sin(2@) <BR>
e1 = mu<SUB>02</SUB>(sin@) <SUP>2</SUP> + mu<SUB>20</SUB>(cpd@) <SUP>2</SUP>
- mu<SUB>11</SUB>cos(2@) </I></P>
</UL>

<P>Now if the coordinate system is rotated through an angle of @ then the
moment of inetria along the major and minor axis will be </P>

<UL>
<P><I>I<SUB>max</SUB> = (cos@) <SUP>2</SUP>mu<SUB>20 </SUB>+ (sin@) <SUP>2</SUP>mu<SUB>02</SUB>
+ sin(2@)mu<SUB>11<BR>
</SUB>I<SUB>min </SUB>= (sin@) <SUP>2</SUP>mu<SUB>20 </SUB>+ (cos@) <SUP>2</SUP>mu<SUB>02</SUB>
- sin(2@)mu<SUB>11</SUB></I></P>
</UL>

<P>However, the major and minor moments of inertia are define by </P>

<UL>
<P><I>I<SUB>max</SUB> = a<SUP>3</SUP>b Pi / 4<BR>
I<SUB>min</SUB> = b<SUP>3</SUP>a Pi / 4 </I></P>
</UL>

<P>and a and b are the major and minor axis respectivley, so that by sunstition
</P>

<UL>
<P><I>a = [ I<SUP>3</SUP><SUB>min</SUB> 4<SUP>2 </SUP>/ I<SUB>max </SUB>Pi<SUP>2</SUP>
] <SUP>0.125<BR>
</SUP>b= [ I<SUP>3</SUP><SUB>max</SUB> 4<SUP>2 </SUP>/ I<SUB>min </SUB>Pi<SUP>2</SUP>
] <SUP>0.125</SUP></I></P>
</UL>

<P>and the aspect ratio is simply the ratio of the major to minor axis.
*/

Pmoment *PIP_shape_moment(Pimage *image, int label)
    {
    int row, col, c1, c2, r1, r2;
    double a1, a2, a3, a4, a5, a6, a7, a8, a9;
    double xm, ym, sx, sy, xm2, ym2, cos2, sin2, sin_2, cos_2;
    double theta, Imin, Imax, puissance;
    double major, minor, eccen;
    Pmoment *moment;

    int nc = image->nc;
    int	nr = image->nr;
    pixel *ip;
    
    PIPcomment("Shape Moments (%d)\n", label);
    if((ip = image->data) == NULL ) NULern;

    c2 = 0; c1 = nc;
    r2 = 0; r1 = nr;

    mu00 = 0.0;
    m00 = 0.0; m01 = 0.0; m10 = 0.0; m11 = 0.0;
    m02 = 0.0; m20 = 0.0; m12 = 0.0; m21 = 0.0;
    m03 = 0.0; m30 = 0.0; a1  = 0.0; a2  = 0.0;

    for(row = 0; row < nr; row++)
    for(col = 0; col < nc; col++)
     if( ip[row * nc + col] == label )
      {
      if(c1 > col) c1 = col;
      if(c2 < col) c2 = col;
      if(r1 > row) r1 = row;
      if(r2 < row) r2 = row;

      a1 = (double)col; 
      a2 = (double)row;

      m00 += 1.0; 
      m01 += a1;
      m10 += a2; 
      m11 += a1 * a2; 
      
      a1 *= a1; 
      a2 *= a2;

      m02 += a1; 
      m20 += a2; 
      m12 += (double)row * a1; 
      m21 += (double)col * a2;
      m03 += (double)col * a1;
      m30 += (double)row * a2; 
      }

    if( m00 == 0 ) PIPern("Zero Area"); 
    if( r1 > r2 || c1 > c2 ) PIPern("Invalid Region");
    image->r1 = r1; image->r2 = r2;
    image->c1 = c1; image->c2 = c2;

    xm = m01 / m00; 
    ym = m10 / m00; 

    mu02 = m02 - ( m00 * POWER2( xm ) ) ;
    mu20 = m20 - ( m00 * POWER2( ym ) ) ;
    mu11 = m11 - ( m00 * xm * ym );
 
    if(mu20 == mu02) theta = 0.0; 
    else theta = (0.5*atan2(2.0*mu11, (mu20-mu02)));

    /* To accomodate Clockwise coordinates */
    /* theta = -theta; */

    if((a1==0.0)||(a2 ==0.0)) eccen = (double) -1.0;
    else if((a1/a2) > 0)      eccen = sqrt(a1/a2);
    else 	  	      eccen = sqrt(-a1/a2);

    cos2 = POWER2((cos(theta)));
    sin2 = POWER2((sin(theta)));
    cos_2 = cos((((double)2.0)*theta));
    sin_2 = sin((((double)2.0)*theta));  

    /* Computes the eccentricity of the shape */

    a1 = mu02*cos2 + mu20*sin2 - mu11*sin_2;
    a2 = mu02*sin2 + mu20*cos2 + mu11*cos_2; 

    sx = sqrt( mu02 / m00 ); 
    sy = sqrt( mu20 / m00 ); 

    /* Computes the moments out the CM wrt Theta */

    Imax = (cos2*mu20+sin2*mu02+sin_2*mu11);
    Imin = (sin2*mu20+cos2*mu02-sin_2*mu11);

    /* Computes the moments variance */

    SMX = sqrt(Imax/m00);
    SMY = sqrt(Imin/m00);

    /* Compute The Major and Minor Axis */

    major = pow( Imin*Imin*Imin/Imax*1.621, 0.125);
    minor = pow( Imax*Imax*Imax/Imin*1.621, 0.125);

    if(central == 1 || invariant == 1) {
      xm2 = POWER2(xm);
      ym2 = POWER2(ym);

      /* The central moments of order 3 are as follows */

      mu10 = 0.0;
      mu01 = 0.0;
      mu30 = m30 - ((double)3.0)*xm*m20 +((double)2.0)*m10*xm2;
      mu03 = m03 - ((double)3.0)*ym*m02 +((double)2.0)*m01*ym2;
      mu21 = m21 - m20*ym - ((double)2.0)*m11*xm+((double)2.0)*xm2*m01; 
      mu12 = m12 - m02*xm - ((double)2.0)*m11*ym+((double)2.0)*ym2*m10; 
      }

    if(invariant == 1) {

      nu00 = (double)1.0;
      nu01 = (double)0.0;
      nu10 = (double)0.0;

      puissance = m00 * m00;
      nu02 = mu02 / puissance;
      nu20 = mu20 / puissance;
      nu11 = mu11 / puissance;

      puissance = pow(m00,(double)2.5);
      nu12 = mu12 / puissance;
      nu21 = mu21 / puissance;
      nu03 = mu03 / puissance;
      nu30 = mu30 / puissance;

      a1 = nu20 - nu02;
      a2 = nu30 + nu12;
      a3 = nu21 + nu03;
      a4 = nu30 - 3.0 * nu12;
      a5 = ((double)3.0) * nu21 - nu03;
      a6 = POWER2(a2);
      a7 = POWER2(a3);
      a8 = a2 * (a6 - ((double)3.0) * a7);
      a9 = a3 * (((double)3.0) * a6 - a7);

      M1 = nu20 + nu02;
      M2 = POWER2 (a1) + ((double)4.0) * POWER2(nu11);
      M3 = POWER2 (a4) + POWER2(a5);
      M4 = a6 +a7;
      M5 = a4 *a8  +a5 *a9;
      M6 = a1 *(a6 -a7) +((double)4.0) *nu11 *a2 *a3;
      M7 = ((double)3.0 *nu21 - nu30) *a8 -a4 *a9;
      }

    if( !pip_verbose ) {
      PIPcomment("Area %.0f ", m00);
      PIPcomment("Xm %.0f Ym %.0f\n", xm,ym);
      }
    else { 
      PIPcomment("\n\t\tArea\t\t%f\n", m00);
      PIPcomment("\t\tRegion\t\t(%d %d) (%d %d)\n", c1,r1,c2,r2);
      PIPcomment("\t\tXCentroid\t%f Sigma %f\n", xm, sx);
      PIPcomment("\t\tYCentroid\t%f Sigma %f\n", ym, sy);
      PIPcomment("\t\tMax Inertia\t%.2e Sigma %f\n", Imax, SMX);
      PIPcomment("\t\tMin Inertia\t%.2e Sigma %f\n", Imin, SMY);
      PIPcomment("\t\tTheta\t\t%f Eccen %f\n", theta*RADIANS, eccen);
      PIPcomment("\t\tMajor\t\t%f Minor %f\n", major*2, minor*2);
      PIPcomment("\t\tAspect Ratio\t%f\n",     major/minor);
      }

    if((moment = (Pmoment *) malloc(sizeof(Pmoment))) == NULL ) MEMern;
    moment->area = m00;
    moment->xm = xm; moment->ym = ym;
    moment->sx = sx; moment->sy = sy; 
    moment->theta = theta;
    moment->eccen = eccen;
    moment->major = major;
    moment->minor = minor;
    return moment;
    }

/**
* Calculate distance from centre of mass (xm,ym)
This functions calculate radius statistics from a chain code (Pchain *) 
that defines the position of particles perimeter. The radius is the
distance from the centroid of the particle (xm, ym) to each of the points
in the chain code. The radius data is loaded into the Psignal data structure,
and passed to the signal_stat routine for the following statistical analysis:

- Min , Max and Average
- Variance and Standard deviation
- Skewness and Peakiness (Kurtosis)

When a boundray representation is use, the total area of a feature is
simply

Area = 0.5 Sum(k=0, n-1) ( X<SUB>k</SUB>.Y<SUB>k+1 </SUB>- X<SUB>k+1</SUB>.Y<SUB>k</SUB>)

In addition to the area, it is often useful to measure the filled area
or the convex area. The filled area is the total area of the feature after
the internal holes have been filled. The convex area us a little more complex.
*/

Psignal *
PIP_shape_radius(Pchain *chain, int xm, int ym) 
    {
    int    i;
    double xdiff, ydiff; 
    double hypot(double, double);
    double *data;
    Psignal *radius;
    int signal_stat(Psignal *);

    PIPcomment("\tShape Radius (%d,%d)",xm,ym);
    if( chain->n == 0 ) PIPern("Invalid Chaincode");
    if((data = (double *) malloc( sizeof(double) * chain->n ))== NULL ) MEMern;

    for(i=0; i<chain->n; i++)
      {
      xdiff = (double)(chain->x[i]) - xm;
      ydiff = (double)(chain->y[i]) - ym;
      data[i] = hypot(xdiff,ydiff);
      }

    radius = (Psignal *) malloc( sizeof( Psignal ));
    if( radius == NULL ) MEMern;

    radius->num = i;
    radius->data = data;
    signal_stat(radius);
    return radius;
    }
