/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/

/** \file
\brief Segementation of an Image	

Before we can extract the features of an image it is necessary to segment
the image into a binary image.
That is to distinguish between parts of the image that are, or are not
the particle we wish to analyses.
To do this we set a threshold on the grey level.
For all pixels that are less than the threshold the image is set to true
(255) otherwise they are set to false (0).
In many case the threshold can be adjusted manually, but errors in the
threshold setting will obviously cause bias in the measured parameter.
Using the same settings on different images from the same or similar sample
may be inappropriate particularly if the lighting conditions change, or
the contrast changes in response to the level of contamination in the oil.
Automatic thresholding can be accomplished in several ways.
*/

#include "local.h"

/** 
Segmentation by descrimanant area
@param image pointer to image
@param hist pointer to histogram
@return threshold level
*/

int 
PIP_segment_darea(Pimage *image, Phist *hist)
  {
  int i, k;
  int area[256];
  int diff[256];
  int max = 0;
  int mid = 0;
  int np = image->np;
  pixel *ip; 

  ip = image->data;

  PIPcomment("\tSegment DArea : ");

  for(k=0; k<256; k++) area[k] = 0;
  for(k=0; k<256; k++) for(i=0; i<np; i++) if( ip[i] < k ) area[k]++;
  for(k=1; k<256; k++) diff[k] = area[k] - area[k-1];
  for(k=1; k<256; k++) if( diff[k] > max ) { max = diff[k]; mid = k; }

  if( pip_debug )
    { 
    FILE *fp;
    fp = fopen("area.dat","w");
    for(k=0; k<256; k++) fprintf(fp,"%d %d %d\n", k,area[k],diff[k]);
    fclose(fp);
    }

  hist->mid = mid;
  hist->pos1 = 0;
  hist->pos2 = 256;
  for(i=0; i<256; i++) hist->data[i] = area[i];
  PIPcomment("(%d,%d,%d)\n", hist->pos1,hist->mid,hist->pos2);
  return PIPOK;
  }

static double  	centre[9],	    /* Cluster Centres */
	    	points[9],          /* Number of points in cluster  */
            	average[9],         /* New cluster centres value */
            	variance[9];        /* Variance of new cluster center value */

static int kmeans(int *, int);

/**
Segmentation of image with K Means.
This function returns threshold level based k-means minimizattion. That
is finding the the centre of two populations n the histogram. This method
works on the assumption that there are two populations. A black population
and a white population. It calculates a threshold between the populations
that minimizes the variance from the centre of the population. This technique
assumes that population distributions are symmetric. In many case they
are not.
@param 	image pointer to image
@param	hist pointer to image histogram
@return	threshdold level
*/

int 
PIP_segment_kmeans(Pimage *image, Phist *hist)
    {
    pixel *ip; 
    int i, j, h[256];
    int	np = image->np;

    PIPcomment("\tSegment Kmeans Clustering : "); 

    ip = (pixel *) image->data;
    if( ip == NULL ) NULerr;

    /* Guess best two centres */

    for(i=0; i<256; i++) h[i] = 0;
    for(i=0; i<np;  i++) h[ip[i]]++;
    h[0] = 0; 

    /* A good starting point */

    centre[0] = (double) 0;	
    centre[1] = (double) 255;

    if( !kmeans(h, 2) ) return PIPFAIL;

    for(i=0; i<256; i++) hist->data[i] = h[i];	

    if( centre[0] < centre[1] ) { i = 0; j = 1; } 
    else 		        { i = 1; j = 0; }

    hist->pos1 = centre[i];
    hist->pop1 = (int) points[i];
    hist->var1 = (int) variance[i];
    hist->max1 = h[hist->pos1];

    hist->pos2 = centre[j];
    hist->var2 = (int) variance[j];
    hist->pop2 = (int) points[j];
    hist->max2 = h[hist->pos2];

    hist->mid = hist->pos1 + (hist->pos2-hist->pos1)/2; 
    hist->min = h[hist->mid];
    hist->max = MAX(hist->max1, hist->max2);
    hist->pop = hist->pop1 + hist->pop2;

    PIPcomment("(%d:%d:%d)\n", hist->pos1, hist->mid, hist->pos2);
    return PIPOK;
    }

/* 
** General purpose one dimensional K-means Cluster Analysis
*/

static int kmeans(int *h, int nk)
    {
    int i, j, class, count, flag; 
    double  d,	dmin; /* Euclidean Distance */

    if( nk >= 9 ) PIPerr("Too many Clusters\n"); 

    class = 0;
    count = 0;
    do {
        for(i = 0; i < nk; i++) 
	  {
          points[i] = 0.0;
          average[i] = 0.0; 
          variance[i] = 0.0; 
          }

        /* assign each vector to a cluster (class) */

	for(i = 0; i < 256; i++)
	  {
          dmin = 100000.0;

          for(j = 0; j < nk; j++) 
	    {
            d = i - centre[j]; 
	    if( d < 0 ) d = -d;
            if( d < dmin ) { dmin = d; class = j; }
	    }

          points[class] += (double) h[i];
          average[class] += (double) i*h[i];
          variance[class] += (double) i*i*h[i];
	  }

        /* compute new cluster centre */

        for(i = 0; i < nk; i++) 
          if( points[i] > 0 ) 
	    {
            average[i] /= points[i];
            variance[i] /= points[i];
            variance[i] -= average[i] * average[i];
            }

        /* check for convergence */

        flag = TRUE;
        for(i = 0; i < nk; i++) 
          if( centre[i] != average[i] ) flag = FALSE;

	/* Update New Centres */

        for(i = 0; i < nk; i++) 
          centre[i] = average[i];

	if( pip_verbose )
          for(i = 0; i < nk; i++) 
            PIPcomment("Cluster #%d Points %8.2f Centre %8.2f Variance %8.2f\n", 
              i,points[i],centre[i],variance[i]);

        } while (!flag && (count++ < 1000));

    return PIPOK;
    }

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#define DeltaTau 	0.5
#define MaxTau 		20.2
#define MinTau 		0.2
#endif

/* Local Function prototypes.  */

static void ScaleSpace(long *,double,float *);
static void ZeroCrossHistogram(float *, short *);
static void DerivativeHistogram(float *, float *);

/**
Segment Image based upon local minima in histogram. 

This function returns the minima between two peaks in the histogram.
Each threshold is adjusted to the nearest minimum in the brightness histogram.
It is the point at which peaks belonging to each phase are most recognizably
separated. This minimum cannot be found by simply looking for the lowest
channel. because it is subject to high numerical noise and it contains
relatively few counts. It is sometimes better to fit a parabola over several
channels, up to the maxima on either side. Of concern is a situation where
the peaks are either very different in size, or far apart with a broad
flat minimum. In this situation it may be very difficult to find a consistent
minimum. A good example of this problem occurs when we have a minor phase
which may only represents a few percent of the total area of the image.
In this case, the minor phase peak may be virtually absent, and the histogram
minimum may wander. 
*/

int 
PIP_segment_minima(Pimage *image, Phist *hdata, int strict)
  {
  int   i,j,np;
  float tau;
  pixel *ip;
  float smooth[256], deriv[256];
  long  hist[256]; short cross[256];
  int   Max[256], Min[256], nmax, nmin;
  int   pop1, pop2, histmax;

  PIPcomment("\tSegment Minima (strict:%d) ",strict);
  ip = (pixel *) image->data;
  np = image->np;

  for(i=0; i<256; i++) { Min[i] = 0; Max[i] = 0; 
	hist[i] = 0; deriv[i] = 0; cross[i] = 0; smooth[i] = 0; }

  /* Calculate Histogram */

  histmax = 0;
  for(i=0; i<np;  i++) hist[ip[i]]++;
  hist[0] = 0; /* ignore 0 points */
  for(i=0; i<256; i++) if( histmax < hist[i] ) histmax = hist[i];

  /* Load data before we start */

  hdata->max = histmax;
  for(i=0; i<256; i++) hdata->data[i] = hist[i];

  /* Start Smoothing Histogram */

  for(tau=MinTau; tau<MaxTau; tau+=DeltaTau)
    {
    ScaleSpace(hist,tau,smooth);
    DerivativeHistogram(smooth,deriv);
    ZeroCrossHistogram(deriv,cross);
    
    nmax = 0; nmin = 0;
    for(i=1; i<255; i++) 
       if( cross[i] == 0 )  {
          if( cross[i-1] == 1 && cross[i+1] == -1 ) Max[nmax++] = i;
          else { if( cross[i-1] == -1 && cross[i+1] == 1 ) Min[nmin++] = i; }
	  }
  
    if( pip_debug )
      {
      char filename[40]; FILE *fp;
      sprintf(filename, "Tau%g.dat", tau);
      fp = fopen(filename, "w");
      for(i=0; i<256; i++) 
       fprintf(fp,"%d %8ld %g %8g %8d\n", 
	i,hist[i],smooth[i],deriv[i],cross[i]*histmax);
      fclose(fp);
      }

    if( pip_verbose )
      {
      PIPcomment("\nSmoothing Tau:%f Nmax:%d Nmin:%d", tau, nmax, nmin);
      PIPcomment("\nMax ");
      for(i=0;i<nmax;i++) PIPcomment("%d:%d ",Max[i],(int)smooth[Max[i]]); 
      PIPcomment("\nMin ");
      for(i=0;i<nmin;i++) PIPcomment("%d:%d ",Min[i],(int)smooth[Min[i]]); 
      }

    if( nmax < 2 ) PIPerr("Cannot find two Maximum\n"); 
    if( nmin < 1 ) PIPerr("Cannot find Minima"); 

    j = 0;
    if( !strict && nmax == 3  ) 
      {
      int depth1, depth2;
      if( Min[1] > Max[2] ) PIPerr("Min outside of Maximum\n");
      depth1 = MAX(Max[0] - Min[0], Max[1] - Min[0]);
      depth2 = MAX(Max[1] - Min[1], Max[2] - Min[1]);
      if( pip_verbose ) PIPcomment("\n depth1 %d depth2 %d\n", depth1, depth2); 
      if( depth1 > depth2 ) j = 0; else j = 1; 
      nmax = 2; /* Fudge to get it to work */
      }

    if( nmax == 2 ) 
	{
	if( strict && nmin != 1 ) continue;
        pop1 = 0; for(i=0; i<Min[j]; i++) pop1 += hist[i]; 
        pop2 = 0; for(i=Min[j]; i<256; i++) pop2 += hist[i]; 
        hdata->pos1=Max[j];   hdata->max1=smooth[Max[j]];   hdata->pop1=pop1;
        hdata->pos2=Max[j+1]; hdata->max2=smooth[Max[j+1]]; hdata->pop2=pop2;

        hdata->max = histmax;
        hdata->mid = Min[j]; 
	hdata->min = smooth[Min[j]]; 
        hdata->pop = pop1 + pop2;

 	PIPcomment("(%d,%d,%d)\n", Max[j], Min[j], Max[j+1]);
	return PIPOK;
	}
    }

  PIPcomment("There are two many maxima ***\n");
  return PIPFAIL;
  }

/* 
** LOCAL functions
*/

static void DerivativeHistogram(float *hist, float *deriv)
  {
  int i, n = 255;

  /* Compute endpoints using second order polynomial interpolation. */
  deriv[0]=(-1.5*hist[0]+2.0*hist[1]-0.5*hist[2]);
  deriv[n]=(0.5*hist[n-2]-2.0*hist[n-1]+1.5*hist[n]);

  /* Compute derivative using central differencing.  */
  for(i=1; i<n; i++) deriv[i]=(hist[i+1]-hist[i-1])/2;
  return;
  }

static void ScaleSpace( long *hist, double tau, float *scaled)
  {
  int i, j;
  double sum;
  float alpha = 1.0/(tau*sqrt((double) (2.0*PI)));
  float beta  = (-1.0/(2.0*tau*tau));

  for(j=0; j<256; j++)
    {
    sum = 0.0;
    for(i=0; i<256; i++) sum+=hist[i]*exp((double)(beta*(j-i)*(j-i)));
    scaled[j] = alpha*sum;
    }
  }

/* Mark zero crossings. */

static void ZeroCrossHistogram(float *second, short *cross)
  {
  int i, parity=0;
  for(i=0; i<256; i++)
    {
    cross[i] = 0;
    if( second[i] < 0.0 ) { if( parity > 0 ) cross[i]=(-1); parity=1; }
    else if( second[i] > 0.0 ) { if( parity < 0 ) cross[i]=1; parity=(-1); }
    }
  }

/**
Simple segmentation based upon maxima
*/

int 
PIP_segment_simple(Pimage *image, Phist *hist)
  {
  int i, min, mid, max;
  int pop1, pop2, h[256];
  int np = image->np;

  pixel *ip; 

  ip = image->data;

  PIPcomment("\tSegment Simple : ");

  for(i=0; i<256; i++) h[i] = 0;
  for(i=0; i<np; i++) h[ip[i]]++;

  for(i=0; i<256; i++) if( h[i] > 0 ) break; min = i;
  for(i=255; i>0; i--) if( h[i] > 0 ) break; max = i;

  mid = max / 3;
  pop1 = 0; for(i=0; i<mid; i++) pop1 += h[i];
  pop2 = 0; for(i=mid; i<256; i++) pop2 += h[i];

  hist->mid = mid;
  hist->pos1 = min; hist->pop1 = pop1;
  hist->pos2 = max; hist->pop2 = pop2;
  hist->pop = pop1 + pop2;
  for(i=0; i<256; i++) hist->data[i] = h[i];
  PIPcomment("(%d,%d,%d)\n", min,mid,max);
  return PIPOK;
  }
