/*********************************************************************
*
* CSIRO Automation
* Queensland Centre for Advanced Technologies
* PO Box 883, Kenmore, QLD 4069, Australia
* www.cat.csiro.au/cmst
*
* Copyright (c) CSIRO Manufacturing Science & Technology
*
*********************************************************************/

/** \file
\brief Morpholgical functions 
*/

#include "local.h"

static int harj[8][9] = {
 {1, 1, 1, 0, 1, 0, 0, 0, 0}, 
 {0, 1, 0, 0, 1, 1, 0, 0, 0}, 
 {1, 0, 0, 1, 1, 0, 1, 0, 0}, 
 {0, 1, 0, 1, 1, 0, 0, 0, 0}, 
 {0, 0, 1, 0, 1, 1, 0, 0, 1}, 
 {0, 0, 0, 1, 1, 0, 0, 1, 0}, 
 {0, 0, 0, 0, 1, 0, 1, 1, 1}, 
 {0, 0, 0, 0, 1, 1, 0, 1, 0} }; 

static int hark[8][9] = {
 {0, 0, 0, 0, 0, 0, 1, 1, 1}, 
 {0, 0, 0, 1, 0, 0, 1, 1, 0},
 {0, 0, 1, 0, 0, 1, 0, 0, 1},
 {0, 0, 0, 0, 0, 1, 0, 1, 1},
 {1, 0, 0, 1, 0, 0, 1, 0, 0},
 {0, 1, 1, 0, 0, 1, 0, 0, 0},
 {1, 1, 1, 0, 0, 0, 0, 0, 0},
 {1, 1, 0, 1, 0, 0, 0, 0, 0} };

/**
Thinning of binary image using Haralick and Shapiro's morphological thinning.
Morphological thinning, described in the textbook
Robert M. Haralick and Linda G. Shapiro: "Computer and Robot
Vision", volume I, Addison-Wesley, Reading, Mass., 1992.,
Chapter 5. The algorithm is based on stuff from the whole
chapter, and is given in section 5.10.1.
@Author	Qian Huang (Michigan State University), 
        Oivind Due Trier (University of Oslo).
@Warning	Rountine has not been tested - it is slow 
*/

int 
PIP_thin(Pimage *input, Pimage *output) 
   { 
   int i, j, n, d, r, c;
   int update = 0, empty;
   int hit_flag, mis_flag;
   int add[9];
   pixel *ip, *tmp, *neg, *ham, *hit, *mis;

   int ng = input->ng;
   int nr = input->nr;
   int nc = input->nc;
   int np = input->np;

   PIPcomment("Thin\n"); 

   if( output->data == NULL) 
     if( PIP_make(output, nc, nr, ng, "Thinning") == PIPFAIL ) 
       PIPerr("Cannot Create Thin Image"); 

   if((ip = (pixel *) input->data) == NULL ) NULerr;
   if((tmp = (pixel *) output->data) == NULL ) NULerr;
   memcpy(tmp, ip, np);

   add[1] = -nc-1; add[2] = -nc; add[3] = -nc+1;
   add[8] = -1;    add[0] = 0;   add[4] = 1;
   add[5] = nc+1;  add[6] = nc;  add[7] = nc-1;

   neg = (pixel *) calloc(sizeof(pixel), np);
   ham = (pixel *) calloc(sizeof(pixel), np);
   hit = (pixel *) calloc(sizeof(pixel), np);
   mis = (pixel *) calloc(sizeof(pixel), np);

   for(n=0; n<100; n++) 
      {
      printf("Pass number %i\n", n);
      for(i=0; i<np; i++) neg[i] = ON - tmp[i];

      for(d=0; d<8; d++)
        {   
        printf("\tDirection %i\n", d);

        empty = TRUE;

        /* Look at internal pixels */

        for(r=1; r<nr-1; r++)
        for(c=1; c<nc-1; c++)
          {   
          i = r*nc + c;
          hit_flag = TRUE;
          mis_flag = TRUE;

	  /* Look at Surrounding Pixels */

          for(j=0; j<9; j++)
	    {
	    if( harj[d][j] && (tmp[i+add[j]] == OFF) ) hit_flag = FALSE ;
	    if( hark[d][j] && (neg[i+add[j]] == OFF) ) mis_flag = FALSE ;
	    }

          if( hit_flag ) hit[i] = ON ; else hit[i] = OFF ;
          if( mis_flag ) mis[i] = ON ; else mis[i] = OFF ;
          if( hit_flag && mis_flag ) { ham[i] = ON ; empty = FALSE ; } 
          else ham[i] = OFF ;
          }
   
        if( !empty ) { 
	  update = TRUE; 
          for(i=0; i<np; i++) 
	    {
	    tmp[i] -= ham[i];
	    neg[i] = ON - tmp[i];
	    }
          }
	}

      if( !update ) break;
      }

   free(neg);
   free(ham);
   free(hit);
   free(mis);

   return PIPOK;
   } 

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#ifndef NULL
#define NULL 0
#endif
#define THN_OK 		1
#define THN_ERROR 	-1
#define THN_NO_MEMORY 	-10
#define THN_NO_DIR 	-256
#define THN_UNKNOWN_INT -32768
#define THN_FOREGROUND 	0
#define THN_BACKGROUND  255
#endif

static int edge(int *, int, int, int, int);

/** 
Zhang-Suen thinning 
@Note This routine has not been tested.
*/

int 
PIP_thinning(Pimage *input, Pimage *output)
    {
    int i, j;
    int flag = TRUE;
    pixel *ip, *op;

    int ng = input->ng;
    int nc = input->nc;
    int nr = input->nr;
    int np = input->np;
    int *x, *y;

    PIPcomment("Thinning\n");

    if( output->data == NULL) 
      if( PIP_make(output, nc, nr, ng, "Thinning") == PIPFAIL )
        PIPerr("Cannot Create Thin Image"); 

    if((ip = input->data) == NULL) NULerr;
    if((op = output->data) == NULL) NULerr; 
    if((x = (int *) malloc( np* sizeof(int) )) == NULL) MEMerr;
    if((y = (int *) malloc( np* sizeof(int) )) == NULL) MEMerr;

    for(i=0; i<np; i++) if( ip[i] > OFF ) x[i] = 9; else x[i] = 0;
    for(i=0; i<np; i++) y[i] = 0;

    /* while(flag) */

       {
       PIPcomment("*");
       flag = FALSE;

       for(i=0; i<nc; i++)
       for(j=0; j<nr; j++) {	
         if( x[j*nc+i] == 0 ) { y[j*nc+i] = 0; continue; }
       	 if( edge(x, i, j, nc, nr) ) { y[j*nc+i] = 1; flag = TRUE; }
       	 else y[j*nc+i] = 0;
         }

       for(i=0; i<np; i++) x[i] -= y[i]; 

       for(i=0; i<nc; i++)
       for(j=0; j<nr; j++) {	
       	 if( x[j*nc+i] == 0 ) { y[j*nc+i] = 0; continue; }
       	 if( edge(x, i, j, nc, nr) ) { y[j*nc+i] = 1; flag = TRUE; }
       	 else y[j*nc+i] = 0;
         }

       for(i=0; i<np; i++) x[i] -= y[i]; 
       }

    for(i=0; i<np; i++) if( y[i] > 0 ) op[i] = ON; else op[i] = OFF;
    free(x); free(y);
    return PIPOK;
    }

static int edge(int *ip, int i, int j, int nc, int nr)
    {
    int n, ar, br, a[8];
    int p1, p2;

    /* A is setup in a grid pattern

	7 6 5
	0 - 4
	1 2 3

    AR is the number of 01 patterns */


    for(n=0; n<8; n++) a[n] = 0;

    if(i-1 >= 0) {
       a[0] = ip[j*nc+(i-1)];
       if(j+1 < nr) a[1] = ip[(j+1)*nc+(i-1)];
       if(j-1 >= 0) a[7] = ip[(j-1)*nc+(i-1)];
       }

    if(i+1 < nc) {
       a[4] = ip[j*nc+(i+1)];
       if (j+1 < nr) a[3] = ip[(j+1)*nc+(i+1)];
       if (j-1 >= 0) a[5] = ip[(j-1)*nc+(i+1)];
       }

    if(j+1 < nr) a[2] = ip[(j+1)*nc+i];
    if(j-1 >= 0) a[6] = ip[(j-1)*nc+i];

    ar = 0; br = 0;
    for(n=0; n<7; n++) { if( !a[n] && a[n+1] ) ar++; br += a[n]; }
    if ((a[7] == 0) && (a[0] == 1)) ar++;
    br = br + a[7];

    p1 = a[0]*a[2]*a[4]; 
    p2 = a[2]*a[4]*a[6];

    if ( (ar == 1) && (p1 == 0) && (p2 == 0) ) return(PIPOK);
    return(PIPFAIL);
    }

/**
Generate backbone from skeleton
@warning this program uses Zimage until I can do it myself
*/

int 
PIP_backbone(Pimage *input, Pimage *output)
    {
    PIP_save(input, "/tmp/mask.pgm");
    PIPcomment("Backbone\n");

    if( system("rsh stan backbone") != 0 ) 
       { printf("System Call Failure\n"); return(PIPFAIL); }

    printf("\n");
    PIP_load(output, "/tmp/skel.pgm");
    return(PIPOK);
    }

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#define MAX_DIST 5000
#endif


/*  
To Start all non-zero internal points are marked with MAX_DIST.
Then starting at the top left corner, scanning down and across,
if current point is non-zero, then look for minimum neighbour, 
If neighbour is vert/horx add 5, if diagonal add 7 (5*5 + 5*5 = 7*7)
Set the current point is (min + distance to cell).

Basically the zero value outside is propogated across the blob)
Thus, starting from the non-zero edge, the minimum is outside (0)
then the value of the edge pixel is set to 0+5 or 0+7, then 
from the next pixel, the min is (5 or 7) plus (5 or 7) again.
*/

/**
* This functuon generates distance transform of binary image
* This function performs a twp pass distance transformation. The result
* is passed back to the \c output image, which has been normalized. The
* maximum value is loaed into Image->scale. 
\image html distance.gif "Binary image before and after distance transformation"
*
*/

int 
PIP_distance(Pimage *input, Pimage *output)
    {
    int i, j;
    pixel *ip, *op;
    unsigned int *tmp, add, min, max, val;
    double scale;

    int nc = input->nc;
    int	nr = input->nr;
    int	ng = input->ng;
    int	np = input->np;

    /* Copy Input Image to Temporary Pointer */

    if(( ip = input->data) == NULL ) NULerr;
    if((tmp = (unsigned int *) malloc(np*sizeof(int))) == NULL) MEMerr;
    for(i=0; i<np; i++) if( ip[i] ) tmp[i] = MAX_DIST; else tmp[i] = OFF;

    /* First Pass */

    for(i=1; i<nr-1; i++) 
    for(j=1; j<nc-1; j++) 
     if( tmp[i*nc+j] )			
      {
      min = tmp[i*nc+(j+1)]; add = 5;
      val = tmp[i*nc+(j-1)]; if( min > val ) min = val;
      val = tmp[(i-1)*nc+j]; if( min > val ) min = val;
      val = tmp[(i+1)*nc+j]; if( min > val ) min = val;
      val = tmp[(i-1)*nc+(j+1)]; if( min > val ) { min = val; add = 7; }
      val = tmp[(i-1)*nc+(j-1)]; if( min > val ) { min = val; add = 7; }
      val = tmp[(i+1)*nc+(j+1)]; if( min > val ) { min = val; add = 7; }
      val = tmp[(i+1)*nc+(j-1)]; if( min > val ) { min = val; add = 7; }
      tmp[i*nc+j] = min + add;
      }

    /* Second Pass */

    for(i=nr-2; i>0; i--) 
    for(j=nc-2; j>0; j--) 
     if( tmp[i*nc+j] )
      {
      min = tmp[i*nc+(j+1)]; add = 5;
      val = tmp[i*nc+(j-1)]; if( min > val ) min = val;
      val = tmp[(i-1)*nc+j]; if( min > val ) min = val;
      val = tmp[(i+1)*nc+j]; if( min > val ) min = val;
      val = tmp[(i-1)*nc+(j+1)]; if( min > val ) { min = val; add = 7; }
      val = tmp[(i-1)*nc+(j-1)]; if( min > val ) { min = val; add = 7; }
      val = tmp[(i+1)*nc+(j+1)]; if( min > val ) { min = val; add = 7; }
      val = tmp[(i+1)*nc+(j-1)]; if( min > val ) { min = val; add = 7; }
      tmp[i*nc+j] = min + add;
      }

    max = (unsigned int) 0;
    for(i=0; i<np; i++) if( tmp[i] != MAX_DIST && tmp[i] > max ) max = tmp[i]; 
    scale = 255.0 / (double) max;

    /* Create Output Pointer */

    if( output->data == NULL )
      if( PIP_make(output, nc, nr, ng, "Distance Transform") == PIPFAIL ) 
        PIPerr("Cannot Create Distance Transform Image"); 

    op = output->data;
    for(i=0; i<np; i++) op[i] = (pixel) (tmp[i]*scale);

    free(tmp);
    output->scale = (double) max / (double) 255;
    PIPcomment("Distance Transform : Max (%d) Scale (%f)\n", max,output->scale);
    return PIPOK; 
    }

/* 
** This is a direction vector: for example, if the direction that we have 
** just made is 1, then we look at the cells 8,1,2 
*/

/* static int D[9][3] = { {0,0,0}, {8,1,2}, {1,2,3}, 
		{2,3,4}, {3,4,5}, {4,5,6}, 
		{5,6,7}, {6,7,8}, {7,8,1}  };
static void down_hill(int pos, int dir);
static void up_hill(int pos, int dir);
*/

static int nr, nc, np, add[9];
static int base(int), apex(int);

static pixel *ip, *op;

/*
Generate skeleton from medial axis
@warning This routine does not work yet
*/

int 
PIP_skeleton(Pimage *input, Pimage *output)
    {
    int i, j, na, nb, pos;
    unsigned int max, dir;
    int	ng = input->ng;

    nc = input->nc;
    nr = input->nr;
    np = input->np;

    PIPcomment("Skeletonization\n");

    if((ip = input->data) == NULL) NULerr;
    if( output->data == NULL) 
      if( PIP_make(output, nc, nr, ng, "Skeleton") == PIPFAIL )
        PIPerr("Cannot Create Skeleton Image\n"); 

    op = (pixel *) output->data;

    if( pip_verbose ) PIPcomment("Find Apex and Base Points\n");

    add[1] = -nc-1; add[2] = -nc; add[3] = -nc+1;
    add[8] = -1;    add[0] = 0;   add[4] = +1;
    add[7] = +nc-1; add[6] = +nc; add[5] = +nc+1;

    na = 0; nb = 0;
    for(i=1; i<nr-1; i++)
    for(j=1; j<nc-1; j++)
      if( ip[i*nc+j] ) 
        {
	pos = i*nc+j; 
        if( apex(pos) ) { op[pos] = (unsigned char) 255; na++; }
        if( base(pos) ) { op[pos] = (unsigned char) 128; nb++; }
	}

    if( pip_verbose ) PIPcomment("Number of Apex %d Base %d points \n", na,nb);

    pos = 0;
    max = 0;
    dir = 0;

    /*
    for(i=1; i<9; i++) 
      {
      val = ip[pos+add[i]];
      if( val > max ) { max = val; dir = i; }
      }

    pos += add[dir];
    up_hill(pos, dir); 
    */

    return(PIPOK);
    }

/* APEX - local maxima */

static int apex(int p)
    {
    int i, val = ip[p];
    for(i=1; i<9; i++) if( val <= ip[p+add[i]] ) return(PIPFAIL);
    return(PIPOK);
    }

/* BASE - is the point with distance 1 that has 4 or more zero neighbours */
/* Or vis versa has less than 4 non-zero neighbours */

static int base(int p)
    {
    int i, n=0;
    for(i=1; i<9; i++) if( ip[p+add[i]] ) n++;
    if( n < 4 ) return(PIPOK); 
    else return(PIPFAIL);
    }

#ifdef UP_HILL
static void up_hill(int pos, int dir)
    {
    int i, d;
    int num, val, max, new;

    if( dir < 0 || dir > 8 ) PIPerr("Invalid Direction\n");
    if( pos < 0 || pos > np ) return;
    if( ip[pos] == 0 ) return;

    PIPcomment("Row %d Col %d Val %d\n", pos%nc, pos/nc, ip[pos]); 

    new = 0;
    max = 0;
    num = 0;

    /* Look for maximum neighbour */

    for(i=0; i<8; i++)
      {
      d = D[dir][i];
      val = ip[pos+add[d]];
      if( val > max ) { max = val; new = d; num = 1; }
      if( val == max ) num++;
      }

    if( max < ip[pos] ) /* Mark point */
      {
      op[pos] = MARK;
      }

    if( num > 1 && max >= ip[pos] ) 
      {
      new = 0;
      max = 0;

      for(i=0; i<3; i++)
        {
        d = D[dir][i];
        val = ip[pos+add[d]];
        if( val > max ) { max = val; new = d; }
        }

      dir = new;
      pos += add[dir];
      up_hill(pos, dir); 
      }
    else
      {
      if( num == 1 && max >= ip[pos] )
	{
        dir = new;
        pos += add[dir];
        up_hill(pos, dir); 
	}
      else
        {
	op[pos] = MARK;
        return;  
	}
      }
    }

static void down_hill(int pos, int dir)
    {
    int i, d, val, new=0, max=0;

    for(i=0; i<3; i++)
      {
      d = D[dir][i];
      val = ip[pos+add[d]];
      if( val > max ) { max = val; new = d; }
      }

    dir = new;
    pos += add[dir];
    if( !op[pos] ) down_hill(pos, dir);
    }
#endif

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#define NB 	2000
#endif

/**
This function generates a Binary watershed for the binary <I>input</I>
image. If merge > 0.0, then Areas can be merged. 
*/

int 
PIP_watershed(Pimage *input, Pimage *output, float merge)
    {
    int i, row, col, pos, add[9];
    pixel *ip, *op, val; 
    int level, type;
    int nc = input->nc;
    int	nr = input->nr;
    int	ng = input->ng;

    PIPcomment("Watershed (%.1f)\n",merge); 

    if( output->data == NULL) 
      if( PIP_make(output, nc, nr, ng, "Watershed") == PIPFAIL )
	PIPerr("Cannot Create Watershed Image");

    if((ip = (pixel *) input->data) == NULL) NULerr;
    if((op = (pixel *) output->data) == NULL) NULerr;
		       
    add[1] = -nc-1; add[2] = -nc; add[3] = -nc+1;
    add[8] = -1;    add[0] = 0;   add[4] = 1;
    add[5] = nc+1;  add[6] = nc;  add[7] = nc-1;

    /* Start at the top of distance transformation */

    for(level=256; level>1; level--)
      {
      for(row=1; row<(nr-1); row++) 
      for(col=1; col<(nc-1); col++) 
	{
	pos = row*nc + col;
	if( ip[pos] == level ) {
	  type = 0;
	  for(i=1; i<9; i++)   {
	    val = op[pos+add[i]];
	    if( val > OFF ) {
	      if( type == 0 ) type = (int) val; 
	      else if( type != (int) val ) { type = -1; }
	      }
 	    }

	  /* If pixel has no neighbours - then it can start its own region */
	  /* else if will inherit the region "type" of its neighbours */
	  /* Unless it is at a boundary (where type = -1) */

	  if( type == 0 ) op[pos] = (pixel) level;
	  else if( type > 0 ) op[pos] = (pixel) type; 
	  }
        }
      }

    /* Merge Border */

    if( merge > 0.0 )
     {
     int fill;
     int j, k; pixel a[9];

     if( pip_verbose ) PIPcomment("Merge Border\n");

     for(row=1; row<(nr-1); row++) 
     for(col=1; col<(nc-1); col++) 
	{
	pos = row*nc + col;
        if( ip[pos] && !op[pos]) 
	  { 
	  /* Find Non-zero Unique neighnours */

	  for(i=1,k=0; i<9; i++) { 
	    val = op[pos+add[i]]; 
	    if( val > OFF ) { 
	      for(j=0; j<k; j++) if( val == a[j] ) break;
              if( j == k ) a[k++] = val;
	      }
	    }

	  /* Only fill if the blobs do not differ by more than +- merge */

	  if( k > 1 ) {
	    fill = TRUE;
	    for(i=0; i<k; i++)
	    for(j=i+1; j<k; j++)
	      if( a[i] < a[j]*(1.0-merge) || 
		  a[i] > a[j]*(1.0+merge) ) fill = FALSE; 
            if( fill == TRUE ) op[pos] = a[0];
	    }
	  }
        }
      }

    return PIPOK;
    }

/**
BINARY_WATERSHED : Seperate Touching Features 
Starting form the UEP (Ulimate Erosion Points)
The image is dilated, with the added logical constraint that no new pixel 
may be turned ON if it caused a connection to form between previously 
seperate features, or if it was not ON in the original image. 
*/

int 
binary_watershed(Pimage *input, Pimage *output)
    {
    int i, row, col, pos, add[9];
    unsigned char *ip, *op; 

    int level, mark, type1, type2, type3, type4;
    int val, local[256];	/* Record Level of starting point */

    int nc = input->nc;
    int	nr = input->nr;
    int	ng = input->ng;

    printf("*** Binary Watershed "); fflush(stdout);

    if((ip = (unsigned char *) input->data) == NULL) NULerr;
    if( output->data == NULL )
      if( PIP_make(output, nc, nr, ng, "Watershed") == PIPFAIL )
	PIPerr("Cannot Create Watershed Image\n"); 
		       
    if((op = (unsigned char *) output->data ) == NULL) NULerr;
    
    /* 
Starting at the brightest value and iteratively decrement this to 1 covers all 
brightness levels. At each level, only those pixels at the current brightness 
level in the distance map need be considered. Thos that DO NOT produce a join 
between feature pixels are added to the image. The process continues until all 
of the pixels in the feature, except for those along boundary lines, have been 
restored */

    add[1] = -nc-1; add[2] = -nc; add[3] = -nc+1;
    add[8] = -1;    add[0] = 0;   add[4] = 1;
    add[5] = nc+1;  add[6] = nc;  add[7] = nc-1;

    mark = 0;
    for(level=256; level>1; level--)
      {
      for(row=1; row<(nr-1); row++) 
      for(col=1; col<(nc-1); col++) 
	{
	pos = row*nc + col;
	if( ip[pos] == level )  /* Only Look at Current Level */
	  {
	  type1 = 0;
	  type2 = 0;
	  type3 = 0;
	  type4 = 0;

	  for(i=1; i<9; i++)   /* Look at surrounding Pixels in Output Image */
	    if( op[pos+add[i]] > OFF )
	      {
	      val = (int) op[pos+add[i]]; 
	      if( type1 == 0 ) type1 = val;
	      else if( val != type1 ) 
		{
		if( type2 == 0 ) type2 = val;
		else if( val != type2 ) 
		  {
		  if( type3 == 0 ) type3 = val;
		  else if( val != type3 ) 
		    {
		    if( type4 == 0 ) type4 = val;
		    else if( val != type4 ) printf("Too many Types\n");
		    }
		  }
		}
	      }

	  /* If pixel has no neighbours - then it can start its own region */
	  /* else if will inherit the region "type" of its neighbours */
	  /* Unless it is at a boundary (ie type1 & type2) */
	  /* if type1 and type2 have similiar maxima then fuse */

	  if( type1 == 0 ) { 
	    /* This is an UEP so we give it a new mark */
	    local[mark] = level; 
	    op[pos] = mark++; }
	  else {
	    if( type2 == 0 ) op[pos] = type1;
	    else {
              if( local[type1] > 0.95*local[type2] 
	       && local[type1] < 1.05*local[type2] )
	      op[pos] = type1;
	      }
	    }

	  if( mark > 256 ) PIPerr("Too many regions"); 
	  }
        }

      if( pip_verbose ) printf("Level %d Mark %d\n", level, mark);
      }

    printf(" %d regions ", mark);
    printf("***\n");
    return(PIPOK);
    }

/**
The merge just quantizes the distance, where level>> merge
*/

int 
PIP_watershed_merge(Pimage *input, Pimage *output, int merge)
    {
    int i, row, col, pos, add[9];
    pixel *ip, *op; 

    int level, type;

    int nc = input->nc;
    int	nr = input->nr;
    int	ng = input->ng>>merge;

    PIPcomment("Watershed (Merge:%d)\n",merge); 

    if((ip = (pixel *) input->data) == NULL) NULerr;
    if( output->data == NULL )
      if( PIP_make(output, nc, nr, ng, "Watershed") == PIPFAIL )
	PIPerr("Cannot Create Watershed Image"); 
		       
    if((op = (pixel *) output->data ) == NULL) NULerr;
    
    add[1] = -nc-1; add[2] = -nc; add[3] = -nc+1;
    add[8] = -1;    add[0] = 0;   add[4] = 1;
    add[5] = nc+1;  add[6] = nc;  add[7] = nc-1;

    for(level=256; level>1; level--)
      {
      for(row=1; row<(nr-1); row++) 
      for(col=1; col<(nc-1); col++) 
	{
	pos = row*nc + col;
	if( ip[pos] == level )  /* Only Look at Current Level */
	  {
	  type = 0;
	  for(i=1; i<9; i++)   /* Look at surrounding Pixels in Output Image */
	    if( op[pos+add[i]] > OFF )
	      {
	      if( type == 0 ) type = (int) op[pos+add[i]];
	      else if( (int)op[pos+add[i]] != type ) type = -1; 
	      }

	  /* If pixel has no neighbours - then it can start its own region */
	  /* else if will inherit the region "type" of its neighbours */
	  /* Unless it is at a boundary (where type = -1) */

	  if( type == 0 ) op[pos] = level>>merge; 
	  else if( type > 0 ) op[pos] = type;
	  }
        }
      }

    return(PIPOK);
    }

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#define SMALL	1
#define LARGE	2
#define NB 	2000
#endif

/**
Split image into two regions
The image is split into two, here level == split
If regions are seperated by a boundary of type (border) 
In this case if border == 0 then there will be a black 
border between regions, but if border == 1, then there will
effectively be no border at all.
\return number of boundary pixels 
*/

int 
PIP_watershed_split(Pimage *input, Pimage *output, 
	int split, int border)
    {
    int i, row, col, pos, add[9];
    pixel *ip, *op, val; 

    int level, type, start;
    int nb, boundary[NB];
    int nc = input->nc;
    int	nr = input->nr;
    int	ng = input->ng;

    PIPcomment("Watershed (split:%d)\n",split); 
    if((ip = (pixel *) input->data) == NULL) NULerr;

    if( output->data == NULL )
      if( PIP_make(output, nc, nr, ng, "Watershed") == PIPFAIL )
	PIPerr("Cannot Create Watershed Image");
		       
    if((op = (pixel *) output->data ) == NULL) NULerr;
    
    add[1] = -nc-1; add[2] = -nc; add[3] = -nc+1;
    add[8] = -1;    add[0] = 0;   add[4] = 1;
    add[5] = nc+1;  add[6] = nc;  add[7] = nc-1;

    /* Start at the top of distance transformation */

    nb = 0;
    for(level=256; level>1; level--)
      {
      if (level > split ) start = LARGE;
      else start = SMALL; 

      for(row=1; row<(nr-1); row++) 
      for(col=1; col<(nc-1); col++) 
	{
	pos = row*nc + col;
	if( ip[pos] == level )  /* Only Look at Current Level */
	  {
	  type = 0;
	  for(i=1; i<9; i++)    /* Look at surrounding Pixels */
	    {
	    val = op[pos+add[i]];
	    if( val > OFF )
	      {
	      if( type == 0 ) type = (int) val; 	/* The first type */
	      else if( type != (int) val ) type = -1; 	/* The second */
	      }
 	    }

	  /* If pixel has no neighbours - then it can start its own region */
	  /* else if will inherit the region "type" of its neighbours */
	  /* Unless it is at a boundary (where type = -1) */

	  if( type == 0 ) op[pos] = (pixel) start;
	  else if( type > 0 ) op[pos] = (pixel) type; 
 	  else { boundary[nb++] = pos; 
	    if( nb > NB ) PIPerr("Too many boundary pixels");  }
	  }
        }
      }

    /* Write pixels back along boundary between regions */

    if( border && nb > 0 ) for(i=0; i<nb; i++) op[boundary[i]] = border;	
    return nb;
    }
