/*********************************************************************
*
* CSIRO Automation
* Queensland Centre for Advanced Technologies
* PO Box 883, Kenmore, QLD 4069, Australia
* www.cat.csiro.au/cmst
*
* Copyright (c) CSIRO Manufacturing Science & Technology
*
*********************************************************************/

/** \file 
\brief  Miscellaneous Image Utilities 
*/

#include "local.h" 

/**
* Generate Histogram Image from Histogram
* 
* A histogram of the <I>input</I> image is created. With scalling it can be 
* converted to an image with a height of 100, and a width of 256. 
* This function is generally used for debugging, to view the results of 
* segmentation.
* \code
Pimage input, output;
PIP_load("test.pgm", &input);
PIP_histogram(&input, &output);
PIP_save("hist.pgm", &output);
* \endcode
*/

int PIP_histogram(Pimage *input, Pimage *output)
    {
    pixel *ip;
    int i, j, max=0, h[256];
    int np = input->np;

    /* Create normalized histogram */

    ip = input->data;
    for(i=0; i<256; i++) h[i] = 0;			/* Clear Histogram */
    for(i=0; i<np;  i++) h[ip[i]]++;			/* Load Histogram */
    for(i=1; i<256; i++) if( max < h[i] ) max = h[i]; 	/* Find Maximum */
    for(i=1; i<256; i++) h[i] = 100-(int)(100 *(double)h[i]/max);  /* Rescale */
    h[0] = 100;

    /* Create Image */
     
    if( PIP_make(output, 256, 100, 255, "histogram") == PIPFAIL)
      PIPerr("Cannot Create Histogram Image"); 

    ip = output->data;
    for(i=0; i<256; i++) 
    for(j=0; j<h[i]; j++) 
      ip[j*256+i] = ON; 

    return PIPOK;
    }

/** 
* Generate and return Covarinace matrix of dimension (ng)
*/

int **PIP_covar(Pimage *input, int ng)
    {
    int i, j, k, I, J;
    pixel *ip;
    int **op;
    int nc = input->nc;
    int	nr = input->nr;

    PIPcomment("Covariance (ng:%d)\n", ng); 

    if((ip = input->data) == NULL) NULern;
    if((op = calloc(ng, sizeof(int))) == NULL) MEMern;
    for(i=0;i<ng; i++) if((op[i] = calloc(ng, sizeof(int))) == NULL) MEMern;

    for(j = 0; j < nr; j++) for(i = 0; i < nc-1; i++) /* Across */
      { k = j*nc+i; I = ip[k]; J = ip[k+1]; op[I][J]++; op[J][I]++; } 
    for(j = 0; j < nr-1; j++) for(i = 0; i < nc; i++) /* Down */
      { k = j*nc+i; I = ip[k]; J = ip[k+nc]; op[I][J]++; op[J][I]++; } 

    return op; 
    }

/**
This function allocates memory for 2D matrix
\return pointer to matrix 
\see PIP_covar()
*/

int **PIP_matrix(
	int ng) 	/**< Size of matrix */
    {
    int i, **op;
    if((op = calloc(ng, sizeof(int))) == NULL) MEMern;
    for(i=0; i<ng; i++) if((op[i] = calloc(ng, sizeof(int))) == NULL) MEMern;
    return op;
    }

/**
This function free memory for a 2D matrix 
*/

int PIP_free_matrix(
	int **ip, 	/**< Pointer to matrix */
	int ng) 	/**< Size of matrix */
    {
    int i;
    if( ip == NULL) return PIPFAIL;
    for(i=0; i<ng; i++) free(ip[i]);
    free(ip);
    return PIPOK;
    }

int PIP_covar2(Pimage *input, int **op)
    {
    pixel *ip;
    int i, j, k, I, J;
    int nc = input->nc;
    int	nr = input->nr;

    PIPcomment("Covariance\n"); 
    if((ip = input->data) == NULL) NULerr;
    if( op == NULL ) NULerr;
    for(j = 0; j < nr; j++) for(i = 0; i < nc-1; i++) /* Across */
      { k = j*nc+i; I = ip[k]; J = ip[k+1]; op[I][J]++; op[J][I]++; } 
    for(j = 0; j < nr-1; j++) for(i = 0; i < nc; i++) /* Down */
      { k = j*nc+i; I = ip[k]; J = ip[k+nc]; op[I][J]++; op[J][I]++; } 

    return PIPOK; 
    }

int PIP_covar_to_image(int **ip, int ng, Pimage *op)
    {
    int i, j;
    if( PIP_make(op, ng, ng, 256, "Covar Matrix") == PIPFAIL ) return PIPFAIL;
    for(i=0; i<ng; i++)
      for(j=0; j<ng; j++)
        op->data[i*ng+j] = (pixel) ip[i][j];
    return PIPOK; 
    }

/**
* Cummulative Sum
*/

int PIP_cumsum(Pimage *image)
    {
    int j, i;
    pixel *ip;
    int nc = image->nc;
    int	nr = image->nr;
    int sum = 0, cumsum = 0;

    PIPcomment("CumSum\n"); 
    if((ip = image->data) == NULL ) NULerr;

    for(j=0; j<nr; j++) {
      for(i=0; i<nc; i++) cumsum += ip[j*nc+i];
      sum /= nc;
      for(i=0; i<nc; i++) ip[j*nc+i] = sum;
      }

    return PIPOK;
    }

/**
* This function generates a Bivariant Histogram Image. In the example below
* there is a bivariant histogram of a Cr image vs Cb image. Note that the 
* the coordinates start from the top left hand corner. In this image:
* -  The peak at the centre represents the White/Black/Grey (Cr 128, Cb 128)
* - \b Red is 3 O'clock (Cr 240, Cb 100) 
* - \b Magenta at 4 O'clock 
* - \b Blue at 7 O'clock (Cr 110, Cb 230)
* - \b Cyan at 9 O'clock 
* - \b Green at 11 O'clock (Cr 40, Cb 60)
* - \b Yellow at 1 O'clock
*
* \image html bivariant.jpg "Bivariant Histogram of Cr vs Cb from test image"
* \see  PIP_RGBtoYCrCb
* \todo This should have its own Type
* \warning This function will use mask image if you have them. 
*/

int PIP_bivariant(
	Pimage *op,	/*!< Output image */
	Pimage *i1, 	/*!< First Image - values form X axis */
	Pimage *i2  	/*!< Second Image - values form Y axis */
	)
  {
  int i,j;
  long max = 0;
  long hist[256][256];
  int np = i1->np;

  for(i=0; i<256; i++)
  for(j=0; j<256; j++) hist[i][j] = 0;

  if( i1->mask != NULL && i2->mask != NULL ) {
    for(i=0; i<np; i++)
      if( i1->mask[i]  && i2->mask[i] )
        hist[(int)i1->data[i]][(int)i2->data[i]]++; 
  } else {
    for(i=0; i<np; i++)
        hist[(int)i1->data[i]][(int)i2->data[i]]++; 
	}

  for(i=0; i<256; i++)
    for(j=0; j<256; j++) 
      if( max < hist[i][j] ) max = hist[i][j];

  PIPcomment("Bivariant Max %d\n", max);
  if( op->data == NULL ) PIP_make(op, 256, 256, 255, "Bivariant Histogram");
  if( op->nc != 256 ) { PIPerror("Invalid Number of Columns "); return PIPFAIL; }
  if( op->nr != 256 ) { PIPerror("Invalid Number of Rows "); return PIPFAIL; }
  op->scale = max;

  for(i=0; i<256; i++)
  for(j=0; j<256; j++)
    op->data[j*256+i] = (unsigned char) (hist[i][j]*256/max);

  return PIPOK;
  }

/**
* Create a labelled image based upon 2 other images, and a lookup MAP. 
* A threshold map is a binary 256x256 image that lookup values. 
* For example, if the value of a pixel in \c image1[i][j] is \c I 
* and \c image2[i][j] is \c J, then \c label[i][j] =  \c map[I][J]
* The advangtage of this technique over conventional thresholds is that 
* the threhold map can have any shape. Furthermore, it can identify different regions. 
* For example, if we were to generate a lookup map image, with two regions

* \code
* PIP_make(map, 256, 256, 255, "Threshold Map");
* PIP_drawbox(map, 210, 80, 40, 40, 255); - For Red in CrCb space
* PIP_drawbox(map, 20, 30, 40, 40, 160); - For Green in CrCb space
* PIP_drawbox(map, 90, 210, 40, 40, 100); - For Blue in CrCb space
* PIP_thresh(label, Cr, Cb, map); 
* \endcode
*
* This would successfully segment the red line and green line from the test image 
*
* \image html map.gif "Map image generated for CrCb lookup "
* \image html thresh2.gif "Labelled image generated from lookup of Cr and Cb image"
*
* \warning This function is experimental and may change without notice
*/

int 
PIP_lookup(
	Pimage *label, 		/*!< Output Label Image Pointer */
	Pimage *image1,		/*!< 1st Image Pointer */
	Pimage *image2,		/*!< 2nd Image Pointer */
	Pimage *map  		/*!< Lookup 256x256 map */
	)
    {
    int i, j, k;
    pixel *ip, *i1, *i2, *mp;
    int nc = image1->nc; 
    int nr = image1->nr; 
    int ng = image1->ng; 

    PIPcomment("PIP lookup \n"); 
    if( label->data == NULL) if( PIP_make(label,nc,nr,ng,"Lookup") == PIPFAIL) MEMerr;

    if((ip = (pixel *) label->data) == NULL) NULerr;
    if((i1 = (pixel *) image1->data) == NULL) NULerr;
    if((i2 = (pixel *) image2->data) == NULL) NULerr;
    if((mp = (pixel *) map->data) == NULL) NULerr;
    if( map->nc != 256 || map->nr != 256 ) { PIPerror("Invalid Lookup Map"); return PIPFAIL; }

    for(i=0; i<nr; i++)  
    for(j=0; j<nc; j++) {
      k = i*nc+j;
      ip[k] = mp[i2[k]*256+i1[k]];
      }

    return PIPOK;
    }

/**
* Create Mask Image based upon 2 other images. In theory this could all be done
* with PIP_unary and PIP_maths, but it would be a bit slow. If the mask does not have any
* image, and new image will be created. 
* \warning This function is has been superseeded by PIP_lookup() 
*/

int 
PIP_thresh2(
	Pimage *mask, 		/*!< Output Mask Image Pointer */
	Pimage *image1,		/*!< 1st threshold Image Pointer */
	Pimage *image2,		/*!< 2nd threshold Image Pointer */
	int lower1, 
	int upper1, 
	int lower2, 
	int upper2 
	)	
    {
    int i;
    pixel *ip, *i1, *i2;
    int np = image1->np; 
    int nc = image1->nc; 
    int nr = image1->nr; 
    int ng = image1->ng; 

    PIPcomment("PIP Threshold 2 \n"); 
    if( mask->data == NULL) if( PIP_make(mask,nc,nr,ng,"Thresh2") == PIPFAIL) MEMerr;

    if((ip = (pixel *) mask->data) == NULL) NULerr;
    if((i1 = (pixel *) image1->data) == NULL) NULerr;
    if((i2 = (pixel *) image2->data) == NULL) NULerr;

    for(i=0; i<np; i++) { 
      if( i1[i] > lower1 && i1[i] < upper1 && i2[i] > lower2 && i2[i] < upper2 ) ip[i] = ON;
      else ip[i] = OFF;
      }

    return PIPOK;
    }
