/*********************************************************************
*
* CSIRO Automation
* Queensland Centre for Advanced Technologies
* PO Box 883, Kenmore, QLD 4069, Australia
* www.cat.csiro.au/cmst
*
* Copyright (c) CSIRO Manufacturing Science & Technology
*
*********************************************************************/

/** \file
\brief Image Manipulation functions

One of the most important functions is PIP_unary(), and PIP_maths(). 
Check them out
*/

#include "local.h" 

/** 
Allocate Memory for Pimage
\return 	pointer to image 
\warning   	No memory is allocated for the actual image
\see 		PIP_make()
\see 		PIP_free()
*/

Pimage *
PIP_new()
    {
    Pimage *ip;
    PIPcomment("Create New Image\n");
    if((ip = calloc(1, sizeof( Pimage )) ) == NULL ) NULern;
    return ip ;
    }

/** 
Set Region of Interest of Image 
\return  PIPOK if Ok 
*/

int 
PIP_roi(Pimage *image, 	/*!< Pointer to Image */
	int 	r1, 		/*!< Row lowerbound */
	int 	c1,		/*!< Column lowerbound */
	int   	r2, 		/*!< Row upperbound */
	int	c2)		/*!> Column upperbound */
    {
    PIPcomment("Create ROI \n");
    if( r1 > r2 ) PIPerr("Row lowerbound > upperbound");
    if( c1 > c2 ) PIPerr("Col lowerbound > upperbound");
    image->r1 = r1; image->r2 = r2;
    image->c1 = c1; image->c2 = c2;
    return PIPOK;
    }

/** 
* Create an Edge Region of Interest 
* \return  PIPOK if Ok 
*/

int 
PIP_roi_edge(
	Pimage *image, 		/*!< Pointer to Image */
	int 	edge) 		/*!< Reduction */
    {
    PIPcomment("Create ROI Edge\n");
    if( 2*edge > image->nc ) PIPerr("Image not wide enough");
    if( 2*edge > image->nr ) PIPerr("Image not high enough");
    image->c1 = edge; image->c2 = image->nc-edge;
    image->r1 = edge; image->r2 = image->nr-edge;
    return PIPOK;
    }

/** 
* Create an Mask from the current Region of Interest 
* \param   image Pointer to Image 
* \return  PIPOK if Ok 
* \todo this could be sped up. 
*/

int 
PIP_roi_mask(
	Pimage *image 		
	) 
    {
    int i, nc = image->nc;
    int j, nr = image->nr;

    PIPcomment("Create ROI Mask\n");
    if( image->mask == NULL ) PIP_mask(image);

    for(i=0; i<nc; i++)
    for(j=0; j<nr; j++) {
      if( i >= image->c1 && i <= image->c2 && 
	  j >= image->r1 && j <= image->r2 ) image->mask[j*nc+i] = ON;
       else  image->mask[j*nc+i] = OFF;
       }

    return PIPOK;
    }

/** 
Swap mask and data within an image
\return PIPOK if OK, PIPFAIL if mask is a NULL image
\see PIP_seed()
*/

int 
PIP_swapmask(Pimage *image) /*!< Pointer to Image */
    {
    pixel *ip;
    ip = image->mask;
    if( ip == NULL ) return PIPFAIL;
    image->mask = image->data;
    image->data = ip;
    return PIPOK;
    }

/** 
Copy image mask, from another image
\return  PIPOK if Ok 
*/

int 
PIP_copymask(
	Pimage *image, /*!< Pointer to Image */
	Pimage *mask)	/*!< Pointer to Source of Mask Image */
    {
    int np = image->np;
    if( mask->data == NULL ) PIPerr("There is no mask image");
    if( mask->np != image->np ) PIPerr("Incompatible Image size");
    if( image->mask != NULL ) free(image->mask);
    if((image->mask = malloc(np * sizeof (pixel))) == NULL) MEMerr;
    memcpy( image->mask, mask->data, np);
    return PIPOK;
    }

/** 
Free Pimage Memory 
\return 	void 
\warning   	Image is destroyed 
\see 		PIP_new()
*/

void 
PIP_free(Pimage *ip)	/*!< Pointer to Image */
    {
    if( ip == NULL ) return;
    if( ip->data != NULL ) free(ip->data);
    if( ip->mask != NULL ) free(ip->mask);
    free( ip );
    }

/** 
Initialize new image.

This function initializes the Pimage structure. 
If there is no image data, memory for the image is allocated. 
The size of the image is defined by the width and height of the image. 
All images have a depth of 255, that is, 8 bits.
This corresponds to the data structure of unsigned char. 

\code
Pimage *ip;
ip = PIP_new();
PIP_init(ip, NULL, 200, 100, "test image");
PIP_info(ip);
\endcode

\warning Only 255 grey level depth is currently supported.
\return	PIPOK if Succussful, PIPFAIL if not 
*/

int 
PIP_init(
  Pimage 	*image,		/*!< Pointer to Pimage Data structure */
  pixel		*data,		/*!< Pointer to image data */
  int 		width, 		/*!< Width of Image (columns) */
  int 		height, 	/*!< Height of Image in Pixels (rows) */
  char 		*name)		/*!< Name associated with Image */
    {
    int np = width*height; 

    if( name != NULL ) image->name = strdup(name);
    else image->name = strdup("Unspecified");

    if( data != NULL) image->data = data;
    else if((image->data = (pixel *) calloc(np, sizeof(pixel)))== NULL) MEMerr; 

    image->ng = 255;    image->np = np; image->mask = NULL;
    image->nc = width;  image->c1 = 0;  image->c2 = width;
    image->nr = height; image->r1 = 0;  image->r2 = height;
    image->sec = 0;     image->nsec = 0; 

    PIPcomment("Init (%d %d) %s\n", width, height, image->name);
    return PIPOK;
    }

/** 
Make new image.

This function allocates memory for the Pimage structure. The size of
the image is defined by the width and height of the image. The depth of
the image is not used. All images have a depth of 255, that is, 8 bits.
This corresponds to the data structure of unsigned char. 

\code
Pimage *ip;
ip = PIP_new();
PIP_make(ip, 200, 100, 255, "test image");
PIP_info(ip);
\endcode

\bug   	Only 255 grey level depth is currently supported.
\warning Image data is reallocated 
\return	PIPOK if Succussful, PIPFAIL if not 
\see	PIP_mask()
\see	PIP_init()
*/

int 
PIP_make(
  Pimage 	*image,		/*!< Pointer to Pimage Data structure */
  int 		width, 		/*!< Width of Image (columns) */
  int 		height, 	/*!< Height of Image in Pixels (rows) */
  int 		depth, 		/*!< Depth of Image (pixel depth) */
  char 		*name)		/*!< Name associated with Image */
    {
    int np = width*height; 

    if( image->data == NULL) 
      if((image->data = (pixel *) calloc(np, sizeof(pixel))) == NULL) MEMerr; 
    if( name != NULL ) image->name = strdup(name);
    else image->name = strdup("Unknown");
    image->ng = depth;  image->np = np; image->mask = NULL;
    image->nc = width;  image->c1 = 0;  image->c2 = width;
    image->nr = height; image->r1 = 0;  image->r2 = height;
    image->sec = 0;     image->nsec = 0; 

    PIPcomment("Make (%d %d %d) %s\n", width, height, depth, image->name);
    return PIPOK;
    }

/**
* Allocate memory to Pimage's Mask
* To reduce memory usage, memory is not allocated to the mask image 
* with PIP_make(). In this function, the size of the current image is
* used to allocate a mask image. By default the mask image contains 0's
*/

int 
PIP_mask(
  	Pimage *image)		/*!< Pointer to Pimage Data structure */
    {
    PIPcomment("Allocate memory for mask %s\n", image->name);
    if( image->mask != NULL) free(image->mask);
    if((image->mask = (pixel *) calloc(image->np, sizeof(pixel)))==NULL) MEMerr;
    return PIPOK;
    }

/**
* Copy one image onto another
* @return PIPOK if successful
* @see PIP_maths()
* \todo check to see if the mask stuff is OK.
*/

int 
PIP_copy(
	Pimage *output,		/*!< Pointer to output image */
	Pimage *input) 		/*!< Pointer to input image */
    {
    if( PIP_make(output,input->nc,input->nr,input->ng,"Copied") == PIPFAIL ) 
      PIPerr("Cannot Make Image");
    if( input->data != NULL ) memcpy(output->data, input->data, input->np);
    if( input->mask != NULL ) memcpy(output->mask, input->mask, input->np);
    output->sec = input->sec; 
    output->nsec = input->nsec;
    output->c1 = input->c1; output->c2 = input->c2;
    output->r1 = input->r1; output->r2 = input->r2;
    return PIPOK;
    }

/**
Display Image Information in comment field
\return PIPOK if successful
\see PIPcomment()
*/

int 
PIP_info( Pimage *ip ) /*!< Pointer to Image */
    {
    if( ip->name == NULL) PIPcomment("Image Name is NULL\n");
    else PIPcomment("Name %s\n", ip->name);
    if( ip->data == NULL) PIPcomment("Image Data is NULL\n");
    if( ip->mask == NULL) PIPcomment("Image Mask is NULL\n");
    PIPcomment("Width %d Height %d Depth %d\n",ip->nc,ip->nr,ip->ng);
    PIPcomment("ROI  (%d,%d) (%d,%d)\n", ip->r1,ip->c1,ip->r2,ip->c2) ;
    PIPcomment("Time (%d:%d)\n", ip->sec,ip->nsec) ;
    return PIPOK;
    }

/** 
\image html invert.gif "Invert image with unary operator (i)"
\image html thresh.gif "Threshold image with unary operator (>128)"
Perform unary mathematical operations on image.

This function performs a unary operation upon each pixel in the
input image. The type of operation is controlled with op.

- o If pixel is ON then add value (Offset) 
- ! If pixel is ON then it is set to OFF, and vice versa
- < If pixel is LESS than value then pixel is ON else OFF
- > If pixel is GREATER than value then pixel is ON else OFF 
- = If pixel EQUALS value then pixel is ON else it is set to OFF 
- ^ If pixel EQUALS value then pixel is switched off. 
- ] Shift pixel Value to the right 
- [ Shift pixel Value to the left 
- : Pixel equals Value
- + Value is added to each pixel 
- - Value is subracted from each pixel 
- * Each pixel is muliplied by value
- / Each pixel is divided by value
- p Pixel is set to pixel index (position in vector)
- c Pixel is set to column index (<256) 
- r Pixel is set to row index (<256) 


\code
ip = PIP_new();
PIP_load(ip, "test.pgm");
PIP_unary(ip, '!', 0);
PIP_save(ip, "invert.pgm");
PIP_free(ip);
\endcode
\warning There are no bounds checking on these operations
\see 	 PIP_maths()
*/

int 
PIP_unary(
	Pimage *image, 		/*!< image image to be manipulated */
	char op, 		/*!< operation to be performed */
	int val) 		/*!< value used in operation */
    {
    int i,j;
    pixel *ip;
    int	np = image->np; 
    int	nc = image->nc; 
    int	nr = image->nr; 

    PIPcomment("Unary (op:%c val:%d)\n",op,val); 
    if((ip = image->data) == NULL ) NULerr;

    switch(op)
      {
    case '_': for(i=0;i<np;i++) if(ip[i]>OFF) ip[i] = val; break;
    case 'o': for(i=0;i<np;i++) if(ip[i]>OFF) ip[i] += val; break;
    case '!': for(i=0;i<np;i++) if(ip[i]>OFF) ip[i]=OFF;else ip[i]=ON; break;
    case '<': for(i=0;i<np;i++) if(ip[i]<val) ip[i]=ON; else ip[i]=OFF; break;
    case '>': for(i=0;i<np;i++) if(ip[i]>val) ip[i]=ON; else ip[i]=OFF; break;
    case '=': for(i=0;i<np;i++) if(ip[i]==val) ip[i]=ON;else ip[i]=OFF; break;
    case '^': for(i=0;i<np;i++) if(ip[i]==val) ip[i]=OFF; break;

    case ':': for(i=0;i<np;i++) ip[i] = val; break;
    case 'i': for(i=0;i<np;i++) ip[i] = ON - ip[i]; break;
    case '+': for(i=0;i<np;i++) ip[i] += val; break;
    case '-': for(i=0;i<np;i++) ip[i] -= val; break;
    case '*': for(i=0;i<np;i++) ip[i] *= val; break;
    case '/': for(i=0;i<np;i++) ip[i] /= val; break;

    case ']': for(i=0; i<np; i++) ip[i] = ip[i]>>val; 
	      image->ng = image->ng>>val; break;

    case '[': for(i=0; i<np; i++) ip[i] = ip[i]<<val; 
	      image->ng = image->ng<<val; break;

    case 'p': for(i=0;i<np;i++) ip[i] = (pixel) i; break;

    case 'r': for(i=0;i<nr;i++) for(j=0;j<nc;j++) ip[i*nc+j] = (pixel) i; break;
    case 'c': for(i=0;i<nr;i++) for(j=0;j<nc;j++) ip[i*nc+j] = (pixel) j; break;

    default: PIPerr("Invalid Arguement"); 
      }

    return PIPOK;
    }

/**
Perform Statistical functions on image

Currently the following \c op codes are supported:

- m minimum
- M maximum
- a average

\return Value of statistical operation 
\see PIP_segment_simple() to get median from histogram
*/

int 
PIP_stats(
	Pimage *image, 		/*!< image image to be manipulated */
	char op) 		/*!< operation to be performed */
    {
    int i, m;
    pixel *ip;
    int	np = image->np;

    PIPcomment("Stats (op:%c)\n",op); 
    if((ip = image->data) == NULL ) NULerr;

    switch(op)
      {
      case 'M': m = 0; 
	for(i=0; i<np; i++) if( ip[i] > m ) m = ip[i]; 
	break;

      case 'm': m = 256; 
	for(i=0; i<np; i++) if( ip[i] < m ) m = ip[i]; 
	break;

      case 'a': m = 0; 
	for(i=0; i<np; i++)  m += ip[i]; 
	m /= np;
	break;

      default: PIPerr("Invalid Arguement"); 
      }

    return m;
    }

/**
Perform mathematical operations between two images.
This function performs a mathematical operation between the first and 
second image. The resulting image is placed back into the first image. 
The operation is controlled the the op character.

- & If pixels from both images is ON, then the output is ON 
- ! If a pixel from either images is ON, then the output is ON 
- | If a pixel from either images is OFF, then the output is ON 
- x If pixel in input is ON, then output is OFF (inverse mask)
- m if pixel in input is OFF, then output is OFF (mask)
- o if pixel in input is ON, then output is ON
- = Copy second image onto the first 
- + Add second and first image 
- - Subract second image from the first 
- * Multiply Images 
- / Divide Images 
- > Return Max of two images
- < Return Min of two image

\see PIP_unary()
*/

int 
PIP_maths(
	Pimage *first, 		/*!< image to be manipulated */
	char flag, 		/*!< operation between two images*/
	Pimage *second) 	/*!< source image to be used */
    {
    int i;
    pixel *ip, *op;
    int	np = first->np; 

    PIPcomment("\tMaths (op:%c)\n", flag); 
    if( (ip = first->data) == NULL ) PIPerr("First Image is Empty");
    if( (op = second->data) == NULL ) PIPerr("Second Image is Empty");
    if( first->nc != second->nc ) PIPerr("Different Widths");
    if( first->nr != second->nr ) PIPerr("Different Heighte");

    switch(flag)
      {
      case '&': for(i=0; i<np; i++) 
 	  if( (ip[i]>OFF) && (op[i]>OFF) ) ip[i]=ON; else ip[i]=OFF; break;

      case '|': for(i=0; i<np; i++) 
 	  if( (ip[i]>OFF) || (op[i]>OFF) ) ip[i]=ON; else ip[i]=OFF; break;

      case '!': for(i=0; i<np; i++) 
 	  if( ((ip[i]==OFF) && (op[i]>OFF))
 	   || ((op[i]==OFF) && (ip[i]>OFF)) ) ip[i]=ON; else ip[i]=OFF; break;

      case 'x': for(i=0; i<np; i++) if( op[i] ) ip[i]=OFF; break;
      case 'm': for(i=0; i<np; i++) if( !op[i] ) ip[i]=OFF; break;

      case 'o': for(i=0; i<np; i++) if( op[i] ) ip[i]=ON; break;
      case 'O': for(i=0; i<np; i++) if( !ip[i] ) ip[i]=op[i]; break;

      case '=': for(i=0; i<np; i++) ip[i] = op[i]; break;
      case '+':
				for(i=0; i<np; i++)
					ip[i] = (unsigned	char)CLIP((int)ip[i]+(int)op[i], 0, 255);
				break;
      case '-':
				for(i=0; i<np; i++)
					ip[i] = (unsigned char)CLIP((int)ip[i]-(int)op[i], 0,255);
				break;
      case '*':
				for(i=0; i<np; i++)
					ip[i] = (unsigned char)CLIP((int)ip[i]*(int)op[i], 0, 255);
				break;
      case '/':
				for(i=0; i<np; i++)
					ip[i] = (unsigned char)CLIP((int)ip[i]/(int)op[i], 0, 255);
				break;
      case '>': for(i=0; i<np; i++) ip[i] = MAX(ip[i],op[i]); break;
      case '<': for(i=0; i<np; i++) ip[i] = MIN(ip[i],op[i]); break;

      default: PIPerr("Invalid Arguement"); 
      }

    return PIPOK;
    }

/**
* Function to halve the pixels in both horizontal and vertical direction
*/
 
int 
PIP_halve(
	Pimage *image 		/*!< OLD Image Pointer */
	)	
    {
    int i, j;
    pixel *ip; 
    int nc = image->nc; 
    int nr = image->nr; 

    ip = image->data;
    for(j=0; j<nr; j+=2) 
    for(i=0; i<nc; i+=2)
      *ip++ = image->data[j*nc + i];

    image->nc = nc/2;
    image->nr = nr/2;
    image->np = nr*nc/4;
    return PIPOK;
    }


/**
* Function to double the pixels in both horizontal and vertical direction.
* Both images need to be initialised first.
*/
int 
PIP_double(
   	Pimage *new,
   	Pimage *old
	)
	{

	int i, j;
	pixel *np = new->data; /* Pointer to the new image data */
	pixel *op = old->data; /* Pointer to the old image data */

	if( (old->nr != new->nr/2) || (old->nc != new->nc/2) )
	{
		PIPerr("PIP_double: image sizes not correct");
		return PIPFAIL;
	}

	for(j = 0; j < new->nr; j++)
	{
		for(i = 0; i < new->nc; i++, np++)
		{
	   		*np = *op;
			if(i % 2) op++;
		}
		if((j % 2) == 0) op -= old->nc;
	}
	return PIPOK;
	}

/**
* Function to double the pixels in the horizontal direction
* \todo All functions that change the image size should use another image 
*/

int 
PIP_stretch(
	Pimage *old ,		/*!< OLD Image Pointer */
	Pimage *new 		/*!< NEW Image Pointer */
	)	
    {
    int i;
    pixel *ip, *op;
    int nc = old->nc; 
    int nr = old->nr; 
    int np = old->np; 

    PIPcomment("PIP Stretch\n"); 
    if((ip = (pixel *) old->data) == NULL) NULerr;
    if((op = (pixel *) new->data) == NULL) {
      PIP_make(new, nc*2, nr, 255, "Stretch");
      if((op = (pixel *) new->data) == NULL) NULerr;
      }

    for(i=0; i<np; i++) { 
      *op++ = ip[i]; 
      *op++ = ip[i]; 
      }

    return PIPOK;
    }

/**
* Function to shrink image in horizontal direction
* \todo All functions that change the image size should use another image 
*/

int 
PIP_shrink(
	Pimage *old ,		/*!< OLD Image Pointer */
	Pimage *new 		/*!< NEW Image Pointer */
	)	
    {
    int i;
    pixel *ip, *op;
    int nc = old->nc; 
    int nr = old->nr; 
    int np = old->np; 

    PIPcomment("PIP Shrink\n"); 
    if((ip = (pixel *) old->data) == NULL) NULerr;
    if((op = (pixel *) new->data) == NULL) {
      PIP_make(new, nc/2, nr, 255, "Shrink");
      if((op = (pixel *) new->data) == NULL) NULerr;
      }

    for(i=0; i<np; i+=2) *op++  = (ip[i]+ip[i+1])/2;

    return PIPOK;
    }
/**
Zoom Image in or out in size 
\warning Only zoom-out works at the moment
*/

int 
PIP_zoom(
	Pimage *image, 		/*!< Image Pointer */
	int z)			/*!< Magnification factor */
    {
    int i, j, I, J;
    pixel *ip, *op;
    int NC, nc = image->nc; 
    int NR, nr = image->nr; 

    if( z < 0 ) { NC = - nc / z; NR = - nr /z; }
    else { NC = nc * z, NR = nr * z; }

    PIPcomment("PIP Zoom %d (%d:%d)\n", z,NC,NR); 
    if((ip = (pixel *) image->data) == NULL) NULerr;
    if((op = (pixel *) calloc(NC*NR,sizeof(pixel))) == NULL) MEMerr;

    if( z < 0 ) /* Zoom IN */
      for(I=0; I<NR; I++) 
      for(J=0; J<NC; J++) 
        { i = -I*z; j = -J*z; op[I*NC+J] = ip[i*nc+j]; }

    else /* Zoom OUT */
      for(I=0; I<NR; I++) 
      for(J=0; J<NC; J++) 
        { i = I/z; j = J/z; op[I*NC+J] = ip[i*nc+j]; }

    free(image->data);
    image->data = op;
    image->nc = NC;
    image->nr = NR;
    image->np = NC*NR;
    return PIPOK;
    }

/**
\image html spin.gif "Spinning in clockwise and anticlockwise direction"
Spin Image 90 degrees in clockwise or anti-clockwise direction.
\see PIP_flip()
*/

int 
PIP_spin(
	Pimage *image, 	/*!< Image Pointer */
	char spin 	/*!< Spin direction \b c Clockwise \b a Anticlockwise */
	)
    {
    int i, j;
    pixel *ip, *op;
    int nc = image->nc;
    int nr = image->nr;
    int np = image->np;

    PIPcomment("\tSpin %d\n", spin); 
    if((ip = (pixel *) image->data) == NULL) NULerr;
    if((op = (pixel *) calloc(np,sizeof(pixel))) == NULL) MEMerr;

    switch( spin ) {

      case 'c': case 'C': /* Clockwise */
        for(i=0; i<nr; i++) 
        for(j=0; j<nc; j++) 
          op[j*nr+(nr-1-i)] = ip[i*nc+j]; 
	break;

      case 'a': case 'A': /* Anitclockwise */
        for(i=0; i<nr; i++) 
        for(j=0; j<nc; j++) 
          op[(nc-1-j)*nr+i] = ip[i*nc+j];
	break;

      default: PIPerr("Unknown Spin\n");
      }


    free(ip);
    image->data = op;
    image->nc = nr; 
    image->nr = nc; 
    return PIPOK;
    }

/**
\image html flip.gif "Flip along horizontal and vertical axis"
Flip Image along horizontal axis 
\see PIP_spin()
*/

int 
PIP_flip(
	Pimage *image, 	/*!< Image Pointer */
	char axis	/*!< Flip along \b h Horizontal \b v Vertical axis */
	)
    {
    int i, j;
    pixel *ip, *op;
    int nc = image->nc;
    int nr = image->nr;
    int np = image->np;

    PIPcomment("\tFlip %d\n", axis); 
    if((ip = (pixel *) image->data) == NULL) NULerr;
    if((op = (pixel *) calloc(np,sizeof(pixel))) == NULL) MEMerr;

    switch( axis ) {

      case 'h': case 'H': /* Horizonatal */
        for(i=0;i<nr;i++) 
	for(j=0;j<nc;j++) 
	  op[(nr-i-1)*nc+j] = ip[i*nc+j]; 
	break;

      case 'v': case 'V': /* Vertical */
        for(i=0;i<nr;i++) 
	for(j=0;j<nc;j++) 
	  op[i*nc+(nc-1-j)] = ip[i*nc+j]; 
	break;

      default: PIPerr("Unknown Axis\n");
      }

    free(ip);
    image->data = op;
    return PIPOK;
    }

/**
This function catenates two images together 
*/

int 
PIP_cat(
	Pimage *image,	/*!< Image pointer */
	Pimage *extra,	/*!< Extra Image */
	char side)	/*!< Side of catenation 
	- \b b below 
	- \b a above 
	- \b r right 
	- \b l left 
	- \b i interleave */
    {
    int i, j, np;
    pixel *ip1, *ip2, *op, *new;
    int np1 = image->np;
    int np2 = extra->np;
    np  = np1 + np2;

    PIPcomment("Cat Image %s and %s side %c \n", 
      image->name, extra->name, side);

    /* if( image->nc != extra->nc) PIPerr("Cannot catenate these images"); */

    if((op  = (pixel *) calloc(np,sizeof(pixel))) == NULL) MEMerr;
    if((ip1 = (pixel *) image->data) == NULL) NULerr;
    if((ip2 = (pixel *) extra->data) == NULL) NULerr;
    new = op;

    switch( side ) {

      case 'b': case 'B': /* Below */
        for(i=0; i<np1; i++) *op++ = *ip1++;
        for(i=0; i<np2; i++) *op++ = *ip2++;
        image->nr += extra->nr;
	break;

      case 'a': case 'A': /* Above */
        for(i=0; i<np2; i++) *op++ = *ip2++;
        for(i=0; i<np1; i++) *op++ = *ip1++;
        image->nr += extra->nr;
	break;

      case 'r': case 'R': /* Right */
        for(i=0; i<image->nr; i++) {
          for(j=0; j<image->nc; j++) *op++ = *ip1++;
          for(j=0; j<extra->nc; j++) *op++ = *ip2++; }
        image->nc += extra->nc;
	break;

      case 'l': case 'L': /* Left */
        for(i=0; i<image->nr; i++) {
          for(j=0; j<extra->nc; j++) *op++ = *ip2++; 
          for(j=0; j<image->nc; j++) *op++ = *ip1++; }
        image->nc += extra->nc;
	break;

      case 'i': case 'I': /* Interleave */
        for(i=0; i<image->nr; i++) {
          for(j=0; j<image->nc; j++) *op++ = *ip1++; 
          for(j=0; j<extra->nc; j++) *op++ = *ip2++; }
        image->nr += extra->nr;
	break;

      default: PIPerr("Unknown Side\n");
      }

    free(image->data);
    image->data = new;
    image->np = np;
    return PIPOK;
    }

/**
This function pads an image out with specified colour 
It is useful to use in conjunction with PIP_cat()
\note This could have been done with PIP_make() and PIP_cat();
\code
PIP_pad(ip, 'r', 10, 128);
PIP_cat(ip, other, 'r');
\endcode

*/

int 
PIP_pad(
	Pimage *image,	/*!< Image pointer */
	char side,	/*!< Side of catenation 
			- \b b below 
			- \b a above 
			- \b r right 
			- \b l left */
 	int length,	/*!< The Length/Width of padding */
	pixel colour)	/*!< The Colour of the extension */
    {
    int i, j, np, NP;
    pixel *ip, *op, *new;
    np = image->np;

    PIPcomment("\tPad Image %s on side %c length %d with colour %d \n", 
      image->name, side, length, colour);

    if( side == 'l' || side == 'r' ) NP = length * image->nr;
    else NP = length * image->nc;

    if((op = (pixel *) calloc(np+NP,sizeof(pixel))) == NULL) MEMerr;
    if((ip = (pixel *) image->data) == NULL) NULerr;
    new = op;

    switch( side ) {

      case 'b': case 'B': /* Below */
        for(i=0; i<np; i++) *op++ = *ip++;
        for(i=0; i<NP; i++) *op++ = colour;
        image->nr += length;
	break;

      case 'a': case 'A': /* Above */
        for(i=0; i<NP; i++) *op++ = colour;
        for(i=0; i<np; i++) *op++ = *ip++;
        image->nr += length;
	break;

      case 'r': case 'R': /* Right */
        for(i=0; i<image->nr; i++) {
          for(j=0; j<image->nc; j++) *op++ = *ip++;
          for(j=0; j<length; j++) *op++ = colour; }
        image->nc += length;
	break;

      case 'l': case 'L': /* Left */
        for(i=0; i<image->nr; i++) {
          for(j=0; j<length; j++) *op++ = colour; 
          for(j=0; j<image->nc; j++) *op++ = *ip++; }
        image->nc += length;
	break;

      default: PIPerr("Unknown Side\n");
      }

    free(image->data);
    image->data = new;
    image->np = np+NP;
    return PIPOK;
    }

/**
This function frames the \a image, outside of the ROI in the Pimage structure.
The example below, would generate a block border 10 pixels wide. 
\code 
PIP_load(&image, "test.pgm");
PIP_roi_edge(&image,10);
PIP_frame(&image, (pixel)0);
\endcode
In this example the original and framed image have been catenated with PIP_cat()
\image html frame.gif "Image which has been framed"
*/

int 
PIP_frame( Pimage *image, 	/*!< Image pointer */
	   pixel value)		/*!< Colour of frame */
    {
    pixel *ip;
    int row, col;   
    int nc = image->nc, r1 = image->r1, r2 = image->r2;
    int nr = image->nr, c1 = image->c1, c2 = image->c2;

    PIPcomment("\tFrame Image (%d:%d) (%d:%d)\n", r1,c1,r2,c2);
    if((ip = (pixel *) image->data) == NULL) NULerr;

    for(row=0; row<nr; row++)
    for(col=0; col<nc; col++)
      if( row < r1 || row > r2 || col < c1 || col > c2 )
          ip[row*nc+col] = value;

    return PIPOK;
    }

/**
This function crops the \a image down to row and column window defined
by the ROI in the Pimage structure.
\code 
PIP_load(&image, "test.pgm");
if( image.nc < 20 || image.nr < 20 ) break;
image.r1 = 10; image.r2 = image.nc-10;
image.c1 = 10; image.c2 = image.nr-10;
PIP_crop(&image);
\endcode
*/

int 
PIP_crop( Pimage *image)		/*!< Image pointer */
    {
    int i=0, row, col;   
    pixel *ip, *op;
    int nc = image->nc, r1 = image->r1, r2 = image->r2, NR = r2-r1; 
    int nr = image->nr, c1 = image->c1, c2 = image->c2, NC = c2-c1; 
    int NP = NC * NR; 

    PIPcomment("Crop Image (%d:%d) (%d:%d)\n", r1,c1,r2,c2);
    if( c1 < 0 || c2 > nc || c1 > c2 ) PIPerr("Col is out of range");
    if( r1 < 0 || r2 > nr || r1 > r2 ) PIPerr("Row is out of range");

    if((ip = (pixel *) image->data) == NULL) NULerr;
    if((op = (pixel *) calloc(NP,sizeof(pixel))) == NULL) MEMerr;

    for(row=r1; row<r2; row++)
    for(col=c1; col<c2; col++)
      op[i++] = ip[row*nc+col];

    free(image->data);
    image->data = op;
    image->nc = NC; image->c1 = 0; image->c2 = NC;
    image->nr = NR; image->r1 = 0; image->r2 = NR;
    image->np = NP;
    return PIPOK;
    }

/**
* Deinterlace Image (place the odd image on top of the even)
\image html deinterlace.gif "Interlace and Deinterlaced Image "
\see PIP_cat() 
*/

int PIP_deinterlace(Pimage *image)
    {
    int i, j;   
    pixel *ip, *op, *tmp;
    int nc = image->nc;
    int nr = image->nr;
    int np = image->np; 

    if((ip = (pixel *) image->data) == NULL) NULerr;
    if((op = (pixel *) calloc(np,sizeof(pixel))) == NULL) MEMerr;

    tmp = op;

    for(i=0; i<nr; i+=2) for(j=0; j<nc; j++) *op++ = ip[i*nc+j];
    for(i=1; i<nr; i+=2) for(j=0; j<nc; j++) *op++ = ip[i*nc+j];

    free(image->data);
    image->data = tmp;
    return PIPOK;
    }

/**
* Deinterlace Image (place the odd image on top of the even)
\image html deinterlace.gif "Interlace and Deinterlaced Image "
\see PIP_cat() 
*/

int PIP_deinterlace2(Pimage *image, Pimage *odd, Pimage *even)
    {
    int i, j;   
    pixel *ip, *op, *ep;
    int nc = image->nc;
    int nr = image->nr;

    if((ip = (pixel *) image->data) == NULL) NULerr;
    if((op = (pixel *)   odd->data) == NULL) NULerr;
    if((ep = (pixel *)  even->data) == NULL) NULerr;

    for(i=0; i<nr; i+=2) for(j=0; j<nc; j++) *ep++ = ip[i*nc+j];
    for(i=1; i<nr; i+=2) for(j=0; j<nc; j++) *op++ = ip[i*nc+j];

    return PIPOK;
    }

/**
* Extract Field from mulitiplexed image
* \see PIP_cat() 
*/

int 
PIP_field(
	Pimage *image,          /*!< Pointer to Muliplexed Image */
	int field, 		/*!< Select Field (0) Odd (1) Even */
	Pimage *new)        	/*!< Output Pointer */
    {
    int i, j;
    int nc = image->nc;
    int nr = image->nr;
    pixel *ip;
    
    if( image->data == NULL ) MEMerr;
    PIP_make(new, nc, nr/2, image->ng, "ODD Field");
    ip = new->data;

    for(i=field; i<nr; i+=2)
    for(j=0; j<nc; j++)
      *ip++ = image->data[i*nc+j];

    new->sec  = image->sec;
    new->nsec = image->nsec;
    return PIPOK;
    }

/**
* Untear interlaced image 
* \see PIP_field() 
*/

int 
PIP_untear(
	Pimage *image,          /*!< Pointer to Image */
	int field, 		/*!< Select Field (0) Odd (1) Even */
	int shift) 		/*!< shift index (must be +ve) */
    {
    int i, j;
    int nc = image->nc;
    int nr = image->nr;
    
    if(! image->data) NULerr;
    for(i=field; i<nr; i+=2)
    for(j=0; j<(nc-shift); j++)
      image->data[i*nc+j] = image->data[i*nc+j + shift];

    return PIPOK;
    }
