/*********************************************************************
*
* CSIRO Automation
* Queensland Centre for Advanced Technologies
* PO Box 883, Kenmore, QLD 4069, Australia
* www.cat.csiro.au/cmst
*
* Copyright (c) CSIRO Manufacturing Science & Technology
*
*********************************************************************/

/** \file
\brief Various Grey scale filters

*/

#include "local.h"

/**
Compute the local mean around each pixel using a given window.

In this function each pixel in the image is replaced with the mean value
of the surrounding x y grid of pixels. If x and y equal 3, then a fast
hardcoded window is used, otherwise a sliding window is used. This routine
will be replaced by the convolution routine. 
@see 	PIP_rank()
@return	PIPOK if successful
@param 	image image to be modified
@param	xsize width of window
@param	ysize height of window
*/

int 
PIP_mean(Pimage *image, int xsize, int ysize)
    {
    int i, j, row, col, sum;   
    pixel *ip, *op;
    int nc = image->nc;
    int nr = image->nr;
    int np = image->np;

    PIPcomment("Mean (%d,%d)\n", xsize,ysize);
    if((ip = (pixel *) image->data) == NULL) NULerr;
    if((op = (pixel *) calloc(np,sizeof(pixel))) == NULL) MEMerr;
    if(((xsize&1)==0)||((ysize&1)==0)) PIPerr("Window size must be odd"); 

    if( xsize == 3 && ysize == 3 )
      {
      int pos, add[9];
      if( pip_verbose ) PIPcomment("Use hard coding for 3x3 window\n");

      add[1] = -nc-1; add[2] = -nc; add[3] = -nc+1;
      add[8] = -1;    add[0] = 0;   add[4] = 1;
      add[5] = nc+1;  add[6] = nc;  add[7] = nc-1;

      for(row=1; row<(nr-1); row++)
      for(col=1; col<(nc-1); col++)
        {
        sum = 0;
        pos = row*nc + col; 
        for(i=0; i<9; i++) sum += ip[pos+add[i]]; 
        op[pos] = (pixel) ( sum / 9 );
        }
      }
    else
      {
      int xend, yend, xoff, yoff, half, size;
      if( pip_verbose ) PIPcomment("Use slower variable window\n");

      xend = nc - xsize; xoff = xsize / 2;
      yend = nr - ysize; yoff = ysize / 2;
      half = xsize * ysize / 2;
      size = xsize * ysize;

      for(row=0; row<yend; row++)
      for(col=0; col<xend; col++) 
        {
        sum = 0;
        for(i=0; i<ysize; i++)
        for(j=0; j<xsize; j++)
           sum += ip[(row+i)*nc+col+j];

        op[(row+yoff)*nc+col+xoff] = (pixel) (sum/size);
        }
      }

    /* Swap Image Back */

    free(image->data);
    image->data = op;
    return PIPOK;
    }

/** 
Perform local rank operation on an image.
In this function each pixel in the image is ranked according to the
surrounding grid of pixels. The value that is returned to each 
pixel is controlled by the variable \a type :
- median  returns the median pixel
- minimim returns the minimum pixel
- maximum returns the maximum pixel
- range   returns the range (the difference between min and max)
- variance returns the variance (not working yet)

If x and y equal 3 then the surrounding pixels are ranked with a straight
selection sort, otherwise a sliding window is used and the pixels are ranked
with a histogram. 

@param	image image to be modified
@param	type  type of rank operation
@param	xsize x dimension of window
@param	ysize y dimension of window
@see 	PIP_mean()
@todo	Variance is not working yet
\image html rank_min.gif "Minimum rank operator with 5x5 window"
\image html rank_max.gif "Maximum rank operator with 5x5 window"
\image html rank_med.gif "Median rank operator with 5x5 window"
*/

int
PIP_rank(Pimage *image, char *type, int xsize, int ysize)
    {
    int i, j, k, row, col, level;   
    int half, xend, yend, xoff, yoff;
    pixel *ip, *op;
    int nc = image->nc;
    int nr = image->nr;

    PIPcomment("Rank %s (%d,%d)\n", type,xsize,ysize); 

    /* Calculate Endpoints and Offset */

    if((ip=(pixel *)image->data) == NULL) NULerr
    if((op=(pixel *)malloc(nc*nr*sizeof(pixel)))==NULL) MEMerr;
    if(((xsize&1)==0)||((ysize&1)==0)) PIPerr("Window size must be odd")
    xend = nc - xsize; xoff = xsize / 2;
    yend = nr - ysize + 1; yoff = ysize / 2; 
    half = xsize * ysize / 2;

    switch( type[0] )
      {
      case 'm': case 'M': switch( type[1] ) {
  	case 'i': case 'I': level = 0; break;
        case 'e': case 'E': level = half; break;
        case 'a': case 'A': level = half*2; break;
        default: PIPerr("Unknown Rank Operator");
	} break;

      case 'r': case 'R': level = half*2+1; break;
      case 'v': case 'V': PIPerr("Variance Not Available"); 
      default: PIPerr("Unknown Rank Operator");
      }

    if( pip_verbose ) PIPcomment("level %d \n", level);

    /**
    * If grid is 3x3, then use Rank Pixels with Straight Selection Sort  
    */

    if( xsize == 3 && ysize == 3 )
      {
      int pos, add[9];
      pixel x, a[10];

      add[1] =-nc-1; add[2] =-nc; add[3] =-nc+1;
      add[8] = -1;   add[0] = 0;  add[4] = 1;
      add[5] = nc+1; add[6] = nc; add[7] = nc-1;

      if( pip_verbose ) PIPcomment("Hard Coded 3x3");
      for(row=1; row<(nr-1); row++)
      for(col=1; col<(nc-1); col++)
        {
        pos = row*nc + col; 
        for(i=0; i<9; i++) a[i] = ip[pos+add[i]]; 
  
        for(i=0; i<8; i++) 
	  { 
	  k = i; x = a[i];
	  for(j = i+1; j < 9; j++) if( a[j] < x ) { k = j; x = a[j]; }
	  a[k] = a[i]; a[i] = x;
	  }

        a[9] = a[8] - a[0]; /* To calculate Range */
        op[pos] = (pixel) a[level];
        }

      }

    /**
    * If not then rank pixels with histogram 
    * ( more efficient when the window is large ) 
    */

    else
      {
      unsigned int sum, hist[256];           

      if( pip_verbose ) PIPcomment("Sliding Window\n");
      for(row = 0; row < yend; row++)
        {
	/* Clear Histogram */

        for(k = 0; k < 256; k++) hist[k] = 0; 
  
        /* Calculate First Window */
  
        for(i = 0; i < ysize; i++) 
          for(j = 0; j < xsize; j++) 
	    hist[ip[nc*(row+i)+j]]++; 

        sum = 0;
        for(k = 0; k < 256; k++) { sum += hist[k]; if( sum > level ) break; }
        op[(row+yoff)*nc+xoff] = (pixel) k;
  
        /* Slide Window Horizontally */

        for(col = 1; col < xend; col++) 
	  {
          for(i=0; i<ysize; i++)
	     {
 	     k = (row+i)*nc + (col-1);
             hist[ip[k]]--;     		/* Remove tail */
             hist[ip[k+xsize]]++; 	/* Add to front end */
             }

          sum = 0;
          for(k=0; k<256; k++) { sum += hist[k]; if( sum > level ) break; }
          op[(row+yoff)*nc+col+xoff] = (pixel) k;
          }
        }
      }
  
    free(image->data);
    image->data = op;
    return PIPOK;
    }

/**
Performs a local rank operation on an image with kernel
\return PIPOK if successful
*/

int 
PIP_rankk(
	Pimage *image, 		/*!< Pointer to Input Image */
	char *type, 		/*!< Type of rank */
	Pimage *kernel)		/*!< Pointer to kernel image */
    {
    int i, j, k, row, col, level;   
    int half, xend, yend, xoff, yoff;
    pixel *ip, *op, *kp;
    int nc = image->nc;
    int nr = image->nr;
    int xsize = kernel->nc;
    int ysize = kernel->nr;

    PIPcomment("Rank %s (%d,%d)\n", type,xsize,ysize); 

    /* Calculate Endpoints and Offset */

    kp = kernel->data;
    if((ip=(pixel *)image->data) == NULL) NULerr
    if((op=(pixel *)malloc(nc*nr*sizeof(pixel)))==NULL) MEMerr;
    if(((xsize&1)==0)||((ysize&1)==0)) PIPerr("Window size must be odd")
    xend = nc - xsize; xoff = xsize / 2;
    yend = nr - ysize + 1; yoff = ysize / 2; 
    half = xsize * ysize / 2;

    switch( type[0] )
      {
      case 'm': case 'M': switch( type[1] ) {
  	case 'i': case 'I': level = 0; break;
        case 'e': case 'E': level = half; break;
        case 'a': case 'A': level = half*2; break;
        default: PIPerr("Unknown Rank Operator");
	} break;

      case 'r': case 'R': level = half*2+1; break;
      case 'v': case 'V': PIPerr("Variance Not Available"); 
      default: PIPerr("Unknown Rank Operator");
      }

    if( pip_verbose ) PIPcomment("level %d\n", level);

    /* 
    ** Rank Pixels with Straight Selection Sort (Page 66) 
    */

    if( xsize == 3 && ysize == 3 )
      {
      int pos, add[9];
      pixel x, a[10];

      add[1] =-nc-1; add[2] =-nc; add[3] =-nc+1;
      add[8] = -1;   add[0] = 0;  add[4] = 1;
      add[5] = nc+1; add[6] = nc; add[7] = nc-1;

      for(row=1; row<(nr-1); row++)
      for(col=1; col<(nc-1); col++)
        {
        pos = row*nc + col; 
        for(i=0; i<9; i++) a[i] = ip[pos+add[i]]; 
  
        for(i=0; i<8; i++) 
	  { 
	  k = i; x = a[i];
	  for(j = i+1; j < 9; j++) if( a[j] < x ) { k = j; x = a[j]; }
	  a[k] = a[i]; a[i] = x;
	  }

        a[9] = a[8] - a[0]; /* To calculate Range */
        op[pos] = (pixel) a[level];
        }

      }

    /* 
    ** Rank Pixels with histogram ( more efficient when the window is large ) 
    ** It is impossible to use sliding window with kernel.
    */

    else
      {
      unsigned int sum, hist[256];           

      for(row = 0; row < yend; row++)
      for(col = 0; col < xend; col++) 
        {
	/* Clear Histogram */

        for(k = 0; k < 256; k++) hist[k] = 0; 
  
        /* Calculate Window */
  
        for(i = 0; i < ysize; i++) 
          for(j = 0; j < xsize; j++) 
	    if( kp[xsize*i+j] ) hist[ip[nc*(row+i)+j]]++; 

        sum = 0;
        for(k = 0; k < 256; k++) { sum += hist[k]; if( sum > level ) break; }
        op[(row+yoff)*nc+xoff] = (pixel) k;
	}
  
      }
  
    free(image->data);
    image->data = op;
    return PIPOK;
    }

static int MEAN[10]    = {  1, 1, 1, 1, 1, 1, 1, 1, 1,  9 }; 
static int SMOOTH[10]  = {  1, 2, 1, 2, 4, 2, 1, 2, 1, 16 }; 
static int SHARP[10]   = {  1, 1, 1, 1,-7, 1, 1, 1, 1, 15 }; 
static int LAPLACE[10] = { -1,-1,-1,-1, 8,-1,-1,-1,-1, 16 };

/**
\breif Convolve Image with kernel

In this function each pixel in the image is convolved with a 3x3 kernel.
The following kernels have been defined: 

- \b mean - returns the mean of 3x3 grid
- \b smoothing 
- \b laplacian
- \b sharpening
*/

int 
PIP_convolve(
	Pimage *image, 		/*!< Pointer to Input Image */
	char *type)		/*!< Flag type of convolution */
    {
    int row, col, sum, i, j;   
    pixel *ip, *op;
    int nc = image->nc;
    int nr = image->nr;
    int np = image->np;
    int k, nk, *kernel;
    int xsize, ysize; 

    PIPcomment("Convolve (%s)\n", type); fflush(stdout); 

    if((ip = (pixel *) image->data) == NULL) NULerr;
    if((op = (pixel *) malloc(np*sizeof(unsigned char))) == NULL) MEMerr;

    /* 
    ** At the moment these have been hard coded, but in the 
    ** future there is no reason why they cannot be more flexible
    ** to include much larger kernels 
    */

    xsize = 3; 
    ysize = 3; 
    nk = xsize * ysize; 

    switch( type[0] )
      {
      case 'm': case 'M': kernel = MEAN; break;
      case 'l': case 'L': kernel = LAPLACE; break;
      case 's': case 'S': switch( type[1] ) {
	case 'm': case 'M': kernel = SMOOTH; break;
        case 'h': case 'H': kernel = SHARP; break;
	default: PIPerr("Unknown type"); } break;
      default: PIPerr("Unknown type"); 
      }

    /* Calculate Endpoints and Offset */

    if( xsize == 3 && ysize == 3 )
      {
      /* Fast Routine with a hard coded 3x3 window */

      int pos, add[9];

      add[1] = -nc-1; add[2] = -nc; add[3] = -nc+1;
      add[8] = -1;    add[0] = 0;   add[4] = 1;
      add[5] = nc+1;  add[6] = nc;  add[7] = nc-1;
  
      /* Perform mean filtering to image */
      /* Last entry in Kernel should be the total */
  
      for(row=1; row<(nr-1); row++)
      for(col=1; col<(nc-1); col++)
        {
        sum = 0; pos = row*nc + col; 
        for(i=0; i<9; i++) sum += ip[pos+add[i]] * kernel[i]; 
        op[pos] = (pixel) ( sum / kernel[i] );
        }
      }
    else
      {
      /* Slower Routine with a variable window */

      int xend, yend, xoff, yoff, half, size;

      xend = nc - xsize; xoff = xsize / 2;
      yend = nr - ysize + 1; yoff = ysize / 2;
      half = xsize * ysize / 2;
      size = xsize * ysize;

      for(row=0; row<yend; row++)
      for(col=0; col<xend; col++) 
        {
        sum = 0; k = 0;
        for(i=0; i<ysize; i++)
        for(j=0; j<xsize; j++)
           sum += ip[(row+i)*nc+col+j] * kernel[k++];

        op[(row+yoff)*nc+col+xoff] = (pixel) (sum/kernel[k]);
        }
      }

    /* Swap Image Back */

    free(image->data);
    image->data = op;
    return PIPOK;
    }

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#define P1 pos-nc-1 
#define P2 pos-nc 
#define P3 pos-nc+1
#define P4 pos-1
#define P5 pos   
#define P6 pos+1    
#define P7 pos+nc-1  
#define P8 pos+nc  
#define P9 pos+nc+1
#define SQROOT2		1.414213562
#endif

/**
Image Gradient with 3x3 hard coded window
\todo  Fix Prewitt_Gradient 
\todo  Document distance operators 
*/

int 
PIP_gradient(
	Pimage *image, 	/*!< Pointer to Input Image */
	char *type, 	/*!< Type of gradient Operator */
	char *dist)
    {
    int x = 0, y = 0;
    int i, row, col, pos;
    double max, *op; 
    pixel *ip;
    int nc = image->nc;
    int nr = image->nr;
    int np = image->np;
    double dx, dy, dr, g;

    PIPcomment("Gradient : type (%s) dist (%s)\n", type, dist); 
    if((ip = (pixel *) image->data) == NULL) NULerr;
    if((op = (double *) malloc(np*sizeof(double))) == NULL) MEMerr;

    /* Perform mean filtering to image */

    max = 0;
    for(row=1; row<(nr-1); row++)
    for(col=1; col<(nc-1); col++)
      {
      pos = row*nc + col; 

      switch(type[0])
	{
        case 'r': case 'R': /* Roberts Gradient */
	  x =  ip[P5] - ip[P6]; 
	  y =  ip[P5] - ip[P8]; 
	  break;

	case 's': case 'S': /* Sobel Gradient */
	  x =  ip[P3] + 2*ip[P6] + ip[P9] - ip[P1] - 2*ip[P4] - ip[P7];
	  y =  ip[P1] + 2*ip[P2] + ip[P3] - ip[P7] - 2*ip[P8] - ip[P9];
	  break;

#if 0
	case 'p': case 'P': /* Prewitt_Gradient */
  	  x =  ip[col+1] + ip[rw + col+1] + ip[2*rw + col+1] 
	        - (ip[col-1] + ip[rw + col-1] + ip[2*rw + col-1]); 
  	  y =  ip[col-1] + ip[col] + ip[col+1] 
	        - (ip[2*rw+col-1] + ip[2*rw+col] + ip[2*rw+col+1]);
          break

        case: 'i': case 'I': /* Isotropic Gradient */
	  x =  ip[col+1] + SQROOT2*ip[rw + col+1] + ip[2*rw + col+1] -
		  (ip[col-1] + SQROOT2*ip[rw + col-1] + ip[2*rw + col-1]);
	  y =  ip[col-1] + SQROOT2*ip[col] + ip[col+1] -
  	   (ip[2*rw + col-1] + SQROOT2*ip[2*rw + col] + ip[2 * rw + col+1]);
	  break;
#endif
	default: PIPerr("Unknown Gradient Operator");
	}

      switch( dist[0] ) 
	{
	case 'x': case 'X': op[pos] = ABS(x); break;
	case 'y': case 'Y': op[pos] = ABS(y); break;
	case 'c': case 'C': op[pos] = ABS(x) + ABS(y); break;
        case 'e': case 'E': op[pos] = (int) sqrt((double)((x*x)+(y*y))); break;
        case 's': case 'S': op[pos] = (int) (sqrt((double)((x*x)+(y*y)))/1.8); 
	  break;

	case 'r': case 'R': /* The Radial Range Operator */

	  dx = col - image->c1;	     /* Distance from Seed Point */
	  dy = row - image->r1;	
          dr = sqrt((dx*dx)+(dy*dy)); 
	  g  = x * dx/dr + y * dy/dr;

          switch( dist[1] ) {
	    case 'r': op[pos] = dr; break;
	    case 'x': op[pos] = ABS(dx/dr); break;
	    case 'y': op[pos] = ABS(dy/dr); break;
	    case 'X': op[pos] = ABS(x*dx/dr); break;
	    case 'Y': op[pos] = ABS(y*dy/dr); break;
	    case 's': op[pos] = (x*dx/dr > 0) ? x*dx/dr : 0; break;
	    case 't': op[pos] = (y*dy/dr > 0) ? y*dy/dr : 0; break;
	    case 'p': op[pos] = ( g > 0 ) ? g : 0; break;
	    case 'n': op[pos] = ( g < 0 ) ? -g : 0; break;
	    case 'a': op[pos] = ABS(y * dy/dr) + ABS(x * dx/dr); break;
	    case 'g': op[pos] = ABS(g); break;
	    default : PIPerr("Unknown Distance Operator"); 
	    }
	  break;

	default: PIPerr("Unknown Distance Operator");
        }

      if( op[pos] > max ) max = op[pos];
      }

    /* Rescale Image back to Unsigned char */

    if( dist[0] == 's' ) 
      for(i=0; i<np; i++) ip[i] = (pixel) (op[i] < 256) ? op[i] : 255;
    else
      for(i=0; i<np; i++) ip[i] = (pixel) (op[i]/max*255); 
    free(op);

    image->scale = max;
    return PIPOK;
    }

/**
Smooth image into (x,y) rectangles with Average Value or Local Max
*/

int PIP_smooth(Pimage *image, char *type, int xsize, int ysize)
    {
    pixel *ip;
    int nc = image->nc;
    int nr = image->nr;
    int i, j, row, col, val, sum;
    int xend, yend, xoff, yoff, half, size;

    PIPcomment("Smooth (%d,%d)\n", xsize,ysize); 
    if((ip = (pixel *) image->data) == NULL) NULerr;

    xend = nc - xsize; xoff = xsize / 2;
    yend = nr - ysize + 1; yoff = ysize / 2;
    half = xsize * ysize / 2;
    size = xsize * ysize;

    if( type[0] == 'a' || type[0] == 'A'  ) /* Average */
      {
      for(row=0; row<yend; row+=ysize)
      for(col=0; col<xend; col+=xsize)
          {
          sum = 0;
          for(i=0; i<ysize; i++)
          for(j=0; j<xsize; j++)
	     sum += ip[(row+i)*nc+col+j];

          val = (pixel) ( sum / size ) ;
          for(i=0; i<ysize; i++)
          for(j=0; j<xsize; j++)
           ip[(row+i)*nc+col+j] = val;
          }
      }

    if( type[0] == 'm' || type[0] == 'm' ) /* Maximum */
      {
      int max, I, J, subsample[100][100];

      for(J=0,row=0; row<yend; row+=ysize,J++)
      for(I=0,col=0; col<xend; col+=xsize,I++)
        {
        max = 0;
        for(i=0; i<ysize; i++)
        for(j=0; j<xsize; j++)
	  { 
	  val = ip[(row+i)*nc+col+j];
	  if( val > max ) max = val;
	  }
        subsample[I][J] = max;
        }

      for(J=0,row=0; row<yend; row+=ysize,J++)
      for(I=0,col=0; col<xend; col+=xsize,I++)
        {
	val = subsample[I][J];
        for(i=0; i<ysize; i++)
        for(j=0; j<xsize; j++)
           ip[(row+i)*nc+col+j] = val;
	}
      }

    return PIPOK;
    }

/**
Local contrast expansion of image
In this function the image undergoes localized (x.y) contrast expansion.
The expansion is performed by stretching the histogram, where the upper
and lower 1% of the population is ignored. If x and y equal zero then the
expansion is performed globally. 

static int hist_expand(unsigned int *, unsigned int *, int, int); 
*/

static int hist_stretch(unsigned int *, unsigned int *, int, int); 


int PIP_expand(Pimage *image, int xsize, int ysize)
    {
    int i, j, row, col;
    pixel *ip;
    int nc = image->nc;
    int nr = image->nr;
    int np = image->np;
    int xend, yend, xoff, yoff, half, size;
    unsigned int hist[256], temp[256];

    if( xsize == 0 && ysize == 0 ) 
      {
      PIPcomment("Expansion : Global \n"); 
      if((ip = (pixel *) image->data) == NULL) NULerr;

      for(i=0; i<256; i++) hist[i] = 0;
      for(i=0; i<np; i++) hist[ip[i]]++;

      hist_stretch(hist, temp, nc, nr);

      if( pip_debug ) { FILE *fp; fp = fopen("expand.dat","w");
        for(i=0; i<256; i++) fprintf(fp, "%d %d %d\n", i, hist[i], temp[i]);
        fclose(fp); }

      for(i=0; i<np; i++) ip[i] = (pixel) temp[(int)ip[i]]; 

      return PIPOK;
      }

    PIPcomment("Expand (%d,%d)\n", xsize,ysize); 
    if((ip = (pixel *) image->data) == NULL) NULerr;

    /* Calculate Endpoints and Offset */

    xend = nc - xsize; xoff = xsize / 2;
    yend = nr - ysize + 1; yoff = ysize / 2;
    half = xsize * ysize / 2;
    size = xsize * ysize;

    for(row=0; row<yend; row+=ysize)
    for(col=0; col<xend; col+=xsize)
        {
        for(i=0; i<256; i++) hist[i] = 0;

        for(i=0; i<ysize; i++)
        for(j=0; j<xsize; j++)
	   hist[ip[(row+i)*nc+col+j]]++;

        hist_stretch(hist, temp, xsize, ysize);

        for(i=0; i<ysize; i++)
        for(j=0; j<xsize; j++)
           ip[(row+i)*nc+col+j] = temp[ip[(row+i)*nc+col+j]];
        }

    return PIPOK;
    }

/*
Histogram Expansion 
*/

#if HIST_EXPAND
static int 
hist_expand( unsigned int *hist, unsigned int *temp, int xsize, int ysize)
    {
    int  i, lo, hi, klo, khi, th;

    /* Find the histogram boundaries by locating the 1 percent levels */

    th = xsize * ysize * 0.01;
    lo  =  hi = -1;
    klo = khi = 0;
    for (i = 0; i < 256; i++)
        {
        klo += hist[i]; khi += hist[255-i];
        if (klo > th && lo == -1) lo = i;
        if (khi > th && hi == -1) hi = 255-i;
        if (lo != -1 && hi != -1) break;
        }

    /* Check to see if we have a reasonable contrast span. */
    /* If not, re-do the bounds with a zero threshold. */

    if (hi == lo)
        {
        th = 0;
        lo  =  hi = -1;
        klo = khi = 0;
        for (i = 0; i < 256; i++)
            {
            klo += hist[i]; khi += hist[255-i];
            if (klo > th && lo == -1) lo = i;
            if (khi > th && hi == -1) hi = 255-i;
            if (lo != -1 && hi != -1) break;
            }
        } 

    /* If still have a zero span bound, quit */

    if(hi == lo) 
      {
      for(i=0; i<256; i++) temp[i] = hist[i]; 
      return 1; 
      }

    /* Remap the histogram */

    PIPcomment("Low %d High %d\n", lo,hi);
    for(i = 0; i < 256; i++)
      {
      if (i < lo) temp[i] = 0;
      else if (i > hi) temp[i] = 255;
      else temp[i] = 255*(i-lo)/(hi-lo);
      }

    return 1;
    }
#endif /* HIST_EXPAND */

/** 
Enhance visual appearance of image
In this function the image is enhanced by replacing each pixel with
a pixel calculated from the standard deviation of the surrounding 
\a x x \a y grid of pixels.
*/

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#define factor 0.5
#define minstd 2.0
#endif

int 
PIP_enhance(Pimage *image, int xsize, int ysize)
    {
    int row, col, ind, adex, bdex,
        xend, yend, xoff, yoff, size,
        z, h, i, j, k, m;   

    pixel *ip; 

    float  *op, *wn;                
    float  val, sum, avg, rms;
    float  mean, gain, var, std;

    int nc = image->nc;
    int nr = image->nr;
    int np = image->np;

    PIPcomment("Enhance %d %d\n", xsize,ysize); 

    if( (xsize & 1) == 0 ) PIPerr("Window must be odd\n");
    if( (ysize & 1) == 0 ) PIPerr("Window must be odd\n");

    xoff = xsize / 2; xend = nc - xsize;
    yoff = ysize / 2; yend = nr - ysize + 1;
    size = xsize * ysize;

    if((ip = (pixel *) image->data) == NULL) MEMerr;
    if((op = (float *) malloc(np * sizeof(float ))) == NULL) MEMerr;
    if((wn = (float *) malloc(size* sizeof(float))) == NULL) MEMerr;

    /* Compute the overall mean */

    sum = 0; 
    ip = (pixel *) image->data;
    for(i = 0; i < np; i++) { sum += ip[i]; }
    mean = sum / (float)np;

    /* Perform enhancement to the image */

    m = 0; h = 0;

    for(row = 0 ; row < yend; row++) 
      {
      /* Setup the first window for the current row */

      for(i = 0; i < ysize; i++)
      for(j = 0; j < xsize; j++)
	 { wn[i*xsize+j] = (float) ip[nc*(row+i)+j]; }

      /* Get the standard deviation value */

      sum = 0; rms = 0;
      for(ind = 0; ind < size; ind++)
	{ sum += wn[ind]; rms += wn[ind] * wn[ind]; }

      avg = sum / size;
      var = rms / size - avg * avg;
      std = sqrt((double)var);
      if(std < minstd) { gain = factor * mean; } 
      else { gain = factor * mean / std; }
      val = gain *(ip[nc*(row+yoff) + xoff + m++ ] - avg ) + avg;
      op[nc*(row+yoff) + xoff + h++] = val;

      /* Step through the current row */

      for(col = 0 ; col < xend ; col++)
        {
        /* Delete the tail end and add the leading edge */

        for(k = 0; k < ysize; k++)
        for(z = 0; z < xsize; z++)
	  {
          adex = z + k*xsize;
          bdex = z + k*nc + col + row*nc;
          wn[adex] = (float) ip [bdex+1];
          }

        /* Get the variance value */

        sum = 0; rms = 0;
	for(ind = 0; ind < xsize * ysize; ind++)
	  { sum += wn[ind]; rms += wn[ind] * wn[ind]; }

        avg = sum /(xsize * ysize);
        var = rms /(xsize * ysize) - avg * avg;
        std = sqrt((double)var);

        if(std < minstd) { gain = factor * mean; } 
	else { gain = factor * mean / std; }
        val = gain *(ip[nc*(row + yoff) + xoff + m++ ] - avg ) + avg;
        op[nc * (row + yoff) + xoff + h++] = val;
        } 
	
      m = 0; h = 0;
      }

    for(i=0; i<np; i++) ip[i] = (char) op[i];
    free(op); free(wn);
    return PIPOK;
    }

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#define WSHED  	 0
#define INIT 	-1
#define MASK 	-2
#define INQUEUE -3
#define BORDER  -4
#endif

/**
Perform filter scale watersheding on the input image. 
The border flag should be set to true if you want to initialize the 
borders to 0. This may be necessary to stop the watershed from moving 
outside the image. 
*/

int 
PIP_gwatershed(	
	Pimage *image, 		/*!< Image Pointer */
	int cv, 		/*!< Type of Connectivity 4 or 8 */
	int lw)			/*!< Border flag */
    {
    int d[8], label, flag;
    unsigned long H[256], HC[257], *im;
    int hmin, hmax, lower_b;
    int i, j, l, r, t, pos;

    item_str p_num;
    queue_str q;
    pixel *ip;
    short *s1;

    int nc = image->nc; 
    int nr = image->nr; 
    int np = image->np;

    PIPcomment("Grey Watershed (cv:%d lw:%d)\n", cv,lw);
    if((ip = (pixel *)image->data) == NULL) NULerr; 
    if((cv != 4) && (cv != 8)) PIPerr("Illegal connectivity"); 

    /* Neighborhood pixels */

    d[0] = -nc; d[1] = 1; d[2] = nc;
    d[3] = -1;  d[4] = 1-nc; d[5] = 1+nc;
    d[6] = nc-1; d[7] = -1-nc; 

    if(lw) 
      {         
      r = (nr - 2) * nc;
      for(i=1; i<nc-1; i++) { ip[i+nc] = 0; ip[i+r] = 0; }
      for(i=nc; i<np-nc; i+=nc) { ip[i+1] = 0; ip[i+nc-2] = 0; }
      }

    if( pip_verbose ) PIPcomment("Allocating memory for watershed\n");

    if((s1 =(short *) malloc(np * sizeof(short)))==NULL) MEMerr;
    if((im =(unsigned long *)malloc(np*sizeof(long)))==NULL) MEMerr;

    /* output image initialization */

    for(i=0; i<np; i++) s1[i] = INIT;

    /* Mark image frame */

    for(i=0; i<nc; i++) { s1[i] = BORDER; s1[i+np-nc] = BORDER; }
    for(i=0; i<np; i+=nc) { s1[i] = BORDER; s1[i+nc-1] = BORDER; }

    queue_init(&q);            /* initialize pixels' queue */

    flag = 0;
    label = 0;

    if( pip_verbose ) 
      PIPcomment("Beginning of the tri-distributive function\n");

    /* Initialize histogram */

    for(i=0; i<256; i++) { H[i] = 0; HC[i] = 0; }

    hmax = hmin = ip[0];

    for(i = 1; i < nc - 1; i++)
    for(j = 1; j < nr - 1; j++) 
       {
       l = i + j * nc;
       H[ip[l]] += 1;
       if(ip[l] < hmin) hmin = ip[l];
       if(ip[l] > hmax) hmax = ip[l];
       }

    HC[hmin] = 0;
    for(i = hmin + 1; i <= hmax + 1; i++) HC[i] = HC[i - 1] + H[i - 1];

    for(i = 1; i < nc - 1; i++)
    for(j = 1; j < nr - 1; j++) 
       {
       l = i + j * nc;
       im[HC[ip[l]]] = l;
       HC[ip[l]] += 1;
       }

    if( pip_verbose ) 
      PIPcomment("End of the tri-distributive function\n");

    for(i = hmin; i <= hmax; i++) 
	{
        if (i != 0) lower_b = HC[i - 1];
        else lower_b = 0;

        for (j = lower_b; j < HC[i]; j++) {
          l = im[j];
          s1[l] = MASK;
          for (r = 0; r < cv; r++) {
            pos = l + d[r];
            if ((s1[pos] > 0) || (s1[pos] == WSHED)) {
              p_num.address = l;
              queue_add(&q, &p_num);
              s1[l] = INQUEUE;
              }
            }
          }

        while (!queue_empty(&q)) {
          queue_first(&q, &p_num);
          l = p_num.address;
          for (r = 0; r < cv; r++) {
            pos = l + d[r];
            if (s1[pos] > 0) {
              if ((s1[l] == INQUEUE) || ((s1[l] == WSHED) && flag))
                s1[l] = s1[pos];
              else
                if ((s1[l] > 0) && (s1[l] != s1[pos])) {
                  s1[l] = WSHED;
                  flag = 0;
                }
            }
            else
              if (s1[pos] == WSHED) { 
                if (s1[l] == INQUEUE) {
                  s1[l] = WSHED;
                  flag = 1;
                }
              }
              else
                if (s1[pos] == MASK) {
                  s1[pos] = INQUEUE;
                  p_num.address = pos;
                  queue_add(&q, &p_num);
                }
          }
        }

        for (j = lower_b; j < HC[i]; j++) {
          l = im[j];
          if (s1[l] == MASK) {
            s1[l] = ++label;
            p_num.address = l;
            queue_add(&q, &p_num);
            while (!queue_empty(&q)) {
              queue_first(&q, &p_num);
              pos = p_num.address;
              for (r = 0; r < cv; r++) {
                t = pos + d[r];
                if (s1[t] == MASK) {
                  p_num.address = t;
                  queue_add(&q, &p_num);
                  s1[t] = label;
                }
              }
            }
          }
        }

      }
           
    if( pip_verbose ) PIPcomment("Write Image back to Image frame\n");
    for(i=0; i<nc; i++) { s1[i] = WSHED; s1[i+np-nc] = WSHED; }
    for(i=0; i<np; i+=nc) { s1[i] = WSHED; s1[i+nc-1] = WSHED; }
    for(i=0; i<np; i++) { ip[i] = (pixel) s1[i]; }
    free(s1); free(im);
    return 1;
    }

static int hist_equalization(unsigned int *, unsigned int *);
static int hist_stretch(unsigned int *, unsigned int *, int, int); 

/**
Grey scale equalization of histogram
In this function the image undergoes localized (x,y) histogram equalization.
*/

int 
PIP_equalize(Pimage *image, int xsize, int ysize)
    {
    int row, col, pos, i, j, k, m, 
        xoff, yoff, xend, yend;

    unsigned int hist[256];             /* holds histogram of window */
    unsigned int temp[256];             /* holds temp histogram of window */
    pixel *ip, *op; 

    int nc = image->nc;
    int nr = image->nr; 
    int np = image->np;

    PIPcomment("Equalize (%d,%d)\n", xsize,ysize); 

    if( (xsize & 1) == 0 ) PIPerr("Window must be odd");
    if( (ysize & 1) == 0 ) PIPerr("Window must be odd"); 

    xoff = xsize/2; xend = nc - xsize;
    yoff = ysize/2; yend = nr - ysize + 1;

    for(i=0; i<256; i++) hist[i] = 0;

    if((ip = (pixel *) image->data) == NULL) NULerr;
    if((op = (pixel *) malloc(np*sizeof(pixel))) == NULL) MEMerr;

    /* Scan each row of the image, step column by column with the window */

    m = 0; 
    for(row = 0 ; row < yend; row++)
      {
      for(i = 0; i < ysize; i++)
      for(j = 0; j < xsize; j++)
        hist[ip[nc*(row+i)+j]]++;

      hist_equalization(hist, temp); 
      hist_stretch(hist, temp, xsize, ysize); 

      /* Get the middle pixel value and assign that image location
         with the middle pixel value mapped through the histogram */

      pos = nc*(row+yoff)+xoff+m;
      op[pos] = (pixel) temp[ip[pos]];
      m++;

      /* step across the current row with the window */

      for(col = 0 ; col < xend ; col++)
	 {
         /* Adjust the histogram for the next window */
	 /* Delete the lefmost column & Add the rightmost */

         for(k = 0; k < ysize; k++)
	   {
           hist[ip[col + nc*(row+k)]]--;
           hist[ip[col + nc*(row+k) + xsize]]++; 
           }
            
         hist_equalization(hist, temp); 

	 pos = nc*(row+yoff)+xoff+m; 
         op[pos] = (pixel) temp[ip[pos]];
	 m++;
         }

       /* Reset histogram and variables */

       for(i=0; i<256; i++) hist[i] = 0;
       m = 0; 
       }

    free(image->data);
    image->data = op;
    return PIPOK;
    }

/*
** Histogram Stretching 
*/

static int 
hist_stretch(unsigned int *hist, unsigned int *temp, int xsize, int ysize)
    {
    int  i, lo, hi, klo, khi, th;

    /* Find the histogram boundaries by locating the 1 percent levels */

    th = xsize * ysize * 0.01;
    lo  =  hi = -1;
    klo = khi = 0;
    for (i = 0; i < 256; i++)
        {
        klo += hist[i]; khi += hist[255-i];
        if (klo > th && lo == -1) lo = i;
        if (khi > th && hi == -1) hi = 255-i;
        if (lo != -1 && hi != -1) break;
        }

    /* Check to see if we have a reasonable contrast span. */
    /* If not, re-do the bounds with a zero threshold. */

    if (hi == lo)
        {
        th = 0;
        lo  =  hi = -1;
        klo = khi = 0;
        for (i = 0; i < 256; i++)
            {
            klo += hist[i]; khi += hist[255-i];
            if (klo > th && lo == -1) lo = i;
            if (khi > th && hi == -1) hi = 255-i;
            if (lo != -1 && hi != -1) break;
            }
        } 

    /* If still have a zero span bound, quit */

    if(hi == lo) 
      {
      for(i=0; i<256; i++) temp[i] = hist[i]; 
      return 1; 
      }

    /* Remap the histogram */

    for(i = 0; i < 256; i++)
      {
      if (i < lo) temp[i] = 0;
      else if (i > hi) temp[i] = 255;
      else temp[i] = 255*(i-lo)/(hi-lo);
      }

    return 1;
    }

/* Integrate the histogram to get the equalization map */
/* Compute scale factor and Scale the cumulative distribution */

static int 
hist_equalization( unsigned int hist[], unsigned int temp[])
    {
    int  i;
    float scale;
    unsigned int map[256], k=0;

    for(i=0; i<256; i++) { k += hist[i]; map[i] = k ; }
    if( map[255] == 0.0 ) { for(i=0; i< 256; i++) temp[i]=hist[i]; return 1; }
    scale = 255.0/map[255];
    for(i=0; i<256; i++) temp[i] = (unsigned int) ((float) map[i] * scale);
    return 1;
    }
