/*********************************************************************
*
* CSIRO Automation
* Queensland Centre for Advanced Technologies
* PO Box 883, Kenmore, QLD 4069, Australia
* www.cat.csiro.au/cmst
*
* Copyright (c) CSIRO Manufacturing Science & Technology
*
*********************************************************************/

/** \file
\brief File I/O Functions

\todo  It may be nice to enable these functions to detect PGM and PPM

*/

#ifndef DOXYGEN_SHOULD_SKIP_THIS

#include "local.h" 
static int use_popen;	

FILE *
PIPopen(char *file, char *type)
    {
    FILE *fp = NULL;
    char s[80];
    use_popen = 0;

    if( type[0] == 'r' ) {
      if( file == NULL ) 
        { sprintf(s, "Loading Stdin");      fp = stdin; }
      else { if( strstr(file, "gz") != NULL ) 
        { sprintf(s, "gunzip -c %s", file); use_popen = 1; }
      else { if( strstr(file, "gif") != NULL ) 
        { sprintf(s, "giftopnm %s", file);  use_popen = 1; }
      else { if( strstr(file, "jpg") != NULL ) 
        { sprintf(s, "djpeg %s", file);     use_popen = 1; }
      else 
        { sprintf(s, "Loading %s", file);   fp = fopen(file,"r"); }}}}}

    else {
     
      if( file == NULL ) 
        { sprintf(s, "Saving Stdout");            fp = stdout; }
      else { if( strstr(file, "gz") != NULL ) 
        { sprintf(s, "gzip --stdout > %s", file); use_popen = 1; }
      else { if( strstr(file, "gif") != NULL ) 
        { sprintf(s, "ppmtogif > %s", file);      use_popen = 1; }
      else { if( strstr(file, "jpg") != NULL ) 
        { sprintf(s, "cjpeg > %s", file);         use_popen = 1; }
      else 
        { sprintf(s, "Saving %s", file);          fp = fopen(file,"w"); }}}}}

    if( use_popen  ) fp = popen(s,type); 
    if( fp != NULL ) { PIPcomment("%s\n",s); return fp; }

    if( file == NULL ) fprintf(stderr, "Cannot Open STDIO\n"); 
    else if( use_popen ) fprintf(stderr, "Cannot Open Pipe : %s\n", s); 
    else fprintf(stderr, "Cannot Open File : %s\n", file); 
    return NULL; 
    }

void 
PIPclose(FILE *fp)
    {
    if( fp == stdout ) return;
    if( use_popen ) pclose(fp); 
    else fclose(fp); 
    }
#endif

/**
Load grey image described by filename. 
If the image does not exist, memory for the image is allocated, 
the name of the file is stored in the \c image.name and the
the ROI is define to the maximum.  
A number of filter are applied to the file before it is read, based upon the 
content of the filename:
- gz  gunzip is applied 
- gif giftopnm is applied 
- jpg djpeg is applied 
- default save as PGM

The image is loaded from stdin if the filename is NULL

- To load from stdin. \c PIP_load(ip,NULL); 
- To load a simple pgm file. \c PIP_load(ip,"test.pgm");
- To load a jpeg file. \c PIP_load(ip"test.jpg"); 

\warning Memory will not be allocated to the image if the image->data is NOT null 
\bug   Since external filters are used to convert the image file, it is 
essential that these filters are made available. 
There is no way to use the compression filters with stdin and stdout.
This is due to the fact that the filters are based upon strings in
the filename, not on the actual contents of the file. <P>
\see 	PIP_save()
\see 	PIP_make()
\todo	Need to sort out uniform timespec
*/

int 
PIP_load(
    	Pimage *image,		/*!< Pointer to Image */
    	char *filename) 	/*!< Filename of Image */
    {
    char c, comment[80], type[10], *buff;
    int i, np, width, height, depth;
    FILE *fp; int sec = 0, nsec = 0;

    if( ( fp = PIPopen(filename, "r")) == NULL ) return PIPFAIL;
    while( (c = getc(fp)) == '#' ) while( getc(fp) != '\n');
    ungetc(c, fp);	/* Flush Leading Comments */

    if( fscanf(fp,"%s\n", type) != 1) PIPerr("Cannot Read File Type");
    if( type[0] != 'P' ) PIPerr("This is not a PNM file");
    if( type[1] != '5' ) PIPerr("This is not a Raw PGM file");

    /* Flush Trailing Comments - Read Timespec */

    while( (c = getc(fp)) == '#' ) {
      i = 0;
      while( (c = getc(fp)) != '\n') comment[i++] = c;
      comment[i] = '\0';
      if( pip_verbose ) PIPcomment(comment);
      buff = strtok(strdup(comment), " ");
      if (!strcmp(buff, "TIMESPEC")) {
         sec  = atoi(strtok(NULL, " "));
         nsec = atoi(strtok(NULL, " "));
         }
       }
    ungetc(c, fp);

    if( fscanf(fp,"%d", &width)  != 1 ) PIPerr("Image Width");
    if( fscanf(fp,"%d", &height) != 1 ) PIPerr("Image Height");
    if( fscanf(fp,"%d", &depth) != 1 ) PIPerr("Image Depth"); 
    while( getc(fp) != '\n' );

    np = width * height;
    PIPcomment("Width %d Height %d Depth %d\n", width, height, depth);  

    if( filename == NULL ) filename = "stdin";
    if( image->data == NULL ) {
      if( PIP_make(image, width, height, depth, filename) == PIPFAIL)
        PIPerr("Cannot Create Image"); 
	}
    else {
      image->name = strdup(filename);
      image->ng = depth;  image->np = np; image->mask = NULL;
      image->nc = width;  image->c1 = 0;  image->c2 = width;
      image->nr = height; image->r1 = 0;  image->r2 = height;
      image->sec = 0;     image->nsec = 0; 
      }

    i = fread(image->data, sizeof(char), np, fp);
    if( i != np ) PIPcomment("Cannot Read Image (%d) (%d)", np, i);

    /* This needs to be done later because Make resets them */
    if( sec != 0 ) image->sec = sec;
    if( nsec != 0 ) image->nsec = nsec;
    PIPclose(fp);
    return PIPOK;
    }

/**
Load RGB Image 

This function has been enhanced to also read monochrome image.
If this happens only the first image buffer is used. 
The other images should not have valid data.

\see	 	PIP_saveRGB()
\return 	PIPOK if successful
\warning	Does not read Time Yet
*/

int 
PIP_loadRGB(
    Pimage *rgb[], 	/*!< pointer to image array */
    char *filename) 	/*!< Filename of colour image */
    {
    FILE *fp;
    char c, type[10];
    int ppm = 1, width, height, depth, i, np;

    fp = PIPopen(filename, "r");
    if( fp == NULL ) return PIPFAIL;
    while( (c = getc(fp)) == '#' ) while( getc(fp) != '\n');
    ungetc(c, fp);

    if( fscanf(fp,"%s\n", type) != 1)
      { printf("Cannot Read File Type\n"); exit(1); }

    if( type[0] != 'P' ) PIPerr("This is not a PNM file");
    switch( type[1] ) {
      case '5': ppm = 0; break;
      case '6': ppm = 1; break;
      default: PIPerr("This is not a Raw PPM file");
      }

    /* Read the Comment Line */
    while( (c = getc(fp)) == '#' ) while( getc(fp) != '\n');
    ungetc(c, fp);

    if( fscanf(fp,"%d", &width) != 1 )  PIPerr("Image Width"); 
    if( fscanf(fp,"%d", &height) != 1 ) PIPerr("Image Height"); 
    if( fscanf(fp,"%d", &depth) != 1 )  PIPerr("Image Depth"); 
    while( getc(fp) != '\n' );
    PIPcomment("Width %d Height %d Depth %d\n", width, height, depth);  

    if( ppm == 0 ) {
      if( PIP_make(rgb[0], width, height, depth, filename) == PIPFAIL)
        PIPerr("Cannot Create Image");
      fread(rgb[0]->data, sizeof(char), width*height, fp);
      PIPclose(fp);
      return PIPOK;
      }

    if( PIP_make(rgb[0], width, height, depth, filename) == PIPFAIL)
      PIPerr("Cannot Create Red Image");
    if( PIP_make(rgb[1], width, height, depth, filename) == PIPFAIL)
      PIPerr("Cannot Create Green Image"); 
    if( PIP_make(rgb[2], width, height, depth, filename) == PIPFAIL)
      PIPerr("Cannot Create Blue Image");

    np = width * height;
    for(i=0; i<np; i++) 
      {
      if( fread(&rgb[0]->data[i], (size_t) sizeof(char), (size_t) 1, fp) != 1 )
	PIPerr("Cannot Read Red Image");
      if( fread(&rgb[1]->data[i], (size_t) sizeof(char), (size_t) 1, fp) != 1 )
	PIPerr("Cannot Read Green Image");
      if( fread(&rgb[2]->data[i], (size_t) sizeof(char), (size_t) 1, fp) != 1 )
	PIPerr("Cannot Read Blue Image");
      }

    PIPclose(fp);
    return PIPOK;
    }

/**
* Save single image to file
*
* The file type depends on the contents of the \c filename
* - gz 	gunzip is applied 
* - gif giftopnm is applied
* - jpg djpeg is applied 
* - default save as PGM file
*
* If the filename is NULL then write PGM image to stdout
* To save a simple pgm file PIP_save("test.pgm", &image); 
* To save to stdout  PIP_save(NULL, &image); 
* To save a jpeg file  PIP_save_rgb("test.jpg", &red, &rgb[1], &rgb[2]);
*
* @see 		PIP_load()
* @return 	PIPOK if successful
* @bug
* Since external filters are used to convert the image file, it is 
* essential that these filters are made available. 
* There is no way to use the compression filters with stdin and stdout.
* This is due to the fact that the filters are based upon strings in
* the filename, not on the actual contents of the file. 
*/

int 
PIP_save(
    Pimage *image,  /*!< pointer image to be saved */
    char *filename) /*!< name of file to be saved */
    {
    FILE *fp;
    if( (fp = PIPopen(filename,"w")) == NULL ) 
      PIPerr("Cannot open file");
    fprintf(fp,"P5\n"); 
    fprintf(fp,"#%s\n", image->name); 
    if( image->sec != 0 ) 
      fprintf(fp,"#TIMESPEC %d %d\n", image->sec, image->nsec);
    fprintf(fp,"%d %d %d\n", image->nc, image->nr, image->ng);
    fwrite(image->data, sizeof(char), image->nc*image->nr,fp);
    PIPclose(fp); 
    return PIPOK;
    }

/**
* Save a raw ppm to file from Pimage array
* @return	PIPOK if successful
* @see		PIP_loadRGB()
* @warning 	No Timespec Yet
*/

int 
PIP_saveRGB(
	Pimage *rgb[], 		/*!< image plane array */
	char *filename) 	/*!< filename name of file to be saved */
    {
    FILE *fp;
    int i, np = rgb[0]->np;

    fp = PIPopen(filename, "w");
    fprintf(fp,"P6\n%d %d\n%d\n", rgb[0]->nc, rgb[0]->nr, rgb[0]->ng);
    for(i=0; i<np; i++) {
      fwrite(&rgb[0]->data[i], (size_t) sizeof(char), (size_t) 1, fp);
      fwrite(&rgb[1]->data[i], (size_t) sizeof(char), (size_t) 1, fp);
      fwrite(&rgb[2]->data[i], (size_t) sizeof(char), (size_t) 1, fp);
      }
    PIPclose(fp);
    return PIPOK;
    }
