/*********************************************************************
*
* CSIRO Automation
* Queensland Centre for Advanced Technologies
* PO Box 883, Kenmore, QLD 4069, Australia
* www.cat.csiro.au/cmst
*
* Copyright (c) CSIRO Manufacturing Science & Technology
*
*********************************************************************/

/** \file
\brief General Drawing Utilities 

This functions were pinched from PNMX.
There needs to be a better explaination of lists. 
*/

#include "local.h" 

/**
* This functions draw a pixel on the target image
*/

int
PIP_DrawPixel(
	Pimage *image, 	/*!< Target image */
	int x, 		/*!< X pixel coordinate */
	int y, 		/*!< Y pixel coordinate */
	int colour)	/*!< Pixel Colour (0 to 255) */
    {
    if( x < 0 || x > image->nc ) PIPerr("X is out of Image");
    if( y < 0 || y > image->nr ) PIPerr("Y is out of Image");
    image->data[y*image->nc+x] = (pixel) colour;
    return PIPOK; 
    }

/**
* This function draws an ellipse on the target image 
*/

int
PIP_DrawEllipse(
	Pimage *image, 	/*!< Pointer to target image */
    	int X, 		/*!< Origin of Ellipse in X axis */
	int Y, 		/*!< Origin of Ellipse in Y axis */
	int a, 		/*!< Width (X axis) */
	int b, 		/*!< Length (Y axis) */
	int colour)	/*!< Colour (0 to 255) of index */
    {
    pixel *ip, c; 
    int x, y, d1, d2, nc;
    int t1, t2, t3, t4, t5, t6, t7, t8, t9;

    c = (pixel) colour;
    ip = image->data;
    nc = image->nc;

    x  = a; 	y = 0;
    t1 = a*a;	t2 = 4*t1; 	t3 = 2*t2;
    t4 = b*b;	t5 = 4*t4;	t6 = 2*t5;
    t7 = a*t5;	t8 = 2*t7;	t9 = 0;
    d1 = t2 - t7 + t4;
    d2 = t1 - t8 + t5;

    while( d2<0 ) {
      ip[(Y+y)*nc+(X+x)] = c;
      ip[(Y-y)*nc+(X+x)] = c;
      ip[(Y+y)*nc+(X-x)] = c;
      ip[(Y-y)*nc+(X-x)] = c;
      y++; t9 = t9 + t3;
      if( d1 < 0 ) { 
        d1 = d1 + t9 + t2; 
	d2 = d2 + t9; }
      else { x--; t8 = t8 - t6;
        d1 = d1 - t8 + t9 + t2;
	d2 = d2 - t8 + t9 + t5;
	}
      }

    while(1) {
      ip[(Y+y)*nc+(X+x)] = c;
      ip[(Y-y)*nc+(X+x)] = c;
      ip[(Y+y)*nc+(X-x)] = c;
      ip[(Y-y)*nc+(X-x)] = c;
      x--; t8 = t8 - t6;
      if( d2 < 0 ) {
	y++; 
	t9 = t9 + t3;
	d2 = d2 - t8 + t5 + t9;
	}
      else
	{
	d2 = d2 - t8 + t5;
	}
      if( x < 0 ) break;
      }

    return PIPOK;
    }
 
/**
* This function draws a line on the target image
*/

int
PIP_DrawLine(
	Pimage *image, 		/*!< Pointer to Target Image */
	int x1, 		/*!< X lowerbound of line */
	int y1, 		/*!< Y lowerbound of line */
	int x2, 		/*!< X upperbound of line */
	int y2, 		/*!< X upperbound of line */
	int colour)		/*!< Colour of line (0 to 255) */
    {
    pixel *ip; 
    int i, a, b, c;
    int x, xd, dx, y, yd, dy;
    int nc = image->nc;
    ip = image->data;
  
    PIPcomment("Draw line %d:%d %d:%d %d\n", x1,y1,x2,y2,colour);
    x = x1; if( x2 > x1 ) { xd = x2-x1; dx = 1; } else { xd = x1-x2; dx = -1; }
    y = y1; if( y2 > y1 ) { yd = y2-y1; dy = 1; } else { yd = y1-y2; dy = -1; }

    if( xd > yd ) {
      a = 2*yd; b = a - xd; c = b - xd;
      for(i=0; i<=xd; i++) {
        ip[y*nc+x] = (pixel)colour;
        if( b < 0 ) { b += a; x += dx; }
        else        { b += c; x += dx; y += dy; }
        }
      }
    else {
      a = 2*xd; b = a - yd; c = b - yd;
      for(i=0; i<=yd; i++) {
        ip[y*nc+x] = (pixel)colour;
        if( b < 0 ) { b += a; y += dy; }
        else        { b += c; y += dy; x += dx; }
        }
      }

    /* From PNMX 
    if( x1 == x2 ) { 
      if(y1<y2) for(y=y1; y<=y2; y++) ip[y*nc+x1] = (pixel)colour;
      else      for(y=y2; y<=y1; y++) ip[y*nc+x1] = (pixel)colour; } 
    else { a = (y1-y2)/(x1-x2); b = y1-a*x1;
      if(x1<x2) for(x=x1; x<=x2; x++) ip[(a*x+b)*nc+x] = (pixel)colour;
      else      for(x=x2; x<=x1; x++) ip[(a*x+b)*nc+x] = (pixel)colour;
      } */

    return PIPOK;
    }
 
/**
* This function draws a box on the target image 
*/

int
PIP_DrawBox(
	Pimage *image, 	/*!< Image to drawn upon */
	int xmin, 	/*!< Origin of box (Left hand corner) */
	int ymin,	/*!< Origin of box (Top Corner ) */
        int width, 	/*!< Width in pixels */
	int height, 	/*!< Height in pixels */
	int colour)	/*!< Colour (grey scale) of box */
{
    int	x, y, xmax, ymax;
    int nc = image->nc;
    int nr = image->nr;
    pixel *ip; ip = image->data;
 
    xmax = xmin+width;
    ymax = ymin+height;

    if ( ((xmin > nc) || (ymin > nr)) 
      || ((xmin < 0 ) || (ymin < 0 )) 
      || ((xmax > nc) || (ymax > nr)) ) 
	PIPerr("Rectangle co-ords not in image");
 
    for (x=xmin; x<xmax; x++) {
	ip[ymin*nc+x] = (unsigned char)colour;
	ip[ymax*nc+x] = (unsigned char)colour;
	}
 
    for (y=ymin; y<ymax; y++) {
	ip[y*nc+xmin] = (unsigned char)colour;
	ip[y*nc+xmax] = (unsigned char)colour;
	}

    return PIPOK;
    }

/**
* This function fills a box on the target image 
*/

int
PIP_FillBox(
	Pimage *image, 	/*!< Image to drawn upon */
	int xmin, 	/*!< Origin of box (Left hand corner) */
	int ymin,	/*!< Origin of box (Top Corner ) */
        int width, 	/*!< Width in pixels */
	int height, 	/*!< Height in pixels */
	int colour)	/*!< Colour (grey scale) of box */
{
    int	x, y, xmax, ymax;
    int nc = image->nc;
    int nr = image->nr;
    pixel *ip; ip = image->data;
 
    xmax = xmin+width;
    ymax = ymin+height;

    if ( ((xmin > nc) || (ymin > nr)) 
      || ((xmin < 0 ) || (ymin < 0 )) 
      || ((xmax > nc) || (ymax > nr)) ) 
	PIPerr("Rectangle co-ords not in image");
 
    for (x=xmin; x<xmax; x++) 
    for (y=ymin; y<ymax; y++) 
      ip[y*nc+x] = (unsigned char)colour;

    return PIPOK;
    }

/**
* Set Pixels on Image from List 
*/

int
PIP_listset(Pimage *image, int r, int c, int *list, int n, int colour)
    {
    int i, j;
    int offset = r*image->nc + c;
    
    for(i=0; i<n; i++) {
      j = offset + list[i];
      if( j < image->np ) image->data[j] = (pixel) colour;
      }

    return PIPOK;
    }

/**
* Calculate the sum of pixel values from image along list 
*/

double
PIP_listsum(Pimage *image, int r, int c, int *list, int n)
    {
    int i, j;
    int offset = r*image->nc + c;
    double sum = 0;
    
    for(i=0; i<n; i++) {
      j = offset + list[i];
      if( j < image->np ) sum += image->data[j];
      }
    return (sum / (double ) n);
    }
      
/**
* This function actually "draws" the list onto an image 
* \warning This function is redundant  - do not use.
*/

Pimage * 
PIP_list2image(int *list, int n, int colour)
    {
    int i, j; Pimage *ip; ip = PIP_new();
    if( PIP_make(ip, n, 256, 255, "List2Image") ) return NULL;
    for(i=0; i<n; i++)  
      for(j=0; j<list[i]; j++) 
	ip->data[j*n+i] = colour; 
    return ip;
    }

/**
* Get list of pixel values from image 
*/

int *
PIP_listget(Pimage *image, int r, int c, int *list, int n)
    {
    int i, j, *ip;
    int offset = r*image->nc + c;
    
    if( (ip = malloc(n*sizeof(int))) == NULL) MEMern;
    
    for(i=0; i<n; i++)  {
      j = offset + list[i];
      if( j > 0 && j < image->np ) ip[i] = (int) image->data[j];
      else ip[i] = OFF;
      }

    return ip;
    }

/**
* Function to generate a line list
* \todo This function needs a better explaination
*/

int
PIP_Line( 
	int nc,		/*!< Number of Columns of target image */
	int xd,		/*!< X coordinate of endpoint */
	int yd,		/*!< Y coordinate of endpoint */
	int *list)	/*!< Pointer to list that contains index to line */
    {
    int i, a, b, c;
    int dx = 1, dy = 1;
    int n = 0, x = 0, y = 0;

    if( xd < 0 ) { xd = -xd; dx = -1; }
    if( yd < 0 ) { yd = -yd; dy = -1; }

    if( xd > yd ) {
      a = 2*yd; b = a - xd; c = b - xd;
      for(i=0; i<=xd; i++) {
        list[n++] = y*nc+x;
        if( b < 0 ) { b += a; x += dx; }
        else        { b += c; x += dx; y += dy; }
        }
      }
    else {
      a = 2*xd; b = a - yd; c = b - yd;
      for(i=0; i<=yd; i++) {
        list[n++] = y*nc+x;
        if( b < 0 ) { b += a; y += dy; }
        else        { b += c; y += dy; x += dx; }
        }
      }

    /* This does not seem to always work 

    y = 0; 
    r = 2*dy - dx;
    for(x=0; x<=dx; x++) {
      list[i++] = x + y*nc;
      if( r >= 0 ) { y++; r -= 2*dx - 2*dy; }
      else r += 2*dy;
      }
    return i;
    */

    return n;
    }

/**
* Function to generate an Ellipse list 
* \todo This function needs a better explaination
*/

int
PIP_Ellipse(
	int nc, 	/*!< Number of columns in target image */
	int a, 		/*!< Length */
	int b,		/*!< Height */
	int *list) 	/*!< List containing position (index) of ellipse */
    {
    int i = 0;
    int x, y, d1, d2;
    int t1, t2, t3, t4, t5, t6, t7, t8, t9;

    x  = a; 	y = 0;
    t1 = a*a;	t2 = 4*t1; 	t3 = 2*t2;
    t4 = b*b;	t5 = 4*t4;	t6 = 2*t5;
    t7 = a*t5;	t8 = 2*t7;	t9 = 0;
    d1 = t2 - t7 + t4; 
    d2 = t1 - t8 + t5;

    while( d2<0 ) {
      list[i++] = y*nc+x; list[i++] = y*nc-x;
      list[i++] = -y*nc+x; list[i++] = -y*nc-x;
      y++; t9 += t3;
      if( d1 < 0 ) { d1 += t9 + t2; d2 += t9; }
      else { x--; t8 -= t6; d1 += -t8+t9+t2; d2 += -t8+t9+t5; }
      }

    while(1) {
      list[i++] = y*nc+x; list[i++] = y*nc-x;
      list[i++] = -y*nc+x; list[i++] = -y*nc-x;
      x--; t8 -= t6;
      if( d2 < 0 ) { y++; t9 += t3; d2 += -t8+t5+t9; }
      else { d2 += -t8+t5; }
      if( x < 0 ) break;
      }

    return i;
    }

/**
* Fill Image with Linear interpolation image. 
* @warning : This function has not been completed
*/

int PIP_linear(
	Pimage *image, 
	float a, float b, float c, int x, int y, int z)
    {
    int j, i;
    pixel *ip;
    int nc = image->nc;
    int	nr = image->nr;

    PIPcomment("Linear\n"); 
    if((ip = image->data) == NULL ) NULerr;

    a = a/c; b = b/c;
    for(j=0; j<nr; j++)
    for(i=0; i<nc; i++)
       ip[j*nc+i] = (pixel) (z - a*(i-x) - b*(j-y) );

    return PIPOK;
    }

/**
* Backlight correction using:
* 	1) vertical bands 
* 	2) interpolation
*/

int PIP_backlight(Pimage *image, char *type)
    {
    int i, j;
    int nc = image->nc;
    int nr = image->nr;
    pixel *ip;

    PIPcomment("Backlight \n");
    if((ip = (pixel *)image->data) == NULL) NULerr;

    if( type[0] == 'v' || type[0] == 'V' ) /* Vertical */
      {
      pixel val, max, band[1000];	
      PIPcomment("(vertical bands)");

      for(j=0; j<nc; j++) { 
        for(i=0, max=0; i<nr; i++)
	  { val = ip[i*nc+j]; if( val > max ) max = val; }
        band[j] = max;
        }

      for(j=0; j<nc; j++) 
      for(i=0; i<nr; i++) 
	ip[i*nc+j] = 255-band[j]-ip[i*nc+j];
      }

    if( type[0] == 'i' || type[0] == 'I' ) /* Interpolation */ 
      {
      double slope, offset;
      pixel p, val, x1=0, x2=0, y1=0, y2=0;
      PIPcomment("(linear interpolation) ");
  
      for(j=0;j<nc/10;j++) for(i=0;i<nr;i++) { p=ip[i*nc+j]; if(p>x1)x1=p; }
      for(i=0;i<nr/10;i++) for(j=0;j<nc;j++) { p=ip[i*nc+j]; if(p>y1)y1=p; }

      for(j=9*nc/10;j<nc;j++) for(i=0;i<nr;i++) { p=ip[i*nc+j]; if(p>x2)x2=p;}
      for(i=9*nr/10;i<nr;i++) for(j=0;j<nc;j++) { p=ip[i*nc+j]; if(p>y2)y2=p;}

      PIPcomment("Max X1 %d X2 %d ", x1, x2);
      PIPcomment("Max Y1 %d Y2 %d ", y1, y2);

      if( x2 > x1 ) { offset = x2 - x1; slope = -offset / (double) nc; }
      else { offset = 0; slope = (x1-x2) / (double) nc; }

      for(j=0; j<nc; j++) 
        {
        val = (pixel) offset + slope * j;
        for(i=0; i<nr; i++) ip[i*nc+j] += val;
        }
      }

    return PIPOK;
    }
