/*********************************************************************
*
* CSIRO Automation
* Queensland Centre for Advanced Technologies
* PO Box 883, Kenmore, QLD 4069, Australia
* www.cat.csiro.au/cmst
*
* Copyright (c) CSIRO Manufacturing Science & Technology
*
*********************************************************************/

/** \file
\brief Various Connectivity Routines 
*/

#include "local.h"

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#define MINVAL (unsigned short) 0
#define MAXVAL (unsigned short) 32767
#endif

/*
* This pointer is allocated memory the first time it is used
*/

static unsigned short *label_buffer;

/**
* Label Connected Components in descending order of size
* This function labels connected regions. The output is written back
* onto the original binary input image. Since it is possible to have more
* than 256 regions, the output is sorted into largest region first, where
* any region that has an area less than \a min is ignored or if its
* label is greater than 256. Blobs that are on the edge of the image are 
* ignored if the the \a flag is \b TRUE. Function returns the number of
* blobs. 
\return	Number of labelled regions
\image html label.gif "Labelled image (with regions colourmap)"
*/

int 
PIP_label(
	Pimage *image, 		/*!< Pointer to binary image */
	int cv, 		/*!< Connectivity 4 or 8 */
	int min, 		/*!< Minimum size of region */
	int remove_edge)	/*!< If TRUE, remove edge blobs */
    {
    item_str p;
    queue_str q;
    pixel *ip;
    unsigned short *op;
    int i, j, l, r, d[8];
    unsigned nb = 1, ns;
    int max, k, *area, *posn;
    int nc = image->nc;
    int	np = image->np;

    PIPcomment("Label (cv:%d min:%d)\n", cv,min); 
    if((cv != 4) && (cv != 8)) PIPerr("Invalid CV"); 
    if((ip = image->data) == NULL ) NULerr;

    if(label_buffer == NULL) {
      if((label_buffer = (unsigned short *) malloc(np*sizeof(short))) == NULL ) 
	MEMerr; }

    op = label_buffer;

    /* Neighborhood Pixels */

    d[0] = -nc; d[1] =  1; d[4] = 1 - nc; d[5] =  1 + nc; 
    d[2] =  nc; d[3] = -1; d[6] = nc - 1; d[7] = -1 - nc; 

    /* Label the cells that have not been visited with MAXVAL */

    for(i = 0; i < np; i++) 
      if( ip[i] > (pixel) 0 ) op[i] = MAXVAL; 
      else op[i] = MINVAL;

    /* Write 0 around the borders of the image frame 
       To prevent routine moving outside of matrix */

    for(i = 0; i < nc; i++) { op[i] = MINVAL; op[i+np-nc] = MINVAL; }
    for(i = 0; i < np; i+=nc) { op[i] = MINVAL; op[i+nc-1] = MINVAL; }

    queue_init(&q);             /* Initialize pixels' queue */

    for(i = 0; i < np; i++)     /* Start labeling */
      if( op[i] == MAXVAL ) 
	{
        p.address = i; 
        queue_add(&q, &p); 
        op[i] = nb;
        while (!queue_empty(&q)) 
	  {
          queue_first(&q, &p);
          j = p.address;
          for(r = 0; r < cv; r++) 
	    {
            l = j + d[r];
            if( op[l] == MAXVAL ) 
	       { p.address = l; queue_add(&q, &p); op[l] = nb; }
            }
          }
        nb++;
        }

    PIPcomment("Blobs %d\n", nb-1); 
    if( nb < 1 ) PIPerr("There is no particle\n"); 

    /* Sort Areas into descending order */
    /* Allocate memory for area and posn array */

    if((area = (int *) calloc(nb, sizeof(int))) == NULL) MEMerr;;
    if((posn = (int *) calloc(nb, sizeof(int))) == NULL) MEMerr;;
    for(i=0; i<np; i++) area[op[i]]++; /* Sum area of each blob */

    if( pip_debug )
      {
      FILE *fp; fp = fopen("label.dat","w");
      for(i=0; i<nb; i++) fprintf(fp,"%d area %d\n",i,area[i]);
      fclose(fp);
      }

    /* Run around edge and remove blobs by setting area to zero */
    /* It must be 2 pixels in because we have flagged edge pixels */

    if( remove_edge ) {
      j = np-2*nc;

   if( remove_edge & PIP_EDGE_NORTH ) for(i=0; i<nc; i++) area[op[i+nc]] = 0; 
   if( remove_edge & PIP_EDGE_SOUTH ) for(i=0; i<nc; i++) area[op[i+j]] = 0;  
   if( remove_edge & PIP_EDGE_EAST ) for(i=0; i<np; i+=nc) area[op[i+nc-2]] = 0;
   if( remove_edge & PIP_EDGE_WEST ) for(i=0; i<np; i+=nc) area[op[i+1]] = 0; 
      }

    /* Search for largest area, mark its POSN and then clear the area, */
    /* then search for the next largest area, continue until either */
    /* the area is less than or equal to MIN area, or we have 256 blobs  */
    /* The backgound should have the value of 0, therefore we start at i=1 */

    for(i=1; i<256; i++) 
      {
      max = min; k = 0;
      for(j=1; j<nb; j++) if( area[j] > max ) { max=area[j]; posn[j]=i; k=j; }
      if( area[k] <= min ) break; 	/* Break out when area is <= min */
      area[k] = 0;		  	/* Clear Area to work on next one */
      }

    ns = i;

    if( pip_debug )
      {
      FILE *fp; fp = fopen("sorted.dat","w");
      for(i=0; i<nb; i++) if( posn[i]>0 ) fprintf(fp,"%d posn %d\n",i,posn[i]);
      fclose(fp);
      }

    /* Write the data back into Image */

    for(i=0; i<np; i++) ip[i] = (pixel) posn[op[i]];

    /* Free up data */

    free(area);
    free(posn);
    return ns-1;
    }

/**
* Inline labelling 
*/

int PIP_label_inline(Pimage *image, int cv, int min, int remove_edge)
    {
    item_str p;
    queue_str q;
    cell_str *cell;
    pixel *ip;
    unsigned short *op;
    int i, j, l, r, d[8];
    unsigned nb = 1, ns;
    int max, k, *area, *posn;
    int nc = image->nc;
    int	np = image->np;

    PIPcomment("Label Inline (cv:%d min:%d) ", cv,min); 
    if((cv != 4) && (cv != 8)) PIPerr("Invalid CV"); 
    if((ip = image->data) == NULL ) NULerr;

    if(label_buffer == NULL) {
      printf("Create Buffer Memory\n");
      if((label_buffer = (unsigned short *) malloc(np*sizeof(short))) == NULL ) 
	MEMerr;
	}

    op = label_buffer;

    /* Neighborhood Pixels */

    d[0] = -nc; d[1] =  1; d[4] = 1 - nc; d[5] =  1 + nc; 
    d[2] =  nc; d[3] = -1; d[6] = nc - 1; d[7] = -1 - nc; 

    /* Label the cells that have not been visited with MAXVAL */

    for(i = 0; i < np; i++) 
      if( ip[i] > (pixel) 0 ) op[i] = MAXVAL; 
      else op[i] = MINVAL;

    /* Write 0 around the borders of the image frame 
       To prevent routine moving outside of matrix */

    for(i = 0; i < nc; i++) { op[i] = MINVAL; op[i+np-nc] = MINVAL; }
    for(i = 0; i < np; i+=nc) { op[i] = MINVAL; op[i+nc-1] = MINVAL; }

    q.first = (cell_str *) malloc(sizeof(cell_str));
    q.last  = q.first;
    q.first->next = (cell_str *) 0;

    for(i = 0; i < np; i++)     /* Start Labeling */
      if( op[i] == MAXVAL ) 
	{
        p.address = i; 
        q.last->next = (cell_str *) malloc(sizeof(cell_str)); 
        q.last = q.last->next; 
        q.last->item = p; 
        q.last->next = (cell_str *) 0;

        op[i] = nb;
        while( q.first != q.last ) 
	  {
          cell = q.first;
          q.first = q.first->next;
          p = q.first->item;
          free(cell);

          j = p.address;
          for(r = 0; r < cv; r++) 
	    {
            l = j + d[r];
            if( op[l] == MAXVAL ) { 
	      p.address = l; 
              q.last->next = (cell_str *) malloc(sizeof(cell_str)); 
              q.last = q.last->next; 
              q.last->item = p; 
              q.last->next = (cell_str *) 0;
	      op[l] = nb; 
	      }
            }
          }
        nb++;
        }

    PIPcomment("Blobs %d ", nb-1); 
    if( nb < 1 ) PIPerr("There is no particle\n"); 

    /* Sort Areas into descending order */
    /* Allocate memory for area and posn array */

    if((area = (int *) calloc(nb, sizeof(int))) == NULL) MEMerr;;
    if((posn = (int *) calloc(nb, sizeof(int))) == NULL) MEMerr;;
    for(i=0; i<np; i++) area[op[i]]++; /* Sum area of each blob */

    if( pip_debug )
      {
      FILE *fp; fp = fopen("label.dat","w");
      for(i=0; i<nb; i++) fprintf(fp,"%d area %d\n",i,area[i]);
      fclose(fp);
      }

    /* Run around edge and remove blobs by setting area to zero */
    /* It must be 2 pixels in because we have flagged edge pixels */

    if( remove_edge == TRUE ) {
      j = np-2*nc;
      for(i=0; i<nc; i++) { area[op[i+nc]] = 0; area[op[i+j]] = 0; } 
      for(i=0; i<np; i+=nc) { area[op[i+1]] = 0; area[op[i+nc-2]] = 0; } 
      }

    /* Search for largest area, mark its POSN and then clear the area, */
    /* then search for the next largest area, continue until either */
    /* the area is less than or equal to MIN area, or we have 256 blobs  */
    /* The backgound should have the value of 0, therefore we start at i=1 */

    for(i=1; i<256; i++) 
      {
      max = min; k = 0;
      for(j=1; j<nb; j++) if( area[j] > max ) { max=area[j]; posn[j]=i; k=j; }
      if( area[k] <= min ) break; 	/* Break out when area is <= min */
      area[k] = 0;		  	/* Clear Area to work on next one */
      }

    ns = i;
    PIPcomment("Sorted %d\n", ns-1); 

    if( pip_debug )
      {
      FILE *fp; fp = fopen("sorted.dat","w");
      for(i=0; i<nb; i++) if( posn[i]>0 ) fprintf(fp,"%d posn %d\n",i,posn[i]);
      fclose(fp);
      }

    /* Write the data back into Image */

    for(i=0; i<np; i++) ip[i] = (pixel) posn[op[i]];

    /* Free up data */

    free(area);
    free(posn);
    return(ns-1);
    }

static int merge[10000];

/**
* Fast Labelling 
*/

int PIP_labelf(Pimage *image)
    {
    pixel *ip, prev, curr;
    unsigned short *op;
    int i, j, l;
    unsigned nb = 1, ns;
    int min = 0, max, k;
    int *area, *posn;
    int nc = image->nc;
    int	nr = image->nr;
    int	np = image->np;

    PIPcomment("\tLabel Fast "); 
    if((ip = image->data) == NULL ) NULerr;

    if( label_buffer == NULL) {
      printf("Create Buffer Memory\n");
      if((label_buffer = (unsigned short *) malloc(np*sizeof(short))) == NULL ) 
	MEMerr; }

    op = label_buffer;

    /* Labelling - On Pixels are Labelled, Increment nb on edge */

    nb = 1;
    for(i=0; i<nr; i++) {
      prev = 0;
      for(j=0; j<nc; j++) {
        l = i*nc+j; 
	curr = ip[l];
        if( curr ) { if( prev ) op[l] = nb; else op[l] = nb++;  }
        else op[l] = 0;
	prev = curr;
	}
      }

    /* Merging - If both current and previous are ON */

    for(i=0; i<nb; i++) merge[i] = i; 

    for(j=0; j<nc; j++) {
      prev = 0;
      for(i=1; i<nr; i++) {
	curr = op[i*nc+j]; 
        if( curr && prev ) merge[prev] = curr;  
	prev = curr;
	}
      }

    /* Resolve Indirections */

    for(i=0; i<nb; i++) { 
      j = i; while( merge[j] != j ) j = merge[j]; merge[i] = j;
      /* printf("Resolved %d %d\n", i, merge[i]); */
      }

    PIPcomment("Blobs %d ", nb-1); 
    if( nb < 1 ) PIPerr("There is no particle\n"); 

    for(i=0; i<np; i++) op[i] = (unsigned char) merge[op[i]];

    /* Sort Areas into descending order */
    /* Allocate memory for area and posn array */

    if((area = (int *) calloc(nb, sizeof(int))) == NULL) MEMerr; 
    if((posn = (int *) calloc(nb, sizeof(int))) == NULL) MEMerr;
    for(i=0; i<np; i++) area[op[i]]++; /* Sum area of each blob */

    /* Run around edge and remove blobs by setting area to zero */
    /* It must be 2 pixels in because we have flagged edge pixels */
    /* if( remove_edge == TRUE ) {
      j = np-2*nc;
      for(i=0; i<nc; i++) { area[op[i+nc]] = 0; area[op[i+j]] = 0; } 
      for(i=0; i<np; i+=nc) { area[op[i+1]] = 0; area[op[i+nc-2]] = 0; } 
      } */

    /* Search for largest area, mark its POSN and then clear the area, */
    /* then search for the next largest area, continue until either */
    /* the area is less than or equal to MIN area, or we have 256 blobs  */
    /* The backgound should have the value of 0, therefore we start at i=1 */

    for(i=1; i<256; i++) 
      {
      max = min; k = 0;
      for(j=1; j<nb; j++) if( area[j] > max ) { max=area[j]; posn[j]=i; k=j; }
      if( area[k] <= min ) break; 	/* Break out when area is <= min */
      area[k] = 0;		  	/* Clear Area to work on next one */
      }

    ns = i;
    PIPcomment("Sorted %d", ns-1); 

    /* Write the data back into Image */
    for(i=0; i<np; i++) ip[i] = (pixel) posn[op[i]];
    free(area); free(posn); /* Free up data  */ 

    PIPcomment("\n");
    return(ns-1);
    }

static short *data1;
static short *data2;
static int nc, nr;
static float mc, mr;     /* Localition of Centroid */
static int connect = 8;	 /* Connectivity */
static int Color = 1;	 /* Current Colour (blob index) */
static int Size = 0;	 /* Currebt Size of blob */

/** 
* Recursive Labelling - not tested 
*/

void blob(int x, int y, int c)
   {
   if( data1[y*nc+x] == 0 ) return;
   data1[y*nc+x] = 0;		/* Mark as Visited */
   data2[y*nc+x] = c;		/* Mark with Colour */
   mc += x; mr += y; Size++;
   if (x + 1 < nc) blob(x + 1, y, c); if (x > 0) blob(x - 1, y, c);
   if (y + 1 < nr) blob(x, y + 1, c); if (y > 0) blob(x, y - 1, c);
   if (connect == 4) return;
   if ((x + 1 < nc) && (y + 1 < nr)) blob(x + 1, y + 1, c);
   if ((x > 0) && (y + 1 < nr)) blob(x - 1, y + 1, c);
   if ((x + 1 < nc) && (y > 0)) blob(x + 1, y - 1, c);
   if ((x > 0) && (y > 0)) blob(x - 1, y - 1, c);
   }

/**
* Recursive Labelling
*/

int PIP_labelr(Pimage *image, int cv) 
   {
   int i, x, y;
   int np = image->np;

   connect = cv;
   nc = image->nc;
   nr = image->nr;

   data1 = (short *) malloc(np * sizeof( short));
   data2 = (short *) malloc(np * sizeof( short));

   for(i=0; i<np; i++) data1[i] = image->data[i];
   for(i=0; i<np; i++) data2[i] = 0;

   for(y=0; y<nr; y++)
   for(x=0; x<nc; x++)
     if( data1[y*nc+x] )
        {
        Size = 0;
	mc = mr = 0.0; 
	blob(x, y, Color);
        mc /= Size; mr /= Size;
	printf("blob %4d size %4d pos %5d %5d\n",
	   Color, Size, (int) (0.5 + mc), (int) (0.5 + mr)); Color++;
	}

   for(i=0; i<np; i++) image->data[i] = data2[i];
   free(data1);
   free(data2);
   return(PIPOK);
   }

/** 
* This function leaves holes if there max value is greater than value
* It is an Ad Hoc function to remove holes in a binary input image based
* the data in the grey image.  The function calculates the mean value 
* of the grey image in each hole image
* If the mean value is less than value then the hole is removed. 
* @param 	image 	pointer to image to be manuipulated
* @param 	grey 	pointer to reference image
* @param 	value 
* @param 	nholes 
* @return	number of holes removed.
* @see 		PIP_fill()
*/

int PIP_holes(Pimage *image, Pimage *grey, int value, int nholes)
    {
    int i; 
    pixel *ip, *gp;
    int	np = image->np;
    int nr = 0, real[256];

    PIPcomment("Holes (value:%d holes:%d) ", value,nholes); 
    if(nholes > 255) PIPerr("Too many Holes");
    if((ip = image->data) == NULL) NULerr;
    if((gp = grey->data) == NULL) NULerr;

    /* Too start, we assume that all holes are unreal */
    for(i=1; i<=nholes; i++) real[i] = FALSE;

    /* If any pixel is > value then it is real */
    for(i=0; i<np; i++) if( ip[i] && gp[i] > value ) real[ip[i]] = TRUE;
    for(i=1; i<=nholes; i++) if( real[i] ) nr++;

    if(pip_verbose) for(i=1; i<=nholes; i++) PIPcomment("%d %2d", i,real[i]);  

    /* Switch hole OFF if it is unreal */
    for(i=0; i<np; i++) if( !real[ip[i]] ) ip[i] = OFF; 

    PIPcomment(" Real Holes (%d)\n", nr);
    return nr;
    }

/** 
This function fills seeded region. It does this 
by recursively marking areas with the same property.
Pixels that are visited are turned ON, whilst points that were not visited 
(i.e all points that lie inside the blob) are turned OFF. 
 
\note  Use mask image, to track seed
\warning edge points are ignored (i.e switched OFF). 
\see   PIP_label()
\see   PIP_fill()
*/

int PIP_seed(
	Pimage *image, 	/*!< image pointer to image to be filled */
	int cv, 	/*!< Connectivity 4 or 8, if negative swap threshold  */
	int level)	/*!< Threshold Level - automatic if less than zero */
    {
    item_str p;
    queue_str q;
    pixel *ip, *op;
    int i, j, l, d[8];
    int nc = image->nc;
    int np = image->np;
    int sum;
    int lessthan = 1;

    if( cv < 0 ) { cv = -cv; lessthan = 0; }
    if((ip = image->data) == NULL) NULerr;
    if((cv != 4) && (cv != 8)) PIPerr("Invalid CV"); 
    if((op = (pixel *) calloc(np, sizeof(pixel))) == NULL) MEMerr;;

    PIPcomment("Seed (%d:%d) cv:%d level:%d\n",
	image->r,image->c,cv,level); 

    /* Neighborhood Pixels */

    d[0] = -nc; d[1] =  1; d[4] = 1 - nc; d[5] =  1 + nc; 
    d[2] =  nc; d[3] = -1; d[6] = nc - 1; d[7] = -1 - nc; 

    /* Mark Pixels  around the borders of the image frame 
       To prevent routine moving outside of matrix */

    for(i=0; i<nc; i++)   { op[i] = MARK; op[i+np-nc] = MARK; }
    for(i=0; i<np; i+=nc) { op[i] = MARK; op[i+nc-1]  = MARK; }

    /* Use Lowerbound Region of interest as seed */

    j = image->nc * image->r + image->c; 
    p.address = j; op[j] = MARK; 
    if( pip_verbose ) PIPcomment("Starting pixel %d\n", j);

    /* Establish local threshold */

    sum = ip[j];
    for(i=0; i<8; i++) sum += ip[j+d[i]];
    sum /= 9; sum *= 1.3;

    if( level > 0 ) sum = level; 
    if( pip_verbose ) PIPcomment("Local Threshold %d\n", sum);

    queue_init(&q);             /* Initialize pixels' queue */
    queue_add(&q, &p); 

    if( lessthan ) { while( !queue_empty(&q) ) {
      queue_first(&q, &p); j = p.address;
      for(i=0; i<cv; i++) { l = j + d[i];
        if( ip[l] < sum && op[l] != MARK ) 
	  { p.address = l; queue_add(&q, &p); op[l] = MARK; }
        }
      } }
    else { while( !queue_empty(&q) ) {
      queue_first(&q, &p); j = p.address;
      for(i=0; i<cv; i++) { l = j + d[i];
        if( ip[l] > sum && op[l] != MARK ) 
	  { p.address = l; queue_add(&q, &p); op[l] = MARK; }
        }
      } }

    /* There is a better way */
    for(i=0; i<nc; i++)   { op[i] = OFF; op[i+np-nc] = OFF; }
    for(i=0; i<np; i+=nc) { op[i] = OFF; op[i+nc-1]  = OFF; }

    /* This should be moved to another function 

    image->c1 = nc; image->c2 = 0;
    image->r1 = nr; image->r2 = 0;
    for(i=0; i<nc; i++)   
      for(j=0; j<nr; j++)   
        if( op[j*nc+i] ) {
	  n++; c += i; r += j;
	  if( i < image->c1 ) image->c1 = i; if( i > image->c2 ) image->c2 = i;
	  if( j < image->r1 ) image->r1 = j; if( j > image->r2 ) image->r2 = j;
          }

    if( n > 0 ) { image->c = c/n; image->r = r/n; }
    */
    
    free(image->data);
    image->data = op;
    return PIPOK;
    }

/** 
This function fills holes in binary blob. It does this 
by recursively marking empty areas from each corner of the 
image. Once the scan has been completed, all points that were
visited are turned OFF, whilst points that were not visited 
(i.e all points that lie inside the blob)are turned ON. 
An example is shown in Figure \ref fillmask.
 
\image html fill.gif "Filling the Mask Image" \anchor fillmask

\todo  use all points on the perimeter as seeds
\warning edge points are ignored (i.e switched OFF). 
\param image pointer to image to be filled
\param cv    Connectivity 4 or 8
\see   PIP_label()
\see   PIP_absorb()
*/

int PIP_fill(Pimage *image, int cv)
    {
    item_str p;
    queue_str q;
    pixel *ip;
    int i, j, l, r, d[8];
    int nc = image->nc;
    int nr = image->nr;
    int np = image->np;

    PIPcomment("Fill (%d)\n",cv); 
    if((ip = image->data) == NULL) NULerr;
    if((cv != 4) && (cv != 8)) PIPerr("Invalid CV"); 

    /* Neighborhood Pixels */

    d[0] = -nc; d[1] =  1; d[4] = 1 - nc; d[5] =  1 + nc; 
    d[2] =  nc; d[3] = -1; d[6] = nc - 1; d[7] = -1 - nc; 

    /* Mark Pixels around the borders of the image frame 
       To prevent routine moving outside of matrix */

    for(i = 0; i < nc; i++)  { ip[i] = MARK; ip[i+np-nc] = MARK; }
    for(i = 0; i < np; i+=nc) { ip[i] = MARK; ip[i+nc-1] = MARK; }

    queue_init(&q);             /* Initialize pixels' queue */

    for(i=0; i<4; i++ ) {

      switch(i) {
	case 0: l = nc+2; break;
	case 1: l = 2*nc-2; break;
	case 2: l = (nr-2)*nc+2; break;
	case 3: l = (nr-1)*nc-2; break;
	}

      if( ip[l] == MARK ) continue; 

      p.address = l;
      queue_add(&q, &p); 
      ip[l] = MARK; 

      while( !queue_empty(&q) ) 
        {
        queue_first(&q, &p); /* Go the beginning of the Queue */
        j = p.address;	     /* Get Index of starting point */

        for(r = 0; r < cv; r++) { 	/* Look in all direction */
          l = j + d[r];
          if( ip[l] == OFF ) 		/* If OFF, add to queue */
	    { p.address = l; queue_add(&q, &p); ip[l] = MARK; }
          }
        }
      }

    for(i = 0; i < np; i++) 
	if( ip[i] == MARK ) ip[i] = OFF; else ip[i] = ON;

    return PIPOK;
    }

/**
* Absorb Holes in Binary Image
* Absorb small blobs less that \a size in a labelled image. 
* This is complicated by not knowing which blob should be absorbed into what. 
*/

int PIP_absorb(Pimage *image, int nb, int size)
    {
    int i, j, max;
    int nfill, fill[256];
    int **covar;

    if( (covar = PIP_matrix(nb)) == NULL ) PIPerr("Cannot Generate Matrix");
    if( PIP_covar2(image, covar) == PIPFAIL ) 
      PIPerr("Cannot get Covariant Matrix");
    PIPcomment("\tAbsorb (nb:%d size:%d) ", nb,size);

    nfill = 0;
    for(i=0; i<nb; i++) fill[i] = i;
		  
    for(i=1; i<nb; i++)
      if( covar[i][i] < 30 )
        { 
        max = 0; nfill++; 
        for(j=1; j<nb; j++)
          if( j != i )
            if( covar[i][j] > max ) 
               { max = covar[i][j]; fill[i] = j; }
        }

    PIP_free_matrix(covar, nb);
    PIPcomment("nfill:%d\n", nfill);
    if( pip_verbose ) 
      for(i=0; i<nb; i++) 
	PIPcomment("\tBlob %d Fill %d\n", i,fill[i]);
	   
    if( nfill > 0 ) {
      int i, np; pixel *ip;
      ip = image->data; 
      np = image->np;
      for(i=0; i<np; i++) ip[i] = (pixel) fill[ip[i]];
      }

    return nfill;
    }

int PIP_sort(Pimage *image)
    {
    int i, j, k, min = 0, max, np; 
    pixel *ip;
    int nb, area[256], posn[256];

    printf("\tBinary Sort ");

    ip = image->data; 
    np = image->np;

    for(i=0; i<256; i++) { area[i] = 0; posn[i] = 0; }
    for(i=0; i<np; i++) area[ip[i]]++; /* Sum area of each blob */

    /* Search for largest area, mark its POSN and then clear the area, */
    /* then search for the next largest area, continue until either */
    /* the area is less than or equal to MIN area, or we have 256 blobs  */
    /* The backgound should have the value of 0, therefore we start at i=1 */

    for(i=1; i<256; i++) 
      {
      max = min; k = 0;
      for(j=1; j<256; j++) if( area[j] > max ) { max=area[j]; posn[j]=i; k=j; }
      if( area[k] <= min ) break; 	/* Break out when area is <= min */
      area[k] = 0;		  	/* Clear Area to work on next one */
      }

    /* Write the data back into Image */

    nb = i-1;
    printf("nblobs %d\n", nb); 
    for(i=0; i<np; i++) ip[i] = (pixel) posn[ip[i]];
    return nb;
    }

/**
Sorry, but I can't remember what this  function does 
*/

int 
PIP_remove(Pimage *input, int *remove)
    {
    int i;
    pixel *ip;
    int	np = input->np;
    PIPcomment("\tRemove\n"); 
    if( (ip = input->data) == NULL ) PIPerr("Input Image is Empty");
    for(i=0; i<np; i++) if( remove[ip[i]] ) ip[i] = OFF;
    return PIPOK;
    }
