/*********************************************************************
*
* CSIRO Automation
* Queensland Centre for Advanced Technologies
* PO Box 883, Kenmore, QLD 4069, Australia
* www.cat.csiro.au/cmst
*
* Copyright (c) CSIRO Manufacturing Science & Technology
*
*********************************************************************/

#include "local.h" 

/** \file
* \brief Colour functions
* Colour images are not native to PIP. The Pimage structure only contains 
* a single 8bit plane. Colour images are handled with an array of three Pimages.
* Such an array can be loaded with PIP_loadRGB() and saved with PIP_saveRGB(). 
* 
* One very useful utility to assist in the visualization of an image, 
* is to introduce a colourmap, with PIP_loadcmap() and then colourize a
* greyscale image, with PIP_colourize(). A good example of this would
* be the colourization of a labelled image. Once the labelled image has
* been created it can be overlayed back onto the original image, with
* PIP_overlay() .
* 
* RGB colour space is used for computer graphics and imaging. 
* It stands for Red, Green and Blue.  For the most commonly used test signals
* (ie. generated by the Phillips test system) the RGB values only
* have an amplitude of 75% - ie a value of 191. 
* 
* \image html test_pattern.jpg "Here we have a standard test pattern"
* \image html R.jpg " R component"
* \image html G.jpg " G component"
* \image html B.jpg " B component"
*
*/

/**
* RGB to CMY transformation
* \param rgb  Pointer to Image Array 
* CMY is a basic colour space used when applying colours to media printed on a white background
* (such as paper). It is based on the subtractive properties of colour, which is caused by 
* wavelength absorption. For example, the colour yellow appears yellow because objects absorb
* blue, and reflect red and green. 
*/

int 
PIP_RGBtoCMY( Pimage *rgb[])
    { 
    int	i, np = rgb[0]->np;

    PIPcomment("RGB to CMY\n"); 
    if( rgb[0]->data == NULL ) PIPerr("There is no data in Red plane");
    if( rgb[1]->data == NULL ) PIPerr("There is no data in Green plane");
    if( rgb[2]->data == NULL ) PIPerr("There is no data in Blue plane");

    for(i=0; i<np; i++)
      {
      rgb[0]->data[i] = (pixel) 255 - rgb[0]->data[i];
      rgb[1]->data[i] = (pixel) 255 - rgb[1]->data[i];
      rgb[2]->data[i] = (pixel) 255 - rgb[2]->data[i];
      }

    return PIPOK;
    }

/**
* RGB to Intense RGB transformation
* \param rgb  Pointer to Image Array 
* \warning Data is lost in the clipping process
*/

int 
PIP_RGBtoRGBI( Pimage *rgb[])
    { 
    int r, g, b, R, G, B;
    int	i, np = rgb[0]->np;

    PIPcomment("RGB to RGBI\n"); 
    if( rgb[0]->data == NULL ) PIPerr("There is no data in Red plane");
    if( rgb[1]->data == NULL ) PIPerr("There is no data in Green plane");
    if( rgb[2]->data == NULL ) PIPerr("There is no data in Blue plane");

    for(i=0; i<np; i++)
      {
      r = (int) rgb[0]->data[i];
      g = (int) rgb[1]->data[i];
      b = (int) rgb[2]->data[i];
      R = (2*r-g-b); if( R < 0 ) R = 0; if( R > 255 ) R = 255;
      G = (2*g-r-b); if( G < 0 ) G = 0; if( G > 255 ) G = 255;
      B = (2*b-g-r); if( B < 0 ) B = 0; if( B > 255 ) B = 255;
      rgb[0]->data[i] = (pixel) (R);
      rgb[1]->data[i] = (pixel) (G); 
      rgb[2]->data[i] = (pixel) (B);
      }

    return PIPOK;
    }

/**
* RGB to Normalized RGB transformation
* \param rgb  Pointer to Image Array 
* \note The authour found this space useful for the classification of wear particles
*/

int 
PIP_RGBtoRGY( Pimage *rgb[])
    { 
    double r, g, b, y;
    int	i, np = rgb[0]->np;

    PIPcomment("RGB to RGY\n"); 
    if( rgb[0]->data == NULL ) PIPerr("There is no data in Red plane");
    if( rgb[1]->data == NULL ) PIPerr("There is no data in Green plane");
    if( rgb[2]->data == NULL ) PIPerr("There is no data in Blue plane");

    for(i=0; i<np; i++)
      {
      r = (double) rgb[0]->data[i];
      g = (double) rgb[1]->data[i];
      b = (double) rgb[2]->data[i];
      y = (r+g+b);
      if( y == 0.0 ) continue;
      rgb[0]->data[i] = (pixel) (256.0 * r / y);
      rgb[1]->data[i] = (pixel) (256.0 * g / y); 
      rgb[2]->data[i] = (pixel) (y/3.0);
      }

    return PIPOK;
    }

/**
* RGB to HLS transformation
* \param rgb Pointer to 3 Image Planes  
* The HLS (hue, luminosity, Saturation) colour spaces was developed to be more intuitive: 
* - \b H Hue Ranges from 0 to 2PI degrees, with red at 0 degrees
* - \b L Luminosity Ranges from 0 to 255
* - \b S Saturation Ranges from 0 to 1
* In this function the colourspace has been normalized for Pixel represenation (0 to 255).
* \warning There is some debate about the value of Luminosity, 
* some use the average value of R, G and B (which is used here) others use the Maximum. 
* \code 
* Pimage *rgb[3];
* PIP_loadRGB(rgb, "test.ppm");
* PIP_RGBtoHLS(rgb);
* PIP_save(rgb[0], "Hue.pgm");
* PIP_save(rgb[1], "Lum.pgm");
* PIP_save(rgb[2], "Sat.pgm");
* \endcode
*/

int 
PIP_RGBtoHLS( Pimage *rgb[] )
    { 
    double r, g, b, y, a;
    int	i, np = rgb[0]->np;

    PIPcomment("RGB to HLS\n"); 
    if( rgb[0]->data == NULL ) PIPerr("There is no data in Red plane");
    if( rgb[1]->data == NULL ) PIPerr("There is no data in Green plane");
    if( rgb[2]->data == NULL ) PIPerr("There is no data in Blue plane");

    for(i=0; i<np; i++)
      {
      r = (double) rgb[0]->data[i];
      g = (double) rgb[1]->data[i];
      b = (double) rgb[2]->data[i];
      y = (r+g+b)/3.0;
      if( y == 0.0 ) continue;
      a = atan2(1.732*(g-b),g+b-2*r);
      rgb[0]->data[i] = (pixel) 255*(3.124-a)/6.283;
      rgb[1]->data[i] = (pixel) y;
      rgb[2]->data[i] = (pixel) 255*(1-MIN(r,MIN(g,b))/y);
      }

    return PIPOK;
    }



double PIP_colour_min(double a, double b, double c)
{
	if(a < b)
	{
		if(a < c) return a;
		else return c;
	} else
	{
		if(b < c) return b;
		else return c;
	}
}

double PIP_colour_max(double a, double b, double c)
{
	if(a > b)
	{
		if(a > c) return a;
		else return c;
	} else
	{
		if(b > c) return b;
		else return c;
	}
}

/**
* RGB to HSV transformation
* \param rgb Pointer to 3 Image Planes  
* The HSV (hue, saturation, value) colour spaces were developed to be more
* intuitive and approximate the way humans perceive colour: 
* Hue desribes pure colour, red is 0, green 120, blue 240
*/
int 
PIP_RGBtoHSV( Pimage *rgb[] )
    { 
    int	i, np = rgb[0]->np;

	float H,S,V;
	double var_R, var_G, var_B, var_Min, var_Max, del_Max;


	unsigned char *rp = rgb[0]->data;
	unsigned char *gp = rgb[1]->data;
	unsigned char *bp = rgb[2]->data;


    PIPcomment("RGB to HSV\n"); 
    if( rgb[0]->data == NULL ) PIPerr("There is no data in red plane");
    if( rgb[1]->data == NULL ) PIPerr("There is no data in green plane");
    if( rgb[2]->data == NULL ) PIPerr("There is no data in blue plane");

    for(i=0; i<np; i++, rp++, gp++, bp++)
      {
		var_R = ( *rp / 255.0 );
		var_G = ( *gp / 255.0 );
		var_B = ( *bp / 255.0 );

		var_Min = PIP_colour_min( var_R, var_G, var_B );   //Min. value of RGB
		var_Max = PIP_colour_max( var_R, var_G, var_B );   //Max. value of RGB
		del_Max = var_Max - var_Min;            //Delta RGB value
	
		V = var_Max;

		if(del_Max > .0001)                     //Chromatic data...
		{
   			S = del_Max / var_Max;

			if (var_R == var_Max)
       		{
       			H = (var_G - var_B) / del_Max;
				if (H < 0.0) H += 6.0;
       		}
    		else if (var_G == var_Max)
       		{
       			H = 2.0 + (var_B - var_R) / del_Max;
      		}
    		else if (var_B == var_Max)
       		{
       			H = 4.0 + (var_R - var_G) / del_Max;
       		}

    		H /= 6.0;

		} else //This is a gray, no chroma...
		{
   			H = 0;                                //HSV results = 0 ÷ 1
   			S = 0;
		}


      *rp = (pixel) (H * 255);
      *gp = (pixel) (S * 255);
      *bp = (pixel) (V * 255);
      }

    return PIPOK;
    }


/**
* HSV to RGB transformation
*/
int 
PIP_HSVtoRGB( Pimage *rgb[] )
    { 

  	double h, s, v, r, g, b;
    int	i, np = rgb[0]->np;

	int ih;
	double f, w, q, t;

	unsigned char *hp = rgb[0]->data;
	unsigned char *sp = rgb[1]->data;
	unsigned char *vp = rgb[2]->data;

    PIPcomment("HSV to RGB fast\n"); 
    if( rgb[0]->data == NULL ) PIPerr("There is no data in red plane");
    if( rgb[1]->data == NULL ) PIPerr("There is no data in green plane");
    if( rgb[2]->data == NULL ) PIPerr("There is no data in blue plane");

    for(i=0; i<np; i++, hp++, sp++, vp++)
      {
      h = (double) *hp / 255.0;
      s = (double) *sp / 255.0;
      v = (double) *vp / 255.0;


 		if (s == 0.0)
    	{
      		r = v;
      		g = v;
      		b = v;
    	}
  		else
    	{
      		if(h >= 1.0) h = 0.0;

      		h *= 6.0;

	    	ih = (int)h;
      		f = h - ih;
      		w = v * (1.0 - s);
      		q = v * (1.0 - (s * f));
      		t = v * (1.0 - (s * (1.0 - f)));

      		switch (ih)
        	{
        		case 0:
          		r = v;
          		g = t;
          		b = w;
          		break;
        		case 1:
          		r = q;
          		g = v;
          		b = w;
          		break;
        		case 2:
          		r = w;
          		g = v;
          		b = t;
          		break;
        		case 3:
          		r = w;
          		g = q;
          		b = v;
          		break;
        		case 4:
          		r = t;
          		g = w;
          		b = v;
          		break;
        		case 5:
          		r = v;
          		g = w;
          		b = q;
          		break;
        	}
    	}
		*hp = (unsigned char)(r * 255.0);
       	*sp = (unsigned char)(g * 255.0);
      	*vp = (unsigned char)(b * 255.0);
	  }
	return PIPOK;
	}


/**
* RGB to YCrCb transformation 
* YCrCb colour space was developed for digital component video. 
* It is the scaled and offset version of the YUV colour space. 
* \image html Yc.jpg "Yc component"
* \image html Cb.jpg "Cb component"
* \image html Cr.jpg "Cr component"
* \image html YCrCb.jpg "All three as RGB"
* \param rgb Pointer to 3 Image Planes  
* - \b Y has a defined range of 16 to 235
* - \b Cr has a defined range of 16 to 240, where 128 equals Zero (no colour)
* - \b Cb has a defined range of 16 to 240, where 128 equals Zero (no colour)
* Therefore for the standard colour bars:
* - White (180, 128, 128)
* - Yellow (162, 142, 44)
* - Cyan (131, 44, 156)
* - Green (112, 58,72)
* - Magenta (84, 198, 184)
* - Red (65, 212, 100)
* - Blue (35,114, 212)
* - Black (16, 128, 128)
*/

int 
PIP_RGBtoYCrCb_original( Pimage *rgb[] )
    { 
    double r, g, b;
    double Y, Cr, Cb;
    int	i, np = rgb[0]->np;

    PIPcomment("RGB to YCrCb - With range restrictions\n");
    if( rgb[0]->data == NULL ) PIPerr("There is no data in Red plane");
    if( rgb[1]->data == NULL ) PIPerr("There is no data in Green plane");
    if( rgb[2]->data == NULL ) PIPerr("There is no data in Blue plane");

    for(i=0; i<np; i++)
      {
      r = (double) rgb[0]->data[i];
      g = (double) rgb[1]->data[i];
      b = (double) rgb[2]->data[i];
      Y = (( 77*r + 150*g +  29*b) / 256);        if( Y <16 ) Y =16; if( Y >235 ) Y =235;
      Cr = ((131*r - 110*g -  21*b) / 256 + 128); if( Cr<16 ) Cr=16; if( Cr>240 ) Cr=240;
      Cb = ((-41*r -  87*g + 131*b) / 256 + 128); if( Cb<16 ) Cb=16; if( Cb>240 ) Cb=240;
      rgb[0]->data[i] = (pixel) Y;
      rgb[1]->data[i] = (pixel) Cr; 
      rgb[2]->data[i] = (pixel) Cb; 
      }

    return PIPOK;
    }
          
/**
* YCrCb to RGB transformation 
* \param YCrCb Pointer to 3 Image Planes  
*/

int 
PIP_YCrCbtoRGB_original( Pimage *YCrCb[] )
    { 
    double Y, Cr, Cb;
    double R, G, B;
    int	i, np = YCrCb[0]->np;

    PIPcomment("YCrCb to RGB \n"); 
    if( YCrCb[0]->data == NULL ) PIPerr("There is no data in Y plane");
    if( YCrCb[1]->data == NULL ) PIPerr("There is no data in Cb plane");
    if( YCrCb[2]->data == NULL ) PIPerr("There is no data in Cr plane");

    for(i=0; i<np; i++)
      {
      Y  = (double) YCrCb[0]->data[i];
      Cr = (double) YCrCb[1]->data[i]-128;
      Cb = (double) YCrCb[2]->data[i]-128;
      R  = (Y + 1.366*Cr - 0.002*Cb); if( R < 0 ) R = 0; if( R > 255 ) R = 255;
      G  = (Y - 0.700*Cr - 0.334*Cb); if( G < 0 ) G = 0; if( G > 255 ) G = 255;
      B  = (Y - 0.006*Cr + 1.732*Cb); if( B < 0 ) B = 0; if( G > 255 ) B = 255;
      YCrCb[0]->data[i] = (pixel) R;
      YCrCb[1]->data[i] = (pixel) G; 
      YCrCb[2]->data[i] = (pixel) B;
      }

    return PIPOK;
    }

/**
* RGB to YCrCb transformation with extended dynamic range
* This is the transformation used by Frame capture cards
* \param rgb Pointer to 3 Image Planes  
*/

int 
PIP_RGBtoYCrCb( Pimage *rgb[] )
    { 
    double r, g, b;
    double Y, Cr, Cb;
    int	i, np = rgb[0]->np;

    PIPcomment("RGB to YCrCb (Extended range to emulate BT848)\n"); 
    if( rgb[0]->data == NULL ) PIPerr("There is no data in Red plane");
    if( rgb[1]->data == NULL ) PIPerr("There is no data in Green plane");
    if( rgb[2]->data == NULL ) PIPerr("There is no data in Blue plane");

    for(i=0; i<np; i++)
      {
      r = (double) rgb[0]->data[i];
      g = (double) rgb[1]->data[i];
      b = (double) rgb[2]->data[i];
      Y  = ( 0.257*r + 0.504*g + 0.098*b +  16); 
      Cr = ( 0.439*r - 0.368*g - 0.071*b + 128); 
      Cb = (-0.148*r - 0.291*g + 0.439*b + 128); 
      if( Y <0 ) Y =0; if( Y >255 ) Y =255;
      if( Cr<0 ) Cr=0; if( Cr>255 ) Cr=255;
      if( Cb<0 ) Cb=0; if( Cb>255 ) Cb=255;

      rgb[0]->data[i] = (pixel) Y; 
      rgb[1]->data[i] = (pixel) Cr; 
      rgb[2]->data[i] = (pixel) Cb;  
      }

    return PIPOK;
    }

/**
* YCrCb to RGB transformation with extended dynamic range
* This transformation is recommended by BT848 capture cards
* \param YCrCb Pointer to 3 Image Planes  
*/

int 
PIP_YCrCbtoRGB( Pimage *YCrCb[] )
    { 
    double Y, Cr, Cb;
    double R, G, B;
    int	i, np = YCrCb[0]->np;

    PIPcomment("YCrCb to RGB (Extended Range for BT848)\n"); 
    if( YCrCb[0]->data == NULL ) PIPerr("There is no data in Y plane");
    if( YCrCb[1]->data == NULL ) PIPerr("There is no data in Cb plane");
    if( YCrCb[2]->data == NULL ) PIPerr("There is no data in Cr plane");

    for(i=0; i<np; i++)
      {
      Y  = (double) YCrCb[0]->data[i] - 16;
      Cr = (double) YCrCb[1]->data[i] - 128;
      Cb = (double) YCrCb[2]->data[i] - 128;
      R = (1.164*Y + 1.596*Cr);		   if( R<0 ) R=0; if( R>255 ) R=255;
      G = (1.164*Y - 0.813*Cr - 0.391*Cb); if( G<0 ) G=0; if( G>255 ) G=255;
      B = (1.164*Y + 2.018*Cb);		   if( B<0 ) B=0; if( B>255 ) B=255;
      YCrCb[0]->data[i] = (pixel) R; 
      YCrCb[1]->data[i] = (pixel) G; 
      YCrCb[2]->data[i] = (pixel) B; 
      }

    return PIPOK;
    }

/**
* RGB to YUV transformation
* \param rgb Pointer to 3 Image Planes  
* YUV color space is the basic colour space used by the PAL color video standard. 
* - \b Y has a range of 0 to 255, 
* - \b U has a range of -112 to 112, and 
* - \b V has a range of -157 to +157. 
* \warning Since the YUV transformation is signed it does not make any sense 
* for an 8 bit unsigned data structure. 
* \todo Add a new data structure with signed integers
*/

int 
PIP_RGBtoYUV( Pimage *rgb[] )
    { 
    double r, g, b, y;
    int	i, np = rgb[0]->np;

    PIPcomment("RGB to YUV - requires signed data\n"); 
    if( rgb[0]->data == NULL ) PIPerr("There is no data in Red plane");
    if( rgb[1]->data == NULL ) PIPerr("There is no data in Green plane");
    if( rgb[2]->data == NULL ) PIPerr("There is no data in Blue plane");

    for(i=0; i<np; i++)
      {
      r = (double) rgb[0]->data[i];
      g = (double) rgb[1]->data[i];
      b = (double) rgb[2]->data[i];
      y = (0.299*r + 0.587*g + 0.114*b);
      rgb[0]->data[i] = (pixel) y;
      rgb[1]->data[i] = (pixel) 0.492*(b-y);
      rgb[2]->data[i] = (pixel) 0.877*(r-y);
      }

    return PIPOK;
    }

/**
* YUV to RGB transformation
* \param yuv Pointer to 3 Image Planes  
*/

int 
PIP_YUVtoRGB( Pimage *yuv[] )
    { 
    double Y, U, V;
    int	i, np = yuv[0]->np;

    PIPcomment("YUV to RGB - requires signed data\n"); 
    if( yuv[0]->data == NULL ) PIPerr("There is no data in Y plane");
    if( yuv[1]->data == NULL ) PIPerr("There is no data in U plane");
    if( yuv[2]->data == NULL ) PIPerr("There is no data in V plane");

    for(i=0; i<np; i++)
      {
      Y = (double) yuv[0]->data[i];
      U = (double) yuv[1]->data[i];
      V = (double) yuv[2]->data[i];
      yuv[0]->data[i] = (pixel) (Y + 1.140*V);
      yuv[1]->data[i] = (pixel) (Y - 0.395*U - 0.581*V);  
      yuv[2]->data[i] = (pixel) (Y + 2.032*U);  
      }

    return PIPOK;
    }

/**
* RGB to YIQ transformation
* \param rgb Pointer to 3 Image Planes  
* YIQ is the color space used by the NTSC composite colour video standard.
* It is similiar to YUV, except that the colourspace is rotated 33 degrees clockwise, 
* so that I is the orange-blue axis, and Q is the purple-green
* - \b Y is luminance 0 to 255,
* - \b I is in-phase and has a range -152 to +152, 
* - \b Q is quadrature and has a range -134 to 134. 
* \image html Y.jpg "Y component"
* \image html I.jpg "I component"
* \image html Q.jpg "Q component"
*/

int 
PIP_RGBtoYIQ( Pimage *rgb[] )
    { 
    double r, g, b, y;
    int	i, np = rgb[0]->np;

    PIPcomment("RGB to YIQ - requires signed data\n"); 
    if( rgb[0]->data == NULL ) PIPerr("There is no data in Red plane");
    if( rgb[1]->data == NULL ) PIPerr("There is no data in Green plane");
    if( rgb[2]->data == NULL ) PIPerr("There is no data in Blue plane");

    for(i=0; i<np; i++)
      {
      r = (double) rgb[0]->data[i];
      g = (double) rgb[1]->data[i];
      b = (double) rgb[2]->data[i];
      y = (0.299*r + 0.587*g + 0.114*b);
      rgb[0]->data[i] = (pixel) y;
      rgb[1]->data[i] = (pixel) 0.736*(r-y) - 0.268*(b-y); 
      rgb[2]->data[i] = (pixel) 0.478*(r-y) + 0.413*(b-y); 
      }

    return PIPOK;
    }

/**
* YIQ to RGB transformation
*/

int 
PIP_YIQtoRGB( Pimage *yiq[] )
    { 
    double Y, I, Q;
    int	i, np = yiq[0]->np;

    PIPcomment("YIQ to RGB - requires signed data \n"); 
    if( yiq[0]->data == NULL ) PIPerr("There is no data in Y plane");
    if( yiq[1]->data == NULL ) PIPerr("There is no data in I plane");
    if( yiq[2]->data == NULL ) PIPerr("There is no data in Q plane");

    for(i=0; i<np; i++)
      {
      Y = (double) yiq[0]->data[i];
      I = (double) yiq[1]->data[i];
      Q = (double) yiq[2]->data[i];
      yiq[0]->data[i] = (pixel) (Y + 0.956*I + 0.621*Q);
      yiq[1]->data[i] = (pixel) (Y - 0.272*I - 0.647*Q);
      yiq[2]->data[i] = (pixel) (Y - 1.106*I + 1.703*Q);
      }

    return PIPOK;
    }

/**
* Overlay Colour Mask on Grey Image.
* When ever neither R,G or B exists then set RGB to grey. 
* In other words if you have a coloured labelled image, then
* overlay the background grey image over it.
* The \a grey image is overlayed onto the three colour planes 
* \a red, \a grn, \a blu.
* This routine is designed to place a labelled image
* on top of the original. 
* \image html overlay.jpg "Original image overlayed with labelled image"
* \code
* Pimage *grey, *work, *rgb[3];
* PIP_load(grey "test.pgm");
* PIP_copy(work, grey);	
* PIP_unary(work, '>', 100); 
* PIP_label(work, 4, 10, 1); 
* PIP_loadcmap("regions");
* PIP_colourize(rgb, work);
* PIP_overlay(rgb, grey);	
* PIP_saveRGB(rgb, "over.ppm");
* \endcode
*/

int PIP_overlay(
	Pimage  *rgb[],		/*!< Colour Image Array to be generated */
	Pimage 	*gry) 		/*!< Source Grey Image */
    { 
    int i;
    int	np = gry->np;

    PIPcomment("Colour Overlay\n");

    if( gry->data == NULL ) PIPerr("Invalid Grey Image"); 
    if( rgb[0]->data == NULL ) PIPerr("Invalid Red Image"); 
    if( rgb[1]->data == NULL ) PIPerr("Invalid Green Image");
    if( rgb[2]->data == NULL ) PIPerr("Invalid Blue Image"); 

    for(i=0; i<np; i++)
      if( !rgb[0]->data[i] && !rgb[1]->data[i] && !rgb[2]->data[i] ) 
        {
        rgb[0]->data[i] = gry->data[i];
        rgb[1]->data[i] = gry->data[i];
        rgb[2]->data[i] = gry->data[i];
        }

    return PIPOK;
    }


static pixel red[256], grn[256], blu[256];

/**
Load Colour Map into memory 
\warning You may need to define the environment variable LIBSX if
the colour map file cannot be found locally.
*/

int 
PIP_loadcmap(
	char *cmap)	/*!< Filename of colour map */
    {
    int i, r, g, b;
    FILE *fp;

    /* Stuff Pinched from LibSX */

    PIPcomment("Loading Cmap (%s)\n", cmap);

    if((fp = fopen(cmap,"r")) == NULL ) {
      char buff[80]; char *basename;
      basename = getenv("LIBSX");
      if( basename == NULL ) PIPerr("LIBSX not defined"); 
        sprintf(buff, "%s/%s", basename,cmap);
        if((fp = fopen(buff,"r")) == NULL )
          PIPerr("Cannot open CMAP file"); 
	}

    for(i=0; i<256; i++) {
      if( fscanf(fp,"%d%d%d", &r,&g,&b) != 3 ) break;
      red[i] = (pixel) r;
      grn[i] = (pixel) g;
      blu[i] = (pixel) b;
      }
    fclose(fp);
    return PIPOK;
    }

/**
Convert grey image to colour 
Translate grey scale image \a grey into three colour planes 
\a red, \a grn and \a blu with the aid of a colour map.
The colourmap is a file specified by \a cmap</B> which contains 
256 colour triplets (r,g,b). 
Each pixel in the image ( 0-255) is now defined by a colour
(r,g,b) value. If the colourmap file cannot be found, 
the environment variable \b LIBSX is prepended to the filename.
\code
Pimage *grey, *rgb[3];
PIP_load(grey, "test.pgm");
PIP_loadcmap("regions");
PIP_colourize(rgb, grey);
PIP_saveRGB(rgb, "test.ppm");
\endcode
*/

int 
PIP_colourize(
	Pimage *rgb[], 	/*!< Output RGB Image plane */
	Pimage *image) 	/*!< Input Grey Image */
    {
    int i, j;
    int nc = image->nc;
    int nr = image->nr;
    int ng = image->ng;
    int np = image->np;

    if( rgb[0]->data == NULL) if( PIP_make(rgb[0],nc,nr,ng,"Red") == PIPFAIL)
      PIPerr("Cannot Create Red Image");
    if( rgb[1]->data == NULL) if( PIP_make(rgb[1],nc,nr,ng,"Green") == PIPFAIL)
      PIPerr("Cannot Create Green Image"); 
    if( rgb[2]->data == NULL) if( PIP_make(rgb[2],nc,nr,ng,"Blue") == PIPFAIL)
      PIPerr("Cannot Create Blue Image");

    for(i=0; i<np; i++) 
      {
      j = image->data[i];
      rgb[0]->data[i] = red[j];
      rgb[1]->data[i] = grn[j];
      rgb[2]->data[i] = blu[j];
      }

    return PIPOK;
    }

/**
* Overlay colourmap in a single hit 
*/

int 
PIP_overcmap( 
	Pimage *rgb[],		/*!< Output Colour Umage */
	Pimage *label, 		/*!< Labelled Image */
	Pimage *grey) 		/*!< Original Grey Image */
    {
    int i, j;
    int nc = label->nc;
    int nr = label->nr;
    int ng = label->ng;
    int np = label->np;

    if( rgb[0]->data == NULL ) if( PIP_make(rgb[0],nc,nr,ng,"Red") == PIPFAIL)
      PIPerr("Cannot Create Red Image");
    if( rgb[1]->data == NULL ) if( PIP_make(rgb[1],nc,nr,ng,"Green") == PIPFAIL)
      PIPerr("Cannot Create Green Image"); 
    if( rgb[2]->data == NULL ) if( PIP_make(rgb[2],nc,nr,ng,"Blue") == PIPFAIL)
      PIPerr("Cannot Create Blue Image");

    for(i=0; i<np; i++) 
      {
      j = label->data[i];
      if( j == 0 ) {
        rgb[0]->data[i] = grey->data[i];
        rgb[1]->data[i] = grey->data[i];
        rgb[2]->data[i] = grey->data[i];
	}
      else 
	{
        rgb[0]->data[i] = red[j];
        rgb[1]->data[i] = grn[j];
        rgb[2]->data[i] = blu[j];
	}
      }

    return PIPOK;
    }
