/*********************************************************************
*
* CSIRO Automation
* Queensland Centre for Advanced Technologies
* PO Box 883, Kenmore, QLD 4069, Australia
* www.cat.csiro.au/cmst
*
* Copyright (c) CSIRO Manufacturing Science & Technology
*
*********************************************************************/

/** \file
 * \brief Chain code functions
 */

#include "local.h"

/**
* Generate chaincode from outline image
* \return length of chain
*/

int PIP_chain(Pimage *input, Pchain *chain)
    {
    pixel *ip;
    int i, row, col, nl;
    int nc = input->nc;
    int	np = input->np;

    if(( ip = input->data) == NULL ) NULerr;

    nl = 0;
    for(i = 0; i < np; i++) if( ip[i] > OFF ) { ip[i] = MARK; nl++; }
    for(i = 0; i < np; i++) if( ip[i] > OFF ) break;
    row = i / nc; col = i % nc; 

    PIPcomment("Chain (%d,%d) Points %d ", row, col, nl);

    if( chain->x != NULL ) free(chain->x);
    if( chain->y != NULL ) free(chain->y);
    if((chain->x = (int *) malloc(sizeof(int)*nl)) == NULL) MEMerr;
    if((chain->y = (int *) malloc(sizeof(int)*nl)) == NULL) MEMerr;

    for(i = 0; i < nl; i++) 
      {
      chain->x[i] = col;
      chain->y[i] = row; 
      ip[row*nc + col] = ON; 

           if( ip[row*nc + (col+1)]   == MARK ) col++;
      else if( ip[(row+1)*nc+(col+1)] == MARK ) { row++; col++; }
      else if( ip[(row+1)*nc+col]     == MARK ) { row++; }
      else if( ip[(row+1)*nc+(col-1)] == MARK ) { row++; col--; }
      else if( ip[row*nc + (col-1)]   == MARK ) col--;
      else if( ip[(row-1)*nc+col]     == MARK ) { row--; }
      else if( ip[(row-1)*nc+(col-1)] == MARK ) { row--; col--; }
      else if( ip[(row-1)*nc+(col+1)] == MARK ) { row--; col++; }
      else break;
      }

    chain->n = i;
    return i;
    }

/* 
** Compute the chain code with 8 way look ahead
*/

int PIP_chain8(Pimage *image, Pchain *chain)
    {
    int i, j;		/* Direction indexes */
    int row=0, col=0;	/* New Position */
    int crow, ccol;	/* Current Position */
    int srow, scol;	/* Starting Position */
    int found;		/* Pixel found flag */
    int n, nl;          /* Current index and Total Length */
    int d, l;		/* Current & Last Direction */
    int di[9], dj[9];	/* Direction Matrices */
    int nc, nr, np;
    pixel *ip;

    dj[0] =  0;	dj[1] = -1; dj[2] = -1;	dj[3] = -1;
    dj[4] =  0;	dj[5] =  1; dj[6] = 1;	dj[7] = 1;
    di[0] =  1;	di[1] =  1; di[2] = 0;	di[3] = -1;
    di[4] = -1;	di[5] = -1; di[6] = 0;	di[7] = 1;

    ip = image->data;
    nr = image->nr;
    nc = image->nc;
    np = image->np;

    /* Find the starting Pixel */

    nl = 0;
    for(i=0; i<np; i++) if( ip[i] > OFF ) { ip[i] = MARK; nl++; }
    for(i=0; i<np; i++) if( ip[i] > OFF ) break;

    /* Initialize for starting pixel */

    n = 0; l = 4;
    crow = srow = i / nc; 
    ccol = scol = i % nc; 

    PIPcomment("Chain (%d,%d) Points %d\n", scol, srow, nl);

    /* Create & Clear the code table */

    if( chain->x != NULL ) free(chain->x);
    if( chain->y != NULL ) free(chain->y);
    if((chain->x = (int *) malloc(sizeof(int)*nl)) == NULL) MEMerr;
    if((chain->y = (int *) malloc(sizeof(int)*nl)) == NULL) MEMerr;

    /* Start Tracking */

    do {
      d = -1;	
      found = FALSE;

      /* Look for next pixel */

      for(i=l+1; i<l+8; i++) { j = i%8; 
	col = di[j]+ccol; if( col < 0 || col > nc ) continue;
	row = dj[j]+crow; if( row < 0 || row > nr ) continue;
    	if( ip[row*nc+col] == MARK ) { d=i; found=TRUE; break; } 
	}

      if( !found ) break;
      if( n >= nl ) return(PIPFAIL); 

      /* If we have found a Pixel */

      crow = row;		
      ccol = col;
      l = (d+5)%8;
      ip[row*nc+col] = ON;

      chain->x[n] = ccol;
      chain->y[n++] = crow; 
      } while ( (crow!=srow) || (ccol!=scol) );	

  chain->n = n;
  return n;
  }

double PIP_chain_perimeter(Pchain *chain)
    {
    int i, x, y;
    double p = 0;	
    double hypot(double, double);

    for(i=1; i<chain->n; i++) 
      {
      if( chain->x[i] == 0 ) break;
      x = chain->x[i] - chain->x[i-1];
      y = chain->y[i] - chain->y[i-1];
      p += hypot((double)x,(double)y);
      }

    return p;
    }

/**
* Generate border image from mask image
*/

int PIP_border(Pimage *input, Pimage *output)
    {
    pixel *ip, *op;
    int i, row, col, val;
    int nc = input->nc;
    int	nr = input->nr;
    int	ng = input->ng;
    int	np = input->np;

    PIPcomment("Border "); 

    if((ip = input->data) == NULL ) NULerr;
    if( output->data == NULL ) 
      if( PIP_make(output, nc, nr, ng, "Outline") == PIPFAIL ) 
        PIPerr("Cannot Create Outline image");

    /* Clear Output Image */

    op = output->data;
    for(i=0; i<np; i++) op[i] = OFF;

    for(row = 1; row < nr-1; row++)
    for(col = 1; col < nc-1; col++) 
      {
      val = ip[row*nc+col];
      if( ip[row*nc+(col+1)] != val ) op[row*nc+col] = ON;
      if( ip[(row+1)*nc+col] != val ) op[row*nc+col] = ON;
      }

    return PIPOK;
    }

/**
** Remove border from mask image
*/

int PIP_binary_unborder(Pimage *input)
    {
    int i, row, col,  pos, add[10];
    pixel *ip, val, min;
    int nc = input->nc;
    int	nr = input->nr;

    printf("Binary UnBorder\n"); 
    if((ip = input->data) == NULL ) NULerr;;

    add[1] = -nc-1; add[2] = -nc; add[3] = -nc+1;
    add[8] = -1;    add[0] = 0;   add[4] = 1;
    add[5] = nc+1;  add[6] = nc;  add[7] = nc-1;

    for(row = 1; row < nr-1; row++)
    for(col = 1; col < nc-1; col++) 
      {
      pos = row*nc+col;
      val = ip[pos];
      if( val == OFF ) 
        {
        min = ON;
        for(i=1; i<9; i++) 
	  if( ip[pos+add[i]] > OFF && ip[pos+add[i]] < min ) 
	    min = ip[pos+add[i]];
        if( min != ON ) { printf("%d %d %d\n", row, col, min); ip[pos] = min; }
        }
      }

    return PIPOK;
    }

/**
* Merge generate border from mask image
* @bug not finished yet
*/

int PIP_binary_merge(Pimage *input, Pimage *water, Pimage *label)
    {
    pixel *ip;
    int row, col, pos, add[10];
    int nc = input->nc;
    int	nr = input->nr;

    printf("Binary Merge\n"); 
    if((ip = input->data) == NULL ) NULerr;;

    add[1] = -nc-1; add[2] = -nc; add[3] = -nc+1;
    add[8] = -1;    add[0] = 0;   add[4] = 1;
    add[5] = nc+1;  add[6] = nc;  add[7] = nc-1;

    for(row = 1; row < nr-1; row++)
    for(col = 1; col < nc-1; col++) 
      {
      pos = row*nc+col;
      }

    return PIPOK;
    }

/**
* Generate Outline
\image html outline.gif "Binary image before and after outline"
\return length of outline
\warning Outline image is marked with 128
*/

int PIP_outline(Pimage *input, Pimage *output)
    {
    pixel *ip, *op;
    int row, col, inc, val, nl;
    int nc = input->nc;
    int	nr = input->nr;
    int	ng = input->ng;

    PIPcomment("Outline\n"); 

    if( input->data == NULL ) NULerr;
    if( output->data == NULL ) 
      if( PIP_make(output, nc, nr, ng, "Outline") == PIPFAIL ) 
        PIPerr("Cannot Create Outline image\n"); 

    nl = 0;
    ip = input->data;
    op = output->data;

    for(row = 1; row < nr-1; row++)
    for(col = 1; col < nc-1; col++) 
      {
      val = ip[row*nc+col];
      if( val ) 
       {
       inc = 0;
       if( ip[row*nc+(col-1)] == 0 ) inc++;
       if( ip[(row-1)*nc+col] == 0 ) inc++;
       if( ip[row*nc+(col+1)] == 0 ) inc++;
       if( ip[(row+1)*nc+col] == 0 ) inc++;

       if( inc > 0 )
       /* if( inc > 0 && inc < 4 ) */
         {
	 op[row*nc+col] = MARK;
	 nl++;
	 }
       }
     }

    return nl;
    }
