/*********************************************************************
*
* CSIRO Automation
* Queensland Centre for Advanced Technologies
* PO Box 883, Kenmore, QLD 4069, Australia
* www.cat.csiro.au/cmst
*
* Copyright (c) CSIRO Manufacturing Science & Technology
*
*********************************************************************/

/** \file 
\brief Binary Image Utilities 

These are utilities that treat the data contained within Pimage as binary. 
That is, any pixel that has a value of 0 is treated as OFF, whilst all other pixels are treated as ON. 

*/

#include "local.h"

/**
This function performs binary erosion of order num. That is, a
pixel that is ON is switched OFF if there are at least num neighbouring
pixels that are also OFF. 
\image html erode.gif "Binary image before and after erosion"
\see PIP_dilate()
\return PIPOK if successful 
*/

int 
PIP_erode(
    Pimage *image, 	/*!< Pointer to Image to be eroded */
    int num)		/*!< Number of Neighbours */
    {
    int i, row, col, sum, pos, add[9];
    pixel *ip, *op; 
    int nc = image->nc;
    int	nr = image->nr;
    int	np = image->np;

    PIPcomment("Erode (%d)\n", num); 
    if((ip = (pixel *) image->data) == NULL ) NULerr;
    if((op = (pixel *) malloc(np*sizeof(pixel))) == NULL) MEMerr;;
    memcpy(op, ip, np);

    add[1] = -nc-1; add[2] = -nc; add[3] = -nc+1;
    add[8] = -1;    add[0] = 0;   add[4] = 1;
    add[5] = nc+1;  add[6] = nc;  add[7] = nc-1;

    for(row=1; row<(nr-1); row++) 
    for(col=1; col<(nc-1); col++) 
      {
      sum = 0;
      pos = (row)*nc + col; 
      for(i=1; i<9; i++) if( ip[pos+add[i]] == OFF) sum++; 
      if( sum > num ) op[pos] = OFF;
      }

    free(image->data);
    image->data = op;
    return PIPOK;
    }

/**
This function performs binary dilation of order num. That is,
a pixel that is OFF is switched ON if there are at least num neighbouring
pixels that are also ON. 
\return PIPOK if successful
\see PIP_erode()
*/

int 
PIP_dilate(
	Pimage *image, 	/*!< Pointer to image to be dilated */
	int num) 	/*!< Number of neighbours necessary for dilation */
    {
    int i, row, col, sum, pos, add[9];
    pixel val, *ip, *op; 
    int nc = image->nc;
    int	nr = image->nr;
    int np = image->np;

    PIPcomment("Dilate (%d)\n", num); 

    if((ip = (pixel *) image->data) == NULL) NULerr;
    if((op = (pixel *) malloc(np*sizeof(pixel))) == NULL) MEMerr;
    memcpy(op, ip, np);
    
    add[1] = -nc-1; add[2] = -nc; add[3] = -nc+1;
    add[8] = -1;    add[0] = 0;   add[4] = 1;
    add[5] = nc+1;  add[6] = nc;  add[7] = nc-1;

    for(row=1; row<(nr-1); row++) 
    for(col=1; col<(nc-1); col++) 
	{
	pos = row*nc + col;
	val = ip[pos];
	if( val == OFF ) 
	  {
	  sum = 0;
	  for(i=1; i<9; i++) if( ip[pos+add[i]] > OFF ) sum++;
	  if( sum > num ) op[pos] = ON;
	  }
        }

    free(image->data);
    image->data = op;
    return PIPOK;
    }

/**
Yet another Standard Dilation 
\return number of pixels removed from image 
\see PIP_dilate()
\see PIP_erode()
*/

int 
PIP_dilate2(Pimage *image) 	/*!< Pointer of image to be dilated */
    {
    int i, row, col, pos, add[9];
    pixel *ip, *op; 
    int npixels = 0;
    int nc = image->nc;
    int	nr = image->nr;
    int np = image->np;

    PIPcomment("Dilate2\n"); 

    if((ip = (pixel *) image->data) == NULL) NULerr;
    if((op = (pixel *) malloc(np*sizeof(pixel))) == NULL) MEMerr;
    memcpy(op, ip, np);
    
    add[1] = -nc-1; add[2] = -nc; add[3] = -nc+1;
    add[8] = -1;    add[0] = 0;   add[4] = 1;
    add[5] = nc+1;  add[6] = nc;  add[7] = nc-1;

    for(row=1; row<(nr-1); row++) 
    for(col=1; col<(nc-1); col++) 
	{
	pos = row*nc + col;
	if( ip[pos] == OFF ) 
	  for(i=1; i<9; i++) 
	    if( ip[pos+add[i]] > OFF ) 
	      { npixels++; op[pos] = ip[pos+add[i]]; }
        }

    free(image->data);
    image->data = op;
    return npixels;
    }

/**
Yet another Dilation ? 
\return number of erosion 
\see PIP_dilate()
*/

int 
PIP_dilate3(Pimage *image)
    {
    int i, row, col, pos, add[9];
    pixel *ip, *op, *tmp; 
    int loop, change;
    int nc = image->nc;
    int	nr = image->nr;
    int np = image->np;

    PIPcomment("Dilate3 : Ultimate Erosion\n"); 

    if((ip = (pixel *) image->data) == NULL) NULerr;
    if((op = (pixel *) malloc(np*sizeof(pixel))) == NULL) MEMerr;
    
    add[1] = -nc-1; add[2] = -nc; add[3] = -nc+1;
    add[8] = -1;    add[0] = 0;   add[4] = 1;
    add[5] = nc+1;  add[6] = nc;  add[7] = nc-1;

    for(loop=0; loop<nr*nc; loop++)
      {
      change = 0;
      memcpy(op, ip, np);
      for(row=1; row<(nr-1); row++) 
      for(col=1; col<(nc-1); col++) 
	{
	pos = row*nc + col;
	if( ip[pos] == OFF ) 
	  for(i=1; i<9; i++) 
	    if( ip[pos+add[i]] > OFF ) 
	      { change++; op[pos] = ip[pos+add[i]]; }
        }
      PIPcomment("Dilate %d times : removing  %d pixels\n", loop, change);
      if( change == 0 ) break;
      tmp = ip; ip = op; op = tmp; /* Swap Pointers */
      }

    free(image->data);
    image->data = op;
    return loop;
    }

/** 
Check to see it blob touches edge

- \c PIP_EDGE_EXTEND copy penultimate edges
- \c PIP_EDGE_REMOVE Turn edge pixels OFF 

\return side of window that blob touches 
The following flags are || together
- \c PIP_EDGE_NONE	No edges touch
- \c PIP_EDGE_NORTH
- \c PIP_EDGE_SOUTH
- \c PIP_EDGE_EAST
- \c PIP_EDGE_WEST
*/

int 
PIP_edge(
	Pimage *image,  /*!< pointer to image */
	Pedge flag) 	/*!< flag to control action */
    {
    int i;
    pixel *ip; 
    int nc = image->nc;
    int	np = image->np;
    int flag1, flag2, flag3, flag4;

    if((ip = (pixel *) image->data) == NULL ) NULerr;
    flag1 = PIP_EDGE_NONE;
    flag2 = PIP_EDGE_NONE;
    flag3 = PIP_EDGE_NONE;
    flag4 = PIP_EDGE_NONE;

    /* By Default this routine will always detect
    whether the occurance of features on the edge */

    for(i=0; i<nc; i++)   { if( ip[i+nc]      ) flag1 = PIP_EDGE_NORTH; }
    for(i=0; i<nc; i++)   { if( ip[i+np-2*nc] ) flag2 = PIP_EDGE_SOUTH; }
    for(i=0; i<np; i+=nc) { if( ip[i+1]       ) flag3 = PIP_EDGE_WEST; }
    for(i=0; i<np; i+=nc) { if( ip[i+nc-2]    ) flag4 = PIP_EDGE_EAST; }

    if( flag == PIP_EDGE_EXTEND )
      {
      PIPcomment("Edge - Extend Edge\n"); 
      for(i=0; i<nc; i++)   ip[i] = ip[i+nc]; 		/* Copy Row below */
      for(i=0; i<nc; i++)   ip[i+np-nc] = ip[i+np-2*nc];/* Copy Row Above */
      for(i=0; i<np; i+=nc) ip[i] = ip[i+1];		/* Copy Western Col */
      for(i=0; i<np; i+=nc) ip[i+nc-1] = ip[i+nc-2];	/* Copy Eastern Col */
      }

    if( flag == PIP_EDGE_REMOVE ) {
      PIPcomment("Edge -  Remove Edge\n"); 
      if( flag1 ) for(i=0; i<nc; i++) ip[i+nc] = OFF; 
      if( flag2 ) for(i=0; i<nc; i++) ip[i+np-2*nc] = OFF; 
      if( flag3 ) for(i=0; i<np; i+=nc) ip[i+1] = OFF; 
      if( flag4 ) for(i=0; i<np; i+=nc) ip[i+nc-2] = OFF; 
      }

    return(flag1|flag2|flag3|flag4);
    }

/**
Medial axis of binary Image
\return Index to the ultimate point of erosion. 
*/

int 
PIP_mediaxis(
	Pimage *input, 		/*!< Pointer to Input Image */
	Pimage *output)		/*!< Pointer to Output Umage */
    {
    int i, j;
    pixel *ip, *op;
    pixel val, max, tmp, num;
    int nc = input->nc;
    int	nr = input->nr;
    int	ng = input->ng;

    PIPcomment("MediAxis Transformation\n"); 

    if((ip = (pixel *) input->data) == NULL) NULerr;
    if( output->data == NULL )
      if( PIP_make(output, nc, nr, ng, "Medial Axis ") == PIPFAIL )
        PIPerr("Cannot Create Medial Axis Image");
 
    op = (pixel *) output->data;

    num = 0;
    for(i=1; i<nr-1; i++)
    for(j=1; j<nc-1; j++)
      { 
      val = ip[i*nc+j];
      if( val == 0 ) continue;

      max = ip[(i-1)*nc+j];
      tmp = ip[(i+1)*nc+j]; if( tmp > max ) max = tmp;
      tmp = ip[i*nc+(j+1)]; if( tmp > max ) max = tmp;
      tmp = ip[i*nc+(j-1)]; if( tmp > max ) max = tmp;

      tmp = ip[(i-1)*nc+(j+1)]; if( tmp > max ) max = tmp;
      tmp = ip[(i-1)*nc+(j-1)]; if( tmp > max ) max = tmp;
      tmp = ip[(i+1)*nc+(j+1)]; if( tmp > max ) max = tmp;
      tmp = ip[(i+1)*nc+(j-1)]; if( tmp > max ) max = tmp;

      if( max <= val ) { op[i*nc+j] = ip[i*nc+j]; num++; }
      else op[i*nc+j] = OFF; 
      }

    return num;
    }


/** 
Final Point of Erosion
\return PIPOK if successful
*/

int 
PIP_final(Pimage *input) 	/*!< Pointer to Input Image */
    {
    int i, row, col, sum, pos, add[9];
    pixel *ip, *op, *tmp; 
    int loop, change;
    int nc = input->nc;
    int	nr = input->nr;
    int	np = input->np;

    PIPcomment("Ultimate Point of Erosion\n"); 

    if((ip = (pixel *) input->data) == NULL ) NULerr;
    if((op = (pixel *) malloc(np*sizeof(pixel))) == NULL) MEMerr;;
    memcpy(op, ip, np);

    add[1] = -nc-1; add[2] = -nc; add[3] = -nc+1;
    add[8] = -1;    add[0] = 0;   add[4] = 1;
    add[5] = nc+1;  add[6] = nc;  add[7] = nc-1;

    for(loop=0; loop<nr*nc; loop++)
      {
      change = 0;
      for(row=1; row<(nr-1); row++) 
      for(col=1; col<(nc-1); col++) 
        { pos = (row)*nc + col; 
        if( op[pos] > OFF) { sum = 0;
          for(i=1; i<9; i++) if( ip[pos+add[i]] == OFF) sum++; 
          if( sum > 0 && sum < 8  ) { change++; op[pos] = OFF; }
	  }
        }

      PIPcomment("Erode %d times : removing  %d pixels\n", loop, change);
      if( change == 0 ) break;
      tmp = ip; ip = op; op = tmp; /* Swap Pointers */
      }

    free(input->data);
    input->data = op;
    return PIPOK;
    }
