/*
 ********************************************************************
 *
 * srpc.c - SRPC server module for the launcher
 *
 *     CSIRO Automation
 *     Queensland Centre for Advanced Technologies
 *     PO Box 883, Kenmore, QLD 4069, Australia
 *     www.cat.csiro.au/cmst
 *
 *      $Id: srpc.c 1776 2007-05-29 03:12:25Z pra077 $
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 ********************************************************************
 */

#ifdef __GNUC__
#define RTX_STATIC_ATTRIBUTE __attribute__((unused))
#else
#define RTX_STATIC_ATTRIBUTE
#endif
static char *rcsid RTX_STATIC_ATTRIBUTE = "$Id: srpc.c 1776 2007-05-29 03:12:25Z pra077 $";

/**
 ********************************************************************
 * 
 * \file srpc.c
 * \brief SRPC server module for the launcher
 * \author Pavan Sikka
 */

#include <unistd.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sched.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <poll.h>
#include <semaphore.h>

#include <rtx/main.h>
#include <rtx/message.h>
#include <rtx/thread.h>
#include <rtx/mutex.h>
#include <rtx/sem.h>
#include <rtx/auth.h>
#include <rtx/lock.h>
#include <rtx/error.h>
#include <rtx/timer.h>
#include <rtx/signal.h>
#include <rtx/list.h>
#include <rtx/command.h>
#include <rtx/srpc.h>

#include "launch.h"

/* command handlers for interactive sessions */

int launch_cmd_up (int argc, char * argv[]);
int launch_cmd_down (int argc, char * argv[]);
int launch_cmd_go (int argc, char * argv[]);
int launch_cmd_top (int argc, char * argv[]);
int launch_cmd_bottom (int argc, char * argv[]);
int launch_cmd_ls (int argc, char * argv[]);
int launch_cmd_auto (int argc, char * argv[]);

/* commands for interactive sessions */

RtxCommandMenu launchCmds[] = {
  {"up", launch_cmd_up, "increment app level by 1"},
  {"down", launch_cmd_down, "decrement app level by 1"},
  {"go", launch_cmd_go, "change app level as specified"},
  {"top", launch_cmd_top, "change app level to max"},
  {"bottom", launch_cmd_bottom, "change app level to 0"},
  {"ls", launch_cmd_ls, "list apps"},
  {"auto", launch_cmd_auto, "start launcher automatically (on|off)"},
  {NULL, NULL, NULL}
};

static Launch * l;

/********************************************************************
 * SRPC functions
 */

int
launch_srpc_get_app_list (
                         char * buf,
                         int len,
			 int argc,
			 char * argv[]
			 )
{
    LaunchAppNode * nd;

    buf[0] = '\0';
    while ((nd = (LaunchAppNode *) rtx_list_iterate 
	    (l->apps)) != NULL) {
        strcat (buf, " ");
	strcat (buf, nd->appName);
    }
    return (0);
}

int
launch_srpc_get_app_status (
			   char * buf,
			   int len,
			   int argc,
			   char * argv[]
			   )
{
    LaunchAppNode * nd;
    int i;
    char locLine[64];

    buf[0] = '\0';
    if (argc < 2) {
        strcpy (buf, "");
	return (0);
    }
    if ((argc == 2) && (strcmp (argv[1], "all") == 0)) {
        while ((nd = (LaunchAppNode *) rtx_list_iterate 
		(l->apps)) != NULL) {
	    sprintf (locLine, "%d %s %d ", nd->level, nd->appName, nd->status);
	    strcat (buf, locLine);
	}
	return (0);
    }
    for (i=1; i<argc; i++)
        if ((nd = (LaunchAppNode *) rtx_list_lookup 
	        (l->apps, argv[i])) != NULL) {
	    sprintf (locLine, "%d %s %d ", nd->level, nd->appName, nd->status);
	    strcat (buf, locLine);
	}
    return (0);
}
	  
int
launch_srpc_get_app_info (
                         char * buf,
                         int len,
			 int argc,
			 char * argv[]
			 )
{
    LaunchAppNode * nd;

    buf[0] = '\0';
    if (argc != 2) {
        strcpy (buf, "ERROR");
	return (0);
    }
    if ((nd = (LaunchAppNode *) rtx_list_lookup 
	     (l->apps, argv[1])) != NULL) {
        sprintf (buf, "%s %s %d %d %d %s", nd->appName, nd->hostName,
		 nd->appType, nd->timeout, nd->level, nd->command);
    } else {
        strcat (buf, "ERROR");
    }
    return (0);

}

int
launch_srpc_get_runlevel (
                         char * buf,
                         int len,
			 int argc,
			 char * argv[]
			 )
{
    sprintf (buf, "%d", l->curAppLevel);
    return (0);
}

int
launch_srpc_set_runlevel (
                         char * buf,
                         int len,
			 int argc,
			 char * argv[]
			 )
{
    int desLevel;

    if (l->verbose)
        rtx_message ("launch_srpc_set_runlevel:");
    buf[0] = '\0';
    if (argc != 2) {
        strcpy (buf, "ERROR");
	return (0);
    }

    desLevel = atoi (argv[1]);
    if ((desLevel < 0) || (desLevel > 255)) {
        strcpy (buf, "ERROR");
	return (0);
    }

    if (l->verbose)
        rtx_message ("launch_srpc_set_runlevel: %d", desLevel);
    if (l->state == LAUNCH_STATE_STABLE) {
	l->desiredAppLevel = desLevel;
	l->userDesLevel = l->desiredAppLevel;
	sprintf (buf, "%d", desLevel);
	if (rtx_sem_post (l->eventSem) == -1) {
	      rtx_error_flush ("launch_srpc_set_runlevel: rtx_sem_post");
	}
    } else
        strcpy (buf, "ERROR");
    return (0);
}

int
launch_srpc_start (
		   Launch * ln
		   )
{
    l = ln;
    if (l->verbose)
        rtx_message ("launch: starting SRPC server");
    if ((l->srpcServer = rtx_srpc_init (l->srpcPortNum)) == NULL)
      return (rtx_error ("launch_srpc_start: rtx_srpc_init() failed"));

    if (rtx_srpc_register (l->srpcServer, "get_applist", 
			   launch_srpc_get_app_list)) 
        rtx_message_warning ("launch: rtx_srpc_register() failed");
    if (rtx_srpc_register (l->srpcServer, "get_status", 
			   launch_srpc_get_app_status)) 
        rtx_message_warning ("launch: rtx_srpc_register() failed");
    if (rtx_srpc_register (l->srpcServer, "get_info", 
			   launch_srpc_get_app_info)) 
        rtx_message_warning ("launch: rtx_srpc_register() failed");
    if (rtx_srpc_register (l->srpcServer, "get_runlevel", 
			   launch_srpc_get_runlevel)) 
        rtx_message_warning ("launch: rtx_srpc_register() failed");
    if (rtx_srpc_register (l->srpcServer, "set_runlevel", 
			   launch_srpc_set_runlevel)) 
        rtx_message_warning ("launch: rtx_srpc_register() failed");

    return (0);
}

/*********************************************************************
 * command handlers for interactive sessions
 *********************************************************************
 */

int 
launch_cmd_up (
	       int argc, 
	       char * argv[]
	       )
{
    int desLevel;

    desLevel = l->curAppLevel;
    if (l->state == LAUNCH_STATE_STABLE) {
	l->desiredAppLevel = launch_get_next_app_level_up (l, desLevel);
	l->userDesLevel = l->desiredAppLevel;
	if (rtx_sem_post (l->eventSem) == -1)
	    return (rtx_error ("launch_cmd_up: rtx_sem_post"));
    }
    return (0);
}

int 
launch_cmd_down (
		 int argc, 
		 char * argv[]
		 )
{
    int desLevel;

    desLevel = l->curAppLevel;
    if (l->state == LAUNCH_STATE_STABLE) {
	l->desiredAppLevel = launch_get_next_app_level_down (l, desLevel);
	l->userDesLevel = l->desiredAppLevel;
	if (rtx_sem_post (l->eventSem) == -1)
	    return (rtx_error ("launch_cmd_down: rtx_sem_post"));
    }
    return (0);
}

int 
launch_cmd_go (
	       int argc, 
	       char * argv[]
	       )
{
    int desLevel;

    desLevel = atoi (argv[1]);
	rtx_error_flush("Desired %d\n",desLevel);
    if ((desLevel < 0) || (desLevel > 255))
        return (rtx_error ("launch_cmd_go: invalid run level (%d)", desLevel));
    if (l->appsLevel[desLevel] != NULL) {
        l->desiredAppLevel = desLevel;
    } else if (desLevel < l->curAppLevel) {
        l->desiredAppLevel = launch_get_next_app_level_down (l, desLevel);
    } else if (desLevel > l->curAppLevel) {
        l->desiredAppLevel = launch_get_next_app_level_up (l, desLevel);
    }
    l->userDesLevel = l->desiredAppLevel;
	rtx_error_flush("Desired %d %d\n",l->userDesLevel,l->desiredAppLevel);
    if (desLevel != l->desiredAppLevel)
        rtx_message( "launch_cmd_go: level %d empty, changing level to %d", 
		 desLevel, l->desiredAppLevel);
    if (l->state == LAUNCH_STATE_STABLE) {
	if (rtx_sem_post (l->eventSem) == -1)
	    return (rtx_error ("launch_cmd_go: rtx_sem_post"));
    }
	getchar();
    return (0);
}

int 
launch_cmd_top (
		int argc, 
		char * argv[]
		)
{
    if (l->state == LAUNCH_STATE_STABLE) {
	l->desiredAppLevel = l->maxAppLevel;
	if (rtx_sem_post (l->eventSem) == -1)
	    return (rtx_error ("launch_cmd_go: rtx_sem_post"));
    }
    return (0);
}

int 
launch_cmd_bottom (
		   int argc, 
		   char * argv[]
		   )
{
    if (l->state == LAUNCH_STATE_STABLE) {
	l->desiredAppLevel = 0;
	if (rtx_sem_post (l->eventSem) == -1)
	    return (rtx_error ("launch_cmd_go: rtx_sem_post"));
    }
    return (0);
}

int 
launch_cmd_ls (
	       int argc, 
	       char * argv[]
	       )
{
    LaunchAppNode * nd = NULL;

    printf ("current level = %d\n\n", l->curAppLevel);
    printf ("level\tname\tstatus\tcommand\n");
    while ((nd = (LaunchAppNode *) rtx_list_iterate 
	    (l->apps)) != NULL) {
        printf ("%d\t%s\t%d [%d]\t%s\n", nd->level, nd->appName, 
		nd->status, (int) nd->appPid, nd->command);
    }
    return (0);
}

int 
launch_cmd_auto (
		 int argc, 
		 char * argv[]
		 )
{
    char * autofile = ".AUTOSTART";

    if (argc > 1) {
        if (strcmp (argv[1], "on") == 0) {
	    if (open (autofile, O_RDWR | O_CREAT, 0664) == -1) {
	        return (rtx_error_errno ("launch_cmd_auto"));
	    } else if (strcmp (argv[1], "off") == 0) {
	        if (unlink (autofile) == -1)
		    return (rtx_error_errno ("launch_cmd_auto"));
	    }
	}
    }
    return (0);
}

