/*
 ********************************************************************
 *
 * msg.c - messages for launch and launchagent
 *
 *     CSIRO Automation
 *     Queensland Centre for Advanced Technologies
 *     PO Box 883, Kenmore, QLD 4069, Australia
 *     www.cat.csiro.au/cmst
 *
 *      $Id: msg.c 1776 2007-05-29 03:12:25Z pra077 $
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 ********************************************************************
 */

#ifdef __GNUC__
#define RTX_STATIC_ATTRIBUTE __attribute__((unused))
#else
#define RTX_STATIC_ATTRIBUTE
#endif
static char *rcsid RTX_STATIC_ATTRIBUTE = "$Id: msg.c 1776 2007-05-29 03:12:25Z pra077 $";

/**
 ********************************************************************
 * 
 * \file msg.c
 * \brief messages for launch and launchagent
 * \author Pavan Sikka
 */

/**
 * \page msg
 *
 * functions to process messages
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#include <rtx/message.h>
#include <rtx/error.h>
#include <rtx/list.h>

#include "launch.h"

static char * launchMsg[] = {
    "NONE",
    "START",
    "LAUNCH",
    "LAUNCH_DONE",
    "LAUNCH_FAILED",
    "KILL",
    "KILL_DONE",
    "KILL_FAILED",
    "DONE"
};

/**
 * parse a message string into a message node
 */
LaunchMsgNode *
launch_parse_msg (
		  char * msg      /**< message string */
		  )
{
    int n = 0, i = 0;
    LaunchMsgNode * nd = NULL;
    char cmdTok[BUFSIZ];

    /* sanity check on the input */
    if (msg == NULL)
        return (rtx_error_null ("launch_parse_msg: NULL input"));
    n = strlen (msg);
    if ((n < 4) || (n > BUFSIZ))
        return (rtx_error_null ("launch_parse_msg: invalid "
				" message length [%d]", n));
    if ((nd = (LaunchMsgNode *) calloc 
	 (1, sizeof (LaunchMsgNode))) == NULL)
        return (rtx_error_null ("launch_parse_msg: no memory"));
    strcpy (nd->msg, msg);
    cmdTok[0] = '\0';
    sscanf (msg, "%s", cmdTok);
    for (i=0; i<LAUNCH_MSG_MAX_NUM; i++)
        if (strcmp (cmdTok, launchMsg[i]) == 0)
	    break;
    nd->msgType = i;
    switch (nd->msgType) {
	case LAUNCH_MSG_START :
	    if (sscanf (msg, "%*s %s %d", nd->hostName,
			(int *) &(nd->appPid)) != 2)
	        return (rtx_error_null ("launch_parse_msg: invalid "
					"START message [%s]", msg));
	    break;
	case LAUNCH_MSG_LAUNCH :
  	    if (sscanf (msg, "%*s %s %d %d %n", nd->appName,
			(int *) &(nd->appType), &(nd->timeout),
			&i) != 3)
	        return (rtx_error_null ("launch_parse_msg: invalid "
					"LAUNCH message [%s]", msg));
	    strncpy (nd->command, &(msg[i]), BUFSIZ-1);
	    break;
	case LAUNCH_MSG_LAUNCH_DONE :
	    if (sscanf (msg, "%*s %s %d", nd->appName,
			(int *) &(nd->appPid)) != 2)
	        return (rtx_error_null ("launch_parse_msg: invalid "
					"LAUNCH_DONE message [%s]",
					msg));
	    break;
	case LAUNCH_MSG_LAUNCH_FAILED :
  	    if (sscanf (msg, "%*s %s %n", nd->appName, &i) != 3)
	        return (rtx_error_null ("launch_parse_msg: invalid "
					"LAUNCH_FAILED message [%s]",
					msg));
	    strncpy (nd->errMsg, &(msg[i]), BUFSIZ-1);
	    break;
	case LAUNCH_MSG_KILL_FAILED :
  	    if (sscanf (msg, "%*s %s %n", nd->appName, &i) != 3)
	        return (rtx_error_null ("launch_parse_msg: invalid "
					"KILL_FAILED message [%s]",
					msg));
	    strncpy (nd->errMsg, &(msg[i]), BUFSIZ-1);
	    break;
	case LAUNCH_MSG_KILL :
	    if (sscanf (msg, "%*s %s", nd->appName) != 1)
	        return (rtx_error_null ("launch_parse_msg: invalid "
					"KILL message [%s]", msg));
	    break;
	case LAUNCH_MSG_KILL_DONE :
	    if (sscanf (msg, "%*s %s", nd->appName) != 1)
	        return (rtx_error_null ("launch_parse_msg: invalid "
					"KILL_DONE message [%s]",
					msg));
	    break;
	case LAUNCH_MSG_DONE :
	    break;
	case LAUNCH_MSG_NONE :
	default :
	    return (rtx_error_null ("launch_parse_msg: invalid "
				    "message [%s]", msg));
	    break;
    }
    return (nd);
}

/**
 * create a message node
 */
LaunchMsgNode *
launch_create_msg (
		   LaunchMsgType msgType,   /**< message type */
		   char * name              /**< app/host name */
		  )
{
    LaunchMsgNode * nd = NULL;

    if ((nd = (LaunchMsgNode *) calloc 
	 (1, sizeof (LaunchMsgNode))) == NULL)
        return (rtx_error_null ("launch_create_msg: no memory"));
    nd->msgType = msgType;
    switch (nd->msgType) {
	case LAUNCH_MSG_START :
	    strncpy (nd->hostName, name, BUFSIZ-1);
	    nd->appPid = getpid ();
	    break;
	case LAUNCH_MSG_LAUNCH_DONE :
	case LAUNCH_MSG_LAUNCH_FAILED :
	case LAUNCH_MSG_KILL_FAILED :
	case LAUNCH_MSG_KILL :
	case LAUNCH_MSG_KILL_DONE :
	case LAUNCH_MSG_DONE :
	case LAUNCH_MSG_LAUNCH :
	    strncpy (nd->appName, name, BUFSIZ-1);
	    break;
	case LAUNCH_MSG_NONE :
	default :
	    return (rtx_error_null ("launch_create_msg: invalid "
				    "message type [%d]", msgType));
	    break;
    }
    return (nd);
}

