/*
 ********************************************************************
 *
 * launch.h - header file for the application launcher
 *
 *     CSIRO Automation
 *     Queensland Centre for Advanced Technologies
 *     PO Box 883, Kenmore, QLD 4069, Australia
 *     www.cat.csiro.au/cmst
 *
 *	$Id $
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 ********************************************************************
 */
/**
 ********************************************************************
 *
 * \file launch.h
 * \brief header file for the application launcher
 * \author Pavan Sikka
 *
 *
 ********************************************************************
 */

#ifndef __launch_h__
#define __launch_h__

#include <rtx/main.h>
#include <rtx/message.h>
#include <rtx/thread.h>
#include <rtx/mutex.h>
#include <rtx/sem.h>
#include <rtx/lock.h>
#include <rtx/error.h>
#include <rtx/timer.h>
#include <rtx/list.h>
#include <rtx/command.h>
#include <rtx/inet.h>
#include <rtx/srpc.h>

#define LAUNCH_PORT       6500
#define LAUNCH_PRIO       21

/**
 * \file launch.h
 * \brief Header file for the system launcher
 */

/**
 * launch controller state
 */
typedef enum _launch_state {
    LAUNCH_STATE_NONE,
    LAUNCH_STATE_INIT,
    LAUNCH_STATE_STABLE,
    LAUNCH_STATE_CHANGE_UP,
    LAUNCH_STATE_CHANGE_DOWN,
    LAUNCH_STATE_MAX_NUM
} LaunchState;

/**
 * enum to identify applications that send a signal to the
 * launcher when they start up successfully 
 */
typedef enum _launch_app_type {
    LAUNCH_APP_NONE,
    LAUNCH_APP_NEUTRAL,
    LAUNCH_APP_AWARE,
    LAUNCH_APP_MAX_NUM
} LaunchAppType;

/**
 * enum for launch messages between launch and launchagent
 */
typedef enum _launch_msg_type {
    LAUNCH_MSG_NONE,
    LAUNCH_MSG_START,
    LAUNCH_MSG_LAUNCH,
    LAUNCH_MSG_LAUNCH_DONE,
    LAUNCH_MSG_LAUNCH_FAILED,
    LAUNCH_MSG_KILL,
    LAUNCH_MSG_KILL_DONE,
    LAUNCH_MSG_KILL_FAILED,
    LAUNCH_MSG_DONE,
    LAUNCH_MSG_MAX_NUM
} LaunchMsgType;

/**
 * node for the host list
 */
typedef struct _launch_host_node {
    char * hostName;
    RtxInetTcpClient * s;
    int status;
    RtxThread * writerTh;
    RtxList * reqQ;
    RtxSem * req;
    struct _launch * l;
} LaunchHostNode;

/**
 * structure used to store information about an application to be
 * launched by the launcher
 */
typedef struct _launch_app_node {
    char * hostName;
    LaunchHostNode * hostNd;
    char * appName;
    LaunchAppType appType;
    int timeout;
    pid_t appPid;
    char * command;
    int status;
    int level;
    RtxMutex mutex;
} LaunchAppNode;

/**
 * message node
 */
typedef struct _launch_msg_node {
    LaunchMsgType msgType;
    char appName[BUFSIZ];
    LaunchAppNode * appNd;
    char hostName[BUFSIZ];
    LaunchAppType appType;
    int timeout;
    pid_t appPid;
    char command[BUFSIZ];
    char errMsg[BUFSIZ];
    char msg[BUFSIZ];
} LaunchMsgNode;

/**
 * main data structure used by the launchagent
 */
typedef struct _launch_agent {
    char myHostName[BUFSIZ];
    pid_t myPid;
    int baseprio;
    int verbose;
    int done;
    int errors;
	int cleanup;
    char * launchHostName;
    char * dir;
	char * confdir;
    RtxInet * launchSock;
    RtxThread * readerTh;
    RtxThread * writerTh;
    RtxThread * childUpTh;
    RtxThread * childDoneTh;
    RtxList * apps;
    RtxList * events;
    RtxMutex * eventQ;
    RtxSem * eventSem;
} LaunchAgent;

/**
 * structure that binds together various bits of information 
 * and buffers used by the launcher
 */
typedef struct _launch {
    char * configFileName;
    char * remoteShell;
    int errors;
    int done;
    int commandLineReqd;
    int printAppInfo;
    int baseprio;
    int srpcPortNum;
    int shutdownTimeout;
    int userSpecifiedLevel;
    int logTransitions;
    char myHostName[BUFSIZ];
    char dir[BUFSIZ];
    char ** defs;
    pid_t myPid;
    RtxList * agents;
    RtxList * apps;
    RtxList * appsLevel[256];
    RtxList * events;
    int curAppLevel;
    int desiredAppLevel;
    int userDesLevel;
    int workingLevel;
    int maxAppLevel;
    int retryMode;
    int verbose;
    int webFlag;
    int webPort;
	int cleanup;
    RtxThread * controlTh;
    RtxInet * agentServer;
    RtxThread * timeoutTh;
    RtxTimer * tmr;
    int timerThreadRunning;
    int timerFired;
    RtxSem * eventSem;
    LaunchState state;
    RtxMutex * eventQ;
    RtxList * appStack;
    LaunchAppNode * curApp;
    char buf[BUFSIZ];
    char buf1[BUFSIZ];
    char buf2[BUFSIZ];
    char buf3[BUFSIZ];
    RtxSrpc * srpcServer;
	char * confdir;
} Launch;

/* functions/variables in other modules */

/* launch.c */

int launch_get_next_app_level_up (Launch * l, int level);
int launch_get_next_app_level_down (Launch * l, int level);

/* config.c */

int launch_read_config_file (Launch * l);

/* msg.c */

LaunchMsgNode * launch_parse_msg (char * msg);
LaunchMsgNode * launch_create_msg (LaunchMsgType msgType,
				   char * name);

/* rtc.c */

extern RtxCommandMenu launchCmds[];

#endif


