/*
 ********************************************************************
 *
 * launch.c - application launcher
 *
 *     CSIRO Automation
 *     Queensland Centre for Advanced Technologies
 *     PO Box 883, Kenmore, QLD 4069, Australia
 *     www.cat.csiro.au/cmst
 *
 *      $Id: launch.c 1777 2007-05-29 03:13:32Z pra077 $
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 ********************************************************************
 */

static char *rcsid = "$Id: launch.c 1777 2007-05-29 03:13:32Z pra077 $";

/**
 ********************************************************************
 * 
 * \file launch.c
 * \brief Application launcher
 * \author Pavan Sikka
 */

/**
 * \page launcher Launcher
 *
 * 
 */

#include <unistd.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sched.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <poll.h>

#include <rtx/main.h>
#include <rtx/message.h>
#include <rtx/thread.h>
#include <rtx/mutex.h>
#include <rtx/sem.h>
#include <rtx/auth.h>
#include <rtx/lock.h>
#include <rtx/error.h>
#include <rtx/timer.h>
#include <rtx/signal.h>
#include <rtx/list.h>
#include <rtx/lock.h>
#include <rtx/command.h>
#include <rtx/getopt.h>

#include "launch.h"

#define LAUNCH_NET_DELAY            10
#define LAUNCH_AGENT_TIMEOUT        30
#define LAUNCH_WEBSERVER_PORT       8080

/* forward declarations */

void * launch_control_thread (void * arg);
void * launch_agent_reader_thread (void * p, void * arg);
void * launch_agent_writer_thread (void * arg);
void * launch_timeout_thread (void * arg);
int launch_srpc_start (Launch * l);
int launch_timeout_set (Launch * l, double tm);
int  launch_start_agent (Launch * l, LaunchHostNode * nd);
void launch_shutdown_thread (Launch * l, int errors);
int launch_startup (Launch * l);
int launch_shutdown (Launch * l);
int launch_start_app (Launch * l, LaunchAppNode * nd);
int launch_stop_app (Launch * l, LaunchAppNode * nd);
int launch_write_log (Launch * l);


/**
 * reader thread
 *
 * this function is launched by the TCP/IP server thread
 * for each launchagent program. This thread handles the
 * messages from the corresponding launchagent program. It
 * parses the messages received, constructs the messages 
 * and then adds them to the event Q.
 */
void *
launch_agent_reader_thread (     
		void * p,     /**< RtxInetTcpClient * */
		void * arg    /**< Launch * */
		)
{
	Launch * l = (Launch *) arg;
	RtxInetTcpClient * clnt = (RtxInetTcpClient *) p;
	char * buf = NULL;
	int n;
	LaunchMsgNode * nd = NULL;
	LaunchHostNode * hostNd = NULL;

	if ((buf = calloc (1, 1024)) == NULL) {
		rtx_error_flush ("unable to allocate buffer");
		launch_shutdown_thread (l, 1);
	}
	if (l->verbose)
		rtx_message ("launch_agent_reader_thread: up");
	while (! l->done) {
		if (l->verbose)
			rtx_message ("launch_agent_reader_thread: "
					"waiting for msg");
		if ((n = rtx_inet_readline (clnt->sock, buf, 1024, NULL))
				== -1) {
			if (l->verbose)
				rtx_message ("rtx_inet_readline [%d], done = %d",
						n, l->done);
			if (! l->done) {
				rtx_error_flush ("rtx_inet_readline");
				launch_shutdown_thread (l, 1);
			} else {
				rtx_error_traceback_depth (0);
				continue;  
			}
		}
		if (l->done)
			continue;
		if (l->verbose)
			rtx_message ("launch_agent_reader_thread: msg [%d]", n);
		if (n == 0) {
			/* end-of-file */
			rtx_error_flush ("socket closed");
			launch_shutdown_thread (l, 1);
		}
		if (l->verbose)
			rtx_message ("launch_agent_reader_thread: rx: %s", buf);
		/* Have a complete message for processing */
		if ((nd = launch_parse_msg (buf)) == NULL) {
			rtx_error_flush ("launch_parse_msg() failed [%s]",
					buf);
			continue;
		}
		if (l->verbose)
			rtx_message ("launch_agent_reader_thread: msg parsed");
		/* Have a valid message for processing */
		switch (nd->msgType) {
			case LAUNCH_MSG_START :
				if (l->verbose)
					rtx_message ("launch_agent_reader_thread: "
							"LAUNCH_MSG_START");
				if ((hostNd = (LaunchHostNode *) rtx_list_lookup 
							(l->agents, nd->hostName)) == NULL) {
					rtx_error_flush ("agent node [%s] not found",
							nd->hostName);
					launch_shutdown_thread (l, 1);
					break;
				}
				if (l->verbose)
					rtx_message ("launch_agent_reader_thread: "
							"host [%s]", hostNd->hostName);
				/* found the node, fill it in */
				hostNd->s = clnt;
				/* start the writer thread */
				if ((hostNd->writerTh = rtx_thread_create 
							("launch[writer]", (l->verbose>1)?1:0, 
							 RTX_THREAD_SCHED_OTHER, 0, 0,
							 RTX_THREAD_CANCEL_DEFERRED, 
							 launch_agent_writer_thread, hostNd, 
							 NULL, NULL)) == NULL) {
					rtx_error ("rtx_thread_create("
							"agent_writer_thread) failed");
					launch_shutdown_thread (l, 1);
					break;
				}
				if (l->verbose)
					rtx_message ("launch_agent_reader_thread: "
							"writer thread started");
				hostNd->status = 1;
				if (rtx_list_enqueue (l->events, nd) == -1) {
					rtx_error_flush ("rtx_list_enqueue() "
							"failed");
					launch_shutdown_thread (l, 1);
					break;
				}
				if (l->verbose)
					rtx_message ("launch_agent_reader_thread: "
							"msg enqueued");
				if (rtx_sem_post (l->eventSem) == -1) {
					launch_shutdown_thread (l, 1);
					break;
				}
				if (l->verbose)
					rtx_message ("launch_agent_reader_thread: "
							"sem posted");
				break;
			case LAUNCH_MSG_LAUNCH_DONE :
				if ((nd->appNd = (LaunchAppNode *) rtx_list_lookup 
							(l->apps, nd->appName)) == NULL) {
					rtx_error_flush ("app node not found [%s]",
							nd->appName);
					launch_shutdown_thread (l, 1);
					break;
				}
				/* found the node, fill it in */
				nd->appNd->status = 1;
				nd->appNd->appPid = nd->appPid;
				if (l->printAppInfo)
					rtx_message ("%s [%d] up", nd->appNd->appName,
							(int) nd->appNd->appPid);
				if (rtx_list_enqueue (l->events, nd) == -1) {
					rtx_error_flush ("rtx_list_enqueue() "
							"failed");
					launch_shutdown_thread (l, 1);
					break;
				}
				if (rtx_sem_post (l->eventSem) == -1) {
					rtx_error_flush ("rtx_sem_post() "
							"failed");
					launch_shutdown_thread (l, 1);
					break;
				}
				break;
			case LAUNCH_MSG_LAUNCH_FAILED :
				if ((nd->appNd = (LaunchAppNode *) rtx_list_lookup 
							(l->apps, nd->appName)) == NULL) {
					rtx_error_flush ("app node not found [%s]",
							nd->appName);
					launch_shutdown_thread (l, 1);
					break;
				}
				/* found the node, fill it in */
				nd->appNd->status = 0;
				if (rtx_list_enqueue (l->events, nd) == -1) {
					rtx_error_flush ("rtx_list_enqueue() "
							"failed");
					launch_shutdown_thread (l, 1);
					break;
				}
				if (rtx_sem_post (l->eventSem) == -1) {
					rtx_error_flush ("rtx_sem_post() "
							"failed");
					launch_shutdown_thread (l, 1);
					break;
				}
				break;
			case LAUNCH_MSG_KILL_FAILED :
				if ((nd->appNd = (LaunchAppNode *) rtx_list_lookup 
							(l->apps, nd->appName)) == NULL) {
					rtx_error_flush ("app node not found [%s]",
							nd->appName);
					launch_shutdown_thread (l, 1);
					break;
				}
				/* found the node, fill it in */
				if (rtx_list_enqueue (l->events, nd) == -1) {
					rtx_error_flush ("rtx_list_enqueue() "
							"failed");
					launch_shutdown_thread (l, 1);
					break;
				}
				if (rtx_sem_post (l->eventSem) == -1) {
					rtx_error_flush ("rtx_sem_post() "
							"failed");
					launch_shutdown_thread (l, 1);
					break;
				}
				break;
			case LAUNCH_MSG_KILL_DONE :
				if ((nd->appNd = (LaunchAppNode *) rtx_list_lookup 
							(l->apps, nd->appName)) == NULL) {
					rtx_error_flush ("app node not found [%s]",
							nd->appName);
					launch_shutdown_thread (l, 1);
					break;
				}
				/* found the node, fill it in */
				nd->appNd->status = 0;
				if (l->printAppInfo)
					rtx_message ("%s [%d] down", nd->appNd->appName,
							(int) nd->appNd->appPid);
				if (rtx_list_enqueue (l->events, nd) == -1) {
					rtx_error_flush ("rtx_list_enqueue() "
							"failed");
					launch_shutdown_thread (l, 1);
					break;
				}
				if (rtx_sem_post (l->eventSem) == -1) {
					rtx_error_flush ("rtx_sem_post() "
							"failed");
					launch_shutdown_thread (l, 1);
					break;
				}
				break;
			case LAUNCH_MSG_LAUNCH :
			case LAUNCH_MSG_KILL :
			case LAUNCH_MSG_DONE :
			case LAUNCH_MSG_NONE :
			default :
				rtx_error_flush ("message not valid [%s]", buf);
				break;
		}
	}
	free (buf);
	return (NULL);

}

/**
 * writer thread
 *
 * this thread is launched by the reader thread. It is responsible
 * for sending messages to the launchagent program. It provides
 * a message Q to which other threads can write the messages they
 * want sent to the launchagent.
 */
void *
launch_agent_writer_thread (
		void * arg     /**< LaunchHostNode * */
		)
{
	LaunchHostNode * nd = (LaunchHostNode *) arg;
	Launch * l = nd->l;
	LaunchMsgNode * msgNd = NULL;
	char * buf = NULL;

	if ((buf = calloc (1, 1024)) == NULL) {
		rtx_error_flush ("unable to allocate buffer");
		launch_shutdown_thread (l, 1);
	}
	if (l->verbose)
		rtx_message ("launch_agent_writer_thread: up");
	while (! l->done) {
		if (rtx_sem_wait (nd->req) == -1) {
			rtx_error_flush ("rtx_sem_wait() failed");
			launch_shutdown_thread (l, 1);
			break;
		}
		if (l->verbose)
			rtx_message ("launch_agent_writer_thread: "
					"request signalled");
		/* a request is now available in the Q */
		if ((msgNd = (LaunchMsgNode *) rtx_list_dequeue 
					(nd->reqQ)) == NULL) {
			rtx_error_flush ("rtx_list_dequeue() --> NULL");
			launch_shutdown_thread (l, 1);
			break;
		}
		if (l->verbose)
			rtx_message ("launch_agent_writer_thread: "
					"got request");
		/* act on the request */
		switch (msgNd->msgType) {
			case LAUNCH_MSG_LAUNCH :
				sprintf (buf, "LAUNCH %s %d %d %s\n", 
						msgNd->appNd->appName,
						msgNd->appNd->appType, 
						msgNd->appNd->timeout, 
						msgNd->appNd->command);
				if (l->verbose)
					rtx_message ("launch_agent_writer_thread: "
							"LAUNCH_MSG_LAUNCH [%s]", buf);
				if (rtx_inet_write (nd->s->sock, buf, strlen (buf), NULL) 
						== -1) {
					rtx_error_flush ("rtx_inet_write() failed");
					launch_shutdown_thread (l, 1);
					break;
				}
				break;
			case LAUNCH_MSG_KILL :
				sprintf (buf, "KILL %s\n", msgNd->appName);
				if (l->verbose)
					rtx_message ("launch_agent_writer_thread: "
							"LAUNCH_MSG_KILL [%s]", buf);
				if (rtx_inet_write (nd->s->sock, buf, strlen (buf), NULL) 
						== -1) {
					rtx_error_flush ("rtx_inet_write() failed");
					launch_shutdown_thread (l, 1);
					break;
				}
				break;
			case LAUNCH_MSG_DONE :
				/*
				   sprintf (buf, "DONE\n");
				   if (l->verbose)
				   rtx_message ("launch_agent_writer_thread: "
				   "LAUNCH_MSG_DONE [%s]", buf);
				   if (rtx_inet_write (nd->s->sock, buf, strlen (buf), NULL) 
				   == -1) {
				   rtx_error_flush ("rtx_inet_write() failed");
				   launch_shutdown_thread (l, 1);
				   break;
				   }
				   break;
				   */
			case LAUNCH_MSG_START :
			case LAUNCH_MSG_LAUNCH_DONE :
			case LAUNCH_MSG_LAUNCH_FAILED :
			case LAUNCH_MSG_KILL_FAILED :
			case LAUNCH_MSG_KILL_DONE :
			case LAUNCH_MSG_NONE :
			default :
				rtx_error_flush ("message not valid [%s]", buf);
				break;
		}
	}	
	return (NULL);
}

/**
 * control thread
 *
 * this thread is responsible for maintaining the run-level
 * desired by the user. It does this by implementing a simple
 * state machine.
 */
void *
launch_control_thread (
		void * arg    /**< Launch * */
		)
{
	Launch * l = (Launch *) arg;
	LaunchMsgNode * msgNd = NULL;
	LaunchAppNode * appNd = NULL;
	int killTimeout = 0;
	int numAgentsUp = 0, numAppsUp = 0, numAppsDown = 0;

	if (l->verbose)
		rtx_message ("launch_control_thread: up");
	if (launch_timeout_set (l, 30.0) == -1) {
		launch_shutdown_thread (l, 1);
	}
	while (! l->done) {
		/* wait for an event */
		if (rtx_sem_wait (l->eventSem) == -1) {
			launch_shutdown_thread (l, 1);
			break;
		}
		if (l->done)
			continue;
		msgNd = (LaunchMsgNode *) rtx_list_dequeue (l->events);
		if (l->verbose)
			rtx_message ("launch_control_thread: processing event");
		/* we have an event to be processed */
		switch (l->state) {
			case LAUNCH_STATE_INIT :
				if (l->verbose)
					rtx_message ("launch_control_thread: "
							"LAUNCH_STATE_INIT");
				l->workingLevel = -1;
				if (l->timerFired) {
					/* an agent is not starting up - exit */
					rtx_error ("timer expired: agent not up");
					launch_shutdown_thread (l, 1);
					l->timerFired = 0;
					break;
				}
				if (msgNd == NULL) {
					if (l->verbose)
						rtx_message ("launch_control_thread: "
								"no message node");
					continue;
				}
				switch (msgNd->msgType) {
					case LAUNCH_MSG_START :
						numAgentsUp++;
						if (l->verbose)
							rtx_message ("launch_control_thread: "
									"LAUNCH_MSG_START [%s]",
									msgNd->hostName);
						if (numAgentsUp == l->agents->n) {
							l->state = LAUNCH_STATE_STABLE;
							if (launch_timeout_set (l, 0.0) == -1) {
								launch_shutdown_thread (l, 1);
								break;
							}
						} else {
							if (launch_timeout_set (l, 30.0) == -1) {
								launch_shutdown_thread (l, 1);
								break;
							}
						}
						if (rtx_sem_post (l->eventSem) == -1) {
							rtx_error_flush ("rtx_sem_post");
							launch_shutdown_thread (l, 1);
							break;
						}
						break;
					case LAUNCH_MSG_LAUNCH :
					case LAUNCH_MSG_KILL :
					case LAUNCH_MSG_DONE :
					case LAUNCH_MSG_LAUNCH_DONE :
					case LAUNCH_MSG_LAUNCH_FAILED :
					case LAUNCH_MSG_KILL_FAILED :
					case LAUNCH_MSG_KILL_DONE :
					case LAUNCH_MSG_NONE :
					default :
						rtx_error_flush ("message not valid [%d]",
								msgNd->msgType);
						break;
				}   
				break;
			case LAUNCH_STATE_STABLE :
				if (l->verbose)
					rtx_message ("launch_control_thread: "
							"LAUNCH_STATE_STABLE");
				if (launch_timeout_set (l, 0.0) == -1) {
					launch_shutdown_thread (l, 1);
					break;
				}
				rtx_message( " stable at level %d", l->curAppLevel);

				l->workingLevel = -1;
				if (l->desiredAppLevel > l->curAppLevel) {
					l->state = LAUNCH_STATE_CHANGE_UP;
					if (rtx_sem_post (l->eventSem) == -1) {
						rtx_error_flush ("rtx_sem_post");
						launch_shutdown_thread (l, 1);
						break;
					}
					break;
				}
				if (l->desiredAppLevel < l->curAppLevel) {
					l->state = LAUNCH_STATE_CHANGE_DOWN;
					if (rtx_sem_post (l->eventSem) == -1) {
						rtx_error_flush ("rtx_sem_post");
						launch_shutdown_thread (l, 1);
						break;
					}
					break;
				}
				if (msgNd == NULL) {
					if (l->retryMode) {
						if (l->desiredAppLevel != l->userDesLevel) {
							rtx_message ("retry mode: restarting %d -> %d after 10 s",
									l->desiredAppLevel, l->userDesLevel);
							sleep (10);
							l->desiredAppLevel = l->userDesLevel;
							if (rtx_sem_post (l->eventSem) == -1) {
								rtx_error_flush ("rtx_sem_post");
								launch_shutdown_thread (l, 1);
								break;
							}
						}
					}
					break;
				}
				switch (msgNd->msgType) {
					case LAUNCH_MSG_KILL_DONE :
						/* an app has died unexpectedly */
						l->desiredAppLevel = 
							launch_get_next_app_level_down 
							(l, msgNd->appNd->level);
						rtx_message( " %s [level %d] "
								"down unexpectedly, going to level %d",
								msgNd->appNd->appName,
								msgNd->appNd->level,
								l->desiredAppLevel);
						if (rtx_sem_post (l->eventSem) == -1) {
							rtx_error_flush ("rtx_sem_post");
							launch_shutdown_thread (l, 1);
							break;
						}
						break;
					case LAUNCH_MSG_START :
					case LAUNCH_MSG_LAUNCH :
					case LAUNCH_MSG_KILL :
					case LAUNCH_MSG_KILL_FAILED :
					case LAUNCH_MSG_DONE :
					case LAUNCH_MSG_LAUNCH_DONE :
					case LAUNCH_MSG_LAUNCH_FAILED :
					case LAUNCH_MSG_NONE :
					default :
						rtx_error_flush ("message not valid [%d]",
								msgNd->msgType);
						break;
				}   
				break;
			case LAUNCH_STATE_CHANGE_UP :
				if (l->verbose)
					rtx_message ("launch_control_thread: "
							"LAUNCH_STATE_CHANGE_UP");
				if (l->workingLevel == -1) {
					l->workingLevel = launch_get_next_app_level_up (l, l->curAppLevel);
					/*
					   for (i=l->curAppLevel+1; i<l->desiredAppLevel;
					   i++)
					   if (l->appsLevel[i]->n > 0)
					   break;
					   l->workingLevel = i;
					   */
					rtx_message( " level %d --> %d",
							l->curAppLevel, l->workingLevel);
					if (l->verbose)
						rtx_message ("launch_control_thread: "
								"cur/working level = %d/%d",
								l->curAppLevel, 
								l->workingLevel);
					numAppsUp = 0;
					if ((l->appStack = rtx_list_init ()) == NULL) {
						rtx_error_flush ("launch_start_app");
						launch_shutdown_thread (l, 1);
					}
					while ((appNd = (LaunchAppNode *) 
								rtx_list_iterate 
								(l->appsLevel[l->workingLevel])) != NULL)
						if (rtx_list_push (l->appStack, appNd) 
								== -1) {
							rtx_error_flush ("launch_start_app");
							launch_shutdown_thread (l, 1);
						}
					if ((l->curApp = (LaunchAppNode *) rtx_list_pop
								(l->appStack)) == NULL) {
						rtx_error_flush ("launch_start_app");
						launch_shutdown_thread (l, 1);
					}
					if (launch_start_app (l, l->curApp) == -1) {
						rtx_error_flush ("launch_start_app");
						launch_shutdown_thread (l, 1);
					}
					if (launch_timeout_set 
							(l, (double) l->curApp->timeout) == -1) {
						launch_shutdown_thread (l, 1);
						break;
					}
				}
				if (l->verbose)
					rtx_message ("launch_control_thread: "
							"working level [%d]", 
							l->workingLevel);
				if (l->timerFired) {
					/* an app is not starting up - give up */
					rtx_message( " %s timedout",
							l->curApp->appName);
					rtx_list_destroy (l->appStack, 0);
					l->desiredAppLevel = l->curAppLevel;
					l->curAppLevel = l->workingLevel;
					l->workingLevel = -1;
					l->state = LAUNCH_STATE_CHANGE_DOWN;
					if (launch_timeout_set (l, 0.0) == -1) {
						launch_shutdown_thread (l, 1);
						break;
					}
					if (rtx_sem_post (l->eventSem) == -1) {
						rtx_error_flush ("rtx_sem_post");
						launch_shutdown_thread (l, 1);
						break;
					}
					continue;
				}
				if (msgNd == NULL) {
					if (l->verbose)
						rtx_message ("empty event Q");
					continue;
				}
				switch (msgNd->msgType) {
					case LAUNCH_MSG_LAUNCH_DONE :
						rtx_message( " %s up [level"
								" %d]", msgNd->appNd->appName,
								msgNd->appNd->level);
						if (msgNd->appNd->level == l->workingLevel)
							numAppsUp++;
						if (numAppsUp == 
								l->appsLevel[l->workingLevel]->n) {
							rtx_list_destroy (l->appStack, 0);
							l->curAppLevel = l->workingLevel;
							l->state = LAUNCH_STATE_STABLE;
							if (launch_timeout_set (l, 0.0) == -1) {
								launch_shutdown_thread (l, 1);
								break;
							}
							if (rtx_sem_post (l->eventSem)
									== -1) {
								rtx_error_flush ("rtx_sem_post");
								launch_shutdown_thread (l, 1);
								break;
							}
						} else {
							if ((l->curApp = (LaunchAppNode *) 
										rtx_list_pop (l->appStack))
									== NULL) {
								rtx_error_flush ("launch_start_app");
								launch_shutdown_thread (l, 1);
							}
							if (launch_start_app (l, l->curApp)
									== -1) {
								rtx_error_flush ("launch_start_app");
								launch_shutdown_thread (l, 1);
							}
							if (launch_timeout_set 
									(l, (double) l->curApp->timeout)
									== -1) {
								launch_shutdown_thread (l, 1);
								break;
							}
						}
						break;
					case LAUNCH_MSG_LAUNCH_FAILED :
						/* bad - an app failed to launch */
						if (launch_timeout_set (l, 0.0) == -1) {
							launch_shutdown_thread (l, 1);
							break;
						}
						rtx_message( " %s [level %d] "
								"failed to launch",
								msgNd->appNd->appName,
								msgNd->appNd->level);
						rtx_list_destroy (l->appStack, 0);
						l->desiredAppLevel = l->curAppLevel;
						l->curAppLevel = l->workingLevel;
						l->workingLevel = -1;
						l->state = LAUNCH_STATE_CHANGE_DOWN;
						if (rtx_sem_post (l->eventSem) == -1) {
							rtx_error_flush ("rtx_sem_post");
							launch_shutdown_thread (l, 1);
							break;
						}
						break;
					case LAUNCH_MSG_KILL_DONE :
						/* bad - an app failed */
						/* an app has died unexpectedly */
						if (launch_timeout_set (l, 0.0) == -1) {
							launch_shutdown_thread (l, 1);
							break;
						}
						rtx_message( " %s [level %d] "
								"failed",
								msgNd->appNd->appName,
								msgNd->appNd->level);
						rtx_list_destroy (l->appStack, 0);
						l->desiredAppLevel = 
							launch_get_next_app_level_down
							(l, msgNd->appNd->level);
						l->curAppLevel = l->workingLevel;
						l->workingLevel = -1;
						l->state = LAUNCH_STATE_CHANGE_DOWN;
						if (rtx_sem_post (l->eventSem) == -1) {
							rtx_error_flush ("rtx_sem_post");
							launch_shutdown_thread (l, 1);
							break;
						}
						break;
					case LAUNCH_MSG_KILL_FAILED :
					case LAUNCH_MSG_START :
					case LAUNCH_MSG_LAUNCH :
					case LAUNCH_MSG_KILL :
					case LAUNCH_MSG_DONE :
					case LAUNCH_MSG_NONE :
					default :
						rtx_error_flush ("message not valid [%d]", msgNd->msgType); 
						break;
				}   
				break;
			case LAUNCH_STATE_CHANGE_DOWN :
				if (l->verbose)
					rtx_message ("launch_control_thread: "
							"LAUNCH_STATE_CHANGE_DOWN");
				if (l->curAppLevel == 0) {
					l->state = LAUNCH_STATE_STABLE;
					if (rtx_sem_post (l->eventSem)
							== -1) {
						rtx_error_flush ("rtx_sem_post");
						launch_shutdown_thread (l, 1);
						break;
					}
					continue;
				}
				if (l->workingLevel == -1) {
					l->workingLevel = launch_get_next_app_level_down (l, l->curAppLevel);
					/*
					   for (i=l->curAppLevel-1; i>l->desiredAppLevel;
					   i--)
					   if (l->appsLevel[i]->n > 0)
					   break;
					   l->workingLevel = i;
					   */
					rtx_message( " level %d --> %d",
							l->curAppLevel, l->workingLevel);
					numAppsDown = 0;
					killTimeout = 0;
					while ((appNd = (LaunchAppNode *) 
								rtx_list_iterate 
								(l->appsLevel[l->curAppLevel])) != NULL)
						if (appNd->status) {
							if (launch_stop_app (l, appNd) == -1) {
								rtx_error_flush ("launch_stop_app");
								launch_shutdown_thread (l, 1);
							}
							numAppsDown++;
							if (appNd->timeout > killTimeout)
								killTimeout = appNd->timeout;
						}
					if (launch_timeout_set 
							(l, (double) killTimeout) == -1) {
						launch_shutdown_thread (l, 1);
						break;
					}
				}
				if (numAppsDown == 0) {
					if (launch_timeout_set (l, 0.0) == -1) {
						launch_shutdown_thread (l, 1);
						break;
					}
					l->curAppLevel = l->workingLevel;
					l->state = LAUNCH_STATE_STABLE;
					if (rtx_sem_post (l->eventSem)
							== -1) {
						rtx_error_flush ("rtx_sem_post");
						launch_shutdown_thread (l, 1);
						break;
					}
				}
				if (l->timerFired) {
					/* one of the apps has failed to shutdown */
					/* try again, this time launchagent will use sigkill */
					rtx_message("Timeout!!! I'll kill them ALL");
					while ((appNd = (LaunchAppNode *) 
								rtx_list_iterate 
								(l->appsLevel[l->curAppLevel])) != NULL) {
						if (appNd->status) {
							if (launch_stop_app (l, appNd) == -1) {
								rtx_error_flush ("launch_stop_app");
								launch_shutdown_thread (l, 1);
							}
						}
					}
				}
				if (msgNd == NULL) {
					continue;
				}
				switch (msgNd->msgType) {
					case LAUNCH_MSG_KILL_FAILED :
						rtx_message( " failed to kill "
								"%s [level %d]",
								msgNd->appNd->appName,
								msgNd->appNd->level);
						numAppsDown--;
						if (numAppsDown == 0) {
							l->curAppLevel = l->workingLevel;
							l->state = LAUNCH_STATE_STABLE;
							if (launch_timeout_set (l, 0.0) == -1) {
								launch_shutdown_thread (l, 1);
								break;
							}
							if (rtx_sem_post (l->eventSem)
									== -1) {
								rtx_error_flush ("rtx_sem_post");
								launch_shutdown_thread (l, 1);
								break;
							}
						}
						if (launch_timeout_set 
								(l, (double) killTimeout) == -1) {
							launch_shutdown_thread (l, 1);
							break;
						}
						break;
					case LAUNCH_MSG_KILL_DONE :
						rtx_message( " %s down [level"
								" %d]", msgNd->appNd->appName,
								msgNd->appNd->level);
						if (msgNd->appNd->level == l->curAppLevel)
							numAppsDown--;
						if (numAppsDown == 0) {
							l->curAppLevel = l->workingLevel;
							l->state = LAUNCH_STATE_STABLE;
							if (launch_timeout_set (l, 0.0) == -1) {
								launch_shutdown_thread (l, 1);
								break;
							}
							if (rtx_sem_post (l->eventSem)
									== -1) {
								rtx_error_flush ("rtx_sem_post");
								launch_shutdown_thread (l, 1);
								break;
							}
						}
						if (launch_timeout_set 
								(l, (double) killTimeout) == -1) {
							launch_shutdown_thread (l, 1);
							break;
						}
						break;
					case LAUNCH_MSG_LAUNCH_DONE :
					case LAUNCH_MSG_LAUNCH_FAILED :
					case LAUNCH_MSG_START :
					case LAUNCH_MSG_LAUNCH :
					case LAUNCH_MSG_KILL :
					case LAUNCH_MSG_DONE :
					case LAUNCH_MSG_NONE :
					default :
						rtx_error_flush ("message not valid [%d]", msgNd->msgType);
						break;
				}   
				break;
			case LAUNCH_STATE_NONE :
			case LAUNCH_STATE_MAX_NUM :
			default :
				break;
		}
	}
	return (NULL);
}

/**
 * timeout thread
 *
 * this thread is used by the control thread to implement timeouts.
 * I wish there was a better way to do this ...
 */
void *
launch_timeout_thread (
		void * arg     /**< launch * */
		)
{
	Launch * l = (Launch *) arg;

	if ((l->tmr = rtx_timer_create (0.0, 0.0, NULL, NULL, 0
					)) == NULL) {
		rtx_error_flush ("rtx_timer_create");
		launch_shutdown_thread (l, 1);
	}
	l->timerThreadRunning = 1;
	if (launch_timeout_set (l, 30.0) == -1) {
		launch_shutdown_thread (l, 1);
	}
	while (! l->done) {
		if (l->verbose)
			rtx_message ("waiting for timer signal");
		if (rtx_timer_wait (l->tmr) == -1) {
			rtx_error_flush ("rtx_timer_wait");
			launch_shutdown_thread (l, 1);
			break;
		}
		if (l->verbose)
			rtx_message ("timer fired");
		l->timerFired++;
		if (l->verbose)
			rtx_message ("posting sem");
		if (rtx_sem_post (l->eventSem) == -1) {
			rtx_error_flush ("rtx_sem_post");
			launch_shutdown_thread (l, 1);
			break;
		}
		if (l->verbose)
			rtx_message ("sem posted");
	}
	l->timerThreadRunning = 0;
	return (NULL);
}

/**
 * set the timeout
 */
int
launch_timeout_set (
		Launch * l,      /**< Launch * */
		double tm        /**< timeout */
		)
{
	if (l->timerThreadRunning == 0)
		return (0);

	if (tm < 0.01) {
		if (rtx_timer_stop (l->tmr) == -1)
			return (rtx_error ("launch_timeout_set: "
						"rtx_timer_stop"));
		l->timerFired = 0;
		return (0);
	}
	if (rtx_timer_set (l->tmr, tm, 0.0) == -1)
		return (rtx_error (" launch_timeout_set: "
					"rtx_timer_set"));
	if (rtx_timer_start (l->tmr) == -1)
		return (rtx_error (" launch_timeout_set: "
					"rtx_timer_start"));

	return (0);
}

/**
 * start launchagent on the specified host
 */
int 
launch_start_agent (
		Launch * l,          /**< Launch * */
		LaunchHostNode * nd  /**< launchagent node */
		)
{
	pid_t childPid;
	char verb[16],clnup[16];

	if (!l->cleanup) {
		/* initialize the request semaphore and list */
		if ((nd->reqQ = rtx_list_init ()) == NULL)
			return (rtx_error_errno ("launch_start_agent (%s): "
						"rtx_list_init", nd->hostName));
		if ((nd->req = rtx_sem_init (NULL, 0, 0)) == NULL)
			return (rtx_error_errno ("launch_start_agent (%s): "
						"rtx_sem_init", nd->hostName));
	}

	if ((childPid = fork ()) == -1)
		return (rtx_error_errno ("launch_start_agent (%s): "
					"fork() failed", nd->hostName));

	if (childPid == 0) { /* child */
		/*
		   if (setsid () == -1) {
		   rtx_error_errno ("launch_start_agent (%s): "
		   "child setsid() failed [%s]",
		   nd->hostName);
		   exit (1);
		   }
		   close (STDIN_FILENO);
		   close (STDOUT_FILENO);
		   close (STDERR_FILENO);
		   */
		sprintf (verb, "%d", l->verbose);
		sprintf (clnup, "%d", l->cleanup);
		if (strcmp (nd->hostName, l->myHostName) != 0) {
			//rtx_message("LAUNCH: in forked process, exec %s (%s, %s)",
			//		l->remoteShell, nd->hostName, l->myHostName);
			int ret = 0;
			ret = execlp (l->remoteShell, l->remoteShell, nd->hostName, 
					"launchagent", "-debug", verb, "-launchhost", 
					l->myHostName, "-directory", l->dir, 
					"-cfgdir",l->confdir, "-preclean", clnup,
					(char *) 0);
			if (ret == -1) {
				rtx_error_errno ("launch_start_agent (%s): "
						"child execlp() failed [%s]",
						nd->hostName);
				exit (1);
			}
		} else {
			//rtx_message ("LAUNCH: in forked process, exec launchagent\n");
			if (execlp ("launchagent", "launchagent", 
						"-debug", verb, "-directory", l->dir,
						"-cfgdir",l->confdir, "-preclean", clnup,
						(char *) 0) == -1) {
				rtx_error_errno ("launch_start_agent (%s): "
						"child execlp() failed [%s]",
						nd->hostName);
				exit (1);
			}
		}
	}
	/* parent */
	return (0);
}

/**
 * add a LAUNCH message to the message Q of the specified app
 */
int
launch_start_app (
		Launch * l,          /**< launch * */
		LaunchAppNode * nd   /**< app to be launched */
		)
{
	LaunchMsgNode * msgNd = NULL;

	if (l->verbose)
		rtx_message ("launch_start_app [%s]", nd->appName);
	if ((msgNd = launch_create_msg (LAUNCH_MSG_LAUNCH, 
					nd->appName)) == NULL)
		return (rtx_error ("launch_start_app: launch_create_msg"));
	/* fill in the relevant fields */
	msgNd->appNd = nd;
	if (l->verbose)
		rtx_message ("launch_start_app [%s]: msg node created",
				nd->appName);
	if (rtx_list_enqueue (nd->hostNd->reqQ, msgNd) == -1)
		return (rtx_error ("launch_start_app: rtx_list_enqueue"));
	if (l->verbose)
		rtx_message ("launch_start_app [%s]: msg node enqueued",
				nd->appName);
	if (rtx_sem_post (nd->hostNd->req) == -1)
		return (rtx_error ("launch_start_app: rtx_sem_post"));
	if (l->verbose)
		rtx_message ("launch_start_app [%s]: req sem posted",
				nd->appName);
	return (0);
}

/**
 * add a KILL message to the message Q of the specified app
 */
int
launch_stop_app (
		Launch * l,           /**< launch * */
		LaunchAppNode * nd    /**< app to be killed */
		)
{
	LaunchMsgNode * msgNd = NULL;

	if ((msgNd = launch_create_msg (LAUNCH_MSG_KILL, 
					nd->appName)) == NULL)
		return (rtx_error ("launch_start_app: launch_create_msg"));
	/* fill in the relevant fields */
	msgNd->appNd = nd;
	if (rtx_list_enqueue (nd->hostNd->reqQ, msgNd) == -1)
		return (rtx_error ("launch_start_app: rtx_list_enqueue"));
	if (rtx_sem_post (nd->hostNd->req) == -1)
		return (rtx_error ("launch_start_app: rtx_sem_post"));
	return (0);
}

/**
 * shutdown launcher
 *
 * a helper function that can be used by threads to signal
 * the launcher to shutdown, usually as a result of an
 * internal error.
 */
void 
launch_shutdown_thread (
		Launch * l,    /**< Launch * */
		int errors     /**< error count */
		)
{
	l->errors += errors;
	rtx_message ("launch_shutdown_thread");
	if (! l->done)
		rtx_main_signal_shutdown ();
	pthread_exit (NULL);
}

/**
 * startup the launcher
 *
 * initialize all the data-structures and create all the
 * threads.
 */
int 
launch_startup (
		Launch * l      /**< Launch * */
		)
{
	LaunchHostNode * nd = NULL;

	/* ger cwd */
	if (getcwd (l->dir, BUFSIZ) == NULL)
		return (rtx_error ("launch_startup: cwd pathname longer "
					"than %d chars", BUFSIZ));

	if (rtx_signal_block_realtime () == -1)
		return (rtx_error ("launch_startup: "
					"rtx_signal_block_realtime"));

	/* block the relevant signals:
	 * CHLD is sent when a child terminates
	 * PIPE is sent if the program attempts to write
	 *      to a socket thats been closed at the other
	 *      end
	 */
	if (rtx_signal_block_realtime () == -1)
		return (rtx_error ("launch_startup: rtx_signal_block_realtime() failed"));
	if (rtx_signal_block (SIGCHLD) == -1)
		return (rtx_error ("launch_startup: rtx_signal_block() failed"));
	if (rtx_signal_block (SIGPIPE) == -1)
		return (rtx_error ("launch_startup: rtx_signal_block() failed"));

	/* initialize the lists */
	if ((l->events = rtx_list_init ()) == NULL)
		return (rtx_error ("launch_startup: rtx_list_init() failed"));

	/* initialize sems/mutexes */
	if ((l->eventSem = rtx_sem_init (NULL, 0, 0)) == NULL)
		return (rtx_error ("rtx_sem_init() failed"));
	if ((l->eventQ = rtx_mutex_init (NULL, RTX_MUTEX_DEFAULT, 
					0)) == NULL)
		return (rtx_error ("launch_startup: rtx_mutex_init() failed"));

	/* startup the timeout thread */
	if ((l->timeoutTh = rtx_thread_create 
				("launch[timeout]", (l->verbose>1)?1:0,
				 RTX_THREAD_SCHED_OTHER,
				 0, 0, RTX_THREAD_CANCEL_DEFERRED, 
				 launch_timeout_thread, l, NULL, NULL)) == NULL)
		return (rtx_error ("launch_startup: rtx_thread_create(control_thread)"));
	/* bring up the control thread */
	l->state = LAUNCH_STATE_INIT;
	if ((l->controlTh = rtx_thread_create 
				("launch[control]", (l->verbose>1)?1:0,
				 RTX_THREAD_SCHED_OTHER,
				 0, 0, RTX_THREAD_CANCEL_DEFERRED, 
				 launch_control_thread, l, NULL, NULL)) == NULL)
		return (rtx_error ("launch_startup: rtx_thread_create(control_thread)"));
	if (l->cleanup) {
		/* start up the launch agents in cleanup modes*/
		while ((nd = (LaunchHostNode *) rtx_list_iterate 
					(l->agents)) != NULL) {
			if (launch_start_agent (l, nd) == -1)
				return (rtx_error ("llaunch_startup: aunch_start_agent()"));
		}
		rtx_message("All launch agents started for cleanup.\n...Waiting 4 secs and going on");
		rtx_timer_sleep(4.0);
		l->cleanup = 0;
	}

	/* start up the TCP server for launch agents */
	if ((l->agentServer = rtx_inet_init 
				(RTX_INET_TCP_SERVER, NULL, LAUNCH_PORT, NULL, 0,
				 launch_agent_reader_thread, NULL, l)) == NULL)
		return (rtx_error ("launch_startup: rtx_inet_start_tcp_server()"));
	/* start up the launch agents */
	while ((nd = (LaunchHostNode *) rtx_list_iterate 
				(l->agents)) != NULL) {
		if (launch_start_agent (l, nd) == -1)
			return (rtx_error ("llaunch_startup: aunch_start_agent()"));
	}
	/* start up the SRPC server */
	if (launch_srpc_start (l) == -1)
		return (rtx_error ("launch_startup: launch_srpc_start() failed"));
	return (0);
}

/**
 * shutdown the launcher
 *
 * kill all the threads and then clean up all data-structures
 */
int 
launch_shutdown (
		Launch * l     /**< Launch * */
		)
{
	int errs = 0;
	LaunchHostNode * nd = NULL;
	char * doneCmd = "DONE\n";

	l->done = 1;

	if (l->verbose)
		rtx_message ("launch_shutdown: starting");
	/* stop the launch agents */
	while ((nd = (LaunchHostNode *) rtx_list_iterate 
				(l->agents)) != NULL)
		if (nd->status) {
			if (rtx_inet_write (nd->s->sock, doneCmd, 
						strlen (doneCmd), NULL) == -1) {
				errs++;
				rtx_error ("launch_shutdown: rtx_inet_write");
			}
		}
	if (l->verbose)
		rtx_message ("launch_shutdown: launchagent told");
	/* kill all threads */
	if (l->timeoutTh != NULL)
		if (rtx_thread_destroy_sync (l->timeoutTh) == -1) {
			rtx_error ("rtx_thread_destroy_sync(timeout_thread)");
			errs++;
		}
	if (l->verbose)
		rtx_message ("launch_shutdown: timeout thread done");
	rtx_sem_post (l->eventSem);
	if (l->controlTh != NULL)
		if (rtx_thread_destroy_sync (l->controlTh) == -1) {
			rtx_error ("rtx_thread_destroy_sync(control_thread)");
			errs++;
		}
	if (l->verbose)
		rtx_message ("launch_shutdown: control thread done");
	/* stop the TCP server */
	if (rtx_inet_done (l->agentServer) == -1) {
		errs++;
		rtx_error ("launch_shutdown: rtx_inet_tcp_server_done");
	}
	if (l->verbose)
		rtx_message ("launch_shutdown: tcp server done");
#if 0
	// The state file is managed by the clients now.
	if (launch_write_log (l) == -1) {
		rtx_error ("launch_write_log");
		errs++;
	}
#endif
	if (l->eventQ != NULL)
		if (rtx_mutex_destroy (l->eventQ)) {
			rtx_error ("rtx_mutex_destroy)");
			errs++;
		}
	if (l->eventSem != NULL)
		if (rtx_sem_destroy (l->eventSem)) {
			rtx_error ("rtx_mutex_destroy)");
			errs++;
		}
	if (l->apps != NULL)
		rtx_list_destroy (l->apps, 0);
	if (l->events != NULL)
		rtx_list_destroy (l->events, 0);

	return (errs);
}

int
launch_write_log (
		Launch * l
		)
{
	int i;
	FILE * fp = NULL;
	LaunchAppNode * nd = NULL;

	if ((fp = fopen ("/tmp/launchapps.kill", "w")) == NULL)
		return (rtx_error ("launch_write_log: fopen"));
	if (fprintf (fp, "#!/bin/sh\n\n") == -1)
		return (rtx_error ("launch_write_log: fprintf"));
	for (i=255; i>=0; i--) {
		if (l->appsLevel[i] != NULL) {
			while ((nd = (LaunchAppNode *) rtx_list_iterate
						(l->appsLevel[i])) != NULL) {
				if (nd->status) {
					rtx_message (" %s [%d] not down", 
							nd->appName, (int) nd->appPid);
					if (strcmp (nd->hostName, l->myHostName) == 0) {
						if (fprintf (fp, "kill %d\n", 
									(int) nd->appPid) == -1)
							return (rtx_error ("launch_write_log: "
										"fprintf"));
						if (fprintf (fp, "sleep 1\n") == -1)
							return (rtx_error ("launch_write_log: "
										"fprintf"));
					} else {
						if (fprintf (fp, "%s %s kill %d\n", 
									l->remoteShell, nd->hostName, 
									(int) nd->appPid) == -1)
							return (rtx_error ("launch_write_log: "
										"fprintf"));
						if (fprintf (fp, "sleep 1\n") == -1)
							return (rtx_error ("launch_write_log: "
										"fprintf"));
					}
				}
			}
		}
	}
	if (fclose (fp) == -1)
		return (rtx_error ("launch_write_log: fclose"));
	return (0);
}

int
launch_get_next_app_level_up (
		Launch * l,
		int level
		)
{
	int i;

	for (i=level+1; i<256; i++)
		if (l->appsLevel[i] != NULL)
			if (l->appsLevel[i]->n > 0)
				break;
	if (i >= 256) {
		return (l->maxAppLevel);
	}
	return (i);
}

int
launch_get_next_app_level_down (
		Launch * l,
		int level
		)
{
	int i;

	for (i=level-1; i>=0; i--)
		if (l->appsLevel[i] != NULL)
			if (l->appsLevel[i]->n > 0)
				break;
	if (i < 0) {
		return (0);
	}
	return (i);
}


/**
 * Initialize the launcher
 * */
static Launch launch;
int launch_init(Launch * l,int argc, char * argv[])
{
	/**
	 * The command-line options
	 */
	static RtxGetopt launchOpts[] = {
		{"interactive", "Interactive mode",
			{
				{RTX_GETOPT_SET, &launch.commandLineReqd, ""},
				RTX_GETOPT_END_ARG
			}
		},
		{"retry", "Retry mode",
			{
				{RTX_GETOPT_SET, &launch.retryMode, ""},
				RTX_GETOPT_END_ARG
			}
		},
		{"web", "Start a web server to access the launcher state",
			{
				{RTX_GETOPT_SET, &launch.webFlag, "bool"},
				RTX_GETOPT_END_ARG
			}
		},
		{"wport", "TCP port for the web server",
			{
				{RTX_GETOPT_INT, &launch.webPort, "int"},
				RTX_GETOPT_END_ARG
			}
		},
		{"print", "Print app name and pid to message daemon",
			{
				{RTX_GETOPT_SET, &launch.printAppInfo, "bool"},
				RTX_GETOPT_END_ARG
			}
		},
		{"preclean", "Ask launch agents to check for leftovers from previous runs",
			{
				{RTX_GETOPT_SET, &launch.cleanup, "bool"},
				RTX_GETOPT_END_ARG
			}
		},
		{"level", "Launch level",
			{
				{RTX_GETOPT_INT, &launch.userSpecifiedLevel, "level"},
				RTX_GETOPT_END_ARG
			}
		},
		{"timeout", "Shutdown timeout (in minutes)",
			{
				{RTX_GETOPT_INT, &launch.shutdownTimeout, "timeout"},
				RTX_GETOPT_END_ARG
			}
		},
		{"remoteshell", "Remote shell login program",
			{
				{RTX_GETOPT_CHAR, &launch.remoteShell, "remote shell program"},
				RTX_GETOPT_END_ARG
			}
		},
		{"paramdir", "Exported config file directory", 
			{
				{RTX_GETOPT_STR, &launch.confdir, "directory"},
				RTX_GETOPT_END_ARG
			}
		},
		{"srpcport", "SRPC program number",
			{
				{RTX_GETOPT_INT, &launch.srpcPortNum, "SRPC port number"},
				RTX_GETOPT_END_ARG
			}
		},
		RTX_GETOPT_END
	};
	char * launchHelpStr = NULL;

	int errs;
	FILE * fp = NULL;
	char * p = NULL;
	/* errors and messages should go to both stderr and messaged */

	if (rtx_main_init ("launch", RTX_ERROR_STDERR | RTX_ERROR_MESSAGE) == -1) {
		rtx_message( " rtx_error_init() failed");
		exit (1);
	}

	if (rtx_lock_get ("/tmp/launch.lock") == -1) {
		rtx_error_traceback_depth (0);
		rtx_message( " failed to get lock on /tmp/launch.lock");
		rtx_message( " a launch process is already running");
		exit (1);
	}

	/* initialize and process command-line options */

	launch.myPid = getpid ();
	launch.commandLineReqd = 0;
	launch.srpcPortNum = 100;
	launch.verbose = 0;
	launch.baseprio = 0;
	launch.timerFired = 0;
	launch.done = 0;
	launch.webFlag = 0;
	launch.webPort = LAUNCH_WEBSERVER_PORT;
	launch.printAppInfo = 0;
	launch.retryMode = 0;
	launch.shutdownTimeout = -1;
	launch.configFileName = "./launch.conf";
	launch.remoteShell = "ssh";
	launch.userSpecifiedLevel = -1;
	launch.cleanup = 0;
	launch.confdir = ".";
	rtx_getopt_set_str ("config", launch.configFileName);
	rtx_getopt_set_int ("priority", launch.baseprio);
	rtx_getopt_disable_config ();
	if ((errs = RTX_GETOPT_CMD (launchOpts, argc, argv, rcsid, 
					launchHelpStr)) <= 0) {
		if (errs == -1)
			RTX_GETOPT_PRINT (launchOpts, argv[0], rcsid, launchHelpStr);
		exit (1);
	}    
	errs = 0;
	memcpy(l,&launch,sizeof(Launch));
	l->verbose = rtx_getopt_get_verbose (0);
	l->baseprio = rtx_getopt_get_int ("priority");
	l->configFileName = rtx_getopt_get_str ("config");
	l->defs = rtx_getopt_get_strcat ("D");

	if (l->shutdownTimeout >= 0) {
		l->shutdownTimeout *= 60;
	}

	/* get own hostname/pid */
	if (gethostname (l->myHostName, BUFSIZ) == -1)
		return (rtx_error_errno ("launch_startup: gethostname()"));
	/* strip the domain name from the hostname */
	if ((p = strchr (l->myHostName, '.')) != NULL)
		p[0] = '\0';

	/* read the config file */
	if (launch_read_config_file (l) == -1)
		return (rtx_error ("launch_startup: launch_read_config_file()"));

	/* post-process command-line options */

	if (l->userSpecifiedLevel != -1)
		l->desiredAppLevel = l->userSpecifiedLevel;
	else if (l->commandLineReqd > 0)
		l->desiredAppLevel = 0;
	else
		l->desiredAppLevel = l->maxAppLevel;
	l->userDesLevel = l->desiredAppLevel;

	if ((l->desiredAppLevel < 0) || 
			(l->desiredAppLevel > 255)) {
		rtx_error_flush ("invalid desired run level "
				"= %d\n", l->desiredAppLevel);
		exit (1);
	}

	/* get going with the launcher */

	if (launch_startup (l) == -1) {
		rtx_error_flush ("launch_startup");
		exit (1);
	}

	if ((fp = fopen ("/tmp/launch.pid", "w")) != NULL) {
		fprintf (fp, "%d\n", (int) getpid ());
		fclose (fp);
	}
	return 0;
}


/**
 * Initialize the launcher
 * */
int launch_terminate(Launch * l)
{
	int i,errs = 0;
	RtxTime ts;
	double t0;

	rtx_time_get(&ts);
	t0 = rtx_time_to_double(&ts);
	/* bring down all the apps */
	if (! l->errors) {
		/* change app level to 0 */
		/* wait for app level to become 0 - poll 1Hz */
		l->desiredAppLevel = 0;
		rtx_sem_post (l->eventSem);
		for (i=0; (i<l->shutdownTimeout) && 
				(l->desiredAppLevel != l->curAppLevel); i++) {
			if (l->desiredAppLevel == l->curAppLevel)
				break;
			sleep (1);
			if (((i+1)%5)==0) {
				double t;
				rtx_time_get(&ts);
				t = l->shutdownTimeout - (rtx_time_to_double(&ts) - t0);
				rtx_message( " waiting for apps... (timeout in %.1fs)", t);
			}
		}
	}

	if (launch_shutdown (l)) {
		rtx_error ("launch_shutdown");
		errs++;
	}
	if (rtx_lock_return ("/tmp/launch.lock") == -1) {
		rtx_error ("rtx_lock_return");
		errs++;
	}
	if (unlink ("/tmp/launch.lock") == -1) {
		rtx_error_errno ("unlink");
		errs++;
	}
	if (unlink ("/tmp/launch.pid") == -1) {
		rtx_error_errno ("unlink");
		errs++;
	}
	
	return errs;
}
