/*
 ********************************************************************
 *
 * config.c - read and process launch config file
 *
 *     CSIRO Automation
 *     Queensland Centre for Advanced Technologies
 *     PO Box 883, Kenmore, QLD 4069, Australia
 *     www.cat.csiro.au/cmst
 *
 *      $Id: config.c 1776 2007-05-29 03:12:25Z pra077 $
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 ********************************************************************
 */

#ifdef __GNUC__
#define RTX_STATIC_ATTRIBUTE __attribute__((unused))
#else
#define RTX_STATIC_ATTRIBUTE
#endif
static char *rcsid RTX_STATIC_ATTRIBUTE = "$Id: config.c 1776 2007-05-29 03:12:25Z pra077 $";

/**
 ********************************************************************
 * 
 * \file config.c
 * \brief read launch config file
 * \author Pavan Sikka
 */

/**
 * \page config
 *
 * functions to read and process the config file
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <netdb.h>

#include <rtx/message.h>
#include <rtx/error.h>
#include <rtx/list.h>
#include <rtx/getopt.h>

#include "launch.h"

/**
 * read the configuration file and setup app, level and host lists
 */
int 
launch_read_config_file (
			 Launch * l     /**< the launch handle */
			 )
{
    int i, n;
    int runLevel, timeout, shutdownTimeout=0;
    char * appName;
    LaunchAppType appType;
    LaunchAppNode * nd;
    LaunchHostNode * hostNd;
    FILE * fp;
    int numDefs = 0;
    char b[BUFSIZ];
    struct hostent * h;
    char * p = NULL;

	fp = fopen(l->configFileName,"r");
	if (fp == NULL) {
		return (rtx_error_errno ("launch_read_config_file (%s): "
					"fopen() failed", 
					l->configFileName));
	}
	fclose(fp);
		

    for (i=0; i<256; i++)
        l->appsLevel[i] = NULL;
    if ((l->apps = rtx_list_init ()) == NULL)
	return (rtx_error_errno ("launch_read_config_file (%s): "
				 "rtx_list_init() failed", 
				 l->configFileName));
    if ((l->agents = rtx_list_init ()) == NULL)
	return (rtx_error_errno ("launch_read_config_file (%s): "
				 "rtx_list_init() failed", 
				 l->configFileName));
    numDefs = rtx_getopt_get_num_strings (l->defs);
#ifdef sparc_solaris
    sprintf (b, "/usr/local/bin/cpp -P ");
#else
#ifdef i486_linux
    sprintf (b, "/usr/bin/cpp -P -Ulinux ");
#else
    sprintf (b, "/usr/bin/cpp -P ");
#endif
#endif
    for (i=0; i<numDefs; i++) {
        strcat (b, "-D");
	strcat (b, l->defs[i]);
	strcat (b, " ");
    }
    strcat (b, l->configFileName);
    if ((fp = popen (b, "r")) == NULL)
        return (rtx_error_errno ("launch_read_config_file (%s): "
				 "popen() failed", 
				 l->configFileName));
    l->maxAppLevel = 0;
    while (fgets (l->buf, BUFSIZ, fp) != NULL) {
        /* skip comment lines */
        if ((l->buf[0] == '%') || (l->buf[0] == '#'))
    	    continue;
	/* skip blank lines */
	n = strlen (l->buf);
	for (i=0; i<n; i++)
	    if (! (isspace ((int) l->buf[i])))
	        break;
	if (i >= n)
	    continue;
	/* we have a line with data, get the data */
	if (sscanf (l->buf, "%d %s %s %d %n", &runLevel, l->buf1,
		    l->buf2, &timeout, &i) != 4)
	    return (rtx_error ("launch_read_config_file (%s): "
			       "invalid config line [%s]",
			       l->configFileName, l->buf));
	for (; i<n; i++)
	    if (! (isspace ((int) l->buf[i])))
	        break;
	if (i >= n)
	    return (rtx_error ("launch_read_config_file (%s): "
			       "command not specified in line [%s]",
			       l->configFileName, l->buf));
	strcpy (l->buf3, &(l->buf[i]));
	/* strip the linefeeds from the command */
	n = strlen (l->buf3);
	for (i=0; i<n; i++)
  	    if (l->buf3[i] == '\n')
	        l->buf3[i] = '\0';
	/* sanity checks on the data */
	if ((runLevel < 1) || (runLevel > 255))
	    return (rtx_error ("launch_read_config_file (%s): "
			       "invalid run level %d [%s]", 
			       l->configFileName, runLevel, 
			       l->buf));
	if ((timeout < 0) || (timeout > 60))
	    return (rtx_error ("launch_read_config_file (%s): "
			       "invalid timeout %d [%s]", 
			       l->configFileName, timeout, 
			       l->buf));
	shutdownTimeout += timeout;
	if (l->verbose) {
		rtx_message("shutdownTimeout: %d", shutdownTimeout);
	}
	if (strcmp (l->buf2, "launch-aware") == 0)
  	    appType = LAUNCH_APP_AWARE;
	else if (strcmp (l->buf2, "launch-neutral") == 0)
  	    appType = LAUNCH_APP_NEUTRAL;
	else
	    return (rtx_error ("launch_read_config_file (%s): "
			       "invalid app type (%s) [%s]",
			       l->configFileName, l->buf2, 
			       l->buf));
	/* get hostname part if provided */
	/* use buf1 to store hostname, buf2 to store app name */
	strcpy (l->buf2, l->buf1);
	if ((appName = strchr (l->buf1, ':')) == NULL) {
	    strcpy (l->buf1, l->myHostName);
	} else {
	    /* we have <hostname>:<appname> which we need to split
	     * up.
	     */
	    /* get rid of the trailing :<appname> */
	    * appName = '\0';
	    appName++;
	    /* check that the hostname is a valid host */
            if ((h = gethostbyname (l->buf1)) == NULL)
                return (rtx_error ("gethostbyname(%s) failed", l->buf1));
	    /* strip the domain name from the hostname */
	    if ((p = strchr (l->buf1, '.')) != NULL)
	        p[0] = '\0';
	}
	/* got a valid line */
	/* add the hostname to the list of agents */
	if ((hostNd = rtx_list_lookup (l->agents, l->buf1))
	        == NULL) {
  	    if ((hostNd = (LaunchHostNode *) calloc 
		 (1, sizeof (LaunchHostNode))) == NULL) 
	        return (rtx_error ("launch_read_config_file (%s): "
				   "calloc() failed",
				   l->configFileName));
  	    if ((hostNd->hostName = strdup (l->buf1)) == NULL) 
	        return (rtx_error ("launch_read_config_file (%s): "
				   "strdup() failed",
				   l->configFileName));
	    hostNd->l = l;
	    if (rtx_list_add (l->agents, l->buf1, hostNd) == -1)
	        return (rtx_error ("launch_read_config_file (%s): "
				   "rtx_list_add() failed",
				   l->configFileName));
	}
	/* add the nodes to the list of apps */
	if ((nd = (LaunchAppNode *) calloc 
	     (1, sizeof (LaunchAppNode))) == NULL)
	    return (rtx_error ("launch_read_config_file (%s): "
			       "calloc() failed",
			       l->configFileName));
	if ((nd->hostName = strdup (l->buf1)) == NULL)
	    return (rtx_error ("launch_read_config_file (%s): "
			       "strdup (hostname) failed",
			       l->configFileName));
	if ((nd->appName = strdup (l->buf2)) == NULL)
	    return (rtx_error ("launch_read_config_file (%s): "
			       "strdup (appname) failed",
			       l->configFileName));
	if ((nd->command = strdup (l->buf3)) == NULL)
	    return (rtx_error ("launch_read_config_file (%s): "
			       "strdup (command) failed",
			       l->configFileName));
	if (rtx_mutex_init (&(nd->mutex), RTX_MUTEX_DEFAULT,
			    0) == NULL)
	    return (rtx_error ("launch_read_config_file (%s): "
			       "rtx_mutex_init()  failed",
			       l->configFileName));
	nd->appType = appType;
	nd->timeout = timeout;
	nd->level = runLevel;
	nd->hostNd = hostNd;
	if (runLevel > l->maxAppLevel)
  	    l->maxAppLevel = runLevel;
	if (rtx_list_add (l->apps, l->buf2, nd) == -1)
	    return (rtx_error ("launch_read_config_file (%s): "
			       "rtx_list_add() failed",
			       l->configFileName));

    }
    /* Now build up the appLevel structure */
    while ((nd = rtx_list_iterate (l->apps)) != NULL) {
        runLevel = nd->level;
	if (l->appsLevel[runLevel] == NULL)
  	    if ((l->appsLevel[runLevel] = rtx_list_init ()) == NULL)
	        return (rtx_error ("launch_read_config_file (%s): "
				   "rtx_list_init() failed",
				   l->configFileName));
	if (rtx_list_add (l->appsLevel[runLevel], nd->appName,
			  nd) == -1)
	    return (rtx_error ("launch_read_config_file (%s): "
			       "rtx_list_add() failed",
			       l->configFileName));
    }
	/* make shutdown timeout the sum of all timeouts, + 5 seconds */
	if (l->shutdownTimeout < 0) {
		l->shutdownTimeout = shutdownTimeout + 5;
	}
    pclose (fp);
    return (0);
}

